#!/usr/bin/env python

from PySide.QtCore    import Qt, QByteArray, QIODevice, QModelIndex, QString, \
                             QTextStream, QUrl, QVariant, SIGNAL
from PySide.QtGui     import QAbstractItemView, QDialog, QDialogButtonBox, \
                             QFileDialog, QGridLayout, QGroupBox, \
                             QHBoxLayout, QInputDialog, QLabel, QLineEdit, \
                             QListView, QMenuBar, QMessageBox, \
                             QProgressDialog, QStringListModel, QTableView, \
                             QToolButton, QVBoxLayout, QWidget
from PySide.QtSql     import QSqlQueryModel
from PySide.QtNetwork import QNetworkAccessManager, QNetworkCookieJar, \
                             QNetworkReply, QNetworkRequest
from LTdatabase       import LTDatabase

version = '0.0.1'
exportUrl = QUrl('http://www.librarything.com/export-tab')
loginUrl  = QUrl('http://www.librarything.com/signup.php')

class LTException(Exception): #{{{1
    def __init__(self, source, error):
        self._source = source
        self._error  = error

    def __str__(self):
        return repr('%s: %s' % (self._source, self._error))

class LTColumnSelectGeneric(QDialog): #{{{1
    def __init__(self, parent, database): #{{{2
        QDialog.__init__(self, parent)
        self._database = database
        self.setMinimumSize(800, 480)

    def setupAvailableList(self, model): #{{{2
        self._availableColumns = QListView()
        self._availableColumns.setModel(model)
        self._availableColumns.setUniformItemSizes(True)
        self._availableColumns.setSelectionMode(QAbstractItemView.SingleSelection)
        self._availableColumns.connect(SIGNAL('doubleClicked(QModelIndex)'), self.moveItemClick)
        self._availableColumns.model().sort(0)

        box = QGroupBox('Available fields')
        lyt = QVBoxLayout()
        lyt.addWidget(self._availableColumns)
        box.setLayout(lyt)

        return box

    def setupSelectedList(self, model, name): #{{{2
        self._selectedColumns = QListView()
        self._selectedColumns.setModel(model)
        self._selectedColumns.setUniformItemSizes(True)
        self._selectedColumns.setSelectionMode(QAbstractItemView.SingleSelection)
        self._selectedColumns.connect(SIGNAL('doubleClicked(QModelIndex)'), self.moveItemClick)
        self._selectedColumns.model() # This is needed to force the display

        box = QGroupBox(name)
        lyt = QVBoxLayout()
        lyt.addWidget(self._selectedColumns)
        box.setLayout(lyt)

        return box

    def setupDialog(self, availBox, selBox): #{{{2
        # Set up the up/down arrows for ordering the display columns {{{3
        orderBtnUp = QToolButton()
        orderBtnUp.setArrowType(Qt.UpArrow)
        orderBtnUp.connect(SIGNAL('clicked(bool)'), self.orderItemUp)

        orderBtnDown = QToolButton()
        orderBtnDown.setArrowType(Qt.DownArrow)
        orderBtnDown.connect(SIGNAL('clicked(bool)'), self.orderItemDown)

        orderBtnBox = QVBoxLayout()
        orderBtnBox.addWidget(orderBtnUp)
        orderBtnBox.addWidget(orderBtnDown)

        # Set up the left/right arrows for moving to/from display {{{3
        selectBtnRight = QToolButton()
        selectBtnRight.setArrowType(Qt.RightArrow)
        selectBtnRight.connect(SIGNAL('clicked()'), self.moveItemRight)

        selectBtnLeft = QToolButton()
        selectBtnLeft.setArrowType(Qt.LeftArrow)
        selectBtnLeft.connect(SIGNAL('clicked()'), self.moveItemLeft)

        selectBtnBox = QVBoxLayout()
        selectBtnBox.addWidget(selectBtnRight)
        selectBtnBox.addWidget(selectBtnLeft)

        # Add the Save/Cancel buttons #{{{3
        exitButtons = QDialogButtonBox(QDialogButtonBox.Save|QDialogButtonBox.Cancel,Qt.Horizontal)
        exitButtons.connect(SIGNAL('accepted()'), self.saveSettings)
        exitButtons.connect(SIGNAL('rejected()'), self.reject)

        # Add all the widgets to the window {{{3
        lyt1 = QHBoxLayout()
        lyt2 = QVBoxLayout()
        lyt1.addWidget(availBox)
        lyt1.addLayout(selectBtnBox)
        lyt1.addWidget(selBox)
        lyt1.addLayout(orderBtnBox)
        lyt2.addLayout(lyt1)
        lyt2.addWidget(exitButtons)
        self.setLayout(lyt2)

    def moveItemLeft(self): #{{{2
        item = self._selectedColumns.currentIndex()
        
        if item.row() == -1:
            return

        self.moveItem(item, self._selectedColumns, self._availableColumns)

    def moveItemRight(self): #{{{2
        item = self._availableColumns.currentIndex()

        if item.row() == -1:
            return

        self.moveItem(item, self._availableColumns, self._selectedColumns)

    def moveItemClick(self, item): #{{{2
        srcList = self.sender()
        dstList = None

        if srcList == self._availableColumns:
            dstList = self._selectedColumns
        elif srcList == self._selectedColumns:
            dstList = self._availableColumns
        else:
            print "Unknown caller"
            return

        self.moveItem(item, srcList, dstList)

    def orderItemUp(self, checked): #{{{2
        self.orderItem(-1)

    def orderItemDown(self, checked): #{{{2
        self.orderItem(1)

    def orderItem(self, offset): #{{{2
        selected = self._selectedColumns.currentIndex()
        selRow = selected.row()
        selColName = selected.data()

        if selRow == -1:
            return

        self._selectedColumns.model().removeRow(selRow)
        self._selectedColumns.model().insertRow(selRow + offset)
        newEntry = self._selectedColumns.model().index(selRow + offset)
        self._selectedColumns.model().setData(newEntry, selColName)
        self._selectedColumns.setCurrentIndex(newEntry)

    def moveItem(self, item, srcList, dstList): #{{{2
        colName = item.data()

        srcList.model().removeRow(item.row())

        newRow = dstList.model().rowCount()
        dstList.model().insertRow(newRow, QModelIndex())
        newItem = dstList.model().index(newRow)
        dstList.model().setData(newItem, colName)

        if dstList == self._availableColumns:
            dstList.model().sort(0)
            newItem = dstList.model().match(dstList.model().index(0), Qt.DisplayRole, QVariant(colName))[0]

        dstList.setCurrentIndex(newItem)

class LTColumnSelectDisplay(LTColumnSelectGeneric): #{{{1
    def __init__(self, parent, database): #{{{2
        LTColumnSelectGeneric.__init__(self, parent, database)

        self.setWindowTitle('Select and order columns to display')

        availableModel = QStringListModel(database.getNonDisplayColumns())
        selectedModel  = QStringListModel(database.getDisplayColumns())

        availableList = self.setupAvailableList(availableModel)
        selectedList  = self.setupSelectedList(selectedModel, 'Displayed fields')

        availableModel.deleteLater()
        selectedModel.deleteLater()

        self.setupDialog(availableList, selectedList)

    def saveSettings(self): #{{{2
        self._database.saveDisplayOrder(self._selectedColumns.model().stringList())
        self.accept()

class LTColumnSelectSort(LTColumnSelectGeneric): #{{{1
    def __init__(self, parent, database): #{{{2
        LTColumnSelectGeneric.__init__(self, parent, database)

        self.setWindowTitle('Select and order columns to sort by')

        availableModel = QStringListModel(database.getNonSortColumns())
        selectedModel  = QStringListModel(database.getSortColumns())

        availableList = self.setupAvailableList(availableModel)
        selectedList  = self.setupSelectedList(selectedModel, 'Sort fields')

        availableModel.deleteLater()
        selectedModel.deleteLater()

        self.setupDialog(availableList, selectedList)

    def saveSettings(self): #{{{2
        self._database.saveSortOrder(self._selectedColumns.model().stringList())
        self.accept()

class LTBookWindow(QDialog): #{{{1
    def __init__(self, parent, database, bookID): #{{{2
        QDialog.__init__(self, parent)

        self._bookInfo = database.getBookInfo(bookID)
        self._fieldNames = database.getFieldNames()

        self.setWindowTitle(self._bookInfo[QString('title')])

        lyt = QGridLayout(self)
        fldTitle = QLabel()
        fldTitle.setTextFormat(Qt.RichText)
        fldTitle.setText('<b>' + self.getBookInfo('title') + '</b>')

        fldAuthor = QLabel(self.getBookInfo('author_fl'))
        fldPublication = QLabel(self.getBookInfo('publication'))

        fldRating = self.createTaggedLabel('stars')
        fldTags = self.createTaggedLabel('tags')
        fldISBN = self.createTaggedLabel('isbn')

        lyt.addWidget(fldTitle, 1, 1, 1, 2)
        lyt.addWidget(fldAuthor, 2, 1, 1, 2)
        lyt.addWidget(fldPublication, 3, 1, 1, 2)
        lyt.addWidget(fldRating, 4, 1, 1, 1)
        lyt.addWidget(fldISBN, 4, 2, 1, 1)
        lyt.addWidget(fldTags, 5, 1, 1, 1)

        isbn = self.getBookInfo('isbn')
        if not isbn.isEmpty():
            fldLink = QLabel()
            fldLink.setOpenExternalLinks(True)
            fldLink.setTextFormat(Qt.RichText)
            fldLink.setText('<a href="http://www.librarything.com/isbn/' + self.getBookInfo('isbn') + '">Open in browser</a>')
            lyt.addWidget(fldLink, 6, 1, 1, 2, Qt.AlignHCenter)

    def getFieldName(self, colName): #{{{2
        return self._fieldNames[QString(colName)]

    def getBookInfo(self, colName): #{{{2
        return self._bookInfo[QString(colName)]

    def createTaggedLabel(self, field): #{{{2
        fldEntry = QLabel()
        fldEntry.setTextFormat(Qt.RichText)
        fldEntry.setText('<font color="brown">' + self.getFieldName(field) + ': </font>' + self.getBookInfo(field))

        return fldEntry

class LTAboutWindow(QDialog): #{{{1
    def __init__(self, parent): #{{{2
        QDialog.__init__(self, parent)

        self.setWindowTitle('About')

        aboutText  = '<h2>pybraryThing v' + version + '</h2>'
        aboutText += '<p>A <a href="http://www.librarything.com/">LibraryThing</a> export file viewer.</p>'
        aboutText += '<p>Written by Robin Hill <a href="mailto:maemo@robinhill.me.uk">&lt;maemo@robinhill.me.uk&gt;</a><br/>'
        aboutText += 'Please email or post to <a href="http://talk.maemo.org/">t.m.o</a> for support</p>'

        lyt = QHBoxLayout(self)
        txt = QLabel()
        txt.setOpenExternalLinks(True)
        txt.setTextFormat(Qt.RichText)
        txt.setText(aboutText)
        lyt.addWidget(txt)

class LTMainWindow(QWidget): #{{{1
    def __init__(self): #{{{2
        QWidget.__init__(self, None)

        # Set to the N900 screen size to aid testing
        self.setMinimumSize(800, 480)
        self.setWindowTitle('LibraryThing Viewer')

    def setupScreen(self): #{{{2
        # Create the required objects
        self._tab      = QTableView()
        self._box      = QHBoxLayout()
        self._menu     = QMenuBar()
        self._model    = QSqlQueryModel()
        self._network  = QNetworkAccessManager(self)
        self._progress = QProgressDialog(self)
        self._tab.setModel(self._model)

        try:
            self._database = LTDatabase()
        except Exception, e:
            raise LTException('Error opening/creating database', str(e))

        # Set a cookie jar
        self._network.setCookieJar(QNetworkCookieJar())

        # Handle network responses - this does not work currently
        # Instead connect the signal to each request
#        self._network.connect(SIGNAL('finished(QNetworkReply)'), self.replyFinished)

        # Set up the menu
        m_fields = self._menu.addAction('Display Fields')
        m_order  = self._menu.addAction('Sort Order')
        m_load   = self._menu.addAction('Load Export File')
        m_dl     = self._menu.addAction('Download Export File')
        m_about  = self._menu.addAction('About')

        m_fields.connect(SIGNAL('triggered()'), self.displayFields)
        m_order.connect(SIGNAL('triggered()'), self.sortOrder)
        m_load.connect(SIGNAL('triggered()'), self.loadFile)
        m_dl.connect(SIGNAL('triggered()'), self.downloadExportFile)
        m_about.connect(SIGNAL('triggered()'), self.about)

        # Set up the view pane
        self._tab.setCornerButtonEnabled(False)
        self._tab.setSelectionMode(QAbstractItemView.NoSelection)
        self._tab.verticalHeader().setVisible(False)
        self._tab.verticalHeader().setDefaultSectionSize(70)
        self._tab.horizontalHeader().setMovable(True)

        # Handle moving/resizing of columns
        self._tab.horizontalHeader().connect(SIGNAL('sectionMoved(int, int, int)'), self.changeColumnOrder)
        self._tab.horizontalHeader().connect(SIGNAL('sectionResized(int, int, int)'), self.changeColumnSize)
        self._tab.horizontalHeader().connect(SIGNAL('sectionDoubleClicked(int)'), self._tab.resizeColumnToContents)

        # Handle double-clicking an entry
        self._tab.connect(SIGNAL('doubleClicked(QModelIndex)'), self.viewBookInfo)
                    
        # Set up the main view
        self._box.setMenuBar(self._menu)
        self._box.addWidget(self._tab)
        self.setLayout(self._box)

    def loadData(self): #{{{2
        # Set the SQL query to use
        col_widths = self._database.getQuery(self._model)

        # Hide the ID column
        self._tab.setColumnHidden(0, True)

        # Set the column widths
        view_size = self._tab.viewport().width()

        var_size_count = len(col_widths)
        for i in range(len(col_widths)):
            if col_widths[i] > 0:
                self._tab.setColumnWidth(i + 1, col_widths[i])
                view_size -= col_widths[i]
                var_size_count -= 1

        for i in range(len(col_widths)):
            if col_widths[i] == 0:
                width = 100
                if view_size > 0:
                    if var_size_count > 1:
                        width = int(view_size / var_size_count)
                        view_size -= width
                        var_size_count -= 1
                    else:
                        width = view_size

                self._tab.setColumnWidth(i + 1, width)
                self.changeColumnSize(i + 1, 0, width)

    def displayFields(self): #{{{2
        winFields = LTColumnSelectDisplay(self, self._database)
        if winFields.exec_():
            self.loadData()

    def sortOrder(self): #{{{2
        winFields = LTColumnSelectSort(self, self._database)
        if winFields.exec_():
            self.loadData()

    def loadFile(self): #{{{2
        filename = QFileDialog.getOpenFileName(self, 'Open LT Export')

        if filename != None:
            self.setupProgressBar('LoadingData')
            self._database.loadFile(filename, self._progress)
            self._model.clear()
            self.loadData()

    def loginLT(self): #{{{2
        # TODO: Move the username/password to a config setting
        # Ask for user name & password
        userName = QInputDialog.getText(self, 'LibraryThing login', 'User name:', QLineEdit.Normal)[0]

        if userName.isEmpty():
            return

        password = QInputDialog.getText(self, 'LibraryThing login', 'Password:', QLineEdit.Password)[0]

        if password.isEmpty():
            return

        # Login
        post_data =  'formusername=%s&formpassword=%s&Submit=Sign+in' % (userName, password)
        reply = self._network.post(QNetworkRequest(loginUrl), QByteArray(post_data))
        reply.connect(SIGNAL('finished()'), self.replyFinished)

        self._loginBox = QDialog(self)
        self._loginBox.setWindowTitle('Logging in')
        lyt = QHBoxLayout(self._loginBox)
        lyt.addWidget(QLabel('Authenticating against the LibraryThing server'))
        self._loginBox.exec_()

    def downloadExportFile(self): #{{{2
        self.disconnect(SIGNAL('downloadComplete()'), self.downloadExportFile)

        if self.haveLTCookie():
            self.setupProgressBar('Downloading file')
            reply = self._network.get(QNetworkRequest(exportUrl))
            reply.connect(SIGNAL('downloadProgress(qint64, qint64)'), self.downloadProgress)

            reply.connect(SIGNAL('finished()'), self.replyFinished)
        else:
            self.connect(SIGNAL('downloadComplete()'), self.downloadExportFile)
            self.loginLT()

    def haveLTCookie(self): #{{{2
        for cookie in self._network.cookieJar().cookiesForUrl(exportUrl):
            if QString(cookie.name()) == QString('cookie_userid'):
                return True

        return False

    def setupProgressBar(self, title): #{{{2
        self._progress.setWindowTitle(title)
        self._progress.setMaximum(100)
        self._progress.setValue(1)

    def downloadProgress(self, done, total): #{{{2
        self._progress.setMaximum(total)
        self._progress.setValue(done)

    def about(self): #{{{2
        aboutWin = LTAboutWindow(self)
        aboutWin.exec_()

    def changeColumnOrder(self, id, oldPos, newPos): #{{{2
        col = self._model.record().fieldName(id)
        self._database.changeColumnOrder(col, oldPos, newPos)

    def changeColumnSize(self, id, oldSize, newSize): #{{{2
        col = self._model.record().fieldName(id)
        self._database.changeColumnSize(col, newSize)

    def viewBookInfo(self, item): #{{{2
        bookID = self._model.record(item.row()).value(0).toInt()[0]

        LTBookWindow(self, self._database, bookID).exec_()

    def replyFinished(self, reply = None): #{{{2
        if reply == None:
            reply = self.sender()

        if reply.error() == QNetworkReply.NoError:
            redir = reply.attribute(QNetworkRequest.RedirectionTargetAttribute).toUrl()
            if not redir.isEmpty():
                url = reply.request().url().resolved(redir)
                reply.deleteLater()

                req = QNetworkRequest(url)
                rep = self._network.get(req)
                rep.connect(SIGNAL('finished()'), self.replyFinished)
            elif reply.url() == exportUrl:
                export = reply.readAll()
                reply.deleteLater()
                self.emit(SIGNAL('downloadComplete()'))

                stream = QTextStream(export, QIODevice.ReadOnly)

                self.setupProgressBar('Loading data')
                self._database.loadStream(stream, self._progress)
                self._model.clear()
                self.loadData()
            else:
                self._loginBox.done(0)
                res = reply.readAll()
                reply.deleteLater()

                if res.indexOf('Wrong user name or password') != -1:
                    self.disconnect(SIGNAL('downloadComplete()'), self.downloadExportFile)
                    QMessageBox.critical(self, 'Login failed', 'Username or password incorrect')

                self.emit(SIGNAL('downloadComplete()'))
        else:
            QMessageBox.critical(self, 'Network error', reply.errorString())
            reply.deleteLater()
            self.emit(SIGNAL('downloadComplete()'))

