/*
 * This file is part of pwsafe
 *
 * Copyright (C) 2005 Nokia Corporation.
 * Copyright (C) 2005 HolisTech Limited.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <src/interface.h>
#include <src/callbacks.h>
#include <src/preferences.h>
#include <gtk/gtk.h>
#include <libintl.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#if HILDON == 1
#include <hildon/hildon.h>
#include <hildon/hildon-file-chooser-dialog.h>
#else
#include <hildon-widgets/hildon-app.h>
/*#include <hildon-fm/hildon-widgets/hildon-file-chooser-dialog.h>*/
#include <hildon-fm-2/hildon/hildon-file-chooser-dialog.h> 
#include <hildon-widgets/hildon-note.h>
#include <hildon-widgets/hildon-font-selection-dialog.h>
#include <hildon-widgets/hildon-get-password-dialog.h>
#endif

#include <mce/dbus-names.h>

#define MAIN_TEXT MAIN_VIEW_NAME

/* Privates: */
static void create_menu( MainView *main );
static void create_toolbar( MainView *main );
void create_textarea( MainView *main );
GtkWidget *show_rec_line(GtkWidget *sizer, char *label, char *txt, int row, int col);

char *emptynull(char *a) { return (a==NULL) ? "" : a;}

/* Creates and initialises a main_view */
MainView* interface_main_view_new( AppData *data ) {
	MainView* result = g_new0( MainView, 1 );
/* Create GTK vbox which will be our main area */
	GtkWidget *scrolled_window;
/*
	GtkWidget *main_vbox = gtk_vbox_new( FALSE, 0 );
*/
#if MAEMO_VERSION_MAJOR >= 5
	scrolled_window = hildon_pannable_area_new();
#else
	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
#endif
	result->clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
/* no record window currently showing */
	result->rec_win=NULL;
	result->file_edited = FALSE;
	result->file_name=NULL;
/* pasword unset */
	result->pass=NULL;
/* Store handle to app's data to view's data and hildon appview */
	result->data = data;
	data->main_view = HILDON_WINDOW(hildon_window_new()); 
	/* Create all our necessary parts */
	create_textarea( result );
	create_toolbar( result ); 
	create_menu( result );
	/* Put mainwindow to GTK vbox and show it */
	gtk_container_add( GTK_CONTAINER( result->data->main_view ), scrolled_window );
	gtk_container_add( GTK_CONTAINER( scrolled_window ), result->treeView);
	gtk_widget_show( scrolled_window );
	gtk_widget_grab_focus( GTK_WIDGET(result->treeView ));
	/* attach handlers */
	osso_hw_set_event_cb( data->osso, NULL,(osso_hw_cb_f *) hw_event_handler, result);
	g_signal_connect (G_OBJECT(data->main_view), "delete_event", G_CALLBACK(exit_event_handler), result);	/* top right X, etc. */
	g_signal_connect(G_OBJECT(result->data->main_view), "key_press_event", G_CALLBACK(callback_key_press), result);

	/* Get the system DBus connection */
	DBusConnection *con = osso_get_sys_dbus_connection(data->osso);
	/* Add the callback, which should be called, once the device is rotated */
	dbus_bus_add_match(con, MCE_MATCH_RULE, NULL);
	dbus_connection_add_filter(con, (DBusHandleMessageFunction *) dbus_handle_mce_message, result, NULL);

#if MAEMO_VERSION_MAJOR >= 5
	result->regexHttp = g_regex_new("http(s?)://[\\S]+", G_REGEX_CASELESS, 0, NULL);
	result->regexWww = g_regex_new("www\\.[\\S]+", G_REGEX_CASELESS, 0, NULL);
#endif
	return result; /* Return new mainview handle */
}


/* clean up the allocated memory */
void interface_main_view_destroy( MainView *main ) {
#if MAEMO_VERSION_MAJOR >= 5
	g_regex_unref(main->regexHttp);
	g_regex_unref(main->regexWww);
#endif
	g_free( main );
}

GtkWidget *show_rec_line(GtkWidget *sizer, char *label, char *txt, int row, int col) {
	GtkWidget *textentry;
#if MAEMO_VERSION_MAJOR < 5
	gtk_table_attach_defaults(GTK_TABLE(sizer), gtk_label_new(label), col, col+1, row, row+1);
	textentry=gtk_entry_new();
	gtk_table_attach_defaults(GTK_TABLE(sizer), textentry, col+1,col+2, row, row+1);
#else
	textentry=hildon_entry_new(HILDON_SIZE_AUTO);
	hildon_entry_set_placeholder(HILDON_ENTRY(textentry), label);
	gtk_table_attach_defaults(GTK_TABLE(sizer), textentry, col,col+2, row, row+1);
#endif
	if (txt!=NULL) gtk_entry_set_text(GTK_ENTRY(textentry), txt);
	return textentry;
}

void unshow_rec(MainView *main) {
	if (main->rec_win!=NULL) {
		gtk_widget_destroy(GTK_WIDGET(main->rec_win->dialog));
		g_free(main->rec_win);
		main->rec_win=NULL;
	}
}

gchar *get_password(MainView *main, gchar *caption, gint pwdDlgType) {
	PasswordUI *pwUi;
	gchar *result=NULL;
#if MAEMO_VERSION_MAJOR >= 5
	GtkWidget *image;
#endif

	pwUi = g_new0(PasswordUI, 1);
	pwUi->type = pwdDlgType;
	pwUi->main = main;
	pwUi->result = NULL;
	
	pwUi->dialog = gtk_dialog_new_with_buttons(caption, GTK_WINDOW(main->data->main_view), GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR, NULL);
#if MAEMO_VERSION_MAJOR < 5
	pwUi->button = gtk_button_new_from_stock(GTK_STOCK_OK);
	pwUi->entry1=gtk_entry_new();
#else
	pwUi->button =  hildon_button_new_with_text(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL, _("OK"), NULL);
	image = gtk_image_new_from_stock (GTK_STOCK_OK, GTK_ICON_SIZE_BUTTON);
	hildon_button_set_image (HILDON_BUTTON (pwUi->button), image);
	pwUi->entry1=hildon_entry_new(HILDON_SIZE_AUTO);
#endif
	gtk_container_add (GTK_CONTAINER (gtk_dialog_get_action_area(GTK_DIALOG(pwUi->dialog))), pwUi->button);
	gtk_widget_set_sensitive(pwUi->button, FALSE);
	g_signal_connect (pwUi->button, "clicked", G_CALLBACK (callback_pwdialog_button), pwUi);

	gtk_entry_set_visibility(GTK_ENTRY(pwUi->entry1), FALSE);
	gtk_container_add (GTK_CONTAINER (gtk_dialog_get_content_area(GTK_DIALOG(pwUi->dialog))), pwUi->entry1);
	g_signal_connect(G_OBJECT(pwUi->entry1), "changed", G_CALLBACK(callback_pwdialog_changed), pwUi);
	g_signal_connect(G_OBJECT(pwUi->entry1), "activate", G_CALLBACK(callback_pwdialog_activate), pwUi);
	if (pwdDlgType != PWD_DLG_GET) {
#if MAEMO_VERSION_MAJOR < 5
		pwUi->entry2=gtk_entry_new();
#else
		pwUi->entry2=hildon_entry_new(HILDON_SIZE_AUTO);
#endif
		gtk_entry_set_visibility(GTK_ENTRY(pwUi->entry2), FALSE);
		gtk_container_add (GTK_CONTAINER (gtk_dialog_get_content_area(GTK_DIALOG(pwUi->dialog))), pwUi->entry2);
		g_signal_connect(G_OBJECT(pwUi->entry2), "changed", G_CALLBACK(callback_pwdialog_changed), pwUi);
		g_signal_connect(G_OBJECT(pwUi->entry2), "activate", G_CALLBACK(callback_pwdialog_activate), pwUi);
		if (pwdDlgType != PWD_DLG_NEW) {
#if MAEMO_VERSION_MAJOR < 5
			pwUi->entry3=gtk_entry_new();
#else
			pwUi->entry3=hildon_entry_new(HILDON_SIZE_AUTO);
#endif
			gtk_entry_set_visibility(GTK_ENTRY(pwUi->entry3), FALSE);
#if MAEMO_VERSION_MAJOR < 5
#else
			hildon_entry_set_placeholder(HILDON_ENTRY(pwUi->entry1), _("Old password"));
			hildon_entry_set_placeholder(HILDON_ENTRY(pwUi->entry2), _("New password"));
			hildon_entry_set_placeholder(HILDON_ENTRY(pwUi->entry3), "Confirm new password");
#endif
			gtk_container_add (GTK_CONTAINER (gtk_dialog_get_content_area(GTK_DIALOG(pwUi->dialog))), pwUi->entry3);
			g_signal_connect(G_OBJECT(pwUi->entry3), "changed", G_CALLBACK(callback_pwdialog_changed), pwUi);
			g_signal_connect(G_OBJECT(pwUi->entry3), "activate", G_CALLBACK(callback_pwdialog_activate), pwUi);
		} else {
#if MAEMO_VERSION_MAJOR < 5
#else
			hildon_entry_set_placeholder(HILDON_ENTRY(pwUi->entry1), _("Password"));
			hildon_entry_set_placeholder(HILDON_ENTRY(pwUi->entry2), _("Confirm password"));
#endif
		}
	} else {
#if MAEMO_VERSION_MAJOR < 5
#else
		hildon_entry_set_placeholder(HILDON_ENTRY(pwUi->entry1), _("Password"));
#endif
	}
	gtk_widget_show_all(GTK_WIDGET(pwUi->dialog));
	gtk_dialog_run(GTK_DIALOG(pwUi->dialog));
	
	// Getting out of the password Dialog
	if (pwUi->result != NULL) {
		result = g_strdup(pwUi->result);
		g_free(pwUi->result);
	}
	g_free(pwUi);
	gtk_widget_destroy (GTK_WIDGET (pwUi->dialog));
	return result;
}

gboolean fillCombo (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, GtkWidget *combo) {
	rec *data;
	gchar *label;
	
	gtk_tree_model_get(model, iter, LABEL_COLUMN, &label, DATA_COLUMN, &data,  -1);
	if (data == NULL && label != NULL) {
#if MAEMO_VERSION_MAJOR < 5
		gtk_combo_box_append_text(GTK_COMBO_BOX(combo), label);
#else
		hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(combo), label);
#endif
	}
	return FALSE;
}

void show_rec(MainView *main, rec *r) {
	GtkWidget *sizer, *scrollwindow;
#if MAEMO_VERSION_MAJOR >= 5
	GtkWidget *selector;
#endif
	GtkTextBuffer *buffer=NULL;
	GtkTextTag *tag;
	RecWindow *rec_win;
	if (main->rec_win!=NULL) return; /* don't show two rec windows at once */
	rec_win=g_malloc0(sizeof(RecWindow));
	main->rec_win=rec_win;
	rec_win->rec=r;
	rec_win->dialog = gtk_dialog_new_with_buttons (_("Edit/View Entry"), GTK_WINDOW(main->data->main_view), GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR, GTK_STOCK_OK, 1, _("Copy User"), 2, _("Copy Pwd"), 3, NULL);
	rec_win->delete_button = gtk_dialog_add_button(GTK_DIALOG(rec_win->dialog), GTK_STOCK_DELETE, 4);
	rec_win->edit_button = gtk_dialog_add_button(GTK_DIALOG(rec_win->dialog), _("Edit"), 5);
/*	gtk_button_set_image(GTK_BUTTON(rec_win->edit_button), GTK_STOCK_EDIT); */
#if MAEMO_VERSION_MAJOR < 5
	g_object_set(rec_win->dialog, "content-area-border", 1, "button_spacing", 20, NULL);
#endif
    /*  where's "button-spacing", "content-area-border", "action_area_border" ? - not in scratchbox, maybe on n800 */
	g_signal_connect(rec_win->dialog, "response", G_CALLBACK ( callback_rec_response), main);
	sizer=gtk_table_new(5, 4, TRUE);
	rec_win->title=show_rec_line(sizer, _("Title"), r->title, 0, 0);
//	rec_win->group=show_rec_line(sizer, "Group", r->group, 0, 2);
	// Change group entry to ComboBox with Entry
#if MAEMO_VERSION_MAJOR < 5
	gtk_table_attach_defaults(GTK_TABLE(sizer), gtk_label_new(_("Group")), 2, 3, 0, 1);
	rec_win->group=gtk_combo_box_entry_new_text();
	if (r->group!=NULL) {
		gtk_entry_set_text(GTK_ENTRY(gtk_bin_get_child(GTK_BIN(rec_win->group))), r->group);
	}
	gtk_table_attach_defaults(GTK_TABLE(sizer), rec_win->group, 3, 4, 0, 1);
	// Fill the combo
	gtk_tree_model_foreach(GTK_TREE_MODEL(main->store), (GtkTreeModelForeachFunc) fillCombo, (gpointer) rec_win->group);
#else
	rec_win->group=hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_title (HILDON_BUTTON(rec_win->group), _("Group"));
	
	selector=hildon_touch_selector_entry_new_text();
	hildon_entry_set_placeholder(hildon_touch_selector_entry_get_entry(HILDON_TOUCH_SELECTOR_ENTRY(selector)), _("Group"));
	if (r->group!=NULL) {
		gtk_entry_set_text(GTK_ENTRY(hildon_touch_selector_entry_get_entry(HILDON_TOUCH_SELECTOR_ENTRY(selector))), r->group);
	}
	hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (rec_win->group), HILDON_TOUCH_SELECTOR (selector));

	gtk_table_attach_defaults(GTK_TABLE(sizer), rec_win->group, 2, 4, 0, 1);
	// Fill the combo
	gtk_tree_model_foreach(GTK_TREE_MODEL(main->store), (GtkTreeModelForeachFunc) fillCombo, (gpointer) selector);
#endif
	
	rec_win->username=show_rec_line(sizer, _("Username"), r->user, 1, 0);
	rec_win->password=show_rec_line(sizer, _("Password"), r->password, 1, 2);

#if MAEMO_VERSION_MAJOR < 5
	rec_win->notes=gtk_text_view_new();
	buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW(rec_win->notes));
	scrollwindow=gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrollwindow), GTK_SHADOW_IN);
#else
	rec_win->notes= hildon_text_view_new();
	hildon_text_view_set_placeholder(HILDON_TEXT_VIEW(rec_win->notes), _("Notes"));
	buffer=hildon_text_view_get_buffer(HILDON_TEXT_VIEW(rec_win->notes));
	hildon_gtk_text_view_set_input_mode(GTK_TEXT_VIEW(rec_win->notes), HILDON_GTK_INPUT_MODE_FULL | HILDON_GTK_INPUT_MODE_MULTILINE);
	scrollwindow=hildon_pannable_area_new();
#endif
	gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(rec_win->notes), GTK_WRAP_WORD_CHAR);
	tag = gtk_text_buffer_create_tag(buffer, "url", "underline", PANGO_UNDERLINE_SINGLE, NULL);
	g_signal_connect( G_OBJECT(buffer), "changed", G_CALLBACK(callback_buffer_changed), main);
	g_signal_connect( G_OBJECT(tag), "event", G_CALLBACK(callback_tag_event), main);
	
	if (r->notes!=NULL) gtk_text_buffer_set_text (buffer, r->notes, -1);
	gtk_text_buffer_set_modified (buffer, FALSE);
	gtk_container_add (GTK_CONTAINER (scrollwindow), rec_win->notes);

	gtk_table_attach(GTK_TABLE(sizer), scrollwindow, 0, 4, 2, 5, GTK_FILL, GTK_FILL, 0, 0);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(rec_win->dialog)->vbox), sizer);
	gtk_widget_show_all (rec_win->dialog);
	interface_set_rec_editable(rec_win, FALSE);
}

void empty_treeView(MainView *main) {
	 gtk_tree_store_clear(main->store);
}


gboolean addEntryToGroup(GtkTreeStore *store, GtkTreeIter *groupIter, rec *data) {
	GtkTreeIter iter;
	
	gtk_tree_store_append(store, &iter, groupIter);
	gtk_tree_store_set(store, &iter, LABEL_COLUMN, data->title, ICON_COLUMN, GTK_STOCK_FILE, DATA_COLUMN, data,  -1);
	return TRUE;
}

gboolean searchGroup (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, SearchGroup *sg) {
	rec *data;
	gchar *label;
	
	gtk_tree_model_get(model, iter, LABEL_COLUMN, &label, DATA_COLUMN, &data,  -1);
	if (data == NULL && label != NULL && g_ascii_strcasecmp( emptynull(sg->data->group), label) == 0) {
		sg->found = addEntryToGroup(GTK_TREE_STORE(model), iter, sg->data);
		
		return TRUE;
	}
	return FALSE;
}

gboolean addEntryToTree(MainView *main, rec *data) {
	SearchGroup *sg;
	GtkTreeIter groupIter;
	
	sg = g_new0(SearchGroup, 1);
	sg->data = data;
	gtk_tree_model_foreach(GTK_TREE_MODEL(main->store), (GtkTreeModelForeachFunc) searchGroup, (gpointer) sg);
	if (sg->found == TRUE) {
		g_free(sg);
		return TRUE;
	}
	g_free(sg);
	
	//Create the group 
	gtk_tree_store_append(main->store, &groupIter, NULL);
	gtk_tree_store_set(main->store, &groupIter, LABEL_COLUMN, emptynull(data->group), ICON_COLUMN, GTK_STOCK_DIRECTORY, DATA_COLUMN, NULL, -1);
	
	//Add the Entry
	addEntryToGroup(main->store, &groupIter, data);
	
	return FALSE;
}

void populate_treeView (MainView *main) {
	GList *onrec=NULL;
	
	g_assert(main != NULL && main->data != NULL );
	empty_treeView(main);

	onrec=g_list_first(recs);
	while (onrec!=NULL) {
	    rec *data=(rec *) onrec->data;
	    if (data->skip == 0) {
		addEntryToTree(main, data);
      	    }
	    onrec=g_list_next(onrec);
	}
//	interface_show_state(main);
}


/* File chooser */
gchar* interface_file_chooser(MainView * mainview, GtkFileChooserAction action) {
	GtkWidget *dialog;
	gchar* filename = NULL;
	const gchar *docsDir = NULL;
	gchar *myDir = NULL;
	
	docsDir = getenv("MYDOCSDIR");
	if (docsDir != NULL) {
		myDir = g_strconcat(docsDir, "/.documents", NULL);
	} else {
		myDir = g_strdup("/home/user/MyDocs/.documents");
	}
	dialog = hildon_file_chooser_dialog_new( GTK_WINDOW(mainview->data->main_view), action);
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), myDir);
	gtk_widget_show_all (GTK_WIDGET(dialog));
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
	}
	gtk_widget_destroy (dialog);
	g_free(myDir);
	return filename;
}

/* yes no note */
gint interface_yes_no_note(MainView * main, gchar *title) {
	HildonNote *hn = NULL;
	gint response = FALSE;

	g_assert(main != NULL && main->data->main_view != NULL);

	hn = HILDON_NOTE(hildon_note_new_confirmation_add_buttons
                     (GTK_WINDOW(main->data->main_view), title, _("Yes"), CONFRESP_YES, _("No"), CONFRESP_NO, NULL, NULL));
	response = gtk_dialog_run(GTK_DIALOG(hn));
	gtk_widget_destroy(GTK_WIDGET(hn));
	return response;
}

/* message Box */
void msgbox(MainView* main, char *msg) {
	GtkWidget *dialog = NULL;
	dialog = hildon_note_new_information( GTK_WINDOW( main->data->main_view ), msg );
	gtk_dialog_run( GTK_DIALOG( dialog ) );
	gtk_widget_destroy( dialog );
}


/* Error note */
void interface_error( PwsafeError me, MainView *main )
{
    GtkWidget *dialog = NULL;
    gchar *msg = NULL;

    switch (me) {

    case PWSAFE_ERROR_OPEN_FAILED:
        msg = g_strdup( _("Open failed!") );
        break;
    case PWSAFE_ERROR_SAVE_FAILED:
        msg = g_strdup( _("Save failed!") );
        break;
    case PWSAFE_ERROR_BADPASS:
        msg = g_strdup( _("Wrong Password!") );
        break;
    case PWSAFE_NO_ERROR:
        return;
    default:
        g_assert_not_reached();
    }
    dialog = hildon_note_new_information( GTK_WINDOW( main->data->main_view ), msg );
    gtk_dialog_run( GTK_DIALOG( dialog ) );
    gtk_widget_destroy( dialog );
    g_free( msg );
}

#if MAEMO_VERSION_MAJOR >= 5
static GtkWidget* add_menu_item (GtkWidget *main_menu, const gchar* title,
		    GCallback clicked_callback, gpointer user_data) {
    HildonAppMenu *app_menu = HILDON_APP_MENU (main_menu);
    /* Create a button, add it, and return it: */ 
    GtkWidget *button = hildon_button_new_with_text (HILDON_SIZE_AUTO, 
	            HILDON_BUTTON_ARRANGEMENT_VERTICAL, title, NULL);
    gtk_widget_show (button);
    g_signal_connect_after (button, "clicked", G_CALLBACK (clicked_callback), user_data);
    hildon_app_menu_append (app_menu, GTK_BUTTON (button));
    return button;
}
#endif

/* Create the menu items needed for the drop down menu */
static void create_menu( MainView *main ) {
	/* Create needed handles */
	GtkWidget *main_menu;

#if MAEMO_VERSION_MAJOR < 5
	GtkWidget *file_menu, *view_menu;
	GtkWidget *separator = NULL;
	GtkWidget *quit = NULL;

	/* Get the menu of our view */
	main_menu = gtk_menu_new ();
	/* Add menu to HildonWindow */
	hildon_window_set_menu(main->data->main_view, GTK_MENU(main_menu)); 
	/* Create new menus for submenus in our drop down menu */
	file_menu = gtk_menu_new ();
	view_menu = gtk_menu_new ();

	/* Create the menu items */
	main->file_item = gtk_menu_item_new_with_label ( _("File") );
	main->new_item = gtk_menu_item_new_with_label ( _("New") );
	main->open_item = gtk_menu_item_new_with_label ( _("Open...") );
	main->open_last_item = gtk_menu_item_new_with_label ( _("Open Last") );
	main->close_item = gtk_menu_item_new_with_label ( _("Close") );
	main->save_item = gtk_menu_item_new_with_label ( _("Save") );
	main->saveas_item = gtk_menu_item_new_with_label ( _("Save As...") );
	main->export_item = gtk_menu_item_new_with_label ( _("Export...") );
	
	main->fullscreen_item = gtk_menu_item_new_with_label ( _("Full Screen") );
	main->add_item = gtk_menu_item_new_with_label ( _("Add Entry") );
	main->change_password = gtk_menu_item_new_with_label ( _("Change Password") );
	separator = gtk_separator_menu_item_new();
	quit = gtk_menu_item_new_with_label( _("Quit") );

	/* Add menu items to right menus */
	gtk_menu_append( GTK_MENU(main_menu), main->file_item );
	gtk_menu_append( GTK_MENU(file_menu), main->new_item );
	gtk_menu_append( GTK_MENU(file_menu), main->open_item );
	gtk_menu_append( GTK_MENU(file_menu), main->open_last_item );
	gtk_menu_append( GTK_MENU(file_menu), main->close_item );
	gtk_menu_append( GTK_MENU(file_menu), main->save_item );
	gtk_menu_append( GTK_MENU(file_menu), main->saveas_item );
	gtk_menu_append( GTK_MENU(file_menu), main->export_item );
 
	gtk_menu_append( GTK_MENU(main_menu), main->add_item );
	gtk_menu_append( GTK_MENU(main_menu), main->change_password );
	gtk_menu_append( GTK_MENU(main_menu), main->fullscreen_item );
	gtk_menu_append( GTK_MENU(main_menu), separator );
	gtk_menu_append( GTK_MENU(main_menu), quit );

	/* Add submenus to the right items */
	gtk_menu_item_set_submenu( GTK_MENU_ITEM (main->file_item), file_menu );
	gtk_menu_item_set_submenu( GTK_MENU_ITEM (main->view_item), view_menu );

	/* Attach the callback functions to the activate signal */
	g_signal_connect( G_OBJECT( main->new_item ), "activate",  G_CALLBACK ( callback_file_new), main );
	g_signal_connect( G_OBJECT( main->open_item ), "activate",  G_CALLBACK ( callback_file_open), main );
	g_signal_connect( G_OBJECT( main->open_last_item ), "activate",  G_CALLBACK ( callback_file_open_last), main );
	g_signal_connect( G_OBJECT( main->close_item ), "activate",  G_CALLBACK ( callback_file_close), main );
	g_signal_connect( G_OBJECT( main->save_item ), "activate",  G_CALLBACK ( callback_file_save), main );
	g_signal_connect( G_OBJECT( main->saveas_item ), "activate",  G_CALLBACK ( callback_file_saveas), main );
	g_signal_connect( G_OBJECT( main->export_item ), "activate",  G_CALLBACK ( callback_export), main );
	g_signal_connect( G_OBJECT( main->add_item ), "activate",  G_CALLBACK ( callback_add_entry), main );
	g_signal_connect( G_OBJECT( main->change_password ), "activate",  G_CALLBACK ( callback_change_password), main );
	g_signal_connect( G_OBJECT( main->fullscreen_item ), "activate",  G_CALLBACK ( callback_fullscreen ), main );
	g_signal_connect( G_OBJECT( quit ), "activate", gtk_main_quit, NULL );
#else
/* Maemo 5+ */
	/* Get the menu of our view */
	main_menu = hildon_app_menu_new ();

	/* Create the menu items */
	main->new_item = add_menu_item (main_menu, _("New"), G_CALLBACK ( callback_file_new), main );
	main->open_item = add_menu_item (main_menu, _("Open..."), G_CALLBACK ( callback_file_open), main );
	main->open_last_item = add_menu_item (main_menu, _("Open Last"), G_CALLBACK ( callback_file_open_last), main );
	main->close_item = add_menu_item (main_menu, _("Close"), G_CALLBACK ( callback_file_close), main );
	main->save_item = add_menu_item (main_menu, _("Save"), G_CALLBACK ( callback_file_save), main );
	main->saveas_item = add_menu_item (main_menu, _("Save As..."), G_CALLBACK ( callback_file_saveas), main );
	main->export_item = add_menu_item (main_menu, _("Export..."), G_CALLBACK ( callback_export), main );
	
	main->change_password = add_menu_item (main_menu, _("Change Password"), G_CALLBACK ( callback_change_password), main );

	/* Add menu to HildonWindow */
	hildon_window_set_app_menu (HILDON_WINDOW (main->data->main_view), HILDON_APP_MENU (main_menu));
#endif

	/* We need to show menu items */
	gtk_widget_show_all( GTK_WIDGET( main_menu ) );
}

void interface_show_item(GtkWidget *menuWidget, gboolean show) {
	if (GTK_IS_WIDGET(menuWidget)) { 
#if MAEMO_VERSION_MAJOR < 5
		gtk_widget_set_sensitive(menuWidget, show);
#else
		if (show) {
			gtk_widget_show(menuWidget);
		} else {
			gtk_widget_hide(menuWidget);
		}
#endif
	}
}

void interface_show_state(MainView *main) {
	gboolean fopen;
	fopen=(main->pass!=NULL);

    	interface_show_item(main->new_item, !fopen);
	interface_show_item(main->open_item, !fopen);
    	interface_show_item(main->open_last_item, main->last_file && (!fopen));
    	interface_show_item(main->close_item, fopen);
    	interface_show_item(main->save_item, fopen);
    	interface_show_item(main->saveas_item, fopen);
        interface_show_item(main->export_item, fopen);
    	interface_show_item(main->change_password, fopen);
	interface_show_item(main->add_item, fopen);
	
	if (GTK_IS_WIDGET(main->add_tb)) {	/* ignore toolbar if it's not there (eg. during app close) */
#if MAEMO_VERSION_MAJOR < 5
		gtk_widget_set_sensitive(GTK_WIDGET(main->new_tb), !fopen);
		gtk_widget_set_sensitive(GTK_WIDGET(main->open_tb), !fopen);
		gtk_widget_set_sensitive(GTK_WIDGET(main->close_tb), fopen);
		gtk_widget_set_sensitive(GTK_WIDGET(main->save_tb), fopen);
		gtk_widget_set_sensitive(GTK_WIDGET(main->saveas_tb), fopen);
#endif
		gtk_widget_set_sensitive(GTK_WIDGET(main->change_password_tb), fopen);
		gtk_widget_set_sensitive(GTK_WIDGET(main->add_tb), fopen);
		gtk_editable_set_editable(GTK_EDITABLE(main->filter_tb), fopen);
		if (!fopen) {
			gtk_entry_set_text(GTK_ENTRY(main->filter_tb), "");
		}
	}
}

void interface_set_rec_editable(RecWindow* rec_win, gboolean editable) {
	rec_win->editable=editable;
	if (editable) {
		gtk_widget_show(GTK_WIDGET(rec_win->delete_button));
		gtk_widget_hide(GTK_WIDGET(rec_win->edit_button));
	} else {
		gtk_widget_hide(GTK_WIDGET(rec_win->delete_button));
		gtk_widget_show(GTK_WIDGET(rec_win->edit_button));
	}
	gtk_editable_set_editable(GTK_EDITABLE(rec_win->title), editable);
#if MAEMO_VERSION_MAJOR < 5
	// Group is a comboBoxEdit
	gtk_editable_set_editable(GTK_EDITABLE(gtk_bin_get_child(GTK_BIN(rec_win->group))), editable);
#else
	gtk_widget_set_sensitive(GTK_WIDGET(rec_win->group), editable);
#endif
	gtk_editable_set_editable(GTK_EDITABLE(rec_win->username), editable);
	gtk_editable_set_editable(GTK_EDITABLE(rec_win->password), editable);
	gtk_text_view_set_editable(GTK_TEXT_VIEW(rec_win->notes), editable);
}

/* Create toolbar to mainview */
static void create_toolbar ( MainView *main ) {
	/* Create new GTK toolbar */
	main->toolbar = gtk_toolbar_new ();

 	/* Set toolbar properties */
	gtk_toolbar_set_orientation( GTK_TOOLBAR(main->toolbar), GTK_ORIENTATION_HORIZONTAL);
	gtk_toolbar_set_style( GTK_TOOLBAR(main->toolbar), GTK_TOOLBAR_BOTH_HORIZ);

	/* Make menus and buttons to toolbar: */
	/* Create toolitems using defined items from stock */
#if MAEMO_VERSION_MAJOR < 5
	main->new_tb = gtk_tool_button_new_from_stock(GTK_STOCK_NEW);
	main->open_tb = gtk_tool_button_new_from_stock(GTK_STOCK_OPEN);
	main->close_tb = gtk_tool_button_new_from_stock(GTK_STOCK_CLOSE);
	main->save_tb = gtk_tool_button_new_from_stock(GTK_STOCK_SAVE);
	main->saveas_tb = gtk_tool_button_new_from_stock(GTK_STOCK_SAVE_AS);
	main->separator_tb1 = gtk_separator_tool_item_new();
	main->fullscreen_tb = gtk_tool_button_new_from_stock( GTK_STOCK_ZOOM_FIT );
    main->separator_tb3 = gtk_separator_tool_item_new();
#endif
	main->add_tb = gtk_tool_button_new_from_stock(GTK_STOCK_ADD);
	main->change_password_tb = gtk_tool_button_new_from_stock(GTK_STOCK_CONVERT);
	main->separator_tb2 = gtk_separator_tool_item_new();
    
    /* create filter text entry in a container */
    main->filter_item_tb = gtk_tool_item_new();
    gtk_tool_item_set_expand(main->filter_item_tb, TRUE); 
#if MAEMO_VERSION_MAJOR < 5
    main->filter_tb = gtk_entry_new();
#else
    main->filter_tb = hildon_entry_new(HILDON_SIZE_AUTO);
    hildon_entry_set_placeholder (HILDON_ENTRY (main->filter_tb), _("Filter"));
#endif

    /* gtk_entry_set_max_length( GTK_ENTRY(main->filter_tb) , (gint) 24);  */
    gtk_container_add( GTK_CONTAINER( main->filter_item_tb ) , main->filter_tb);
	
    /* Insert items to toolbar */
#if MAEMO_VERSION_MAJOR < 5
	gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->new_tb, -1);
	gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->open_tb, -1);
	gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->close_tb, -1);
	gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->save_tb, -1);
	gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->saveas_tb, -1);
	gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->separator_tb1, -1);
#endif
	gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->add_tb, -1);
	gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->change_password_tb, -1);
	gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->separator_tb2, -1);
#if MAEMO_VERSION_MAJOR < 5
	gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->fullscreen_tb, -1);
	gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->separator_tb3, -1);
#endif
	gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->filter_item_tb, -1);

	/* Connect signals to buttons */
#if MAEMO_VERSION_MAJOR < 5
	g_signal_connect(G_OBJECT(main->new_tb), "clicked", G_CALLBACK(callback_file_new), main);
	g_signal_connect(G_OBJECT(main->open_tb), "clicked", G_CALLBACK(callback_file_open), main);
	g_signal_connect(G_OBJECT(main->close_tb), "clicked", G_CALLBACK(callback_file_close), main);
	g_signal_connect(G_OBJECT(main->save_tb), "clicked", G_CALLBACK(callback_file_save), main);
	g_signal_connect(G_OBJECT(main->saveas_tb), "clicked", G_CALLBACK(callback_file_saveas), main);
	g_signal_connect(G_OBJECT(main->fullscreen_tb), "clicked", G_CALLBACK(callback_fullscreen), main);
#endif
	g_signal_connect(G_OBJECT(main->add_tb), "clicked", G_CALLBACK(callback_add_entry), main);
	g_signal_connect(G_OBJECT(main->change_password_tb), "clicked", G_CALLBACK(callback_change_password), main);
	g_signal_connect(G_OBJECT(main->filter_tb), "changed", G_CALLBACK(callback_filter), main);
	
	/* Attach toolbar to main view */
	hildon_window_add_toolbar(main->data->main_view, GTK_TOOLBAR(main->toolbar)); 

	/* Show toolbar */
	gtk_widget_show_all (GTK_WIDGET(main->toolbar));
	gtk_widget_show_all (GTK_WIDGET(main->data->main_view) );
}

/* Create the main grid */
void create_textarea( MainView *main ) {
	GtkCellRenderer *renderer;
        GtkTreeViewColumn *column;
	GtkTreeSelection *select;
		
	main->font_desc = pango_font_description_from_string ("Nokia Sans 16");
	main->store=gtk_tree_store_new (N_COLUMNS, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_POINTER);
	main->treeView=gtk_tree_view_new_with_model(GTK_TREE_MODEL(main->store));
#if MAEMO_VERSION_MAJOR >= 5
	gtk_tree_view_set_show_expanders(GTK_TREE_VIEW (main->treeView), TRUE);
#endif
	renderer = gtk_cell_renderer_pixbuf_new();
	column = gtk_tree_view_column_new_with_attributes (_("Icon"), renderer, "stock-id", ICON_COLUMN, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (main->treeView), column);

	renderer = gtk_cell_renderer_text_new ();
        column = gtk_tree_view_column_new_with_attributes (_("Label"), renderer, "text", LABEL_COLUMN, NULL);
        gtk_tree_view_append_column (GTK_TREE_VIEW (main->treeView), column);
	gtk_widget_show(main->treeView);
        /* Setup the selection handler */
        select = gtk_tree_view_get_selection (GTK_TREE_VIEW (main->treeView));
        gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
        gtk_tree_view_columns_autosize(GTK_TREE_VIEW(main->treeView));
				
    /* Connect signals */
#if MAEMO_VERSION_MAJOR >= 5
	g_signal_connect (G_OBJECT(main->treeView), "hildon-row-tapped", G_CALLBACK (callback_hildon_activate_treeView), main);
#else
	g_signal_connect (G_OBJECT(main->treeView), "row-activated", G_CALLBACK (callback_activate_treeView), main);
#endif
}

/* toggle fullscreen */
void interface_fullscreen(MainView *mainview) {
	g_assert(mainview != NULL && mainview->data != NULL );
	mainview->fullscreen = !mainview->fullscreen;
	if (mainview->fullscreen) {
		gtk_window_fullscreen(GTK_WINDOW(mainview->data->main_view));
	} else {
		gtk_window_unfullscreen(GTK_WINDOW(mainview->data->main_view));
	}
}

void interfaceCallLastItem(MainView *mainView) {
#if MAEMO_VERSION_MAJOR < 5
	g_signal_emit_by_name(G_OBJECT(mainView->open_last_item), "activate");
#else
	g_signal_emit_by_name(G_OBJECT(mainView->open_last_item), "clicked");
#endif
}

