#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "memorycardmanager.h"
#include "savestateselect.h"
#include "keysconfigselect.h"

QString game;
#ifdef Q_WS_MAEMO_5
QString cfg_path ="/home/user/.config/Team16/psx4m/";
QString keys_path="/home/user/.config/Team16/keys/";
#else
QString cfg_path  = QString("%1/.config/Team16/psx4m/").arg(QDir::homePath());
QString keys_path = QString("%1/keys/").arg(cfg_path);
#endif
QString bios_path="/home/user/MyDocs/";
QString rom_path="/home/user/MyDocs";
QString sstate_path="/home/user/.pcsx/sstates/";
QString screenshots_path="/opt/maemo/usr/games/screenshots/";

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

#ifdef Q_WS_MAEMO_5
    //Copy defaul keys config if does not exist:
    if (!QFile::exists(keys::defaultFile())){
        QFile::copy("/opt/psx4m/keys", keys::defaultFile());
    }
#endif

    myProcess = NULL;
    settings=new QSettings();
    bios_path=settings->value("bios_path","/home/user/MyDocs/").toString();
    if (bios_path.right(1)!="/") bios_path+="/";
    rom_path=settings->value("rom_path","/home/user/MyDocs/").toString();
    if (rom_path.right(1)!="/") rom_path+="/";

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
    cfg=new options(this);
    cfg->hide();
    game="";
    cfg->load_cfg();


    QDir cfg_dir(cfg_path);
    if (!cfg_dir.exists()){
        if (!cfg_dir.mkpath(cfg_path))
            qDebug("Can't create cfg dir, permission error?");
        else
            cfg->save_cfg();
    }

    QDir keys_dir(keys_path);
    if (!keys_dir.exists()){
        if (!keys_dir.mkpath(keys_path))
            qDebug("Can't create keys dir, permission error?");
    }

    ui->listWidget->setStyleSheet("QListView::item{ height: 86px; }");
    ui->listWidget->setAutoFillBackground(true);
    ui->listWidget->setIconSize(QSize(144,80));

    get_roms();

    connect (ui->pushButton_2,SIGNAL(clicked()),this,SLOT(rom_config()));
    connect (ui->pushButton,SIGNAL(clicked()),this,SLOT(rom_exec()));
    connect (ui->pbnSstate,SIGNAL(clicked()),this,SLOT(rom_exec_savestate()));

    this->menuBar()->addAction("Default config",this,SLOT(def_cfg()));
    this->menuBar()->addAction("GUI config",this,SLOT(gui_cfg()));
    this->menuBar()->addAction("Configure keys",this,SLOT(set_keys()));
    this->menuBar()->addAction("Edit Memory cards",this,SLOT(memory_cards()));
    this->menuBar()->addAction("Help",this,SLOT(open_help()));

#ifdef Q_WS_MAEMO_5
    connect(ui->listWidget, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(rom_selected(QListWidgetItem*)));
#else
    connect(ui->listWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(rom_selected(QListWidgetItem*)));
#endif
}

void MainWindow::get_roms()
{
    qDebug() << "Reading roms from" << rom_path;
    m_Roms.clear();
    ui->listWidget->clear();
    QDir dir(rom_path);
    dir.setFilter(QDir::Files);
    dir.setSorting(QDir::Name | QDir::IgnoreCase);
    dir.setNameFilters(QStringList()<<"*.bin"<<"*.iso"<<"*.img"<<"*.z"<<"*.znx"<<"*.pbp"<<"*.mdf");

    QFileInfoList list = dir.entryInfoList();
    foreach (QFileInfo fileInfo, list){
        if (fileInfo.fileName()=="scph1001.bin") continue;

        RomInfo* rom = new RomInfo();
        qDebug() << "  Rom:" << fileInfo.fileName();

        QIcon icon(":/images/ICON0.PNG");
        if (QFile::exists(QString("%1.png").arg(fileInfo.filePath())))
            icon = QIcon(QString("%1.png").arg(fileInfo.filePath()));
        else if (QFile::exists(QString("%1png").arg(fileInfo.filePath().replace(fileInfo.suffix(), ""))))
            icon = QIcon(QString("%1png").arg(fileInfo.filePath().replace(fileInfo.suffix(), "")));
        else if (QString::compare(fileInfo.suffix(), "pbp", Qt::CaseInsensitive) == 0){
            QByteArray iconData = get_pbp_icon(fileInfo.filePath());
            if (iconData.size() > 0){
                QImage image = QImage::fromData(iconData);
                icon = QIcon(QPixmap::fromImage(image));
            }
        }
        rom->FilePath = fileInfo.absoluteFilePath();
        rom->FileName = fileInfo.fileName();
        rom->DisplayName = fileInfo.baseName().replace("_", " ");
        m_Roms.append(rom);

        ui->listWidget->addItem(new QListWidgetItem(icon, rom->DisplayName));
    }
    if (m_Roms.count() > 0){
        ui->listWidget->setCurrentRow(0);
        rom_selected(NULL);
    }
}

void MainWindow::rom_exec_savestate()
{
    SavestateSelect* dlg = new SavestateSelect(this, m_Roms.at(ui->listWidget->currentRow()));
    if (dlg->exec() == QDialog::Accepted){
        int slot = dlg->Selected();
        int disc = dlg->SelectedDisc();
        delete dlg;

        if (slot >= 0){
            qDebug() << "Savestate" << slot << "disc" << disc;
            rom_exec(disc, slot);
        }
    }else
        delete dlg;
    rom_selected(NULL);
}

void MainWindow::rom_exec(){
    rom_exec(0, -1);
}

void MainWindow::rom_exec(int disc, int slot)
{
    if (ui->listWidget->currentRow()!=-1){
        RomInfo* rom = m_Roms.at(ui->listWidget->currentRow());
        game = rom->FileName;
        cfg->load_cfg();

        if (myProcess == NULL){
            if (!options::isSupported(current_core, rom->FilePath)){
                QMessageBox msg(this);
                msg.setWindowTitle(tr("Error"));
                msg.setText(tr("Image format not supported by the selected core."));
                msg.exec();
                return;
            }

            ui->pushButton->setEnabled(false);
            ui->pushButton_2->setEnabled(false);
            ui->pbnSstate->setEnabled(false);
            myProcess = new QProcess(this);
            myProcess->setWorkingDirectory("/home/user/");
            connect(myProcess, SIGNAL(finished(int)), this, SLOT(process_finished(int)));

            QStringList args = current_config;
            args << rom->FilePath;
            if (disc > 0)
                args << "-disc" << QString::number(disc);
            if (slot >= 0)
                args << "-load" << QString::number(slot);

            qDebug (QString("%1 %2").arg("/opt/maemo/usr/games/"+current_core).arg(args.join(" ")).toAscii());
            myProcess->start("/opt/maemo/usr/games/"+current_core, args);
            if (!myProcess->waitForStarted(10000)){
                qDebug() << "Failed to start process.";
                process_finished(-1);
            }
        }
    }
}

void MainWindow::process_finished(int exitCode)
{
    if (exitCode){
        QString error = myProcess->readAllStandardError();
        QMessageBox msg(this);
        msg.setWindowTitle(tr("Error"));
        if (error.isEmpty())
            msg.setText(tr("Unknown error"));
        else
            msg.setText(error);
        msg.exec();
    }

    ui->pushButton->setEnabled(true);
    ui->pushButton_2->setEnabled(true);
    ui->pbnSstate->setEnabled(true);
    myProcess->deleteLater();
    myProcess = NULL;
}

void MainWindow::rom_selected(QListWidgetItem*){
    if (myProcess == NULL && ui->listWidget->currentRow()!=-1){
        RomInfo* rom = m_Roms.at(ui->listWidget->currentRow());
        game="";
        cfg->load_cfg();
        game = rom->FileName;
        cfg->load_cfg();

        if (rom->SaveStateNames.count() == 0){
            if (current_core == "pcsx"){
                QProcess* p = new QProcess(this);
                p->setWorkingDirectory("/home/user/");
                QStringList args = QStringList() << "-cdfile" << rom->FilePath << "-get_sstatename" << "0";
                p->start("/opt/maemo/usr/games/pcsx", args);
                if (p->waitForFinished(3000)){
                    if (p->exitCode() == 0){
                        QString out = p->readAll();
                        QStringList lines = out.split("\n", QString::SkipEmptyParts);
                        foreach (QString line, lines){
                            line = line.trimmed();
                            if (line.startsWith("SAVESTATE:")){
                                QFileInfo f(line.remove(0, 8).trimmed());
                                rom->SaveStateNames.append(f.fileName());
                            }
                        }
                    }
                }else{
                    p->kill();
                }
                delete p;
            }else{
                QFileInfo f(rom->FileName);
                rom->SaveStateNames.append(QString("%1.save.0").arg(f.fileName()));
            }
        }

        int count = count_save_states(rom->SaveStateNames);
        ui->pbnSstate->setEnabled(count > 0);

/*        QString scr = QString("%1/%2_bkg.png").arg(rom_path).arg(game);
        if (QFile::exists(scr)){
            ui->listWidget->setStyleSheet(QString("QListView { \
                                                    background-image: url(%1); background-position: center; \
                                                    background-repeat: no-repeat; \
                                                    background-color: black; \
                                                    color: white; \
                                                   }    \
                                                   QListView::item { \
                                                       color: white; \
                                                       border-left: 1px solid black; \
                                                       border-top: 1px none black; \
                                                       border-bottom: 1px none black; \
                                                   } \
                                                   QListView::item:selected { \
                                                       color: white; \
                                                       background-color: rgba(0,0,0,125); \
                                                       border: 1px solid white; \
                                                   }").arg(scr));
        }else{
            ui->listWidget->setStyleSheet(QString("QListView { \
                                                    background-image: url(:/images/Gameboot.png); background-position: center; \
                                                    background-repeat: no-repeat; \
                                                    background-color: black; \
                                                    color: white; \
                                                   }    \
                                                   QListView::item { \
                                                       color: white; \
                                                       border-left: 1px solid black; \
                                                       border-top: 1px none black; \
                                                       border-bottom: 1px none black; \
                                                   } \
                                                   QListView::item:selected { \
                                                       color: white; \
                                                       background-color: rgba(0,0,0,125); \
                                                       border: 1px solid white; \
                                                   }"));
        }*/
    }
}

void MainWindow::set_keys(){

    int count = keys::keysConfigCount();
    if (count == 0){
        keyopt=new keys(this);
        keyopt->setWindowTitle(tr("Default config"));
        keyopt->show();
        keyopt = NULL;
    }else{
        KeysConfigSelect* dlg = new KeysConfigSelect(this);
        if (dlg->exec() == QDialog::Accepted){
            keyopt=new keys(this, dlg->Selected(), dlg->SelectedName());
            keyopt->setWindowTitle(dlg->SelectedName());
            keyopt->show();
            keyopt = NULL;
        }
        delete dlg;
    }
}

void MainWindow::def_cfg()
{
    game="";
    cfg->load_cfg();
    cfg->setWindowTitle("Config: default");
    cfg->show();
}

void MainWindow::gui_cfg()
{
    QDialog *dlg=new QDialog(this);
    dlg->setMinimumWidth(200);
    dlg->setWindowTitle(tr("Gui config"));
    QGridLayout *lay=new QGridLayout(dlg);

    lbl1 = new QLabel(settings->value("rom_path").toString());
    lbl1->setWordWrap(true);
    lbl2 = new QLabel(settings->value("bios_path").toString());
    lbl2->setWordWrap(true);

    QPushButton *btn1=new QPushButton(tr("Rom directory"),dlg);
    QPushButton *btn2=new QPushButton(tr("Bios directory"),dlg);
    connect (btn1,SIGNAL(clicked()),this,SLOT(set_rom_dir()));
    connect (btn2,SIGNAL(clicked()),this,SLOT(set_bios_dir()));
    lay->addWidget(lbl1,1,0);
    lay->addWidget(btn1,2,0);
    lay->addWidget(lbl2,3,0);
    lay->addWidget(btn2 ,4,0);
    dlg->exec();
    delete dlg;
}

void MainWindow::set_rom_dir()
{
    QString dir=QFileDialog::getExistingDirectory(this,tr("ROM Directory"),"/home/user/MyDocs/");
    if (!dir.isEmpty()){
        rom_path=dir;
        settings->setValue("rom_path",rom_path);
        settings->sync();
        if (settings->status() != QSettings::NoError){
            QMessageBox msg(this);
            msg.setText("Can't save gui config, permission error?");
            msg.exec();
        }
        lbl1->setText(rom_path);
        get_roms();
    }
}

void MainWindow::set_bios_dir()
{
    QString dir=QFileDialog::getExistingDirectory(this,tr("BIOS Directory"),"/home/user/MyDocs/");
    if (!dir.isEmpty()){
        bios_path=dir;
        if (bios_path.right(1)!="/")bios_path+="/";
        settings->setValue("bios_path",bios_path);
        settings->sync();
        if (settings->status() != QSettings::NoError){
            QMessageBox msg(this);
            msg.setText("Can't save gui config, permission error?");
            msg.exec();
        }
        lbl2->setText(bios_path);
    }
}

void MainWindow::open_help()
{
    QDesktopServices::openUrl(QUrl("http://maemo.team16.ru/index.php?page=7"));
}


MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::rom_config()
{
    if (ui->listWidget->currentRow()!=-1){
        RomInfo* rom = m_Roms.at(ui->listWidget->currentRow());
        QString core = current_core;
        game = rom->FileName;
        cfg->load_cfg();
        cfg->setWindowTitle(tr("Config: %1").arg(game));
        cfg->show();
        if (core != current_core){
            rom->SaveStateNames.clear();
            rom_selected(NULL);
        }
    }
}

void MainWindow::memory_cards()
{
    MemoryCardManager* dlg = new MemoryCardManager(this);
    dlg->show();
    dlg = NULL;
}

int MainWindow::count_save_states(QStringList names)
{
    int res = 0;

    if (names.count() > 0){
        foreach (QString name, names){
            QFileInfo f(name);
            QDir dir;
            if (current_core == "pcsx")
                dir.setPath(sstate_path);
            else
                dir.setPath(rom_path);
            dir.setFilter(QDir::Files);
            dir.setSorting(QDir::Name | QDir::IgnoreCase);
            dir.setNameFilters(QStringList()<< QString("%1*").arg(f.completeBaseName()));
            QFileInfoList list = dir.entryInfoList();

            res += list.count();
        }
    }
    return res;
}

QByteArray MainWindow::get_pbp_icon(QString filename)
{
    QByteArray res;

    QFile inFile(filename);
    if (!inFile.open(QIODevice::ReadOnly))
        return res;

    QDataStream inStream( &inFile );

    char signature[4];
    inStream.readRawData((char *)(signature), sizeof(signature));
    if ((signature[0] != 0x00) || (signature[1] != 0x50) || (signature[2] != 0x42) || (signature[3] != 0x50)) {
        inFile.close();
        return res;
    }

    unsigned int version;
    inStream >> version;

    uint32_t offset[8];
    inStream.readRawData((char *)(offset), sizeof(offset));

    offset[8] = inFile.size();

    int iconSize = offset[2] - offset[1];
    int iconOffset = offset[1] - offset[0];

    char buffer[iconSize];
    if (inStream.skipRawData(iconOffset) != iconOffset) {
        inFile.close();
        return res;
    }

    if (inStream.readRawData(buffer, iconSize) != iconSize) {
        inFile.close();
        return res;
    }

    res = QByteArray(buffer, iconSize);
    inFile.close();
    return res;
}
