/*
 *  profilesx
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include "profilesx-status-panel-plugin.h"
#include <libprofile.h>
#include <keys_nokia.h>
#include <libintl.h>
#include <locale.h>
#include "profile_changed_marshal.h"

#define PROFILESX_STATUS_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
									      TYPE_PROFILESX_STATUS_PLUGIN, ProfilesxStatusPluginPrivate))

#define PROFILESX_PLUGIN_SETTINGS_FILE "/.profilesx"

#define 	PROFILED_SERVICE   "com.nokia.profiled"
#define 	PROFILED_PATH   "/com/nokia/profiled"
#define 	PROFILED_INTERFACE   "com.nokia.profiled"
#define 	PROFILED_CHANGED   "profile_changed"
struct _ProfilesxStatusPluginPrivate
{
  GtkWidget* button;
  DBusGConnection *dbus_conn;
  DBusGProxy *dbus_proxy;
};

HD_DEFINE_PLUGIN_MODULE(ProfilesxStatusPlugin, profilesx_status_plugin, HD_TYPE_STATUS_MENU_ITEM);


gchar* get_profile_display_name(const gchar* profile_name)
{
  if(g_strcmp0(profile_name, "silent")==0)
  {
    return dgettext("osso-profiles", "profi_bd_silent");
  }
  else if(g_strcmp0(profile_name, "general")==0)
  {
    return dgettext("osso-profiles", "profi_bd_general");
  }
  else
  {
    return g_strdup(profile_name);
  }
}

void 
set_active_profile_from_radio_button(GtkRadioButton* button, gchar** profileList)
{
  GSList* list = gtk_radio_button_get_group(button);
  gboolean found = FALSE;
  gint index = g_strv_length(profileList);

  while(list && !found)
  {
    found = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(list->data));
    if(found)
    {
      profile_set_profile(profileList[index-1]);
    }
    list = list->next;
    index--;
  }
}

static void
profilesx_status_plugin_class_finalize(ProfilesxStatusPluginClass* klass)
{
}

static void
show_profile_selection_dlg(GtkButton* button, ProfilesxStatusPlugin* plugin)
{
  gtk_widget_hide(gtk_widget_get_toplevel(GTK_WIDGET(button)));

  GtkWidget* pannable_area;
  GtkWidget* content_area;
  GtkWidget* dialog = gtk_dialog_new_with_buttons("Profiles",
						  NULL,
						  GTK_DIALOG_MODAL,
						  dgettext("hildon-libs", "wdgt_bd_done"), GTK_RESPONSE_OK, NULL);
  gtk_window_set_default_size(GTK_WINDOW(dialog), -1, 300);  

  content_area = gtk_vbox_new(FALSE, 3);
  pannable_area = hildon_pannable_area_new();
  gchar** profileList = profile_get_profiles();
  GtkWidget** vibs;
  GtkWidget* previousRadioButton = NULL;
  if(profileList)
  {
    gint profile_list_length = g_strv_length(profileList);
    gint idx = 0;
    vibs = g_new0(GtkWidget*, profile_list_length);

    gchar** profilePtr;
    gchar* activeProfile = profile_get_profile();

    for(profilePtr = profileList;
	*profilePtr;
	profilePtr++)
    {
      GtkWidget* hbox = gtk_hbox_new(FALSE, 3);
      GtkWidget* button = hildon_gtk_radio_button_new_from_widget(HILDON_SIZE_FINGER_HEIGHT,
								  GTK_RADIO_BUTTON(previousRadioButton));
      gtk_button_set_label(GTK_BUTTON(button),
			   get_profile_display_name(*profilePtr));

      gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(button), FALSE);
      gchar* mode = NULL;
      if(g_strcmp0(*profilePtr, activeProfile)==0)
      {
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);  
      }
      previousRadioButton = button;
      GtkWidget* vib_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
      hildon_check_button_set_active(HILDON_CHECK_BUTTON(vib_button), profile_get_value_as_bool(*profilePtr,
												PROFILEKEY_VIBRATING_ALERT_ENABLED));
      vibs[idx] = vib_button;
      idx++;
      gtk_button_set_label(GTK_BUTTON(vib_button),
			   dgettext("osso-profiles", "profi_fi_general_vibrate"));
      gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
      gtk_box_pack_end(GTK_BOX(hbox), vib_button, FALSE, FALSE, 0);
      gtk_box_pack_start(GTK_BOX(content_area), hbox, FALSE, FALSE, 3);
    }
  }


  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pannable_area), content_area); 
  gtk_container_add(GTK_CONTAINER(gtk_dialog_get_content_area(GTK_DIALOG(dialog))), pannable_area);
  gtk_widget_show_all(dialog);

  int ret = gtk_dialog_run(GTK_DIALOG(dialog));

  if(ret == GTK_RESPONSE_OK)
  {
    set_active_profile_from_radio_button(GTK_RADIO_BUTTON(previousRadioButton), profileList);

    if(profileList)
    {
      gint profile_list_length = g_strv_length(profileList);
      for(;profile_list_length>0;--profile_list_length)
      {
	if(hildon_check_button_get_active(HILDON_CHECK_BUTTON(vibs[profile_list_length-1])))
	{
	  profile_set_value_as_bool(profileList[profile_list_length-1], PROFILEKEY_VIBRATING_ALERT_ENABLED, TRUE);
	}
	else
	{
	  profile_set_value_as_bool(profileList[profile_list_length-1], PROFILEKEY_VIBRATING_ALERT_ENABLED, FALSE);
	}
      }
    }
  }
  profile_free_profiles(profileList);
  g_free(vibs);  

  gtk_widget_destroy(dialog);
}

static gchar* 
get_profile_icon_name(const gchar* profile_name)
{
  GKeyFile *keyFile;
  FILE *iniFile;
  gchar *filename;
  gchar* icon_name = NULL;
  keyFile = g_key_file_new();
  filename = g_strconcat (g_get_home_dir(), PROFILESX_PLUGIN_SETTINGS_FILE, NULL);
  g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);
  icon_name = g_key_file_get_string(keyFile, profile_name, "icon", NULL);
  g_key_file_free (keyFile); 
  g_free (filename); 
  return icon_name;
}

static void
set_profile_status(ProfilesxStatusPlugin* plugin, const gchar* profile)
{
  GtkWidget *image;
  image = gtk_image_new_from_icon_name("general_profile", 48);
  gtk_image_set_pixel_size(GTK_IMAGE(image), 48);
  gchar* icon_name = get_profile_icon_name(profile);
  if(icon_name != NULL)
  {
    GdkPixbuf *pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (), icon_name, 18, 0, NULL);
    if(pixbuf==NULL)
    {
      GtkWidget* image = gtk_image_new_from_file(icon_name);
      pixbuf = gtk_image_get_pixbuf(GTK_IMAGE(image));
      hd_status_plugin_item_set_status_area_icon (HD_STATUS_PLUGIN_ITEM (plugin), pixbuf);
      g_object_unref(image);  
    }
    else
    {
      hd_status_plugin_item_set_status_area_icon (HD_STATUS_PLUGIN_ITEM (plugin), pixbuf);
      g_object_unref(pixbuf);
    }
    g_free(icon_name);
  }
  else
  {
    hd_status_plugin_item_set_status_area_icon (HD_STATUS_PLUGIN_ITEM (plugin), NULL);
  }
  hildon_button_set_image(HILDON_BUTTON (plugin->priv->button), image);
  hildon_button_set_style(HILDON_BUTTON (plugin->priv->button), HILDON_BUTTON_STYLE_PICKER);
  hildon_button_set_alignment(HILDON_BUTTON(plugin->priv->button), 
			      0, 0,
			      1, 1);
  hildon_button_set_value(HILDON_BUTTON(plugin->priv->button), 
			  get_profile_display_name(profile));
}

static void
handle_profile_changed(DBusGProxy *object,
		      gboolean changed, 
		      gboolean active, 
		      const char* profile, 
		      GPtrArray* values, 
		      ProfilesxStatusPlugin* plugin)
{
  gchar* new_profile = profile_get_profile();
  set_profile_status(plugin, new_profile);
}

static void
profilesx_status_plugin_finalize(GObject* object)
{
  ProfilesxStatusPlugin* plugin = PROFILESX_STATUS_PLUGIN(object);
  
  if (plugin->priv->dbus_proxy)
    dbus_g_proxy_disconnect_signal(plugin->priv->dbus_proxy, 
				   PROFILED_CHANGED, 
				   G_CALLBACK(handle_profile_changed), 
				   plugin);

}

static void
register_dbus_signal_on_profile_change(ProfilesxStatusPlugin* plugin)
{
  plugin->priv->dbus_conn = NULL;
  plugin->priv->dbus_proxy = NULL;
  plugin->priv->dbus_conn = hd_status_plugin_item_get_dbus_g_connection (HD_STATUS_PLUGIN_ITEM(&plugin->parent),
									 DBUS_BUS_SESSION, 
									 NULL);

  if(plugin->priv->dbus_conn)
  {
    plugin->priv->dbus_proxy = dbus_g_proxy_new_for_name(plugin->priv->dbus_conn,
							 PROFILED_SERVICE,
							 PROFILED_PATH,
							 PROFILED_INTERFACE);
    // this took me
    GType ARRAY_STRING_STRING_STRING_TYPE = 
      dbus_g_type_get_collection("GPtrArray",    
				 dbus_g_type_get_struct("GValueArray",    
							G_TYPE_STRING,    
							G_TYPE_STRING,    
							G_TYPE_STRING,   
							G_TYPE_INVALID));
    // hours to figure out
    dbus_g_object_register_marshaller(_profile_changed_marshal_VOID__BOOLEAN_BOOLEAN_STRING_BOXED,
				      G_TYPE_NONE,
				      G_TYPE_BOOLEAN,
				      G_TYPE_BOOLEAN,
				      G_TYPE_STRING,
				      ARRAY_STRING_STRING_STRING_TYPE, 
				      G_TYPE_INVALID);   
    // how to do
    dbus_g_proxy_add_signal(plugin->priv->dbus_proxy,    
   			    PROFILED_CHANGED,    
   			    G_TYPE_BOOLEAN,    
   			    G_TYPE_BOOLEAN,    
   			    G_TYPE_STRING,    
   			    ARRAY_STRING_STRING_STRING_TYPE,
   			    G_TYPE_INVALID );    
    // this!
    dbus_g_proxy_connect_signal(plugin->priv->dbus_proxy,
				PROFILED_CHANGED,
				G_CALLBACK(handle_profile_changed), plugin, NULL);
  }
}

static void
profilesx_status_plugin_class_init(ProfilesxStatusPluginClass* klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = (GObjectFinalizeFunc)profilesx_status_plugin_finalize;
  g_type_class_add_private(klass, sizeof (ProfilesxStatusPluginPrivate));
}

static void
profilesx_status_plugin_init(ProfilesxStatusPlugin* plugin)
{
  plugin->priv = PROFILESX_STATUS_PLUGIN_GET_PRIVATE(plugin);

  register_dbus_signal_on_profile_change(plugin);

  plugin->priv->button = hildon_button_new(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
					   HILDON_BUTTON_ARRANGEMENT_VERTICAL);

  hildon_button_set_title(HILDON_BUTTON(plugin->priv->button), dgettext("osso-profiles", "profi_ti_menu_plugin"));
  g_signal_connect(plugin->priv->button, "clicked", G_CALLBACK(show_profile_selection_dlg), plugin);
  gtk_widget_show_all(plugin->priv->button);

  gchar* profile = profile_get_profile();
  if(profile)
  {
    set_profile_status(plugin, profile);
    g_free(profile);
  }
  gtk_container_add(GTK_CONTAINER(plugin), plugin->priv->button);
  gtk_widget_show(GTK_WIDGET(plugin));
}
