#include "profilesx-profile-editor-dialog.h"
#include "profilesx-control-panel-plugin.h"
#include <gtk/gtk.h>
#include <libprofile.h>
#include <keys_nokia.h>
#include <libintl.h>
#include <locale.h>
#include <hildon/hildon.h>
#include <hildon/hildon-file-chooser-dialog.h>

#define IM_SECTION "[IM]"
#define SMS_SECTION "[SMS]"
#define EMAIL_SECTION "[EMAIL]"
#define RINGING_SECTION "[RINGTONE]" // [sic!]

#define PROFILESX_PLUGIN_SETTINGS_FILE "/.profilesx"

struct _editor_data_t
{
  GtkWidget* iconButton;
  GtkWidget* ringtoneSelector;
  GtkWidget* ringtoneButton;
  GtkWidget* emailtoneSelector;
  GtkWidget* emailtoneButton;
  GtkWidget* imtoneSelector;
  GtkWidget* imtoneButton;
  GtkWidget* smstoneSelector;
  GtkWidget* smstoneButton;
  GtkWidget* ringtoneVolumeSlider;
  GtkWidget* smsVolumeSlider;
  GtkWidget* imVolumeSlider;
  GtkWidget* emailVolumeSlider;

  GtkWidget* keypadsoundLevelButton;
  GtkWidget* touchscreensoundLevelButton;
  GtkWidget* systemsoundLevelButton;

  GtkListStore* keypadsoundLevel;
  GtkListStore* touchscreensoundLevel;
  GtkListStore* systemsoundLevel;

  GtkWidget* vibratingCheckbutton;
  GtkWidget* silentCheckbutton;
  /* unused
  GtkWidget* clockalarmCheckbutton;
  GtkWidget* calendaralarmCheckbutton;
  */
  GtkListStore* ringingTone;
  GtkListStore* smsTone;
  GtkListStore* imTone;
  GtkListStore* emailTone;  
  GstElement* sound_player;
};

void play_sound(const gchar* file, editor_data_t* data)
{
  int ret = gst_element_set_state(data->sound_player, GST_STATE_NULL);
  GstState state;
  GstState pending;
  int stret = gst_element_get_state(data->sound_player, &state, &pending, GST_CLOCK_TIME_NONE);
  g_object_set(data->sound_player, "uri", g_filename_to_uri(file, NULL, NULL));
  int ret2 = gst_element_set_state(data->sound_player, GST_STATE_PLAYING);
}

gchar* get_profile_display_name(const gchar* profile_name)
{
  if(g_strcmp0(profile_name, "silent")==0)
  {
    return dgettext("osso-profiles", "profi_bd_silent");
  }
  else if(g_strcmp0(profile_name, "general")==0)
  {
    return dgettext("osso-profiles", "profi_bd_general");
  }
  else
  {
    return g_strdup(profile_name);
  }
}

void read_predefined_ringtones(editor_data_t* data)
{
  GIOChannel* channel;
  GRegex* regex;
  GError* error = NULL;

  channel = g_io_channel_new_file("/etc/ringtones", "r", &error);
  regex = g_regex_new("^\"([^\"]+)\"[ \t]+(.*)$",
		      0,
		      0,
		      NULL);
  
  gchar* line;
  gchar* line_without_linebreak;
  int length;
  gsize line_terminator;
  GtkListStore* current_list_store = NULL;

  while(g_io_channel_read_line(channel,
			       &line,
			       &length,
			       &line_terminator,
			       &error)== G_IO_STATUS_NORMAL)
  {
    if(length>2)
    {
      line_without_linebreak = g_strndup(line,
					 line_terminator);
      g_free(line);
      line = line_without_linebreak;
      if(g_strcmp0(line, SMS_SECTION)==0)
      {
	current_list_store = data->smsTone;
      }
      else if(g_strcmp0(line, IM_SECTION)==0)
      {
	current_list_store = data->imTone;
      }
      else if(g_strcmp0(line, EMAIL_SECTION)==0)
      {
	current_list_store = data->emailTone;
      }
      else if(g_strcmp0(line, RINGING_SECTION)==0)
      {
	current_list_store = data->ringingTone;
      }
      else
      {
	if(current_list_store!=NULL)
	{
	  GtkTreeIter iter;
	  GMatchInfo* match_info;
	  g_regex_match(regex, line, 0, &match_info);
	  gchar* key = g_match_info_fetch(match_info, 1);
	  gchar* value = g_match_info_fetch(match_info, 2);
	  gtk_list_store_append(current_list_store, &iter);
	  gtk_list_store_set(current_list_store, &iter, 0, key, 1, value, -1);
	  g_match_info_free(match_info);
	}
      }
    }
    g_free(line);
  }
  g_io_channel_unref(channel);
}

gchar*
get_display_value_from_file_name(gchar* value, GtkListStore* list_store, GtkTreeIter* iter)
{
  GFile* file;
  gchar* basename;
  file = g_file_new_for_path(value);
  basename = g_file_get_basename(file);
  g_object_unref(file);
  gtk_list_store_prepend(list_store, iter);
  gtk_list_store_set(list_store, iter, 0, basename, 1, value, -1);
  return basename;
}

gchar*
get_display_value(gchar* value, GtkListStore* list_store, GtkTreeIter* iter)
{
  gboolean valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(list_store), iter);
  while (valid)
  {
    gchar* key;
    gchar* list_value;
    gtk_tree_model_get (GTK_TREE_MODEL(list_store), iter,
			0, &key,
			1, &list_value,
			-1);
    if(g_strcmp0(list_value, value)==0)
    {
      g_free (list_value);
      return key;
    }
    g_free (list_value);
    g_free (key);
    valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(list_store), iter);
  }
  return get_display_value_from_file_name(value, list_store, iter);
}

void 
set_profile_tone_value(const gchar* profile_name, const gchar* key, GtkWidget* button, GtkListStore* list_store)
{
  gchar* value = profile_get_value(profile_name, key);
  GtkTreeIter iter;
  gchar* display_value = get_display_value(value, list_store, &iter);
  //  hildon_touch_selector_select_iter(HILDON_TOUCH_SELECTOR(selector), 0, &iter, TRUE);
  hildon_button_set_value(HILDON_BUTTON(button), display_value);
  g_free(value);
  g_free(display_value);
}

void
set_profile_volume_value(const gchar* profile_name, const gchar* key, GtkWidget* slider)
{
  int slider_value;
  slider_value = profile_get_value_as_int(profile_name,
					  key);
  gtk_range_set_value(GTK_RANGE(slider), slider_value);

}

void
set_profile_sound_level_value(const gchar* profile_name, const gchar* key, GtkWidget* button)
{
  int value = profile_get_value_as_int(profile_name, key);

  GtkTreeIter iter;
  hildon_picker_button_set_active(HILDON_PICKER_BUTTON(button), value);
  HildonTouchSelector* selector = 
    hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(button));
  
  GtkListStore* list_store = 
    
    GTK_LIST_STORE(hildon_touch_selector_get_model(hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(button)), 0));

  gboolean valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list_store), &iter);

  while(valid)
  {
    gchar* list_key;
    int list_value;
    gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter, 
		       0, &list_key, 1, &list_value, -1);
    if(list_value==value)
    {
      hildon_button_set_value(HILDON_BUTTON(button), list_key);
      g_free(list_key);
      return;
    }
    g_free(list_key);
    valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(list_store), &iter);
  }
}

void
set_profile_checkbutton_value(const gchar* profile_name, const gchar* key, GtkWidget* checkbutton)
{
  gboolean active = profile_get_value_as_bool(profile_name, key);
  if(active)
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(checkbutton), TRUE);
  else
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(checkbutton), FALSE);
}

void
set_profile_silent_checkbutton_value(const gchar* profile_name, const gchar* key, GtkWidget* checkbutton)
{
  gchar* type = profile_get_value(profile_name, key);
  if(g_strcmp0(type, "ringing")==0)
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(checkbutton), FALSE);
  else
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(checkbutton), TRUE);
}

void set_profile_status_icon_value(const gchar* profile_name, GtkWidget* button)
{
  GKeyFile *keyFile;
  FILE *iniFile;
  gchar *filename;
  gchar* icon_name = NULL;
  keyFile = g_key_file_new();
  filename = g_strconcat (g_get_home_dir(), PROFILESX_PLUGIN_SETTINGS_FILE, NULL);
  g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);
  icon_name = g_key_file_get_string(keyFile, profile_name, "icon", NULL);
  if(icon_name!=NULL)
  {
    GtkWidget* image;
    if(gtk_icon_theme_has_icon(gtk_icon_theme_get_default (), 
			       icon_name))
      image = gtk_image_new_from_icon_name(icon_name, 18);
    else
      image = gtk_image_new_from_file(icon_name);
    
    g_object_set(G_OBJECT(image), "name", icon_name);
    gtk_button_set_image(GTK_BUTTON(button), image);
  }
  g_key_file_free (keyFile); 
  g_free (filename); 
  g_free(icon_name);
}

void
set_profile_values(const gchar* profile_name, editor_data_t* data)
{
  set_profile_tone_value(profile_name, PROFILEKEY_RINGING_ALERT_TONE, data->ringtoneButton, data->ringingTone);
  set_profile_tone_value(profile_name, PROFILEKEY_SMS_ALERT_TONE, data->smstoneButton, data->smsTone);
  set_profile_tone_value(profile_name, PROFILEKEY_EMAIL_ALERT_TONE, data->emailtoneButton, data->emailTone);
  set_profile_tone_value(profile_name, PROFILEKEY_IM_ALERT_TONE, data->imtoneButton, data->imTone);
  set_profile_volume_value(profile_name, PROFILEKEY_RINGING_ALERT_VOLUME, data->ringtoneVolumeSlider);
  set_profile_volume_value(profile_name, PROFILEKEY_EMAIL_ALERT_VOLUME, data->emailVolumeSlider);
  set_profile_volume_value(profile_name, PROFILEKEY_IM_ALERT_VOLUME, data->imVolumeSlider);
  set_profile_volume_value(profile_name, PROFILEKEY_SMS_ALERT_VOLUME, data->smsVolumeSlider);
  set_profile_sound_level_value(profile_name, PROFILEKEY_KEYPAD_SOUND_LEVEL, data->keypadsoundLevelButton);
  set_profile_sound_level_value(profile_name, PROFILEKEY_TOUCHSCREEN_SOUND_LEVEL, data->touchscreensoundLevelButton);
  set_profile_sound_level_value(profile_name, PROFILEKEY_SYSTEM_SOUND_LEVEL, data->systemsoundLevelButton);
  set_profile_checkbutton_value(profile_name, PROFILEKEY_VIBRATING_ALERT_ENABLED, data->vibratingCheckbutton);
  //  set_profile_silent_checkbutton_value(profile_name, PROFILEKEY_RINGING_ALERT_TYPE, data->silentCheckbutton);
  /* unused
  set_profile_checkbutton_value(profile_name, PROFILEKEY_CLOCK_ALARM_ENABLED, data->clockalarmCheckbutton);
  set_profile_checkbutton_value(profile_name, PROFILEKEY_CALENDAR_ALARM_ENABLED, data->calendaralarmCheckbutton);
  */
  set_profile_status_icon_value(profile_name, data->iconButton);
}

void
get_iter_for_value(GtkListStore* list_store, const gchar* value, GtkTreeIter* iter)
{
  gboolean valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list_store), iter);
  while(valid)
  {
    gchar* list_key;
    gchar* list_value;
    gtk_tree_model_get(GTK_TREE_MODEL(list_store), iter,
		       0, &list_key,
		       1, &list_value,
		       -1);
    if(g_strcmp0(list_key, value)==0)
    {
      g_free(list_value);
      g_free(list_key);
      return;
    }
    g_free(list_value);
    g_free(list_key);
    valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(list_store), iter);
  }
}

void
save_profile_checkbutton_value(const gchar* profile_name, const gchar* key, GtkWidget* checkbutton)
{
  gboolean active = hildon_check_button_get_active(HILDON_CHECK_BUTTON(checkbutton));
  profile_set_value_as_bool(profile_name, key, active);
}

void
save_profile_silent_checkbutton_value(const gchar* profile_name, const gchar* key, GtkWidget* checkbutton)
{
  gboolean active = hildon_check_button_get_active(HILDON_CHECK_BUTTON(checkbutton));
  if(active)
  {
    profile_set_value(profile_name, key, "silent");
  }
  else
  {
    profile_set_value(profile_name, key, "ringing");
  }
}

void 
save_profile_tone_value(const gchar* profile_name, const gchar* key, GtkWidget* button, GtkListStore* list_store)
{
  GtkTreeIter iter;
  const gchar* value = hildon_button_get_value(HILDON_BUTTON(button));
  get_iter_for_value(list_store, value, &iter);
  gchar* list_value;
  gchar* key_value;
  gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
		     0, &key_value,
		     1, &list_value,
		     -1);
  profile_set_value(profile_name, key, list_value);
  g_free(list_value);
}
void
save_profile_volume_value(const gchar* profile_name, const gchar* key, GtkWidget* slider)
{
  int slider_value;
  slider_value = (int)gtk_range_get_value(GTK_RANGE(slider));
  profile_set_value_as_int(profile_name,
			   key, slider_value);
}
void
save_profile_sound_level_value(const gchar* profile_name, const gchar* key, GtkWidget* button)
{
  int value = hildon_touch_selector_get_active(hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(button)),
					       0);
  profile_set_value_as_int(profile_name, key, value);
}

void save_profile_status_icon_value(const gchar* profile_name, GtkWidget* button)
{
  gchar* icon_name = NULL;
  GtkImage* image = GTK_IMAGE(GTK_BUTTON(gtk_button_get_image(GTK_BUTTON(button))));
  if(image!=NULL)
  {
    g_object_get(G_OBJECT(image), "icon-name", &icon_name);
    if(icon_name==NULL)
    {
      g_object_get(G_OBJECT(GTK_BUTTON(gtk_button_get_image(GTK_BUTTON(button)))), "name", &icon_name);
    }
  }

  GKeyFile *keyFile;
  gchar *fileData;
  FILE *iniFile;
  gsize size;
  gchar *filename;

  keyFile = g_key_file_new();
  filename = g_strconcat (g_get_home_dir(), PROFILESX_PLUGIN_SETTINGS_FILE, NULL);
  g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);
  if(icon_name!=NULL)
    g_key_file_set_string(keyFile, profile_name, "icon", icon_name);
  else
    g_key_file_remove_key(keyFile, profile_name, "icon", NULL);

  fileData = g_key_file_to_data (keyFile, &size, NULL);
  g_file_set_contents(filename, fileData, size, NULL);

  g_key_file_free (keyFile); 
  g_free (fileData); 
  g_free (filename); 
  g_free(icon_name);

}

void
save_profile_values(const gchar* profile_name, editor_data_t* data)
{
  save_profile_tone_value(profile_name, PROFILEKEY_RINGING_ALERT_TONE, data->ringtoneButton, data->ringingTone);
  save_profile_tone_value(profile_name, PROFILEKEY_SMS_ALERT_TONE, data->smstoneButton, data->smsTone);
  save_profile_tone_value(profile_name, PROFILEKEY_EMAIL_ALERT_TONE, data->emailtoneButton, data->emailTone);
  save_profile_tone_value(profile_name, PROFILEKEY_IM_ALERT_TONE, data->imtoneButton, data->imTone);
  save_profile_volume_value(profile_name, PROFILEKEY_RINGING_ALERT_VOLUME, data->ringtoneVolumeSlider);
  save_profile_volume_value(profile_name, PROFILEKEY_EMAIL_ALERT_VOLUME, data->emailVolumeSlider);
  save_profile_volume_value(profile_name, PROFILEKEY_IM_ALERT_VOLUME, data->imVolumeSlider);
  save_profile_volume_value(profile_name, PROFILEKEY_SMS_ALERT_VOLUME, data->smsVolumeSlider);
  save_profile_sound_level_value(profile_name, PROFILEKEY_KEYPAD_SOUND_LEVEL, data->keypadsoundLevelButton);
  save_profile_sound_level_value(profile_name, PROFILEKEY_TOUCHSCREEN_SOUND_LEVEL, data->touchscreensoundLevelButton);
  save_profile_sound_level_value(profile_name, PROFILEKEY_SYSTEM_SOUND_LEVEL, data->systemsoundLevelButton);
  save_profile_checkbutton_value(profile_name, PROFILEKEY_VIBRATING_ALERT_ENABLED, data->vibratingCheckbutton);
  //  save_profile_silent_checkbutton_value(profile_name, PROFILEKEY_RINGING_ALERT_TYPE, data->silentCheckbutton);
  /* unused
  save_profile_checkbutton_value(profile_name, PROFILEKEY_CLOCK_ALARM_ENABLED, data->clockalarmCheckbutton);
  save_profile_checkbutton_value(profile_name, PROFILEKEY_CALENDAR_ALARM_ENABLED, data->calendaralarmCheckbutton);
  */
  save_profile_status_icon_value(profile_name, data->iconButton);
}

void add_tone_to_list_store(GtkButton* button, GtkWidget* selector)
{
  GtkWidget* dialog;
  GtkWidget* parent = gtk_widget_get_parent(GTK_WIDGET(button));
  GtkListStore* list_store = GTK_LIST_STORE(hildon_touch_selector_get_model(HILDON_TOUCH_SELECTOR(selector), 0));
										    
  parent = gtk_widget_get_parent(parent);
  dialog = hildon_file_chooser_dialog_new(GTK_WINDOW(parent),
					  GTK_FILE_CHOOSER_ACTION_OPEN);
  GtkFileFilter* fileFilter = gtk_file_filter_new();
  gtk_file_filter_add_mime_type(fileFilter, "audio/mpeg3");
  gtk_file_filter_add_mime_type(fileFilter, "audio/mp3");
  gtk_file_filter_add_mime_type(fileFilter, "audio/aac");
  gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(dialog), fileFilter);
  gtk_widget_show_all(GTK_WIDGET(dialog));
  
  int ret = gtk_dialog_run(GTK_DIALOG(dialog));
  if(ret == GTK_RESPONSE_OK)
  {
    gchar* filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    GtkTreeIter iter;
    gchar* display_value = get_display_value(filename, list_store, &iter);

    hildon_touch_selector_select_iter(HILDON_TOUCH_SELECTOR(selector),
				      0, &iter, TRUE);
    g_free(filename);
    g_free(display_value);
  }
  gtk_widget_destroy(dialog);
}

void
play_sound_sample(HildonTouchSelector* selector, gint column, editor_data_t* user_data)
{
  GtkTreeIter iter;
  hildon_touch_selector_get_selected(selector,
				     0,
				     &iter);
  gchar* list_value;
  gtk_tree_model_get(hildon_touch_selector_get_model(selector, 0), &iter,
		     1, &list_value,
		     -1);
  play_sound(list_value, user_data);
}

void
show_tone_selector(GtkListStore* list_store, GtkWidget* button, GtkWidget* selector, editor_data_t* data)
{
  GtkTreeIter iter;
  GtkWidget* more_button;
  get_iter_for_value(list_store, hildon_button_get_value(HILDON_BUTTON(button)), &iter);
  
  hildon_touch_selector_select_iter(HILDON_TOUCH_SELECTOR(selector),
				    0,
				    &iter, TRUE);
  int sound_handler_id = g_signal_connect(selector, "changed", G_CALLBACK(play_sound_sample), data);
  GtkWidget* dialog = gtk_dialog_new_with_buttons(dgettext("osso-profiles", "profi_fi_ringingtone_tone"),
						  NULL,
						  GTK_DIALOG_MODAL,
						  dgettext("hildon-libs", "wdgt_bd_save"),
						  GTK_RESPONSE_OK,
						  NULL);
  more_button = hildon_button_new(HILDON_SIZE_AUTO,
				  HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  gtk_button_set_label(GTK_BUTTON(more_button), dgettext("hildon-libs", "wdgt_bd_add"));
  gtk_container_add(GTK_CONTAINER(gtk_dialog_get_action_area(GTK_DIALOG(dialog))), more_button);
  g_signal_connect(more_button, "clicked", G_CALLBACK(add_tone_to_list_store), selector);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), selector);

  gtk_window_set_default_size(GTK_WINDOW(dialog), -1, 300);  
  gtk_widget_show_all(dialog);
  int ret = gtk_dialog_run(GTK_DIALOG(dialog));
  if(ret == GTK_RESPONSE_OK)
  {
    GtkTreeIter iter;
    hildon_touch_selector_get_selected(HILDON_TOUCH_SELECTOR(selector), 0, &iter);
    gchar* v;
    gtk_tree_model_get(GTK_TREE_MODEL(list_store),
		       &iter,0, &v, -1);
    hildon_button_set_value(HILDON_BUTTON(button), v);
    g_free(v);
  }
  //  g_signal_handler_disconnect(selector, sound_handler_id);
  ret = gst_element_set_state(data->sound_player, GST_STATE_NULL);
  GstState state; 
  GstState pending; 
  int stret = gst_element_get_state(data->sound_player, &state, &pending, GST_CLOCK_TIME_NONE); 
  gtk_widget_destroy(dialog);
}

GtkWidget* create_tone_selector_widget(GtkListStore* list_store)
{
  GtkWidget* widget = hildon_touch_selector_new();
  GtkCellRenderer* renderer;
  HildonTouchSelectorColumn* column;

  renderer = gtk_cell_renderer_text_new(); 
  column = hildon_touch_selector_append_column(HILDON_TOUCH_SELECTOR(widget), 
 					       GTK_TREE_MODEL(list_store), 
 					       renderer, 
 					       "text", 0, NULL); 

  g_object_set(G_OBJECT(column), 
 	       "text-column", 0, 
 	       NULL); 
  return widget; 
}

void
show_ringtone_selector(GtkWidget* button, editor_data_t *data)
{
  g_object_set(data->sound_player, "volume", gtk_range_get_value(GTK_RANGE(data->ringtoneVolumeSlider))/100.0);
  data->ringtoneSelector=create_tone_selector_widget(data->ringingTone);
  show_tone_selector(data->ringingTone, data->ringtoneButton, data->ringtoneSelector, data);
}
void
show_smstone_selector(GtkWidget* button, editor_data_t *data)
{
  g_object_set(data->sound_player, "volume", gtk_range_get_value(GTK_RANGE(data->smsVolumeSlider))/100.0);
  data->smstoneSelector=create_tone_selector_widget(data->smsTone);
  show_tone_selector(data->smsTone, data->smstoneButton, data->smstoneSelector, data);
}

void
show_imtone_selector(GtkWidget* button, editor_data_t *data)
{
  g_object_set(data->sound_player, "volume", gtk_range_get_value(GTK_RANGE(data->imVolumeSlider))/100.0);
  data->imtoneSelector=create_tone_selector_widget(data->imTone);
  show_tone_selector(data->imTone, data->imtoneButton, data->imtoneSelector, data);
}

void
show_emailtone_selector(GtkWidget* button, editor_data_t *data)
{
  g_object_set(data->sound_player, "volume", gtk_range_get_value(GTK_RANGE(data->emailVolumeSlider))/100.0);
  data->emailtoneSelector=create_tone_selector_widget(data->emailTone);
  show_tone_selector(data->emailTone, data->emailtoneButton, data->emailtoneSelector, data);
}


void init_editor_tone_selector_widgets(editor_data_t* app_data)
{
  app_data->imTone = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
  app_data->emailTone = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
  app_data->ringingTone = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
  app_data->smsTone = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
  read_predefined_ringtones(app_data);
  app_data->ringtoneButton = hildon_button_new(HILDON_SIZE_FINGER_HEIGHT,
					       HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  app_data->smstoneButton = hildon_button_new(HILDON_SIZE_FINGER_HEIGHT,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  app_data->imtoneButton = hildon_button_new(HILDON_SIZE_FINGER_HEIGHT,
					     HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  app_data->emailtoneButton = hildon_button_new(HILDON_SIZE_FINGER_HEIGHT,
						HILDON_BUTTON_ARRANGEMENT_VERTICAL);

  hildon_button_set_title(HILDON_BUTTON(app_data->ringtoneButton),
			  dgettext("osso-profiles", "profi_fi_ringingtone_tone"));
  hildon_button_set_title(HILDON_BUTTON(app_data->emailtoneButton),
			  dgettext("osso-profiles", "profi_ti_email_alert_tone"));
  hildon_button_set_title(HILDON_BUTTON(app_data->smstoneButton),
			  dgettext("osso-profiles", "profi_ti_sub_text_message_tone"));
  hildon_button_set_title(HILDON_BUTTON(app_data->imtoneButton),
			  dgettext("osso-profiles", "profi_ti_instant_messaging_tone"));
    
  hildon_button_set_alignment(HILDON_BUTTON(app_data->ringtoneButton),
			      0, 0,
			      1, 1);
  hildon_button_set_alignment(HILDON_BUTTON(app_data->emailtoneButton),
			      0, 0,
			      1, 1);
  hildon_button_set_alignment(HILDON_BUTTON(app_data->smstoneButton),
			      0, 0,
			      1, 1);
  hildon_button_set_alignment(HILDON_BUTTON(app_data->imtoneButton),
			      0, 0,
			      1, 1);
  hildon_button_set_style(HILDON_BUTTON (app_data->ringtoneButton), HILDON_BUTTON_STYLE_PICKER);
  hildon_button_set_style(HILDON_BUTTON (app_data->emailtoneButton), HILDON_BUTTON_STYLE_PICKER);
  hildon_button_set_style(HILDON_BUTTON (app_data->smstoneButton), HILDON_BUTTON_STYLE_PICKER);
  hildon_button_set_style(HILDON_BUTTON (app_data->imtoneButton), HILDON_BUTTON_STYLE_PICKER);

  g_signal_connect(app_data->ringtoneButton, "clicked", G_CALLBACK(show_ringtone_selector), app_data);
  g_signal_connect(app_data->smstoneButton, "clicked", G_CALLBACK(show_smstone_selector), app_data);
  g_signal_connect(app_data->imtoneButton, "clicked", G_CALLBACK(show_imtone_selector), app_data);
  g_signal_connect(app_data->emailtoneButton, "clicked", G_CALLBACK(show_emailtone_selector), app_data);
  
}

void init_tone_volume_slider(GtkWidget** slider)
{
  GtkWidget* widget = hildon_gtk_hscale_new();
  gtk_scale_set_digits(GTK_SCALE(widget), 0);
  gtk_adjustment_configure(gtk_range_get_adjustment(GTK_RANGE(widget)),
			   0,
			   0,
			   100,
			   1,
			   0,
			   0);
			   
  (*slider) = widget;
}

void init_editor_tone_volume_widgets(editor_data_t* app_data)
{
  init_tone_volume_slider(&app_data->ringtoneVolumeSlider);
  init_tone_volume_slider(&app_data->smsVolumeSlider);
  init_tone_volume_slider(&app_data->emailVolumeSlider);
  init_tone_volume_slider(&app_data->imVolumeSlider);
}

void init_checkbutton_widget(GtkWidget** checkbutton, gchar* title)
{
  GtkWidget* widget = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(widget),
		       title);
  hildon_check_button_set_active(HILDON_CHECK_BUTTON(widget), TRUE);
  (*checkbutton) = widget;
}

void init_editor_checkbutton_widgets(editor_data_t* app_data)
{
  init_checkbutton_widget(&app_data->vibratingCheckbutton,
			  dgettext("osso-profiles", "profi_fi_general_vibrate"));
/*   init_checkbutton_widget(&app_data->silentCheckbutton, */
/* 			  dgettext("osso-profiles", "profi_bd_silent")); */
  /* unused
  init_checkbutton_widget(&app_data->clockalarmCheckbutton,
			  dgettext("osso-clock","cloc_ti_alarms"));
  init_checkbutton_widget(&app_data->calendaralarmCheckbutton,
			  dgettext("calendar", "cal_ap_name"));
  */
}

GtkWidget* create_sound_level_widget(GtkListStore* list_store, gchar* title)
{
  GtkWidget* selector;
  GtkCellRenderer* renderer;
  HildonTouchSelectorColumn* column;
  GtkWidget* widget = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT,
					       HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_alignment(HILDON_BUTTON(widget),
			      0, 0,
			      1, 1);

  renderer = gtk_cell_renderer_text_new();
  selector = hildon_touch_selector_new();
  column = hildon_touch_selector_append_column(HILDON_TOUCH_SELECTOR(selector),
					       GTK_TREE_MODEL(list_store), renderer,
					       "text", 0, NULL);
  g_object_set(G_OBJECT(column), "text-column", 0, NULL);
  
  hildon_button_set_title(HILDON_BUTTON(widget),
			  title);
  hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(widget), 
				    HILDON_TOUCH_SELECTOR(selector));
  return widget;
}

void init_sound_level_list_store(GtkListStore* list_store)
{
  GtkTreeIter iter;
  gtk_list_store_append(list_store, &iter);
  gtk_list_store_set(list_store, &iter, 0, dgettext("osso-profiles", "profi_va_key_sounds_off"), 1, 0, -1);
  gtk_list_store_append(list_store, &iter);
  gtk_list_store_set(list_store, &iter, 0, dgettext("osso-profiles", "profi_va_key_sounds_level1"), 1, 1, -1);
  gtk_list_store_append(list_store, &iter);
  gtk_list_store_set(list_store, &iter, 0, dgettext("osso-profiles", "profi_va_key_sounds_level2"), 1, 2, -1);
}

void init_editor_sound_level_widgets(editor_data_t* app_data)
{
  app_data->keypadsoundLevel = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);
  app_data->touchscreensoundLevel = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);
  app_data->systemsoundLevel = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);

  init_sound_level_list_store(app_data->keypadsoundLevel);
  init_sound_level_list_store(app_data->systemsoundLevel);
  init_sound_level_list_store(app_data->touchscreensoundLevel);

  app_data->keypadsoundLevelButton = 
    create_sound_level_widget(app_data->keypadsoundLevel,
			      dgettext("osso-profiles", "profi_fi_key_sounds"));
  app_data->touchscreensoundLevelButton = 
    create_sound_level_widget(app_data->touchscreensoundLevel,
			      dgettext("osso-profiles", "profi_fi_touch_screen_sounds"));

  app_data->systemsoundLevelButton = 
    create_sound_level_widget(app_data->systemsoundLevel,
			      dgettext("osso-profiles", "profi_fi_system_sounds"));
}

void load_custom_icon(GtkButton* button, editor_data_t* app_data)
{
  if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)))
  {
    GtkWidget* dialog;
    GtkWidget* parent = gtk_widget_get_toplevel(GTK_WIDGET(button));
    dialog = hildon_file_chooser_dialog_new(GTK_WINDOW(parent),
					    GTK_FILE_CHOOSER_ACTION_OPEN);
    GtkFileFilter* fileFilter = gtk_file_filter_new();
    //    gtk_file_filter_add_pattern(fileFilter, "*.png");
    gtk_file_filter_add_pixbuf_formats(fileFilter);
    gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(dialog), fileFilter);

    gtk_widget_show_all(GTK_WIDGET(dialog));
    int ret = gtk_dialog_run(GTK_DIALOG(dialog));
    if(ret == GTK_RESPONSE_OK)
    {
      gchar* filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
      GtkWidget* image = gtk_image_new_from_file(filename);
      g_object_set(G_OBJECT(image), "name", filename);
      gtk_button_set_image(button, image);
      //      g_free(filename);
    }
    gtk_widget_destroy(dialog);
  }
}

void icon_selection_dialog(GtkButton* button, editor_data_t* app_data)
{
  GtkWidget* dialog = gtk_dialog_new_with_buttons("Statusbar icon",
						  NULL,
						  GTK_DIALOG_MODAL,
						  dgettext("hildon-libs", "wdgt_bd_save"),
						  GTK_RESPONSE_OK,
						  NULL);
  GtkWidget* icon_box = gtk_hbox_new(TRUE, 0);
  GtkWidget* icon_button_none = hildon_gtk_radio_button_new_from_widget(HILDON_SIZE_FINGER_HEIGHT,
									NULL);
  GtkWidget* icon_button_silent = hildon_gtk_radio_button_new_from_widget(HILDON_SIZE_FINGER_HEIGHT,
									  GTK_RADIO_BUTTON(icon_button_none));
  GtkWidget* icon_button_home = hildon_gtk_radio_button_new_from_widget(HILDON_SIZE_FINGER_HEIGHT,
									GTK_RADIO_BUTTON(icon_button_silent));
  GtkWidget* icon_button_im = hildon_gtk_radio_button_new_from_widget(HILDON_SIZE_FINGER_HEIGHT,
								      GTK_RADIO_BUTTON(icon_button_home));
  GtkWidget* icon_button_outside = hildon_gtk_radio_button_new_from_widget(HILDON_SIZE_FINGER_HEIGHT,
									   GTK_RADIO_BUTTON(icon_button_im));
  GtkWidget* icon_button_loud = hildon_gtk_radio_button_new_from_widget(HILDON_SIZE_FINGER_HEIGHT,
									GTK_RADIO_BUTTON(icon_button_outside));
  GtkWidget* icon_button_custom = hildon_gtk_radio_button_new_from_widget(HILDON_SIZE_FINGER_HEIGHT,
									  GTK_RADIO_BUTTON(icon_button_loud));
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(icon_button_none), FALSE);
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(icon_button_silent), FALSE);
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(icon_button_home), FALSE);
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(icon_button_im), FALSE);
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(icon_button_outside), FALSE);
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(icon_button_loud), FALSE);
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(icon_button_custom), FALSE);

  //  gtk_button_set_image(GTK_BUTTON(icon_button_none), gtk_image_new_from_icon_name("statusarea_silent", 18));
  gtk_button_set_image(GTK_BUTTON(icon_button_silent), gtk_image_new_from_icon_name("statusarea_silent", 18));
  gtk_button_set_image(GTK_BUTTON(icon_button_home), gtk_image_new_from_icon_name("statusarea_profilesx_home", 18));
  gtk_button_set_image(GTK_BUTTON(icon_button_im), gtk_image_new_from_icon_name("statusarea_profilesx_im", 18));
  gtk_button_set_image(GTK_BUTTON(icon_button_outside), gtk_image_new_from_icon_name("statusarea_profilesx_outside", 18));
  gtk_button_set_image(GTK_BUTTON(icon_button_loud), gtk_image_new_from_icon_name("statusarea_profilesx_loud", 18));
  gtk_button_set_image(GTK_BUTTON(icon_button_custom), gtk_image_new_from_icon_name("statusarea_profilesx_custom", 18));
  gtk_container_add(GTK_CONTAINER(icon_box), icon_button_none);
  gtk_container_add(GTK_CONTAINER(icon_box), icon_button_silent);
  gtk_container_add(GTK_CONTAINER(icon_box), icon_button_home);
  gtk_container_add(GTK_CONTAINER(icon_box), icon_button_im);
  gtk_container_add(GTK_CONTAINER(icon_box), icon_button_outside);
  gtk_container_add(GTK_CONTAINER(icon_box), icon_button_loud);
  gtk_container_add(GTK_CONTAINER(icon_box), icon_button_custom);
  g_signal_connect(icon_button_custom, "clicked", G_CALLBACK(load_custom_icon), app_data);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), icon_box);
  
  gtk_widget_show_all(dialog);
  int ret = gtk_dialog_run(GTK_DIALOG(dialog));
  if(ret == GTK_RESPONSE_OK)
  {
    GSList* list = gtk_radio_button_get_group(GTK_RADIO_BUTTON(icon_button_custom));
    gboolean found = FALSE;
    while(list && !found)
    {
      found = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(list->data));
      if(found)
      {
	gtk_button_set_image(button, gtk_button_get_image(GTK_BUTTON(list->data)));
      }
      list = list->next;
    }
    
  }
  gtk_widget_destroy(dialog);
}

void init_editor_status_icon_widget(editor_data_t* app_data)
{
  app_data->iconButton = hildon_button_new(HILDON_SIZE_FINGER_HEIGHT,
					   HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  g_signal_connect(app_data->iconButton, "clicked", G_CALLBACK(icon_selection_dialog), app_data);
}

void show_editor_dialog(const gchar* profile_name, GtkWidget* main_dialog, GstElement* sound_player)
{
  GtkWidget* dialog;
  GtkWidget* content_area;
  GtkWidget* pannable_area;
  editor_data_t* data = g_new0(editor_data_t, 1);


  init_editor_status_icon_widget(data);
  init_editor_tone_selector_widgets(data);
  init_editor_tone_volume_widgets(data);
  init_editor_sound_level_widgets(data);
  init_editor_checkbutton_widgets(data);
  set_profile_values(profile_name, data);
  data->sound_player = sound_player;
  dialog = gtk_dialog_new_with_buttons(get_profile_display_name(profile_name),
				       GTK_WINDOW(main_dialog),
				       GTK_DIALOG_MODAL,
				       dgettext("hildon-libs", "wdgt_bd_save"), GTK_RESPONSE_OK,
				       NULL);
  
  gtk_window_set_default_size(GTK_WINDOW(dialog), -1, 300);
  content_area = gtk_vbox_new(FALSE, 3);
  GtkWidget* iconBox = gtk_hbox_new(TRUE, 0);
  GtkWidget* ringtoneBox = gtk_hbox_new(TRUE, 0);
  GtkWidget* smsBox = gtk_hbox_new(TRUE, 0);
  GtkWidget* imBox = gtk_hbox_new(TRUE, 0);
  GtkWidget* emailBox = gtk_hbox_new(TRUE, 0);
  GtkWidget* icon_label = gtk_label_new("Statusbar icon");
  
  gtk_box_pack_start(GTK_BOX(iconBox), icon_label, TRUE, TRUE, 0); 
  gtk_box_pack_end(GTK_BOX(iconBox), data->iconButton, TRUE, TRUE, 0); 
  gtk_box_pack_start(GTK_BOX(ringtoneBox), data->ringtoneButton, TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(ringtoneBox), data->ringtoneVolumeSlider, TRUE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(smsBox), data->smstoneButton, TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(smsBox), data->smsVolumeSlider, TRUE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(imBox), data->imtoneButton, TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(imBox), data->imVolumeSlider, TRUE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(emailBox), data->emailtoneButton, TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(emailBox), data->emailVolumeSlider, TRUE, TRUE, 0);

  gtk_container_add(GTK_CONTAINER(content_area), iconBox);
  gtk_container_add(GTK_CONTAINER(content_area), data->vibratingCheckbutton);
  gtk_container_add(GTK_CONTAINER(content_area), data->silentCheckbutton);
  gtk_container_add(GTK_CONTAINER(content_area), ringtoneBox);
  gtk_container_add(GTK_CONTAINER(content_area), smsBox);
  gtk_container_add(GTK_CONTAINER(content_area), imBox);
  gtk_container_add(GTK_CONTAINER(content_area), emailBox);
  gtk_container_add(GTK_CONTAINER(content_area), data->keypadsoundLevelButton);
  gtk_container_add(GTK_CONTAINER(content_area), data->touchscreensoundLevelButton);
  gtk_container_add(GTK_CONTAINER(content_area), data->systemsoundLevelButton);

  /* unused
  gtk_container_add(GTK_CONTAINER(content_area), data->clockalarmCheckbutton);
  gtk_container_add(GTK_CONTAINER(content_area), data->calendaralarmCheckbutton);
  */
  pannable_area = hildon_pannable_area_new();
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pannable_area), content_area);
  gtk_container_add(GTK_CONTAINER(gtk_dialog_get_content_area(GTK_DIALOG(dialog))),
		    pannable_area);
  gtk_widget_show_all(dialog);

  int ret = gtk_dialog_run(GTK_DIALOG(dialog));
  if(ret == GTK_RESPONSE_OK)
  {
    save_profile_values(profile_name, data);
  }
  gtk_widget_destroy(dialog);
  gtk_list_store_clear(data->imTone);
  gtk_list_store_clear(data->smsTone);
  gtk_list_store_clear(data->ringingTone);
  gtk_list_store_clear(data->emailTone);
  gtk_list_store_clear(data->keypadsoundLevel);
  gtk_list_store_clear(data->touchscreensoundLevel);
  gtk_list_store_clear(data->systemsoundLevel);
  g_object_unref(data->imTone);
  g_object_unref(data->smsTone);
  g_object_unref(data->ringingTone);
  g_object_unref(data->emailTone);
  g_object_unref(data->keypadsoundLevel);
  g_object_unref(data->touchscreensoundLevel);
  g_object_unref(data->systemsoundLevel);
  g_free(data);
}
