#include <hildon/hildon.h>
#include "intl.h"
#include "query-text.h"
#include "query-dialog.h"
#include "query-tree-model.h"

#define OSSO_BROWSER_SERVICE    "com.nokia.osso_browser"
#define OSSO_BROWSER_PATH       "/com/nokia/osso_browser/request"
#define OSSO_BROWSER_IFACE      "com.nokia.osso_browser"
#define OSSO_BROWSER_OPEN_URL   "load_url"
#define OSSO_BROWSER_NEW_WINDOW "open_new_window"

static void
open_browser(HDHomePluginItem *hdhpi, const char *str)
{
	DBusError err;
	DBusConnection *dbus_connection = NULL;

	dbus_error_init(&err);
	dbus_connection = hd_home_plugin_item_get_dbus_connection(hdhpi, DBUS_BUS_SESSION, &err);
	if (dbus_error_is_set(&err)) {
		g_warning("run_query_button_clicked_cb: Failed to grab session bus: %s\n", err.message);
		dbus_error_free(&err);
	}

	if (dbus_connection) {
		DBusMessage *msg = dbus_message_new_method_call(
			OSSO_BROWSER_SERVICE,
			OSSO_BROWSER_PATH,
			OSSO_BROWSER_IFACE,
			OSSO_BROWSER_NEW_WINDOW);

		if (msg) {
			if (dbus_message_append_args(msg, DBUS_TYPE_STRING, &str, DBUS_TYPE_INVALID)) {
				dbus_connection_send(dbus_connection, msg, NULL);
				dbus_connection_flush(dbus_connection);
			}
			dbus_message_unref(msg);
		}
		dbus_connection_close(dbus_connection);
		dbus_connection_unref(dbus_connection);
	}
}

static void
run_query_button_clicked_cb(GtkWidget *button, GtkTextView *text)
{
	GtkWidget *parent, *toplevel;
	char *str = NULL;
	GtkTextBuffer *buffer = gtk_text_view_get_buffer(text);
	GtkTextIter beg, end;
	HDHomePluginItem *hdhpi = g_object_get_data(G_OBJECT(button), "hdhpi");

	gtk_text_buffer_get_start_iter(buffer, &beg);
	gtk_text_buffer_get_end_iter(buffer, &end);

	str = gtk_text_buffer_get_text(buffer, &beg, &end, FALSE);
	open_browser(hdhpi, str);
	g_free(str);

	toplevel = gtk_widget_get_toplevel(button);
	if (toplevel && GTK_IS_WINDOW(toplevel))
		parent = GTK_WIDGET(gtk_window_get_transient_for(GTK_WINDOW(toplevel)));
	if (toplevel)
		gtk_widget_destroy(toplevel);
	if (parent)
		gtk_widget_destroy(parent);
}

static void
unset_borders(GtkWidget *widget)
{
	if (GTK_IS_CONTAINER(widget)) {
		g_object_set(G_OBJECT(widget), "border-width", 0, NULL);
		if (GTK_IS_BOX(widget))
			g_object_set(G_OBJECT(widget), "spacing", 0, NULL);
		else
		if (GTK_IS_ALIGNMENT(widget))
			g_object_set(G_OBJECT(widget),
				"xscale", 1.0, "yscale", 1.0,
				"left-padding", 0, "right-padding", 0, 
				"top-padding", 0, "bottom-padding", 0, NULL);
		gtk_container_foreach(GTK_CONTAINER(widget), (GtkCallback)unset_borders, NULL);
	}
}

static void
run_query(GtkWidget *parent, GtkTreeModel *tm, GtkTreeIter *itr, HDHomePluginItem *hdhpi)
{
	char *name, *query;

	gtk_tree_model_get(tm, itr,
		QUERY_TREE_MODEL_NAME,  &name,
		QUERY_TREE_MODEL_QUERY, &query,
		-1);

	if (g_strstr_len(query, -1, "{}")) {
		GtkWidget *dlg;
		GtkWidget *text, *button;

		dlg = g_object_new(GTK_TYPE_DIALOG, "transient-for", parent, "title", name, NULL);
	//	gtk_widget_set_size_request(dlg, -1, 420);

		text = query_text_new(query);
		gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dlg)->vbox), text);

		button = g_object_new(HILDON_TYPE_BUTTON,
			"size",        HILDON_SIZE_FINGER_HEIGHT,
			"arrangement", HILDON_BUTTON_ARRANGEMENT_VERTICAL,
			"visible",     TRUE,
			"label",       GTK_STOCK_EXECUTE,
			"use-stock",   TRUE,
			NULL);
		g_object_set_data(G_OBJECT(button), "hdhpi", hdhpi);

		gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), button, FALSE, TRUE, 0);
		g_signal_connect(G_OBJECT(button), "clicked", (GCallback)run_query_button_clicked_cb, text);

		gtk_widget_show(dlg);
	}
	else {
		open_browser(hdhpi, query);
		if (parent)
			gtk_widget_destroy(parent);
	}

	g_free(name);
	g_free(query);
}

typedef struct {
	GtkTreeModel *tm;
	guint row_inserted_id;
	guint row_deleted_id;
	GtkTreeView *tv;
	GObject *label;
	GObject *hpa;
} QueryTreeViewParams;

static void
tm_changed(QueryTreeViewParams *qtvp)
{
	GtkTreeModel *tm = gtk_tree_view_get_model(qtvp->tv);
	gboolean has_rows = FALSE;

	if (tm) {
		GtkTreeIter itr;
		has_rows = gtk_tree_model_get_iter_first(tm, &itr);
	}

	g_object_set(qtvp->label, "visible", !has_rows, NULL);
	g_object_set(qtvp->hpa,   "visible",  has_rows, NULL);
}

static void
remove_ids_and_free(QueryTreeViewParams *qtvp)
{
	g_signal_handler_disconnect(qtvp->tm, qtvp->row_deleted_id);
	g_signal_handler_disconnect(qtvp->tm, qtvp->row_inserted_id);
	g_free(qtvp);
}

GtkWidget *
query_dialog_make_tv(GtkTreeModel *tm, GtkWidget **p_tv)
{
	GtkWidget *vbox;
	GtkTreeViewColumn *col;
	GtkCellRenderer *cr;
	QueryTreeViewParams *qtvp = g_new0(QueryTreeViewParams, 1);

	(*p_tv) = g_object_new(GTK_TYPE_TREE_VIEW, "visible", TRUE, "model", tm, NULL);
	qtvp->tv = GTK_TREE_VIEW((*p_tv));
	qtvp->tm = tm;

	col = gtk_tree_view_column_new();

	cr = g_object_new(GTK_TYPE_CELL_RENDERER_PIXBUF, "xpad", HILDON_MARGIN_TRIPLE, NULL);
	gtk_tree_view_column_pack_start(col, cr, FALSE);
	gtk_tree_view_column_set_attributes(col, cr, "pixbuf", QUERY_TREE_MODEL_IMAGE, NULL);

	cr = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col, cr, TRUE);
	gtk_tree_view_column_set_attributes(col, cr, "text", QUERY_TREE_MODEL_NAME, NULL);

	gtk_tree_view_append_column(qtvp->tv, col);

	qtvp->hpa = g_object_new(HILDON_TYPE_PANNABLE_AREA, "visible", TRUE, "child", (*p_tv), NULL);

	qtvp->label = g_object_new(GTK_TYPE_LABEL, "visible", FALSE, "xalign", 0.5, "yalign", 0.5, "label", _("(No Queries)"), NULL);
	hildon_helper_set_logical_font(GTK_WIDGET(qtvp->label), "LargeSystemFont");
	hildon_helper_set_logical_color(GTK_WIDGET(qtvp->label), GTK_RC_FG, GTK_STATE_NORMAL, "SecondaryTextColor");

	vbox = g_object_new(GTK_TYPE_VBOX, "visible", TRUE, NULL);

	gtk_container_add(GTK_CONTAINER(vbox), GTK_WIDGET(qtvp->label));
	gtk_container_add(GTK_CONTAINER(vbox), GTK_WIDGET(qtvp->hpa));

	qtvp->row_inserted_id = g_signal_connect_swapped(G_OBJECT(tm), "row-inserted", (GCallback)tm_changed, qtvp);
	qtvp->row_deleted_id  = g_signal_connect_swapped(G_OBJECT(tm), "row-deleted",  (GCallback)tm_changed, qtvp);
	tm_changed(qtvp);

	g_object_set_data_full(G_OBJECT(vbox), "qtvp", qtvp, (GDestroyNotify)remove_ids_and_free);

	return vbox;
}

static void
tv_row_activated(GtkWidget *tv, GtkTreePath *tp, GtkTreeViewColumn *col, HDHomePluginItem *hdhpi)
{
	GtkTreeIter itr;
	GtkTreeModel *tm = gtk_tree_view_get_model(GTK_TREE_VIEW(tv));
	if (gtk_tree_model_get_iter(tm, &itr, tp))
		run_query(gtk_widget_get_toplevel(tv), tm, &itr, hdhpi);
}

void
query_dialog_show(HDHomePluginItem *hdhpi, GtkTreeModel *tm)
{
	GtkWidget *query_tv;
	GtkWidget *tv;
	GtkWidget *dlg;

	dlg = g_object_new(GTK_TYPE_DIALOG, "visible", TRUE, "title", _("Query"), NULL);
	gtk_widget_set_size_request(dlg, -1, 420);
	unset_borders(dlg);

	query_tv = query_dialog_make_tv(tm, &tv);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dlg)->vbox), query_tv);

	g_signal_connect(G_OBJECT(tv), "row-activated", (GCallback)tv_row_activated, hdhpi);
	g_signal_connect(G_OBJECT(dlg), "response", (GCallback)gtk_widget_destroy, NULL);
}

