// -*- coding: utf-8; -*-
// (c) Copyright 2010, Nick Slobodsky (Николай Слободской)
// This file is part of PlansPlant.
//
// PlansPlant is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// PlansPlant is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PlansPlant.  If not, see <http://www.gnu.org/licenses/>.
//
#ifndef PLANS_PLANT_TASKS_HPP
#define PLANS_PLANT_TASKS_HPP
#include <QMap>
#include <QList>
#include <QQueue>
#include <QString>
#include <QDateTime>
#include <QXmlStreamReader>
#include <QXmlStreamWriter>
#if defined( PLANSPLANT_LIB_SHARED )
#define PLANSPLANT_CLASS Q_DECL_EXPORT
#elif defined( PLANSPLANT_STATIC )
#define PLANSPLANT_CLASS
#else // Use dynamic library by default.
#define PLANSPLANT_CLASS Q_DECL_IMPORT
#endif
#define PLANSPLANT_NO_UNDO

namespace PlansPlant
{
  //! \todo Move time objects and functions to Timeshop.
  typedef int Time;
  class TasksFile;
  QDateTime read_time( QXmlStreamReader& Stream );
  QString time_string( const QDateTime& TimeVal );
  class PLANSPLANT_CLASS Task
  {
  public:
    typedef Task* Pointer;
    typedef QList<Pointer> List; //!< \todo Try to use vector to use less memory.
    class ID
    {
    public:
      ID( quint32 Value0 = 0, int Origin0 = 0 );
      ID( const QString& Str );
      bool valid() const { return /* Origin >= 0 && */ Value > 0; }
      int origin() const { return Origin; }
      int seq() const { return Value; }
      operator bool() const { return valid(); }
      ID operator=( const QString& Str );
      QString str() const;
      operator QString() const { return str(); }
      bool operator==( const ID& Other ) const { return Origin == Other.Origin && Value == Other.Value; }
      bool operator!=( const ID& Other ) const { return !( Other == *this ); }
      bool operator<( const ID& Other ) const { return Origin < Other.Origin || ( Origin == Other.Origin && Value < Other.Value ); }
      bool operator>( const ID& Other ) const { return Other < *this; }
      bool operator>=( const ID& Other ) const { return !( Origin < Other.Origin ); }
      bool operator<=( const ID& Other ) const { return !( Origin > Other.Origin ); }
    protected:
      //! \note Uncomment Origin check in "valid()" method above if you change type to signed.
      quint32 Origin : 8;
      quint32 Value : 24;
    }; // ID
    typedef QMap<ID,Pointer> Map;
    class ChangesList;
    class Watcher;
    // Info about some change in task.
    class PLANSPLANT_CLASS Change
    {
    public:
      typedef Change* Pointer;
      typedef QList<Pointer> List;
      enum FieldID
      { None, Name, SuperTask, Description, Comment, Completed, SubTasks, Blockers, Dependents, PlanStart, PlanFinish, Estimation, EstimationUnits, Priority, Times, FieldsNum };
      Change( FieldID Field0 = None ) : Field( Field0 ) {}
      virtual ~Change() {}
      FieldID field() const { return Field; } // Maybe use virtual function and don't save data.
      void operator()( Task& Object, TasksFile& File ) const { apply( Object, File ); }
      virtual void apply( Task& Object, TasksFile& File ) const = 0;
#ifndef PLANSPLANT_NO_UNDO
      virtual void save_undo( ChangesList& Undo ) const = 0;
      inline void apply_with_undo( ChangesList& Undo ) const;
#endif // PLANSPLANT_NO_UNDO
      virtual void inform( Watcher& Dest, Task& Object ) const;
      virtual void write( QXmlStreamWriter& Stream ) const;
    protected:
      virtual void write_fields( QXmlStreamWriter& Stream ) const;
      FieldID Field;
    }; // Change
    class Changes; // Internal namespace for changes objects (derived from Change). Defined in file tasks_changes.hpp.
    class PLANSPLANT_CLASS ChangesList
    {
    public:
#warning Add copy constructor and operator=
      ChangesList( Task& Object0 ) : Object( Object0 ) {}
      ChangesList( Task& Object0, Change& FirstChange );
      ~ChangesList();
      const Task& task() const { return Object; }
      Task& task() { return Object; }
      const Change::List& changes() const { return Changes; }
      void add( Change& NewChange ) { Changes.push_back( &NewChange ); }
      void inform( Watcher& Dest ) const;
      bool load( QXmlStreamReader& Stream, TasksFile& Tasks );
      void write( QXmlStreamWriter& Stream ) const;
    protected:
      Task& Object;
      Change::List Changes;
    }; // ChangesList
    // Watcher receives notifications about tasks changes.
    class PLANSPLANT_CLASS Watcher
    {
    public:
      typedef Watcher* Pointer;
      typedef QList<Pointer> List;
      class Event
      {
      public:
	typedef Event* Pointer;
	typedef Task::ID ID;
	enum Type { None, TaskAdded, TaskRemoved, TaskChanged, RootTaskMoved };
	Event( ID Ident0 ) : Ident( Ident0 ) {}
	virtual ~Event() {}
	ID id() const { return Ident; }
	virtual Type type() const;
	virtual void apply( TasksFile& Tasks ) const;
	virtual void inform( Watcher& Dest ) const;
	virtual void write( QXmlStreamWriter& Stream ) const;
      protected:
	virtual void write_fields( QXmlStreamWriter& Stream ) const;
	ID Ident;
      }; // Event
      class Events;
    protected:
      virtual ~Watcher();
    public:
      void add_informer( TasksFile& NewInformer );
      void remove_informer( TasksFile& OldInformer );
      virtual void file_closed( TasksFile& ClosedFile ) { remove_informer( ClosedFile ); }
      virtual void event( const Event& Ev );
      virtual void task_added( Task& NewTask );
      virtual void task_removed( Task::ID TaskID ); //!< This method is called right before the Task object destruction.
      virtual void task_changed( const ChangesList& Changes );
      virtual void task_changed( Task& Object, Task::Change::FieldID Field );
      virtual void task_moved( const Task& Object, int OldIndex, int NewIndex );
      virtual void blocker_added( const Task& Object, Task& NewBlocker );
      virtual void blocker_removed( const Task& Object, const Task& OldBlocker );
      virtual void blocker_moved( const Task& Object, const Task& Blocker, int OldIndex, int NewIndex );
      virtual void blockers_replaced( const Task& Object );
    protected:
      QList<TasksFile*> Informers;
    }; // Watcher
    class PLANSPLANT_CLASS TimeSlice
    {
    public:
      typedef QList<TimeSlice> List;
      TimeSlice( const QDateTime& Start0 = QDateTime(), const QDateTime& Finish0 = QDateTime() ) : Start( Start0 ), Finish( Finish0 ) {}
      const QDateTime& start() const { return Start; }
      void start( const QDateTime& NewStart );
      void start_now();
      const QDateTime& finish() const { return Finish; }
      void finish( const QDateTime& NewFinish );
      void finish_now();
      bool active() const { return Start.isValid() && !Finish.isValid(); }
      int secs() const { return ( Start.isValid() && Finish.isValid() ) ? Start.secsTo( Finish ) : 0; }
      void load( QXmlStreamReader& Stream );
      void write( QXmlStreamWriter& Stream ) const;
    protected:
      QDateTime Start;
      QDateTime Finish;
    }; // TimeSlice
    enum TimeUnits { Seconds = 1, Minutes = 60, Hours = Minutes*60, WorkDays = Hours*8, Days = Hours*24,
		     WorkWeeks = WorkDays*5, Weeks = Days*7, Months = Days*30, Quarters = Months*3, Years = int( Days*365.25 ) };
    static QString units_name( TimeUnits Units ); //!< \todo Save names in a dictionary.
    static QString units_short_name( TimeUnits Units ); //!< \todo Save names in a dictionary.
    static QString priority_name( int Pri );
    static QDateTime add_time( const QDateTime& Start, Time Amount, TimeUnits Units );
    Task( const QString& Name0 = "NoName", Pointer SuperTask0 = 0, ID ID0 = 0 );
    virtual ~Task();
    //*** Accessors
    ID id() const { return Ident; }
    void id( ID NewID ) { if( !Ident.valid() ) Ident = NewID; }
    //! \todo Make all methods that change the object protected (Change's subclasses will call them).
    const QString& name() const { return Name; }
    void name( const QString& NewName ) { Name = NewName; }
    Task* supertask() const { return SuperTask; }
    void supertask( Task* NewTask ); // No check for loops here - do it in the interface objects
    const List& subtasks() const { return SubTasks; }
    Task* subtask( int Index ) const { return Index < 0 ? 0 : ( Index < SubTasks.size() ? SubTasks[ Index ] : 0 ); }
    bool move_subtask( int From, int To );
    // Dependencies
    const List& blockers() const { return Blockers; }
    Task* blocker( int Index ) const { return Index < 0 ? 0 : ( Index < Blockers.size() ? Blockers[ Index ] : 0 ); }
    void add_blocker( Task& Blocker );
    void remove_blocker( Task& Upon );
    bool move_blocker( int From, int To );
    const List& dependents() const { return Dependents; }
    Task* dependent( int Index ) const { return Index < 0 ? 0 : ( Index < Dependents.size() ? Dependents[ Index ] : 0 ); }
    int priority() const { return Priority; }
    void priority( int NewPriority ) { Priority = NewPriority; }
    const QString& description() const { return Description; }
    void description( const QString& NewDescription ) { Description = NewDescription; }
    const QString& comment() const { return Comment; }
    void comment( const QString& NewComment ) { Comment = NewComment; }
    const QDateTime& plan_start() const { return PlanStart; } // When we plan to start this task
    void plan_start( const QDateTime& NewPlanStart ) { PlanStart = NewPlanStart; } // When we plan to start this task
    const QDateTime& plan_finish() const { return PlanFinish; } // When we plan to finish the task
    void plan_finish( const QDateTime& NewPlanFinish ) { PlanFinish = NewPlanFinish; } // When we plan to finish the task
    double completed() const { return Completed; } // 0 = not started 1 = completed
    inline void completed( double NewCompleted );
    Time estimation() const { return Estimation; } // Net time estimation in units (see below), <= 0 - no estimation set
    void estimation( Time NewEstimation ) { Estimation = NewEstimation; }
    TimeUnits estimation_units() const { return EstimationUnits; } // Units for net time estimation
    void estimation_units( TimeUnits NewUnits ) { EstimationUnits = NewUnits; }
    Time estimation_seconds() const { return Estimation * EstimationUnits; } // Net time estimation in seconds
    const TimeSlice::List& times() const { return Times; }
    bool active() const;
    QDateTime start_time() const;
    int secs( const QDateTime& ToTime = QDateTime() ) const;
    void start( const QDateTime& StartTime = QDateTime::currentDateTime() );
    void stop( const QDateTime& StopTime = QDateTime::currentDateTime() );
    void insert_time( const TimeSlice& Slice, int Index = -1 );
    void remove_time( int Index );
    void move_time( int From, int To );
    TimeSlice& time( int Index ) { return Times[ Index ]; }
    //*** Read/write
    void write( QXmlStreamWriter& Stream, bool Standalone = false );
    void write_blockers( QXmlStreamWriter& Stream );
    void load( QXmlStreamReader& Stream, TasksFile& Tasks );
    //*** Helpers
    Task* find_task( Task::ID ForID ) const;
    bool blocked() const;
    bool check_loop( Task& NewChild ) const;
    bool check_loop( const Task::List& Blockers, Task* NewChild = 0 ) const; // Check loop for child with different blockers
    void add_to_map( Map& TasksMap );
  protected:
    bool blockers_list_helper( Task::List& List, const Task::List& Source ) const;
    ID Ident;
    Pointer SuperTask;
    List SubTasks;
    List Blockers;
    List Dependents;
    QString Name;
    QString Description;
    int Priority;
    QString Comment;
    QDateTime PlanStart; // When we plan to start this task
    QDateTime PlanFinish; // When we plan to finish the task
    double Completed; // 0 = not started 1 = completed
    Time Estimation; // Net time estimation in units
    TimeUnits EstimationUnits; // Units for net time estimation. Means seconds in the unit. 
    TimeSlice::List Times;
  }; // Task
  void Task::completed( double NewCompleted )
  {
    Completed = NewCompleted;
    if( Completed < 0 ) Completed = 0;
    else if( Completed > 1 ) Completed = 1;
  } // completed( double )
#ifndef PLANSPLANT_NO_UNDO
  void Task::Change::apply_with_undo( Task::ChangesList& Undo ) const
  {
    save_undo( Undo );
    apply( Undo.task() );
  } // apply_with_undo( ChangesList& ) const
#endif // PLANSPLANT_NO_UNDO
  
  class Synchronizer;
  class PLANSPLANT_CLASS TasksFile
  {
  public:
    TasksFile( const QString& FileName0 = QString() );
    virtual ~TasksFile();
    int id() { return Ident; }
    const QString& uuid() const { return UUID; }
    const QString& file_name() const { return FileName; }
    bool good() const { return Good; }
    bool is_modified() const { return Modified; }
    void modified( bool NewModified = true ) { Modified = NewModified; }
    const Task::List& roots() const { return Roots; }
    Task* find_task( Task::ID ForID ) const;
    Task* active_task() const { return ActiveTask; }
    bool update_root_status( Task& NewRootTask );
    //! \todo Move file loading here:    bool load( const QString& NewFileName = QString() );
    bool save( const QString& NewFileName = QString() );
    // Tasks manipulation (don't change tasks directly or Watchers will not receive signals and synchronization will become broken).
    void add_task( Task& NewTask, const Task::Watcher::Event* Ev = 0 );
    void delete_task( Task& OldTask, const Task::Watcher::Event* Ev = 0 );
    void start_work( Task& ToStart, const QDateTime& When = QDateTime::currentDateTime() );
    void stop_work( const QDateTime& When = QDateTime::currentDateTime() );
    void change_task( Task::ChangesList& Changes, const Task::Watcher::Event* Ev = 0 );
    // These are usual changes to a task. They just generate specific Changes objects, but don't do actual work.
    //! \note Change object will must be NOT used after function returns (it can be saved or destroyed).
    void change_task( Task& Object, Task::Change* Change, const Task::Watcher::Event* Ev = 0 );
    //    bool change_parent( Task* ForTask, Task* NewParent );
    bool move_task( Task& Object, int From, int To, const Task::Watcher::Event* Ev = 0 ); //!< \note First objects is not supertask, but the task to move.
    bool add_blocker( Task& To, Task& Block );
    void remove_blocker( Task& From, Task& Block );
    bool move_blocker( Task& For, Task& Blocker, int From, int To );
    bool change_blockers( Task& Object, const Task::List& NewBlockers, Task* SuperTask = 0 );
    const Task::Watcher::List& watchers() const { return Watchers; }
    Task::Watcher* watcher( int Index ) const { return Index >= 0 && Index < Watchers.size() ? Watchers[ Index ] : 0; }
    void add_watcher( Task::Watcher& NewWatcher );
    void remove_watcher( Task::Watcher& OldWatcher );
  protected:
    void scan_tasks( Task& Obj );
    void broadcast( const Task::Watcher::Event& Ev ) const;
    Task::List Roots;
    Task* ActiveTask;
    QString FileName;
    bool Modified;
    bool Good;
    Task::ID new_id() { return Task::ID( ++LastID, Ident ); }
    Task::Watcher::Event::ID new_event_id() { return Task::Watcher::Event::ID( ++LastEventID, Ident ); }
    int LastID;
    int LastEventID;
    Task::Watcher::List Watchers;
    int Ident;
    QString UUID;
    Synchronizer* Sync;
  }; // TasksFile
} // PlansPlant
#endif // PLANS_PLANT_TASKS_HPP
