// -*- coding: utf-8; -*-
// (c) Copyright 2010, Nick Slobodsky (Николай Слободской)
// This file is part of PlansPlant.
//
// PlansPlant is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// PlansPlant is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PlansPlant.  If not, see <http://www.gnu.org/licenses/>.
//
#ifndef PLANS_PLANT_SYNC_SERVER_HPP
#define PLANS_PLANT_SYNC_SERVER_HPP

#include <plansplant/tasks.hpp>
#include <QTcpSocket>
#include <QTcpServer>
#include <QTimer>

namespace PlansPlant
{
  class PLANSPLANT_CLASS Synchronizer : public Task::Watcher
  {
  public:
    class QueuedEvent
    {
    public:
      typedef QQueue<QueuedEvent> Queue;
      QueuedEvent( const Event& Ev );
      QueuedEvent( Event::ID Ident0, const QByteArray& Data0 );
      Event::ID id() const { return Ident; }
      const QByteArray& data() const { return Data; }
    protected:
      Event::ID Ident;
      QByteArray Data;
    }; // QueuedEvent
    typedef Synchronizer* Pointer;
    typedef QList<Pointer> List;
    Synchronizer( TasksFile& Tasks0 );
    ~Synchronizer();
    TasksFile& tasks() const { return Tasks; }
    virtual void start();
    Event* read_event( QXmlStreamReader& Stream );
    virtual bool load( QXmlStreamReader& Stream );
    virtual void write( QXmlStreamWriter& Stream ) const;
    void event( const Task::Watcher::Event& Ev ); // Watcher's overload
    // Event from the outer world
    virtual void event_in( const Event& Ev );
  protected:
    // Event from the TasksFile.
    virtual void event_out( const Event& Ev );
    virtual bool load_field( QXmlStreamReader& Stream, const QStringRef& Tag, bool SkipUnknown = true );
    virtual void new_event_in_queue();
    TasksFile& Tasks;
    QueuedEvent::Queue Log;
  }; // Synchronizer

  //! \todo Move events processing and persistance to new subclass. 
  class PLANSPLANT_CLASS Transport : public QObject
  {
    Q_OBJECT
  public:
    struct State { enum ID { Disconnected, Connected, UUID, Prompt, Password, Ready }; };
    Transport( QTcpSocket* Socket0 = 0, QObject* Parent = 0 );
    ~Transport();
    bool is_connected() const { return Socket && Socket->state() == QAbstractSocket::ConnectedState; }
    bool ready() const;
    int offset() const { return Offset; }
    virtual void socket( QTcpSocket* Socket );
    virtual void process_queue( const Synchronizer::QueuedEvent::Queue& Ev );
    virtual void disconnect( const QString& ReasonText = QString() );
    virtual bool load( QXmlStreamReader& Stream );
  protected slots:
    virtual void connected();
    virtual void disconnected();
    virtual void ready_to_read();
    virtual void bytes_written( qint64 Count );
    virtual void timeout();
  protected:
    virtual void text_line( const QByteArray& Text ); // Text line (not XML) from the socket with trailing '\n' char
    virtual void xml_block( const QByteArray& Block ); // Complete XML block from the opening to the closing tag without leading and trailing spaces.
    virtual void event_is_sent();
    virtual void write( QXmlStreamWriter& Stream ) const;
    virtual bool load_field( QXmlStreamReader& Stream, const QStringRef& Tag, bool SkipUnknown = true );
    QString Prompt;
    QString Password;
    State::ID CurrState;
    QTcpSocket* Socket;
    QTimer Timer;
    QByteArray Buffer;
    int ToSend;
    int Offset; // From the top of the queue to current event.
  }; // Transport

  // Handshake & auth steps:
  // 1.Node connects to Server
  // 2.Server: "Plans Plant Server\n"
  // 3.Client: "<node id=Node's ID><file><uuid>File's UUID</uuid></file></node>"
  // 4.Server: if right UUID: "<prompt>the node's Prompt</prompt>
  //           else if wrong UUID: <not_found />
  //           else (no <node> block) drop connection
  // 5.Node: if right prompt from server: "<password>node's password</password>"
  //         else drop connection
  // 6.Server: if right password: "<ready/>
  // 7.Server: process queue.
  class PLANSPLANT_CLASS Node : public Synchronizer, public Transport
  {
  public:
    Node( TasksFile& Tasks0, const QString& HostName0 = "localhost", quint16 Port0 = 8953 );
    void start();
    void event_out( const Event& Ev );
    void write( QXmlStreamWriter& Stream ) const;
  protected:
    bool load_field( QXmlStreamReader& Stream, const QStringRef& Tag, bool SkipUnknown = true );
    void new_event_in_queue();
    void event_is_sent();
    void text_line( const QByteArray& Text );
    void xml_block( const QByteArray& Block );
    void connected();
    QString HostName;
    quint16 Port;
  }; // Node

  class MetaServer;
  class PLANSPLANT_CLASS Server : public Synchronizer
  {
  public:
    typedef Server* Pointer;
    typedef QList<Pointer> List;
    class Client : public Transport
    {
    public:
      typedef Client* Pointer;
      typedef QList<Pointer> List;
      Client( Server& Server0, int Ident0 = -1 );
      virtual ~Client();
      int id() const { return Ident; }
      int adjust_offset( int NewTop = 1 ) { return( Offset -= NewTop ); }
      void write( QXmlStreamWriter& Stream ) const;
      void xml_block( const QByteArray& Block );
      void process_queue( const Synchronizer::QueuedEvent::Queue& Ev );
    protected:
      void event_in( const Event& Ev );
      void event_is_sent();
      Server& Srv;
      int Ident;
    }; // Client
    Server( TasksFile& Tasks0, quint16 Port0 = 8953 );
    ~Server();
    qint16 port() const { return Port; }
    void start();
    const Client::List& clients() const { return Clients; }
    void event_is_sent( Client& Cli );
    void write( QXmlStreamWriter& Stream ) const;
  protected:
    void new_event_in_queue();
    bool load_field( QXmlStreamReader& Stream, const QStringRef& Tag, bool SkipUnknown = true );
    Client::List Clients;
    quint16 Port;
    static MetaServer* Coordinator;
  public:
    static const char* Signature;
  }; // Server

  //! \todo Find better name. :)
  class PLANSPLANT_CLASS MetaServer : public QObject
  {
    Q_OBJECT
    class PLANSPLANT_CLASS Porter : public Transport
    {
    public:
      typedef Porter* Pointer;
      typedef QList<Pointer> List;
      Porter( MetaServer& Master0, QTcpSocket* Socket0 );
      ~Porter();
    protected:
      void disconnected();
      void xml_block( const QByteArray& Block );
      MetaServer& Master;
    }; // Porter
  public:
    MetaServer( QObject* Parent = 0 );
    ~MetaServer();
    void add_server( Server& NewServer );
    void remove_server( Server& OldServer );
    void remove_porter( Porter& OldPorter );
    Server::Client* client_identified( const QString& FileUUID, int ClientID );
  protected slots:
    void connection();
  protected:
    Porter::List Porters;
    Server::List Servers;
    QList<QTcpServer*> Listeners;
  }; // MetaServer
} // PlansPlant
#endif // PLANS_PLANT_SYNC_SERVER_HPP
