//
// fujitsuacprotocol.cpp
//
// Copyright 2016 by John Pietrzak (jpietrzak8@gmail.com)
//
// This file is part of Pierogi.
//
// Pierogi is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// Pierogi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
//

#include "fujitsuacprotocol.h"

#include "pirinfraredled.h"

#include <QString>
#include <QMutex>
extern bool commandInFlight;
extern QMutex commandIFMutex;

//
// This is a stateful protocol; rather than sending a single setting to the
// air conditioner in one command, it sends _all_ the AC settings in each
// command.  This makes things quite a bit more complicated than other IR
// controllers...
//
// A "zero" is encoded with a 432 usec pulse, 432 usec space.
// A "one" is encoded with a 432 usec pulse, 1296 usec space.
// The header is a 3456 usec pulse, 1728 usec space.
// As with other stateful AC controls, these commands don't seem to be
// automatically repeated.  (Need to confirm this...)
// The carrier frequency is probably 38 kHz.

FujitsuACProtocol::FujitsuACProtocol(
  QObject *guiObject,
  unsigned int index)
  : SpaceProtocol(
      guiObject, index,
      432, 432,
      432, 1296,
      3456, 1728,
      432,
      0, false)
{
}


void FujitsuACProtocol::startSendingCommand(
  unsigned int threadableID,
  PIRKeyName command)
{
  // First, check if we are meant to be the recipient of this command:
  if (threadableID != id) return;

  KeycodeCollection::const_iterator i = keycodes.find(command);

  // Do we even have this key defined?
  if (i == keycodes.end())
  {
    QMutexLocker cifLocker(&commandIFMutex);
    commandInFlight = false;
    emit errorMessage("Key not defined in this keyset.");
    return;
  }

  // construct the device:
  PIRInfraredLED led(carrierFrequency, dutyCycle);

  connect(
    &led,
    SIGNAL(errorMessage(QString)),
    this,
    SIGNAL(errorMessage(QString)));

  switch (command)
  {
  // A few commands do not send state info:
  case PowerOff_Key:
  case StepLouverVertical_Key:
  case StepLouverHorizontal_Key:
    generateStatelessCommand(command, led);
    break;

/*
  case ACSetTimer_Key:
    generateTimerCommand(led);
    break;
*/

  default:
    generateCommand(led);
  }

  // Now, tell the device to send the whole command:
  led.sendCommandToDevice();

  QMutexLocker cifLocker(&commandIFMutex);
  commandInFlight = false;
}


void FujitsuACProtocol::generateCommand(
  PIRInfraredLED &led)
{
  // First, the "header" pulse:
  led.addPair(headerPulse, headerSpace);

  if (!statefulPrefixBits.size())
  {
    setupStatefulPrefixBits();
  }

  pushBits(statefulPrefixBits, led);

  // All the bits of the following values are transmitted in reverse order:

  // temperature (8 bits):
  pushReverseBits(temperature, led);

  // Next, the timer mode (4 bits):
  pushReverseBits(timerMode, led);

  // Next, the master control (4 bits):
  pushReverseBits(masterControl, led);

  // Next, the swing mode (4 bits):
  pushReverseBits(swingMode, led);

  // Next, the fan speed (4 bits):
  pushReverseBits(fanSpeed, led);

  // Next, off timer duration (11 bits):
  pushReverseBits(timerOffDuration, led);

  // Next, change off timer value (1 bit):
  pushReverseBits(timerOffFlag, led);

  // Next, on timer duration (11 bits):
  pushReverseBits(timerOnDuration, led);

  // Next, change on timer value (1 bit):
  pushReverseBits(timerOnFlag, led);

  // Next, a fixed 8 bit value I don't yet understand:
  if (!statefulPostfixBits.size()) setupStatefulPostfixBits();
  pushBits(statefulPostfixBits, led);

  // Next, the checksum:
  CommandSequence checksum;
  calculateChecksum(checksum);
  pushBits(checksum, led);

  // Finally add the "trail":
  led.addSingle(trailerPulse);
}


void FujitsuACProtocol::generateStatelessCommand(
  PIRKeyName command,
  PIRInfraredLED &led)
{
  if (!statelessPrefixBits.size())
  {
    setupStatelessPrefixBits();
  }

  switch (command)
  {
  case PowerOff_Key:
    pushBits(statelessPrefixBits, led);
    if (!powerOffBits.size())
    {
      setupPowerOffBits();
    }
    pushBits(powerOffBits, led);
    break;

  case StepLouverVertical_Key:
    pushBits(statelessPrefixBits, led);
    if (!stepLouverVerticalBits.size())
    {
      setupStepLouverVerticalBits();
    }
    pushBits(stepLouverVerticalBits, led);
    break;

  case StepLouverHorizontal_Key:
    pushBits(statelessPrefixBits, led);
    if (!stepLouverHorizontalBits.size())
    {
      setupStepLouverHorizontalBits();
    }
    pushBits(stepLouverHorizontalBits, led);
    break;

  default:
    // Command not supported!
    break;
  }
}

void FujitsuACProtocol::setupStatefulPrefixBits()
{
  // The stateful Fujitsu commands all have the same prefix: 0x28C60008087F900C
  // (8 bytes)

  if (statefulPrefixBits.size()) statefulPrefixBits.clear();

  appendBitsToSeq(statefulPrefixBits, 0x28);
  appendBitsToSeq(statefulPrefixBits, 0xC6);
  appendBitsToSeq(statefulPrefixBits, 0x00);
  appendBitsToSeq(statefulPrefixBits, 0x08);
  appendBitsToSeq(statefulPrefixBits, 0x08);
  appendBitsToSeq(statefulPrefixBits, 0x7F);
  appendBitsToSeq(statefulPrefixBits, 0x90);
  appendBitsToSeq(statefulPrefixBits, 0x0C);
}


void FujitsuACProtocol::setupStatefulPostfixBits()
{
  if (statefulPostfixBits.size()) statefulPostfixBits.clear();

  appendBitsToSeq(statefulPostfixBits, 0x04);
}


void FujitsuACProtocol::setupStatelessPrefixBits()
{
  // The stateless Fujitsu commands all have the same prefix: 0x28C6000808

  // first, a sanity check:
  if (statelessPrefixBits.size()) statelessPrefixBits.clear();

  appendBitsToSeq(statelessPrefixBits, 0x28);
  appendBitsToSeq(statelessPrefixBits, 0xC6);
  appendBitsToSeq(statelessPrefixBits, 0x00);
  appendBitsToSeq(statelessPrefixBits, 0x08);
  appendBitsToSeq(statelessPrefixBits, 0x08);
}


void FujitsuACProtocol::setupPowerOffBits()
{
  // The code for the power off command is: 0x40BF
  if (powerOffBits.size()) powerOffBits.clear();

  appendBitsToSeq(powerOffBits, 0x40);
  appendBitsToSeq(powerOffBits, 0xBF);
}


void FujitsuACProtocol::setupStepLouverVerticalBits()
{
  // The code for a vertical step is: 0x9E61
  if (stepLouverVerticalBits.size()) stepLouverVerticalBits.clear();

  appendBitsToSeq(stepLouverVerticalBits, 0x9E);
  appendBitsToSeq(stepLouverVerticalBits, 0x61);
}


void FujitsuACProtocol::setupStepLouverHorizontalBits()
{
  // The code for a horizontal step is: 0x36C9
  if (stepLouverHorizontalBits.size()) stepLouverHorizontalBits.clear();

  appendBitsToSeq(stepLouverHorizontalBits, 0x36);
  appendBitsToSeq(stepLouverHorizontalBits, 0xC9);
}


void FujitsuACProtocol::appendBitsToSeq(
  CommandSequence &seq,
  int eightBitValue)
{
  int i = 0;
  while (i < 8)
  {
    seq.push_back(eightBitValue & 0x80);
    eightBitValue <<= 1;
    ++i;
  }
}


void FujitsuACProtocol::calculateChecksum(
  CommandSequence &checksum)
{
  // The checksum byte is the sum of the previous seven bytes mod 256, then
  // inverted.

  // Sanity check:
  checksum.clear();

  // First byte: the temperature.  Needs to be reversed.
  CommandSequence::const_iterator i = temperature.end();
  while (i != temperature.begin())
  {
    --i;
    checksum.push_back(*i);
  }

  // Second byte: master control and timer mode, combined and reversed.
  bool carry = false;
  CommandSequence::iterator checksumIter = checksum.begin();
  i = masterControl.end();
  while ((i != masterControl.begin()) && (checksumIter != checksum.end()))
  {
    --i;
    addBinary(checksumIter, i, carry);
    ++checksumIter;
  }

  i = timerMode.end();
  while ((i != timerMode.begin()) && (checksumIter != checksum.end()))
  {
    --i;
    addBinary(checksumIter, i, carry);
    ++checksumIter;
  }

  // Third byte: swing mode and fan speed, combined and reversed:
  carry = false;
  checksumIter = checksum.begin();
  i = fanSpeed.end();
  while ((i != fanSpeed.begin()) && (checksumIter != checksum.end()))
  {
    --i;
    addBinary(checksumIter, i, carry);
    ++checksumIter;
  }

  i = swingMode.end();
  while ((i != swingMode.begin()) && (checksumIter != checksum.end()))
  {
    --i;
    addBinary(checksumIter, i, carry);
    ++checksumIter;
  }

  // Fourth byte: last eight bits of the timer off duration:
  carry = false;
  i = timerOffDuration.end();
  checksumIter = checksum.begin();
  // Since checksumIter should be exactly 8 bits, we'll use it to end the loop:
  while ((i != timerOffDuration.begin()) && (checksumIter != checksum.end()))
  {
    --i;
    addBinary(checksumIter, i, carry);
    ++checksumIter;
  }

  // Fifth byte: first three bits of checksumIter, timer off flag, and last
  // four bits of timerOnDuration:
  carry = false;
  checksumIter = checksum.begin();
  while ((i != timerOffDuration.begin()) && (checksumIter != checksum.end()))
  {
    --i;
    addBinary(checksumIter, i, carry);
    ++checksumIter;
  }

  // timerOffFlag should be a single bit:
  i = timerOffFlag.begin();
  if ((i != timerOffFlag.end()) && (checksumIter != checksum.end()))
  {
    addBinary(checksumIter, i, carry);
    ++checksumIter;
  }

  i = timerOnDuration.end();
  while ((i != timerOnDuration.begin()) && (checksumIter != checksum.end()))
  {
    --i;
    addBinary(checksumIter, i, carry);
    ++checksumIter;
  }

  // Sixth byte: first seven bits of timerOnDuration, followed by timerOnFlag:
  carry = false;
  checksumIter = checksum.begin();
  while ((i != timerOnDuration.begin()) && (checksumIter != checksum.end()))
  {
    --i;
    addBinary(checksumIter, i, carry);
    ++checksumIter;
  }

  i = timerOnFlag.begin();
  if ((i != timerOnFlag.end()) && (checksumIter != checksum.end()))
  {
    addBinary(checksumIter, i, carry);
  }

  // Seventh byte: the stateful postfix byte.  No need to reverse this one...
  carry = false;
  checksumIter = checksum.begin();
  i = statefulPostfixBits.begin();
  while ((i != statefulPostfixBits.end()) && (checksumIter != checksum.end()))
  {
    addBinary(checksumIter, i, carry);
    ++i;
    ++checksumIter;
  }

  // Finally, invert the checksum:
  checksumIter = checksum.begin();
  while (checksumIter != checksum.end())
  {
    *checksumIter = !*checksumIter;
    ++checksumIter;
  }
}


void FujitsuACProtocol::addBinary(
  CommandSequence::iterator &checksumIter,
  CommandSequence::const_iterator &i,
  bool &carry)
{
  if (*i)
  {
    if (*checksumIter)
    {
      // Value is a 2, so need to carry one over.
      // If carry is already full, value is 3, so leave both carry and
      // *checksumIter full.  Otherwise, fill the carry and empty the 
      // *checksumIter:
      if (!carry)
      {
        *checksumIter = false;
        carry = true;
      }
    }
    else
    {
      // need to change *checksumIter from empty to full:
      *checksumIter = true;
    }
  }
  else if (carry)
  {
    if (*checksumIter)
    {
      // Value is a two; empty the *checksumIter, but leave the carry full:
      *checksumIter = false;
    }
    else
    {
      // Dump the value into *checksumIter, and empty the carry:
      *checksumIter = true;
      carry = false;
    }
  }
}
