#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <string.h>
#include "debug.h"
#include "icdconnmon.h"

#define DBUS_SIGNAL_MATCH_RULE "interface='com.nokia.icd',type='signal',member='status_changed'"

typedef struct {
	char *conn_type;
	gboolean is_connected;
} CONN_TYPE_STATUS;

typedef struct {
	DBusConnection *dbus_connection;
	gboolean filter_added;
	ICDConnMonStatus connect_status;
	GList *conn_type_statuses;
	char *ip, *netmask, *gw;
} ICDConnMonPrivate;

#define ICD_CONN_MON_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE ((object), ICD_CONN_MON_TYPE, ICDConnMonPrivate))

enum {
	FIRST_PROPERTY = 0,
	CONNECTED_PROPERTY,
	CONNECTED_ENABLED_PROPERTY,
	LAST_PROPERTY
};

static void icd_conn_mon_class_init(gpointer g_class, gpointer class_data);
static void icd_conn_mon_instance_init(GTypeInstance *instance, gpointer g_class);

static void finalize(GObject *obj);
static void get_property(GObject *obj, guint property_id, GValue *value, GParamSpec *pspec);

static DBusHandlerResult dbus_connection_filter(DBusConnection *connection, DBusMessage *message, void *user_data);

static void process_status_change_event(ICDConnMon *conn_mon, const char *conn_type, const char *conn_state);
static gboolean get_current_connection_info(DBusConnection *dbus_connection, char **ip, char **netmask, char **gw);
static gboolean compare_and_set_connection_info(ICDConnMonPrivate *priv, char *new_ip, char *new_netmask, char *new_gw);
static gboolean compare_and_set_string(char **p_set, char *new_val);
static char *set_string(char *old_val, char *new_val);

GType
icd_conn_mon_status_get_type (void)
{
	static GType the_type = 0;

	if (!the_type) {
		static GEnumValue enum_values[] = {
			{ICD_CONN_MON_DISCONNECTED, "ICD_CONN_MON_DISCONNECTED", "Disconnected"},
			{ICD_CONN_MON_CONNECTED,    "ICD_CONN_MON_CONNECTED",    "Connected"},
			{ICD_CONN_MON_RECONNECTED,  "ICD_CONN_MON_RECONNECTED",  "Reconnected"},
			{0, NULL, NULL},
		};

		the_type = g_enum_register_static(ICD_CONN_MON_STATUS_TYPE_STRING, enum_values);
	}

	return the_type;
}

GType
icd_conn_mon_get_type (void)
{
	static GType the_type = 0;

	if (!the_type) {
		static GTypeInfo the_type_info = {
			.class_size     = sizeof (ICDConnMonClass),
			.base_init      = NULL,
			.base_finalize  = NULL,
			.class_init     = icd_conn_mon_class_init,
			.class_finalize = NULL,
			.class_data     = NULL,
			.instance_size  = sizeof (ICDConnMon),
			.n_preallocs    = 0,
			.instance_init  = icd_conn_mon_instance_init,
			.value_table    = NULL
		};

		the_type = g_type_register_static(G_TYPE_OBJECT, ICD_CONN_MON_TYPE_STRING, &the_type_info, 0);
	}

	return the_type;
}

static void
icd_conn_mon_class_init(gpointer g_class, gpointer class_data) {
	G_OBJECT_CLASS (g_class)->finalize = finalize;
	G_OBJECT_CLASS (g_class)->get_property = get_property;

	g_object_class_install_property(G_OBJECT_CLASS(g_class), CONNECTED_PROPERTY,
		g_param_spec_enum("connect-status", "Connectivity status", "Whether ICD is aware of a connection",
			ICD_CONN_MON_STATUS_TYPE, ICD_CONN_MON_DISCONNECTED, G_PARAM_READABLE));

	g_type_class_add_private(g_class, sizeof(ICDConnMonPrivate));
	}

static void
icd_conn_mon_instance_init(GTypeInstance *instance, gpointer g_class)
{
	ICDConnMonPrivate *priv = ICD_CONN_MON_GET_PRIVATE (instance);

	priv->dbus_connection = NULL;
	priv->filter_added = FALSE;
	priv->connect_status = ICD_CONN_MON_DISCONNECTED;
	priv->conn_type_statuses = NULL;
	priv->ip = NULL;
	priv->netmask = NULL;
	priv->gw = NULL;

	if ((priv->dbus_connection = dbus_bus_get (DBUS_BUS_SYSTEM, NULL)) != NULL) {
		DBusError dbus_error;

		dbus_connection_setup_with_g_main (priv->dbus_connection, NULL);

		dbus_error_init (&dbus_error) ;
		dbus_bus_add_match (priv->dbus_connection, DBUS_SIGNAL_MATCH_RULE, &dbus_error);
		if (dbus_error_is_set (&dbus_error))
			purple_debug_error("icd", "D-Bus match \"" DBUS_SIGNAL_MATCH_RULE "\" added, but with error: %s\n", dbus_error.message);
		dbus_error_free (&dbus_error);

		if (!(priv->filter_added = dbus_connection_add_filter (priv->dbus_connection, dbus_connection_filter, instance, NULL)))
			purple_debug_error("icd", "Failed to add filter to D-Bus connection\n");

		priv->connect_status = get_current_connection_info (priv->dbus_connection, &(priv->ip), &(priv->netmask), &(priv->gw))
			? ICD_CONN_MON_CONNECTED : ICD_CONN_MON_DISCONNECTED ;
	}
}

static void
get_property(GObject *obj, guint property_id, GValue *value, GParamSpec *pspec)
{
	ICDConnMonPrivate *priv = ICD_CONN_MON_GET_PRIVATE(obj);

	if (CONNECTED_PROPERTY == property_id)
		g_value_set_enum (value,(priv->connect_status));
	else
		G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
}

static void
finalize(GObject *obj)
{
	GList *llItr = NULL;
	ICDConnMonPrivate *priv = ICD_CONN_MON_GET_PRIVATE (obj) ;

	if (NULL != priv->dbus_connection) {
		if (priv->filter_added)
			dbus_connection_remove_filter(priv->dbus_connection, dbus_connection_filter, obj);
		dbus_bus_remove_match(priv->dbus_connection, DBUS_SIGNAL_MATCH_RULE, NULL);
	}

	for (llItr = priv->conn_type_statuses ; llItr ; llItr = llItr->next) {
		g_free(((CONN_TYPE_STATUS *)(llItr->data))->conn_type);
		g_free(llItr->data);
	}
	g_list_free(priv->conn_type_statuses);
}

static DBusHandlerResult
dbus_connection_filter(DBusConnection *connection, DBusMessage *dbus_message, void *user_data)
{
	DBusHandlerResult result = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	const char *str = NULL;

	if (DBUS_MESSAGE_TYPE_SIGNAL == dbus_message_get_type(dbus_message)) {
		if ((str = dbus_message_get_interface(dbus_message)) != NULL) {
			if (!strcmp(str, "com.nokia.icd")) {
				if ((str = dbus_message_get_member(dbus_message)) != NULL) {
					if (!strcmp(str, "status_changed")) {
						const char *description, *conn_type, *conn_state, *conn_error;

						if (dbus_message_get_args(dbus_message, NULL, DBUS_TYPE_STRING, &description,
																												  DBUS_TYPE_STRING, &conn_type,
																												  DBUS_TYPE_STRING, &conn_state,
																												  DBUS_TYPE_STRING, &conn_error,
																												  DBUS_TYPE_INVALID)) {

							process_status_change_event(ICD_CONN_MON(user_data), conn_type, conn_state);
							result = DBUS_HANDLER_RESULT_HANDLED;
						}
					}
				}
			}
		}
	}

	return result;
}

static void
process_status_change_event(ICDConnMon *conn_mon, const char *conn_type, const char *conn_state)
{
	char *new_ip = NULL;
	char *new_netmask = NULL;
	char *new_gw = NULL;
	ICDConnMonPrivate *priv = ICD_CONN_MON_GET_PRIVATE (conn_mon);
	GList *llItr = NULL;
	gboolean things_have_changed = FALSE;
	gboolean is_connected = TRUE;
	gboolean have_info = FALSE;
	CONN_TYPE_STATUS *existing_status = NULL;

	if (strcmp(conn_state, "CONNECTED") && (is_connected = strcmp(conn_state, "IDLE"))) return ;

	have_info = get_current_connection_info(priv->dbus_connection, &new_ip, &new_netmask, &new_gw);

	for (llItr = priv->conn_type_statuses ; llItr ; llItr = llItr->next) {
		existing_status = ((CONN_TYPE_STATUS *)(llItr->data));
		if (!strcmp(existing_status->conn_type, conn_type)) {
			if (is_connected != existing_status->is_connected) {
				purple_debug_info("icd", "process_status_change_event: conn_type %s has become %sconnected\n", conn_type, is_connected ? "" : "dis");
				things_have_changed = TRUE;
				existing_status->is_connected = is_connected;
			}
			break ;
		}
	}

	/* Couldn't find the conn_type among the existing conn_type entries */
	if (!llItr) {
		CONN_TYPE_STATUS *new_conn_type_status = g_new0(CONN_TYPE_STATUS, 1);

		new_conn_type_status->conn_type = g_strdup(conn_type);
		new_conn_type_status->is_connected = is_connected;
		priv->conn_type_statuses = g_list_prepend(priv->conn_type_statuses, new_conn_type_status);
		things_have_changed = (is_connected != (ICD_CONN_MON_CONNECTED == priv->connect_status || ICD_CONN_MON_RECONNECTED == priv->connect_status));
		purple_debug_info("icd", "process_status_change_event: New conn_type %s is %sconnected\n", conn_type, is_connected ? "" : "dis");
	}

	/* if connection information hasn't changed, than nothing has really changed */
	if (things_have_changed && compare_and_set_connection_info(priv, new_ip, new_netmask, new_gw)) {
		gboolean have_connection = FALSE;

		for (llItr = priv->conn_type_statuses ; llItr ; llItr = llItr->next)
			if ((have_connection = ((CONN_TYPE_STATUS *)(llItr->data))->is_connected))
				break ;

		if (have_connection)
			priv->connect_status = (ICD_CONN_MON_DISCONNECTED == priv->connect_status) ? ICD_CONN_MON_CONNECTED : ICD_CONN_MON_RECONNECTED;
		else
			priv->connect_status = ICD_CONN_MON_DISCONNECTED;

		purple_debug_info("icd", "New connect-status is %s\n", 
			ICD_CONN_MON_DISCONNECTED == priv->connect_status ? "ICD_CONN_MON_DISCONNECTED" :
			ICD_CONN_MON_RECONNECTED == priv->connect_status ? "ICD_CONN_MON_RECONNECTED" :
			ICD_CONN_MON_CONNECTED == priv->connect_status ? "ICD_CONN_MON_CONNECTED" : "???");

		g_object_notify(G_OBJECT(conn_mon), "connect-status");
	}

	if (have_info) {
		g_free(new_ip);
		g_free(new_netmask);
		g_free(new_gw);
	}
}

static gboolean
get_current_connection_info(DBusConnection *dbus_connection, char **ip, char **netmask, char **gw)
{
	gboolean ret = FALSE;

	if (dbus_connection) {
		DBusMessage *method_call = NULL, *reply = NULL;

		if ((method_call = dbus_message_new_method_call("com.nokia.icd", "/com/nokia/icd", "com.nokia.icd", "get_ipinfo")) != NULL) {
			if ((reply = dbus_connection_send_with_reply_and_block (dbus_connection, method_call, 1000, NULL)) != NULL) {
				char *iap_name = NULL;

				if (dbus_message_get_args(reply, NULL, DBUS_TYPE_STRING, &iap_name, 
																							 DBUS_TYPE_STRING, ip, 
																							 DBUS_TYPE_STRING, netmask,
																							 DBUS_TYPE_STRING, gw,
																							 DBUS_TYPE_INVALID)) {
					(*ip)      = g_strdup ((*ip)) ;
					(*netmask) = g_strdup ((*netmask)) ;
					(*gw)      = g_strdup ((*gw)) ;
					ret = TRUE ;
				}
				dbus_message_unref (reply) ;
			}
			dbus_message_unref (method_call) ;
		}
	}
	return ret ;
}

static gboolean
compare_and_set_connection_info(ICDConnMonPrivate *priv, char *new_ip, char *new_netmask, char *new_gw)
{
	gboolean things_have_changed = FALSE;

	if ((things_have_changed = compare_and_set_string((&(priv->ip)), new_ip))) {
		priv->netmask = set_string(priv->netmask, new_netmask);
		priv->gw = set_string(priv->gw, new_gw);
	} else if ((things_have_changed = compare_and_set_string((&(priv->netmask)), new_netmask))) {
		priv->gw = set_string(priv->gw, new_gw);
	} else {
		things_have_changed = compare_and_set_string((&(priv->gw)), new_gw);
	}

	return things_have_changed;
}

static gboolean
compare_and_set_string(char **p_set, char *new_val)
{
	gboolean things_have_changed = FALSE;

	if (NULL == (*p_set)) {
		if ((things_have_changed = (NULL != new_val)))
			(*p_set) = g_strdup(new_val);
		/* else we're fine */
	} else {
		if ((things_have_changed = (NULL == new_val))) {
			g_free((*p_set));
			(*p_set) = NULL;
		} else if ((things_have_changed = strcmp((*p_set), new_val))) {
			g_free ((*p_set));
			(*p_set) = g_strdup(new_val);
		}
	}

	return things_have_changed;
}

static char *
set_string(char *old_val, char *new_val)
{
	if (NULL != old_val)
		g_free(old_val);
	return (NULL == new_val) ? NULL : g_strdup(new_val);
}
