/*
 * This file is a part of pidgin-maemo-docklet
 *
 * Copyright (C) 2009-2010 Gabriel Schulhof <nix@go-nix.ca>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <signal.h>

#include <stdlib.h>
#include <gtk/gtk.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <hildon/hildon.h>
#include "global.h"
#include "pidgin-status-menu-plugin.h"

#define HILDON_STATUS_MENU_PIC_CX 18
#define HILDON_STATUS_MENU_PIC_CY 18

struct _PidginStatusMenuPlugin {
	HDStatusMenuItem __parent_instance__;
	char *pidgin_plugin_dbus_id;
	GtkWidget *image;
	GtkWidget *label;
	DBusConnection *conn;
	GdkPixbuf *status_pb;
	GdkPixbuf *transp_pb;
	gboolean blink_is_visible;
};

struct _PidginStatusMenuPluginClass {
	HDStatusMenuItemClass __parent_class__;
};

HD_DEFINE_PLUGIN_MODULE(PidginStatusMenuPlugin, pidgin_status_menu_plugin, HD_TYPE_STATUS_MENU_ITEM);

static void
pidgin_maemo_docklet_clicked(PidginStatusMenuPlugin *menu_item, int button)
{
	DBusMessage *message = dbus_message_new_method_call(menu_item->pidgin_plugin_dbus_id, 
		PIDGIN_SYSTRAY_DBUS_SERVICE_PATH,
		NULL,
		PIDGIN_SYSTRAY_DBUS_SERVICE_METHOD_CLICKED);
	if (G_LIKELY(message)) {
		DBusMessage *reply;

		dbus_message_append_args(message, DBUS_TYPE_INT32, &button, DBUS_TYPE_INVALID);
		debug_print("|-H- *** pidgin_maemo_docklet_clicked: Calling plugin (@%s) with button = %d\n", menu_item->pidgin_plugin_dbus_id, button);
		reply = dbus_connection_send_with_reply_and_block(menu_item->conn, message, -1, NULL);
		if (reply)
			dbus_message_unref(reply);
	}
}

static void
pidgin_status_menu_plugin_class_finalize(PidginStatusMenuPluginClass *klass)
{
}

static void
finalize(GObject *obj)
{
}

static void
pidgin_status_menu_plugin_class_init(PidginStatusMenuPluginClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->finalize = finalize;
}

static gboolean
blink_image(PidginStatusMenuPlugin *menu_item)
{
	if (menu_item->blink_is_visible)
		hd_status_plugin_item_set_status_area_icon(HD_STATUS_PLUGIN_ITEM(menu_item), menu_item->transp_pb);
	else
		hd_status_plugin_item_set_status_area_icon(HD_STATUS_PLUGIN_ITEM(menu_item), menu_item->status_pb);

	menu_item->blink_is_visible = !(menu_item->blink_is_visible);
	return TRUE;
}

static GdkPixbuf *
pixbuf_from_buffer(gchar *buffer, gsize buffer_size)
{
	GdkPixbuf *pb = NULL;
	GdkPixbufLoader *pb_loader = gdk_pixbuf_loader_new_with_type("png", NULL);

	if (pb_loader) {
		if (gdk_pixbuf_loader_write(pb_loader, (const guchar *)buffer, buffer_size, NULL)) {
			if (gdk_pixbuf_loader_close(pb_loader, NULL))
				pb = g_object_ref(gdk_pixbuf_loader_get_pixbuf(pb_loader));
		}
		g_object_unref(pb_loader);
	}

	if (pb)
		debug_print("|-H- *** pixbuf_from_buffer: [%dx%d]\n", gdk_pixbuf_get_width(pb), gdk_pixbuf_get_height(pb));
	else
		debug_print("|-H- *** pixbuf_from_buffer: FAIL\n");

	return pb;
}

static void
pidgin_status_menu_item_disconnect_plugin(PidginStatusMenuPlugin *menu_item)
{
	debug_print("|-H- *** pidgin_status_menu_item_disconnect_plugin: disconnecting \"%s\"\n", menu_item->pidgin_plugin_dbus_id);
	g_free(menu_item->pidgin_plugin_dbus_id);
	menu_item->pidgin_plugin_dbus_id = NULL;
	g_object_set(G_OBJECT(menu_item->image), "pixbuf", NULL, NULL);
	gtk_widget_hide(GTK_WIDGET(menu_item));
}

static DBusHandlerResult
message_filter(DBusConnection *connection, DBusMessage *message, PidginStatusMenuPlugin *menu_item)
{
	DBusMessage *reply = NULL;
	DBusHandlerResult handler_return = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

	/* connect() */
	if (DBUS_MESSAGE_TYPE_METHOD_CALL == dbus_message_get_type(message) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_NAME, dbus_message_get_destination(message)) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_PATH, dbus_message_get_path(message)) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_METHOD_CONNECT, dbus_message_get_member(message))) {
		gboolean method_return = FALSE;

		if (!(menu_item->pidgin_plugin_dbus_id)) {
			const char *id = dbus_message_get_sender(message);

			if (id) {
				menu_item->pidgin_plugin_dbus_id = g_strdup(id);
				if (menu_item->pidgin_plugin_dbus_id) {
					debug_print("|-H- *** HildonStatusMenu::message_filter: showing menu item\n");
					gtk_widget_show(GTK_WIDGET(menu_item));
					method_return = TRUE;
				}
			}
		}

		debug_print("|-H- *** HildonStatusMenu::message_filter: replying to \"connect\" with %s\n", method_return ? "TRUE" : "FALSE");

		reply = dbus_message_new_method_return(message);
		dbus_message_append_args(reply, DBUS_TYPE_BOOLEAN, &method_return, DBUS_TYPE_INVALID);
	}
	else
	/* disconnect() */
	if (DBUS_MESSAGE_TYPE_METHOD_CALL == dbus_message_get_type(message) &&
		!g_strcmp0(menu_item->pidgin_plugin_dbus_id, dbus_message_get_sender(message)) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_NAME, dbus_message_get_destination(message)) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_PATH, dbus_message_get_path(message)) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_METHOD_DISCONNECT, dbus_message_get_member(message))) {
		const char *id = dbus_message_get_sender(message);

		if (!g_strcmp0(id, menu_item->pidgin_plugin_dbus_id))
			pidgin_status_menu_item_disconnect_plugin(menu_item);

		debug_print("|-H- *** HildonStatusMenu::message_filter: replying to \"disconnect\"\n");
		reply = dbus_message_new_method_return(message);
	}
	else
	/* NameOwnerChanged(name, old, new) */
	if (DBUS_MESSAGE_TYPE_SIGNAL == dbus_message_get_type(message) &&
		!g_strcmp0("org.freedesktop.DBus", dbus_message_get_sender(message)) &&
		!g_strcmp0("org.freedesktop.DBus", dbus_message_get_interface(message)) &&
		!g_strcmp0("/org/freedesktop/DBus", dbus_message_get_path(message)) &&
		!g_strcmp0("NameOwnerChanged", dbus_message_get_member(message))) {
		const char *name, *old_owner, *new_owner;

		debug_print("|-H- *** HildonStatusMenu::message_filter: NameOwnerChanged\n");

		if (dbus_message_get_args(message, NULL, 
			DBUS_TYPE_STRING, &name,
			DBUS_TYPE_STRING, &old_owner,
			DBUS_TYPE_STRING, &new_owner,
			DBUS_TYPE_INVALID))
			if (!g_strcmp0(name, menu_item->pidgin_plugin_dbus_id) &&
			    !g_strcmp0(new_owner, ""))
				pidgin_status_menu_item_disconnect_plugin(menu_item);

		handler_return = DBUS_HANDLER_RESULT_HANDLED;
	}
	/* set_tooltip(string) */
	else
	if (DBUS_MESSAGE_TYPE_METHOD_CALL == dbus_message_get_type(message) &&
		!g_strcmp0(menu_item->pidgin_plugin_dbus_id, dbus_message_get_sender(message)) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_NAME, dbus_message_get_destination(message)) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_PATH, dbus_message_get_path(message)) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_METHOD_SET_TOOLTIP, dbus_message_get_member(message))) {
		const char *tip_text;

		if (dbus_message_get_args(message, NULL, DBUS_TYPE_STRING, &tip_text, DBUS_TYPE_INVALID))
			g_object_set(G_OBJECT(menu_item->label), "label", tip_text, NULL);

		debug_print("|-H- *** HildonStatusMenu::message_filter: replying to set_tooltip(\"%s\")\n", tip_text);

		reply = dbus_message_new_method_return(message);
	}
	/* update_icon(png_buffer, buffer_size) */
	else
	if (DBUS_MESSAGE_TYPE_METHOD_CALL == dbus_message_get_type(message) &&
		!g_strcmp0(menu_item->pidgin_plugin_dbus_id, dbus_message_get_sender(message)) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_NAME, dbus_message_get_destination(message)) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_PATH, dbus_message_get_path(message)) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_METHOD_UPDATE_ICON, dbus_message_get_member(message))) {
		gchar *buffer;
		gsize buffer_size;
		gboolean blink = FALSE;

		if (dbus_message_get_args(message, NULL, DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE, &buffer, &buffer_size, DBUS_TYPE_BOOLEAN, &blink, DBUS_TYPE_INVALID)) {
			GdkPixbuf *pb = pixbuf_from_buffer(buffer, buffer_size);
			if (pb) {
				g_object_set(G_OBJECT(menu_item->image), "pixbuf", pb, NULL);
				g_object_set_data_full(G_OBJECT(menu_item->image), "blink-to",
					blink ? GINT_TO_POINTER(g_timeout_add_seconds(1, (GSourceFunc)blink_image, menu_item)) : 0,
					(GDestroyNotify)g_source_remove);
				g_object_unref(pb);
			}
		}

		debug_print("|-H- *** HildonStatusMenu::message_filter: replying to update_pixbuf\n");

		reply = dbus_message_new_method_return(message);
	}

	if (reply) {
		debug_print("|-H- *** HildonStatusMenu::message_filter: Sending reply\n");
		dbus_connection_send(connection, reply, NULL);
		dbus_connection_flush(connection);
		dbus_message_unref(reply);
		handler_return = DBUS_HANDLER_RESULT_HANDLED;
	}
	else
		debug_print("|-H- *** HildonStatusMenu::message_filter: NO REPLY\n");
	return handler_return;
}

static void
btn_clicked(GtkWidget *button, PidginStatusMenuPlugin *menu_item)
{
	if (menu_item->pidgin_plugin_dbus_id)
		pidgin_maemo_docklet_clicked(menu_item, 1);
}

static void
btn_tap_and_hold(GtkWidget *button, PidginStatusMenuPlugin *menu_item)
{
	if (menu_item->pidgin_plugin_dbus_id)
		pidgin_maemo_docklet_clicked(menu_item, 3);
}

static void
image_notify_pixbuf(GObject *image, GParamSpec *pspec, PidginStatusMenuPlugin *menu_item)
{
	GdkPixbuf *pb_src = NULL;

	g_object_get(image, "pixbuf", &pb_src, NULL);

	if (menu_item->status_pb) {
		g_object_unref(menu_item->status_pb);
		menu_item->status_pb = NULL;
	}
	if (pb_src) {
		menu_item->status_pb = gdk_pixbuf_scale_simple(pb_src, HILDON_STATUS_MENU_PIC_CX, HILDON_STATUS_MENU_PIC_CY, GDK_INTERP_HYPER);
		g_object_unref(pb_src);
	}

	debug_print("|-H- *** image_notify_pixbuf: Setting status area icon\n");

	hd_status_plugin_item_set_status_area_icon(HD_STATUS_PLUGIN_ITEM(menu_item), menu_item->status_pb);
}

static void
pidgin_status_menu_plugin_init(PidginStatusMenuPlugin *menu_item)
{
	GtkWidget *button = NULL;
	GtkWidget *hbox = NULL;

	menu_item->transp_pb = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, HILDON_STATUS_MENU_PIC_CX, HILDON_STATUS_MENU_PIC_CY);
	menu_item->blink_is_visible = TRUE;

	menu_item->conn = hd_status_plugin_item_get_dbus_connection(HD_STATUS_PLUGIN_ITEM(menu_item), DBUS_BUS_SESSION, NULL);
	menu_item->pidgin_plugin_dbus_id = NULL;

	if (menu_item->conn) {
		dbus_connection_setup_with_g_main(menu_item->conn, NULL);
		dbus_connection_add_filter(menu_item->conn, (DBusHandleMessageFunction)message_filter, menu_item, NULL);
		dbus_bus_add_match(menu_item->conn, "type='signal',sender='org.freedesktop.DBus',interface='org.freedesktop.DBus',path='/org/freedesktop/DBus',member='NameOwnerChanged'", NULL);
		if(dbus_bus_request_name(menu_item->conn, PIDGIN_SYSTRAY_DBUS_SERVICE_NAME, DBUS_NAME_FLAG_DO_NOT_QUEUE, NULL) != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER)
			_Exit(1);
	}

	menu_item->image =	g_object_new(GTK_TYPE_IMAGE,
			"visible", TRUE,
			"xalign",  0.5,
			"yalign",  0.5,
			NULL);
	gtk_widget_set_size_request(menu_item->image, 48, 48);
	g_signal_connect(G_OBJECT(menu_item->image), "notify::pixbuf", (GCallback)image_notify_pixbuf, menu_item);

	button = hildon_button_new(HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_image(HILDON_BUTTON(button), menu_item->image);
	gtk_widget_show(button);
	gtk_widget_tap_and_hold_setup(button, NULL, NULL, 0);
	g_signal_connect(G_OBJECT(button), "clicked", (GCallback)btn_clicked, menu_item);
	g_signal_connect(G_OBJECT(button), "tap-and-hold", (GCallback)btn_tap_and_hold, menu_item);

	menu_item->label = g_object_new(GTK_TYPE_LABEL,
		"visible", TRUE,
		"ellipsize", PANGO_ELLIPSIZE_MIDDLE,
		"justification", GTK_JUSTIFY_LEFT,
		NULL);

	hildon_helper_set_logical_color(menu_item->label, GTK_RC_TEXT, GTK_STATE_NORMAL, "SecondaryTextColor");
	hildon_helper_set_logical_color(menu_item->label, GTK_RC_FG, GTK_STATE_NORMAL, "SecondaryTextColor");
	hildon_helper_set_logical_font(menu_item->label, "SmallSystemFont");

	hbox = g_object_new(GTK_TYPE_HBOX, "visible", TRUE, "spacing", HILDON_MARGIN_DEFAULT, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), 
		g_object_new(GTK_TYPE_ALIGNMENT, "visible", TRUE,
			"child", menu_item->label, 
			"xalign", 0.0, "yalign", 0.5,
			"xscale", 1.0, "yscale", 1.0,
			NULL),
		TRUE, TRUE, 0);
	gtk_container_add(GTK_CONTAINER(menu_item), hbox);
}
