#include "Date.h"

#include <assert.h>

#include <Lum/Base/String.h>

#include "../Database.h"

namespace Data {
  Date::DateType *Date::dateType=new Date::DateType();

  Date::DateType::DateType()
  : Type(L"Date")
  {
    RegisterAttribute(L"description",description);
    RegisterAttribute(L"startDate",startDate);
    RegisterAttribute(L"endDate",endDate);
    RegisterAttribute(L"repeat",repeat);
    RegisterAttribute(L"repeatEndDate",repeatEndDate);
    RegisterAttribute(L"repeatInterval",repeatInterval);
    RegisterAttribute(L"wholeDay",wholeDay);
    RegisterAttribute(L"startTime",startTime);
    RegisterAttribute(L"endTime",endTime);
  }

  unsigned long Date::DateType::GetMaxAttribute() const
  {
    return endTime;
  }

  bool Date::DateType::IsPrimaryType() const
  {
    return true;
  }

  ::Base::DataPtr Date::DateType::Create() const
  {
    return new Date;
  }

  Date::Date()
   : Data(dateType),repeat(DateType::noRepeat),repeatInterval(1),wholeDay(true)
  {
    // TODO
    repeatEndDate.SetDate(31,12,2037);
  }

  std::wstring Date::GetDisplayText() const
  {
    return GetDateTimeString()+L" "+description;
  }

  void Date::SetAttribute(::Base::Id id, const std::wstring& value)
  {
    unsigned long         tmp;
    Lum::Base::Time       time;
    Lum::Base::DateTime   now;
    Lum::Base::DateTime   date;
    Lum::Base::SystemTime nDate;

    switch (id) {
    case DateType::description:
      description=value;
      break;
    case DateType::startDate:
      startDate.SetISO8601(value);
      break;
    case DateType::endDate:
      endDate.SetISO8601(value);
      break;
    case DateType::repeat:
      if (Lum::Base::WStringToNumber(value,tmp)) {
        repeat=(DateType::RepeatMode)tmp;
      }
      break;
    case DateType::repeatEndDate:
      repeatEndDate.SetISO8601(value);
      break;
    case DateType::repeatInterval:
      if (Lum::Base::WStringToNumber(value,tmp)) {
        repeatInterval=tmp;
      }
      break;
    case DateType::wholeDay:
      wholeDay=(value==L"true");
      break;
    case DateType::startTime:
      if (time.SetISO8601(value)) {
        nDate.GetLocalDateTime(now);
        nDate.SetUTCDateTime(time.GetHour(),
                             time.GetMinute(),
                             time.GetSecond(),
                             0,
                             startDate.GetDayOfMonth(),
                             startDate.GetMonth(),
                             now.year);
        nDate.GetLocalDateTime(date);
        startTime.Set(date.hour,date.minute,date.second);
      }
      break;
    case DateType::endTime:
      if (time.SetISO8601(value)) {
        nDate.GetLocalDateTime(now);
        nDate.SetUTCDateTime(time.GetHour(),
                             time.GetMinute(),
                             time.GetSecond(),
                             0,
                             endDate.GetDayOfMonth(),
                             endDate.GetMonth(),
                             now.year);
        nDate.GetLocalDateTime(date);
        endTime.Set(date.hour,date.minute,date.second);
      }
      break;
    default:
      assert(false);
    }
  }

  std::wstring Date::GetAttribute(Base::Id id) const
  {
    switch (id) {
    case DateType::description:
      return description;
    case DateType::startDate:
      return startDate.GetISO8601();
    case DateType::endDate:
      return endDate.GetISO8601();
    case DateType::repeat:
      return Lum::Base::NumberToWString((unsigned long)repeat);
    case DateType::repeatEndDate:
      return repeatEndDate.GetISO8601();
    case DateType::repeatInterval:
      return Lum::Base::NumberToWString(repeatInterval);
    case DateType::wholeDay:
      if (wholeDay) {
        return L"true";
      }
      else {
        return L"false";
      }
    case DateType::startTime:
      if (wholeDay) {
        return L"";
      }
      else {
        Lum::Base::SystemTime time;
        Lum::Base::DateTime   sDate;
        Lum::Base::DateTime   now;
        Lum::Base::Time       sTime;

        time.GetLocalDateTime(now);
        time.SetLocalDateTime(startTime.GetHour(),
                              startTime.GetMinute(),
                              startTime.GetSecond(),
                              0,
                              startDate.GetDayOfMonth(),
                              startDate.GetMonth(),
                              now.year);

        time.GetUTCDateTime(sDate);
        sTime.Set(sDate.hour,sDate.minute,sDate.second);
        return sTime.GetISO8601();
      }
    case DateType::endTime:
      if (wholeDay) {
        return L"";
      }
      else {
        Lum::Base::SystemTime time;
        Lum::Base::DateTime   eDate;
        Lum::Base::DateTime   now;
        Lum::Base::Time       eTime;

        time.GetLocalDateTime(now);
        time.SetLocalDateTime(endTime.GetHour(),
                              endTime.GetMinute(),
                              endTime.GetSecond(),
                              0,
                              endDate.GetDayOfMonth(),
                              endDate.GetMonth(),
                              now.year);

        time.GetUTCDateTime(eDate);
        eTime.Set(eDate.hour,eDate.minute,eDate.second);
        return eTime.GetISO8601();
      }
    default:
      assert(false);
    }
  }

  std::wstring Date::GetDateTimeString() const
  {
    std::wstring res;

    res=startDate.GetLocaleDate();

    if (!wholeDay) {
      res+=L" "+startTime.GetLocaleTime();
    }

    if (startDate!=endDate) {
      res+=L"-"+endDate.GetLocaleDate();

      if (!wholeDay) {
        res+=L" "+endTime.GetLocaleTime();
      }
    }
    else if (!wholeDay) {
      res+=L"-"+endTime.GetLocaleTime();
    }

    return res;
  }

  bool Date::operator>(const Date& date) const
  {
    return startDate>date.startDate;
  }
}
