#ifndef BASE_DATA_H
#define BASE_DATA_H

#include <list>
#include <map>
#include <string>

#include <Lum/Base/Object.h>

namespace Base {
  class Link;
  typedef Link *LinkPtr;

  class Data;
  typedef Data* DataPtr;

  class Type;
  typedef Type *TypePtr;

  typedef std::list<DataPtr>                 DataList;
  typedef std::list<DataPtr>::iterator       DataIterator;
  typedef std::list<DataPtr>::const_iterator DataConstIterator;

  typedef std::list<LinkPtr>                 LinkList;
  typedef std::list<LinkPtr>::iterator       LinkIterator;
  typedef std::list<LinkPtr>::const_iterator LinkConstIterator;

  typedef unsigned long                      Id;

  class Type
  {
  private:
    std::wstring                          name;
    std::map<std::wstring,Id>             attrIds;
    std::map<Id,std::wstring>             attrNames;

    static std::map<std::wstring,TypePtr> typeNameMap;

  public:
    static std::list<TypePtr>             typeList;

  public:
    Type(const std::wstring& name);
    virtual ~Type();

    std::wstring GetName() const;
    virtual unsigned long GetMaxAttribute() const = 0;
    bool GetIdFromName(const std::wstring& name, Id &id);
    bool GetNameFromId(Id id, std::wstring& name);
    virtual bool IsPrimaryType() const = 0;

    void RegisterAttribute(const std::wstring& name, Id id);

    virtual DataPtr Create() const = 0;

    static void RegisterType(TypePtr type);
    static TypePtr GetTypeByName(const std::wstring& name);
    static TypePtr GetTypeByIndex(size_t index);
  };


  class Data
  {
  public:
    enum State {
      created,    //< The object has been created, but is not yet stored
      registered, //< The object has been registered (and will be stored)
      deleted     //< The object has been deleted and should not be accesed anymore
    };

  private:
    Id       id;
    TypePtr  type;
    State    state;
    LinkList links;

  public:
    Data(TypePtr type);
    virtual ~Data();

    virtual bool IsGreater(Data* other) const;

    void SetId(Id id);
    Id GetId() const;
    TypePtr GetType() const;

    virtual void PostLoadCallback();

    void SetState(State state);
    State GetState() const;
    bool IsRegistered() const;
    virtual bool IsSaveable() const;

    void LinkData(const std::wstring& name, DataPtr other);
    const LinkList& GetLinks() const;
    const LinkList& GetLinks();
    size_t GetLinkCount() const;
    void Remove(LinkPtr link);

    virtual std::wstring GetDisplayText() const = 0;

    virtual void SetAttribute(Id id, const std::wstring& value) = 0;
    virtual std::wstring GetAttribute(Id id) const = 0;
    virtual bool IsMultilineAttribute(Id id) const;
  };

  class Link
  {
  private:
    std::wstring name;
    DataPtr      first;
    DataPtr      second;
  public:
    Link(const std::wstring& name, DataPtr first, DataPtr second);

    DataPtr GetFirst() const;
    DataPtr GetSecond() const;

    std::wstring GetName() const;
    void SetName(const std::wstring& name);
  };
}

#endif
