#include "Database.h"

#include <Lum/Config/Config.h>

Database *database=new Database();

#include <iostream>
#include <Lum/Base/String.h>

void Database::Insert(::Base::DataPtr data)
{
  database.push_back(data);
  data->SetState(::Base::Data::registered);
}

void Database::Remove(::Base::DataPtr data)
{
  database.remove(data);
  data->SetState(::Base::Data::deleted);
}

bool Database::Load(const std::wstring& file)
{
  Lum::Config::Node                     *top,*objects,*refs;
  Lum::Config::ErrorList                errors;
  std::map< ::Base::Id,::Base::DataPtr> idToObject;

  top=Lum::Config::LoadConfigFromXMLFile(file,errors);

  if (top==NULL) {
    Lum::Config::ErrorList::const_iterator iter;

    iter=errors.begin();
    while (iter!=errors.end()) {
      std::cerr << Lum::Base::WStringToString(iter->GetError()) << std::endl;
      ++iter;
    }
    return false;
  }

  objects=top->GetChild(L"Objects");
  if (objects!=NULL) {
    for (Lum::Config::Node::NodeList::const_iterator iter=objects->GetChildren().begin(); iter!=objects->GetChildren().end(); ++iter) {
      Lum::Config::Node* object=*iter;
      Base::DataPtr      data;
      Base::TypePtr      type;
      size_t             id;

      type=Base::Type::GetTypeByName(object->GetName());
      if (type==NULL) {
        continue;
      }

      data=type->Create();
      if (data==NULL) {
        continue;
      }

      if (!object->GetAttribute(L"id",id)) {
        continue;
      }

      data->SetId(id);

      for (Lum::Config::Node::NodeList::const_iterator iter=object->GetChildren().begin(); iter!=object->GetChildren().end(); ++iter) {
        Lum::Config::Node* attr=*iter;
        Base::Id           id;

        if (type->GetIdFromName(attr->GetName(),id)) {
          if (!attr->GetValue().empty()) {
            data->SetAttribute(id,attr->GetValue());
          }
        }
      }

      idToObject[data->GetId()]=data;
      Insert(data);
    }
  }

  refs=top->GetChild(L"References");
  if (refs!=NULL) {
    for (Lum::Config::Node::NodeList::const_iterator iter=refs->GetChildren().begin(); iter!=refs->GetChildren().end(); ++iter) {
      Lum::Config::Node *ref=*iter;
      size_t            a,b;
      std::wstring      label;

      if (!ref->GetAttribute(L"a",a) || !ref->GetAttribute(L"b",b) || !ref->GetAttribute(L"label",label)) {
        continue;
      }

      std::map< ::Base::Id,::Base::DataPtr>::iterator aIter,bIter;

      aIter=idToObject.find(a);
      bIter=idToObject.find(b);

      if (aIter!=idToObject.end() && bIter!=idToObject.end()) {
        aIter->second->LinkData(label,bIter->second);
      }
    }
  }

  idToObject.clear();

  // First, give all entries an unique id
  ::Base::DataConstIterator iter;

  iter=database.begin();
  while (iter!=database.end()) {
    (*iter)->PostLoadCallback();

    ++iter;
  }

  return true;
}

bool Database::Save(const std::wstring& file)
{
  ::Base::DataConstIterator iter;
  Lum::Config::Node         *top,*objects,*refs;
  ::Base::Id                id=0;

  top=new Lum::Config::Node();
  top->SetName(L"PID");

  objects=new Lum::Config::Node();
  objects->SetName(L"Objects");
  top->Add(objects);

  // First, give all entries an unique id
  iter=database.begin();
  while (iter!=database.end()) {
    if ((*iter)->IsSaveable()) {
      (*iter)->SetId(id);
      id++;
    }

    ++iter;
  }

  iter=database.begin();
  while (iter!=database.end()) {
    Lum::Config::Node *data;

    if ((*iter)->IsSaveable()) {
      data=new Lum::Config::Node();
      data->SetName((*iter)->GetType()->GetName());
      data->SetAttribute(L"id",(*iter)->GetId());

      for (size_t x=0; x<=(*iter)->GetType()->GetMaxAttribute(); x++) {
        std::wstring name;

        if ((*iter)->GetType()->GetNameFromId(x,name)) {
          std::wstring value;

          value=(*iter)->GetAttribute(x);

          if (!value.empty()) {
            data->SetAttribute(name,value,(*iter)->IsMultilineAttribute(x));
          }
        }
      }

      objects->Add(data);
    }

    ++iter;
  }

  refs=new Lum::Config::Node();
  refs->SetName(L"References");
  top->Add(refs);

  iter=database.begin();
  while (iter!=database.end()) {

    if ((*iter)->IsSaveable()) {
      ::Base::LinkConstIterator refIter;

      const ::Base::LinkList &refList=(*iter)->GetLinks();

      refIter=refList.begin();
      while (refIter!=refList.end()) {
        if ((*refIter)->GetFirst()==(*iter)) { // Store links only once
          Lum::Config::Node *ref;

          ref=new Lum::Config::Node();
          ref->SetName(L"Ref");
          ref->SetAttribute(L"a",(*refIter)->GetFirst()->GetId());
          ref->SetAttribute(L"b",(*refIter)->GetSecond()->GetId());
          ref->SetAttribute(L"label",(*refIter)->GetName());

          refs->Add(ref);
        }

        ++refIter;
      }
    }

    ++iter;
  }

  return Lum::Config::SaveConfigToXMLFile(file,top);
}
