import QtQuick 1.0
import QtMobility.sensors 1.2
import QtMultimediaKit 1.1

import "sizer.js" as Sizer;
import "logic.js" as Logic

Rectangle {
    id: board
    color: "#000000"
    width: 640
    height:  360
    focus: true

    signal exitClicked

    property string backgroundImage
    property int score
    property int currentLevel
    property int numLevels
    property int numBales
    property int numPlanks
    property int numBricks
    property real timeTaken

    Image {
        id: background
        source: backgroundImage
        smooth: true
        z: 0
        fillMode: Image.Stretch
        anchors.fill: parent
    }

    MouseArea {
           id: themousearea
           x: 0
           y: 30

           drag.minimumY: -1000
           drag.minimumX: -1000
           drag.maximumY: 1000
           drag.maximumX: 1000
           anchors.rightMargin: 0
           anchors.bottomMargin: 1
           anchors.leftMargin: 0
           anchors.topMargin: 30
           anchors.fill: board
           onPressed: {
                Logic.screenTap();
           }
       }

    Text {
        id: scoretext
        text: "Score: " + score
        color: "#ffffff"
        x: 5
        y: 5
        font.pixelSize: Sizer.smallFontSize();
    }

    Text {
        id: leveltext
        text: "Level: " + currentLevel
        color: "#ffffff"
        y: 5
        font.pixelSize: Sizer.smallFontSize();

        anchors.horizontalCenter: board.horizontalCenter
    }


    Text {
        id: statetext
        text: gameState
        color: "white"
        y:5
        anchors.left: scoretext.right
        anchors.leftMargin: 10
        visible: false
    }

    Text {
        id: starttext
        text: "Tap screen to start"
        color: "#000000"
        anchors.horizontalCenter: board.horizontalCenter
        anchors.verticalCenter: board.verticalCenter
        font.pixelSize: Sizer.largeFontSize();
    }

    Item {
        id: messagecontainer
        height: 60
        width:  board.width - 100
        anchors.horizontalCenter: board.horizontalCenter
        visible: true
        state: "HIDDEN"

        z:20

        states: [
            State {
                name: "HIDDEN"
                PropertyChanges { target: messagecontainer; y: board.height + 5}
            },
            State {
                name: "VISIBLE"
                PropertyChanges { target: messagecontainer; y: (board.height - messagecontainer.height)}
            }
        ]

        transitions: Transition {
            NumberAnimation { properties: "y"; easing.type: Easing.InOutQuad }
        }

        Image {
            height: parent.height
            width: parent.width
            source: "pics/sign.svg"
        }

        Text {
            id: messagetext
            text: ""
            font.pixelSize: 20
            font.bold: true
            color: "#ffffff"
            anchors.horizontalCenter: messagecontainer.horizontalCenter
            anchors.verticalCenter: messagecontainer.verticalCenter
            visible: true

            Timer {
                id: messagetimer
                interval:  1000
                running: false
                repeat: false
                onRunningChanged: {
                    if (messagetimer.running) {
                        messagecontainer.state = "VISIBLE";
                    }
                }

                onTriggered: {
                    messagecontainer.state = "HIDDEN";
                }
            }
        }
    }

    Paddle {
        id: thepaddle
        x: (board.width - thepaddle.width) / 2
        y: board.height - (thepaddle.height * 3) - 3
        width: Sizer.paddleWidth()
        height: Sizer.paddleHeight()
    }
    Chick {
        id: thechick
        x: (thepaddle.x + thepaddle.width / 2) - width/2
        y: thepaddle.y - height
    }

    Timer {
         id: heartbeat;
         interval: 40
         running: false;
         repeat: true;
         onTriggered: Logic.mainEvent();
     }

    Timer  {
        id: deadtimer
        interval: 5000
        running: false
        repeat: false
        onTriggered: {
            Logic.cmdNotRunning();
        }
    }

    Image {
        id: closebutton

        source: "pics/x.svg"
        width: 48
        height: 48
        anchors.right: parent.right
        anchors.rightMargin: 0
        anchors.top: parent.top
        anchors.topMargin: 0
        fillMode: Image.PreserveAspectFit
        smooth: true

        MouseArea {
               id: closearea
               anchors.fill: parent
               }
           }

    Rectangle {
        id: topLine;
        width: parent.width
        height:  1
        x: 0
        y: 32
        color: "#ffffff"
        visible: false
    }

    Accelerometer  {
        id: accelerometer
        Component.onCompleted: start()
        onReadingChanged: {
            var r = reading
            //Logic.scheduleDirection((r.y) * 9); //Symbian/Meego
            Logic.scheduleDirection((r.x) * -9); //Maemo
        }
    }

    SoundEffect {
        id: splatSound
        //source: "sounds/splat.wav"
    }

    LevelScoreBoard {
        id: levelScoreBoard
        anchors.centerIn: parent
        z: 1000
        width: 400
        height: 300
        state: "HIDDEN"
    }

//===================Functions================================


    Component.onCompleted: startupFunction();

    function startupFunction() {
        closearea.clicked.connect(exitClicked)
        loadSettings();
        Logic.cmdNotRunning();
        levelScoreBoard.hideBoard();
    }

    //Called from the main window when X clicked
    function endGame()
    {
        Logic.cmdDead();
    }

    Keys.onPressed: {
        if (event.key == Qt.Key_P) {
            if (gameState == "RUNNING") {
                Logic.cmdPause();
            } else if (gameState == "PAUSED") {
                Logic.cmdResume();
            }
        } else if (event.key == Qt.Key_Q) {
            if (gameState != "NOTRUNNING") {
                Logic.cmdDead();
            }
        } else if (event.key == keyLeft) {
            Logic.scheduleDirection(-15);
        } else if (event.key == keyRight) {
            Logic.scheduleDirection(15);
        } else if (event.key == keyFire) {
            Logic.screenTap();
        } else if (event.key == Qt.Key_F) {
            Logic.cmdLevelComplete();
        }

        event.accepted = true;
    }

    Keys.onReleased: {
        if (event.key == keyLeft || event.key == keyRight) {
            Logic.scheduleDirection (0)
        }
    }

    function displayMessage(msg) {
        messagetext.text = msg;
        messagecontainer.state = "VISIBLE";
    }

    function hideMessage() {
        messagecontainer.state = "HIDDEN";
    }

    function showScoreBoard() {
        levelScoreBoard.currentLevel = currentLevel;
        levelScoreBoard.numBales = numBales;
        levelScoreBoard.numPlanks = numPlanks;
        levelScoreBoard.numBricks = numBricks;
        levelScoreBoard.timeTaken = timeTaken;

        levelScoreBoard.showBoard();
        levelScoreBoard.calculateTotals();
    }

    function hideScoreBoard() {
        levelScoreBoard.currentLevel = 0;
        levelScoreBoard.numBales = 0;
        levelScoreBoard.numPlanks = 0;
        levelScoreBoard.numBricks = 0;
        levelScoreBoard.totalBales = 0;
        levelScoreBoard.totalPlanks = 0;
        levelScoreBoard.totalBricks = 0;
        levelScoreBoard.timeBonus = 0;
        levelScoreBoard.timeTaken = 0;

        levelScoreBoard.hideBoard();
    }

}

