#include <QtCore>
#include <QGraphicsView>
#include <QMessageBox>
#include <QFileDialog>
#include <QDesktopServices>
#include <QKeyEvent>
#include "pageeditorwindow.h"
#include "ui_pageeditorwindow.h"
#include "maemo5paperbackgrounddialog.h"

#ifdef Q_WS_MAEMO_5
#include <QMaemo5InformationBox>
#include <QtGui/QX11Info>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#endif

PageEditorWindow::PageEditorWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::PageEditorWindow)
{
    ui->setupUi(this);

    zooming = false;

    ui->toolBar->setContextMenuPolicy(Qt::PreventContextMenu);
    QList<QAction*> actions = ui->drawpad->actions();
    ui->toolBar->addActions(actions);

//    ui->toolBar->addSeparator();
    actions = ui->drawpad->directionActions();
    ui->toolBar->addActions(actions);

    ui->toolBar->addAction(ui->actionFullScreen);

    m_background.setColor(Qt::white);

    m_page = new StandardPage(this);
    open(-1);

    menuBar()->addAction(ui->actionBackground);
    menuBar()->addAction(ui->actionDelete);
    menuBar()->addAction(ui->actionExport);
    setMenuVisible(false);

    connect(ui->drawpad,SIGNAL(modeChanged()),
            this,SLOT(onModeChanged()));

#ifdef Q_WS_MAEMO_5
    if (!winId()) {
        qWarning("Can't grab keys unless we have a window id");
    }

    unsigned long val = 1;
    Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
    if (!atom) {
        qWarning("Unable to obtain _HILDON_ZOOM_KEY_ATOM. This example will only work "
                 "on a Maemo 5 device!");
        return;
    }

    XChangeProperty (QX11Info::display(),
            winId(),
            atom,
            XA_INTEGER,
            32,
            PropModeReplace,
            reinterpret_cast<unsigned char *>(&val),
            1);
#endif
}

PageEditorWindow::~PageEditorWindow()
{
    delete ui;
}

void PageEditorWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void PageEditorWindow::closeEvent ( QCloseEvent * event ){
    Q_UNUSED(event);
    if (ui->drawpad->paper()->dirty()) {
        qDebug() << "Auto save on quit";
        save();
    }
    setMenuVisible(false);
}

void PageEditorWindow::showEvent(QShowEvent *event){
    Q_UNUSED(event);
    setMenuVisible(true);
}

void PageEditorWindow::keyPressEvent(QKeyEvent* event){
    qreal step = 0.2;
    switch (event->key()) {
        case Qt::Key_F7:
            zoom( 1 + step);
            break;

        case Qt::Key_F8:
            zoom(1 - step);
            break;
    }
    QWidget::keyPressEvent(event);

}

void PageEditorWindow::setMenuVisible(bool val){
    ui->actionBackground->setVisible(val);
    ui->actionDelete->setVisible(val);
    ui->actionExport->setVisible(val);
}

void PageEditorWindow::setViewportSize(QSize val){
    viewportSize = val;
    int margin = 30;
    QSize grid(viewportSize.width() - margin *2 ,
               viewportSize.height() - margin *2 );
    ui->drawpad->setGridSize(grid);
}

bool PageEditorWindow::open(int pageId) {
    clear();

    bool res = m_page->load(pageId);
    Q_ASSERT(res);

    SketchPaper* paper = m_page->content();
    ui->drawpad->setPaper(paper);
    setCenter();
    zoomNormal();

    return true;
}

void PageEditorWindow::newPage(int bookId){
    clear();
    m_page->newPage(bookId);

    SketchPaper* paper = m_page->content();

    paper->setBackground(m_background);
    ui->drawpad->setPaper(paper);
    setCenter();
    zoomNormal();
}

void PageEditorWindow::clear(){
}

bool PageEditorWindow::save() {
    bool res = m_page->save(ui->drawpad->paper(),
                            QRect(0,0,800,670)
                            );

    if (res){
        emit saved(m_page->entity()->id());
    }

    return res;
}

Drawpad* PageEditorWindow::drawpad(){
    return ui->drawpad;
}

void PageEditorWindow::setDefaultBackground(PPBackground val){
    m_background = val;
}

PPBackground PageEditorWindow::lastSelectedBackground(){
    return m_background;
}

void PageEditorWindow::on_actionDelete_triggered()
{
    QMessageBox msg(QMessageBox::Information, "",
                      tr("Delete selected sketch?"),
                      QMessageBox::Yes	 | QMessageBox::No	,
                      this);

    if ( msg.exec() == QMessageBox::Yes ) {
        int id = m_page->entity()->id();
        m_page->remove();
        ui->drawpad->paper()->setDirty(false);
        close();
        emit deleted(id);
    }
}

void PageEditorWindow::on_actionExport_triggered()
{

    QDateTime datetime = QDateTime::currentDateTime();
    QDate date = datetime.date();
    QTime time = datetime.time();

    QString filename = "%7/%1%2%3-%4%5%6.png";
    qDebug() << filename;

    filename = filename.arg(date.year())
                   .arg(date.month(),2,10,QChar('0'))
                   .arg(date.day(),2,10,QChar('0'))
                   .arg(time.hour(),2,10,QChar('0'))
                   .arg(time.minute(),2,10,QChar('0'))
                   .arg(time.second(),2,10,QChar('0'))
                   .arg(QDir::homePath() + QLatin1String("/MyDocs"));

    QString path = QFileDialog::getSaveFileName(this,
                                             tr("Export PNG image"),
                                             filename
                                             );
    if (!path.isNull()) {
        if (!ui->drawpad->save(path)){
#ifdef Q_WS_MAEMO_5

            QMaemo5InformationBox::information(this,
                                               tr("Failed to save to %1").arg(path),
                                               QMaemo5InformationBox::NoTimeout);

#endif
        } else {
            ui->drawpad->paper()->setDirty(false);
        }
    }
}

void PageEditorWindow::setCenter() {
    QRect rect(0,0,viewportSize.width(),viewportSize.height());
    qDebug() << __func__ << rect;

//    ui->drawpad->view()->ensureVisible(rect,0);
    ui->drawpad->view()->centerOn(rect.center());
}

void PageEditorWindow::zoom(qreal factor){
    zooming = true;

    ui->drawpad->view()->scale(factor,factor);
    ui->drawpad->setMode(Drawpad::FreeMoveMode);
}

void PageEditorWindow::zoomNormal(){
    ui->drawpad->view()->resetMatrix();
    zooming = false;
}

void PageEditorWindow::on_actionBackground_triggered()
{
    Maemo5PaperBackgroundDialog dialog;

    PPBackground bg = ui->drawpad->paper()->background();

    dialog.setDefaultBackground(bg);
    if (dialog.exec() == QDialog::Accepted ) {
        m_background = dialog.selectedBackground();
        ui->drawpad->paper()->setBackground(m_background);
        ui->drawpad->refresh();
    }
}

void PageEditorWindow::on_actionFullScreen_triggered()
{
    if (isFullScreen()){
        showNormal();
    } else {
        showFullScreen();
    }
}

void PageEditorWindow::onModeChanged(){
    if (zooming) {
        Drawpad::Mode mode = ui->drawpad->mode();
        if (mode != Drawpad::FreeMoveMode) {
            zoomNormal();
        }
    }
}
