#include <QtCore>
#include <QPainter>
#include <QEvent>
#include <QGraphicsSceneEvent>
#include "graphicsstroke.h"

GraphicsStroke::GraphicsStroke(QGraphicsItem *parent) :
    QGraphicsItem(parent)
{
    m_pen.setColor(Qt::red);
    m_pen.setStyle(Qt::SolidLine);
    m_pen.setCapStyle(Qt::RoundCap);
    m_pen.setJoinStyle(Qt::RoundJoin);
    m_pen.setWidth(3);
}

void GraphicsStroke::paint ( QPainter * painter, const QStyleOptionGraphicsItem * option, QWidget * widget )  {
    Q_UNUSED(option);
    Q_UNUSED(widget);

    if (m_polygon.size() > 0){
        painter->save();
//        qDebug() << __func__ << m_pen.joinStyle() << m_pen.capStyle();
        painter->setRenderHint(QPainter::Antialiasing , true);

        painter->setPen(m_pen);
        painter->drawPolyline(m_polygon);

        painter->restore();
    }
}

QRectF GraphicsStroke::boundingRect() const {
    return m_boundingRect;
}

QPainterPath GraphicsStroke::shape () const {
    QPainterPath path;
    path.addPolygon(m_polygon);
    return path;
}

QString GraphicsStroke::tagName(){
    return "stroke";
}

QDomElement GraphicsStroke::createDomElement(QDomDocument &doc){
    QDomElement element = doc.createElement(tagName());

    QStringList list;
    QVector<QPointF> pts = points();
    foreach (QPointF pt , pts) {
        QString str = QString("%1,%2").arg(pt.x()).arg(pt.y());
        list << str;
    }

    element.setAttribute("points",list.join(" "));
    element.setAttribute("color",m_pen.color().name());
    element.setAttribute("width",m_pen.width());
    return element;
}

bool GraphicsStroke::setFromDomElement(QDomElement &elem){
    if (elem.tagName() != tagName())
        return false;

    prepareGeometryChange();

    clear();
    QString color = elem.attribute("color","#000000");
    QColor c(color);
    m_pen.setColor(c);

    int width = elem.attribute("width","1").toInt();
    m_pen.setWidth(width);

    QString val = elem.attribute("points","");
    QStringList list = val.split(" ");

//    QVector<QPointF> pts;
//    QPointF last;

    for (int i = 0 ; i < list.size() ; i++) {
        QStringList xy = list.at(i).split(",");

        if (xy.size() == 2) {
            qreal x = xy.at(0).toFloat();
            qreal y = xy.at(1).toFloat();
            QPointF pt(x,y);

            append(pt,false);
            /*
            if (i != 0) {
                append(last,false);
                append(pt,false);
            } else {
                append(pt,false);
                append(pt,false);
            }

            last = pt;
            */
        }

    }

    updateBoundingRect();
    return true;
}

QPolygonF GraphicsStroke::polygon(){
    return m_polygon;
}

void GraphicsStroke::setPolygon(QPolygonF val){
    prepareGeometryChange();

    m_polygon = val;
    updateBoundingRect();
}

QVector<QPointF> GraphicsStroke::points(){
    QVector <QPointF> res;
    int n = m_polygon.size();
    if (n > 0){
        QPointF prev = m_polygon.at(0);
        res << prev;

        for (int i = 1 ; i< n ;i++){ // Remove duplicated points
            QPointF pt = m_polygon.at(i);
//            qDebug() << prev << pt ;
            if (pt!= prev) {
//            if ( !(pt.x() == prev.x() && pt.y() == prev.y())){
                res << pt;
                prev = pt;
            }
        }
    }

    return res;
}


void GraphicsStroke::append(const QPointF pt,bool refresh){

    m_polygon.append(pt);

    if (refresh){
        prepareGeometryChange();

        qreal x1,y1,x2,y2;
        m_boundingRect.getCoords(&x1,&y1,&x2,&y2);
        m_boundingRect.setCoords(qMin<qreal>(x1,pt.x()),
                                 qMin<qreal>(y1,pt.y()),
                                 qMax<qreal>(x2,pt.x()),
                                 qMax<qreal>(y2,pt.y()));
        update();
    }
}

void GraphicsStroke::appendLast(const QPointF pt){
    prepareGeometryChange();

    append(pt,false);
    updateBoundingRect();
}

void GraphicsStroke::clear() {
    prepareGeometryChange();

    m_polygon.clear();
    updateBoundingRect();
}

void GraphicsStroke::setPen(const QPen& val){
    m_pen = val;
}

QPen GraphicsStroke::pen(){
    return m_pen;
}

void GraphicsStroke::updateBoundingRect(){
    m_boundingRect = m_polygon.boundingRect();
}
