// Include C header files before Qt header files
// to avoid name collisions with Qt's reserved words
// such as "signal".
//
// Qt header files
#include "panicbuttonmainwindow.h"
#include "ui_panicbuttonmainwindow.h"
#include <QContactDetailFilter>
#include <QContactGuid>
#include <QContactIntersectionFilter>
#include <QContactPhoneNumber>
#include <QContactThumbnail>
#include <QDebug>
#include <QDesktopServices>
#include <QErrorMessage>
#include <QInputDialog>
#include <QListWidgetItem>
#include <QMessage>
#include <QMessageAddress>
#include <QMessageBox>
#include <QPixmap>
#include <QSettings>
#include <QStandardItem>
#include <QtDebug>
#include <QUrl>

PanicButtonMainWindow::PanicButtonMainWindow(QWidget *parent) :
        QMainWindow(parent),
        ui(new Ui::PanicButtonMainWindow),
        allContactsIndex(0),
        emergencyContactsIndex(0),
        gpsUpdateIntervalSet(false)
{
    ui->setupUi(this);
    ui->panicButton->setStyleSheet("color:green");

    // Draw the icon on the right instead of the left
    chooserDialog.setLayoutDirection(Qt::RightToLeft);
    chooserDialog.hide();

    QSettings settings("Whatever", "PanicButton", this);
    emergencyContacts = settings.value("Emergency Contacts").toMap();
    panicMessage = settings.value("Panic Message", tr("This is my location")).toString();
    gpsUpdateInterval = settings.value("GPS Update Interval", 60).toInt();

    menuBar = new QMenuBar();
    aboutAction = new QAction(tr("About Panic Button"), this);
    connect(aboutAction, SIGNAL(triggered()), this, SLOT(about()));
    menuBar->addAction(aboutAction);

    // TODO: Disable this entry until we have updated emergencyContacts.
    // TODO: Leave the entry enabled and if it's selected before
    // the updates arrive, ask if the users wants to cancel the
    // update.
    setContactsAction = new QAction(tr("Choose Contacts"), this);
    connect(setContactsAction, SIGNAL(triggered()), this, SLOT(setContacts()));
    menuBar->addAction(setContactsAction);

    clearContactsAction = new QAction(tr("Clear Contacts"), this);
    connect(clearContactsAction, SIGNAL(triggered()), this, SLOT(clearContacts()));
    menuBar->addAction(clearContactsAction);

    setPanicMessageAction = new QAction(tr("Set Panic Message"), this);
    connect(setPanicMessageAction, SIGNAL(triggered()), this, SLOT(setPanicMessage()));
    menuBar->addAction(setPanicMessageAction);

    setGPSUpdateIntervalAction = new QAction(tr("Set GPS Update Interval"), this);
    connect(setGPSUpdateIntervalAction, SIGNAL(triggered()), this, SLOT(setGPSUpdateInterval()));
    menuBar->addAction(setGPSUpdateIntervalAction);

    helpAction = new QAction(tr("Help"), this);
    connect(helpAction, SIGNAL(triggered()), this, SLOT(help()));
    menuBar->addAction(helpAction);

    setMenuBar(menuBar);

    service = new QMessageService(this);
    if (service) {
        connect(service, SIGNAL(stateChanged(QMessageService::State)),
                this, SLOT(messageServiceStateChanged(QMessageService::State)));
        messageServiceReady = true;
    }

    source  = QGeoPositionInfoSource::createDefaultSource(this);
    // TODO: schedule another attempt to createDefaultSource()
    // if this one fails.
    //
    // TODO: Display a yellow banner "Getting GPS fix"
    // until the first time positionUpdated() is called.
    if (source) {
        connect(source, SIGNAL(positionUpdated(QGeoPositionInfo)),
                this, SLOT(positionUpdated(QGeoPositionInfo)));
        source->startUpdates();
    }

    connect(&chooserDialog, SIGNAL(accepted()), this, SLOT(accept()));
    connect(&chooserDialog, SIGNAL(rejected()), this, SLOT(reject()));

    connect(&allContactsFetchRequest, SIGNAL(stateChanged(QContactAbstractRequest::State)),
            this, SLOT(allContactsFetchRequestStateChanged(QContactAbstractRequest::State)));
    connect(&allContactsFetchRequest, SIGNAL(resultsAvailable()),
            this, SLOT(allContactsFetchRequestResultsAvailable()));
    allContactsFetchRequest.setManager(&cm);

    connect(&emergencyContactsFetchRequest, SIGNAL(stateChanged(QContactAbstractRequest::State)),
            this, SLOT(emergencyContactsFetchRequestStateChanged(QContactAbstractRequest::State)));
    connect(&emergencyContactsFetchRequest, SIGNAL(resultsAvailable()),
            this, SLOT(emergencyContactsFetchRequestUpdateResultsAvailable()));

    emergencyContactsFetchRequest.setManager(&cm);

    if (buildEmergencyContactsFetchRequest()) {
        // TODO: Display a yellow banner "Getting stored emergency contacts from contact manager"
        // until emergencyContactsFetchRequestStateChanged() is called with
        // QContactAbstractRequest::FinishedState
        emergencyContactsFetchRequest.start();
    }

}

PanicButtonMainWindow::~PanicButtonMainWindow()
{

    QSettings settings("Whatever", "PanicButton", this);
    settings.setValue("Emergency Contacts", emergencyContacts);
    settings.setValue("Panic Message", panicMessage);
    settings.setValue("GPS Update Interval", gpsUpdateInterval);

    delete ui;

    if (service)
        delete service;

    if (source)
        delete source;
}

void PanicButtonMainWindow::about() {
    QMessageBox::about(this,
                       tr("About Panic Button"),
                       tr("<h1>Panic Button &copy; 2010 by David Talmage</h1><p>Version 0.3</p>\
                          <p>Send your location to your important people when you press the Panic Button.</p>"));
}

void PanicButtonMainWindow::accept()
{
    emergencyContacts.clear();
    QList<QStandardItem *> newEmergencyContacts = chooserDialog.selectedItems();
    foreach(QStandardItem *item, newEmergencyContacts) {
        QString guid = item->data(Qt::UserRole).toString();
        qDebug() << "accept(): Adding " << guid << ":"
                << allContacts.value(guid) << " to emergencyContacts";
        emergencyContacts.insert(guid, allContacts.value(guid));
    }

    chooserDialog.clear();
    allContacts.clear();
}

void PanicButtonMainWindow::addToChooserDialog(QContactFetchRequest &request)
{
    QList<QContact> results = request.contacts();
    for (; allContactsIndex < results.size(); allContactsIndex++) {
        QContact contact = results.at(allContactsIndex);
        QContactGuid guid = contact.detail<QContactGuid>();
        QList<QContactPhoneNumber> numbers =
                contact.details<QContactPhoneNumber>();

        // Choose only mobile numbers in numbers.
        foreach(QContactPhoneNumber number, numbers) {
            if (number.subTypes().contains(QContactPhoneNumber::SubTypeMobile)) {

                QStandardItem *item = new QStandardItem(contact.displayLabel());

                QContactThumbnail thumbnail = contact.detail<QContactThumbnail>();
                item->setIcon(QIcon(QPixmap::fromImage(thumbnail.thumbnail())));
                item->setData(guid.guid(), Qt::UserRole);
                // Alignment trick:
                // Make the text flush left.  This requires Qt::AlignAbsolute
                // because we set chooserDialog's layout direction to Qt::RightToLeft
                // in the ctor.
                item->setTextAlignment(Qt::AlignLeft|Qt::AlignVCenter|Qt::AlignAbsolute);
                allContacts.insert(guid.guid(), number.number());
                chooserDialog.insertItem(item, emergencyContacts.contains(guid.guid()));
            }
        }
    }

    qDebug() << "addToChooserDialog(): allContactsIndex=" << allContactsIndex;
}

void PanicButtonMainWindow::addToEmergencyContacts(QContactFetchRequest &request)
{
    QList<QContact> contacts = request.contacts();
//    qDebug()<< "emergencyContacts: some results!";
    for(; emergencyContactsIndex < contacts.length(); emergencyContactsIndex++) {
        QContactGuid guid = contacts.at(emergencyContactsIndex).detail<QContactGuid>();
        QList<QContactPhoneNumber> numbers =
                contacts.at(emergencyContactsIndex).details<QContactPhoneNumber>();

        qDebug() << "addToEmergencyContacts(): Retrieved contact:" << guid << " "
                << contacts.at(emergencyContactsIndex).displayLabel();

        // Choose only mobile numbers in numbers.
        foreach(QContactPhoneNumber number, numbers) {
            if (number.subTypes().contains(QContactPhoneNumber::SubTypeMobile)) {
                emergencyContacts.insert(guid.guid(), number.number());
                qDebug() << "Mobile number:" << guid << " " << number;
          }
        }
    }
}

void PanicButtonMainWindow::allContactsFetchRequestResultsAvailable()
{
    QContactFetchRequest *request = qobject_cast<QContactFetchRequest*>(QObject::sender());
//    qDebug()<< "allContacts: some results!";
    addToChooserDialog(*request);
}

void PanicButtonMainWindow::allContactsFetchRequestStateChanged(QContactAbstractRequest::State newState) {
    QContactFetchRequest *request = qobject_cast<QContactFetchRequest*>(QObject::sender());

    // TALMAGE: This might not be the right way to handle
    // an error in FinishedState
    if (newState == QContactAbstractRequest::FinishedState) {
        if (request->error() != QContactManager::NoError) {
            qDebug() << "Error" << request->error() << "occurred during fetch request!";
            return;
        }

        addToChooserDialog(*request);
        allContactsIndex = 0;

        if (chooserDialog.count() > 0) {
            chooserDialog.show();
        } else {
            QMessageBox::information(this, tr("No SMS-compatible contacts!"),
                                     tr("There are no contacts with mobile phones in your address book."));
        }
    } else if (newState == QContactAbstractRequest::CanceledState) {
        qDebug() << "Fetch operation canceled!";
    }

}

void PanicButtonMainWindow::clearContacts()
{
    // Require permission before clearing.
    int r = QMessageBox::warning(this, tr("Confirm"),
                                 tr("Clear emergency contacts?"),
                                 QMessageBox::Yes | QMessageBox::No);
    if (QMessageBox::Yes == r)
        emergencyContacts.clear();
}

bool PanicButtonMainWindow::buildEmergencyContactsFetchRequest()
{
    bool answer = false;
    if (emergencyContacts.size() > 0) {
        QContactIntersectionFilter *intersectionFilter = 0;
        QContactDetailFilter *detailFilter = 0;
        QList<QString> keys = emergencyContacts.keys();

        if (emergencyContacts.size() > 1) {
            // TODO: Track memory leak here
            intersectionFilter = new QContactIntersectionFilter();
        }

        // Make a QContactDetailFilter for the first emergencyContact
        //
        // If there is more than 1, then make a QContactIntersectionFilter
        // append() the QContactDetailFilter to it, make a QContactDetailFilter
        // for each of the remaining emergencyContacts, and append() each of
        // them to the QContactIntersectionFilter.
        //
        foreach(QString key, keys) {
            // TODO: Track memory leak here
            detailFilter = new QContactDetailFilter();
            detailFilter->setDetailDefinitionName(QContactGuid::DefinitionName, QContactGuid::FieldGuid);
            detailFilter->setValue(key);
            qDebug() << "Added guid " << key << " to filter for emergencyContactsFetchRequest";
            if (intersectionFilter) {
                intersectionFilter->append(*detailFilter);
            }
        }

        if (intersectionFilter) {
            emergencyContactsFetchRequest.setFilter(*intersectionFilter);
        } else {
            emergencyContactsFetchRequest.setFilter(*detailFilter);
        }

        answer = true;
    }

    return answer;
}

void PanicButtonMainWindow::emergencyContactsFetchRequestUpdateResultsAvailable()
{
    QContactFetchRequest *request = qobject_cast<QContactFetchRequest*>(QObject::sender());

    // TALMAGE: Must I check error() here?  Is it OK to assume that
    // error() == QContactManager::NoError will be handed to
    // emergencyContactsFetchRequestStateChanged() with
    // newState == QContactAbstractRequest::FinishedState?
    addToEmergencyContacts(*request);
}

void PanicButtonMainWindow::emergencyContactsFetchRequestStateChanged(QContactAbstractRequest::State newState) {
    QContactFetchRequest *request = qobject_cast<QContactFetchRequest*>(QObject::sender());

    // TALMAGE: This might not be the right way to handle
    // an error in FinishedState
    if (newState == QContactAbstractRequest::FinishedState) {
        if (request->error() != QContactManager::NoError) {
            qDebug() << "Error" << request->error() << "occurred during fetch request!";
            return;
        }

        addToEmergencyContacts(*request);
        emergencyContactsIndex = 0;

    } else if (newState == QContactAbstractRequest::CanceledState) {
        qDebug() << "Fetch operation canceled!";
    }
}

void PanicButtonMainWindow::help()
{
    // Help me, Spock!
    // TODO: Discover the documentation in a platform independent way.
    // TODO: Use QtAssistant to display the documentation.
    QUrl url("/opt/usr/share/panicbutton/panicbutton.html");
    url.setScheme(("file"));
    qDebug() << "help(): " << url;
    bool answer = QDesktopServices::openUrl(url);
    qDebug() << "help(): openUrl returned " << answer;
}

void PanicButtonMainWindow::messageServiceStateChanged(QMessageService::State newState)
{
    qDebug() << "messageServiceStateChanged() " << newState;

    messageServiceReady = false;

    if (QMessageService::FinishedState == newState) {
        messageServiceReady = true;
        sendMessage();
    }
}

void PanicButtonMainWindow::on_panicButton_clicked()
{
    QString completePanicMessage = QString("%1 %2").arg(panicMessage).arg(location.coordinate().toString());
    work *w = new work::work();

    w->message = completePanicMessage;
    w->recipients = phoneNumbers();
    if (w->recipients.size() > 0) {
        ui->panicButton->setStyleSheet("color:red");
        ui->panicButton->setText(tr("Panic!"));
        workQueue.append(w);
        sendMessage();
    } else {
        QMessageBox::information(this, tr("Choose a contact first"), tr("No one to send to!"));
        qDebug() << "No one to send to!";
    }
}

QList<QVariant> PanicButtonMainWindow::phoneNumbers() {
    return emergencyContacts.values();
}

// TODO: Display a yellow banner "Getting GPS fix"
// until the first time positionUpdated() is called.
void PanicButtonMainWindow::positionUpdated(const QGeoPositionInfo &info)
{
    location = info;
    if (!gpsUpdateIntervalSet) {
        source->setUpdateInterval(gpsUpdateInterval * 1000);
        gpsUpdateIntervalSet = true;
    }
}

void PanicButtonMainWindow::printEmergencyContacts()
{
    qDebug() << "Emergency Contacts";
    foreach(QString key, emergencyContacts.keys()) {
        qDebug() << "emergencyContact guid: " << key
                << " phone: " << emergencyContacts.value(key);
    }
}

void PanicButtonMainWindow::reject()
{
    // No change to emergencyContacts
    chooserDialog.clear();
    allContacts.clear();
}

/*
    Use the QMessageService to send one message from the
    work queue.
 */
void PanicButtonMainWindow::sendMessage()
{
    if (! workQueue.isEmpty()) {
        if (messageServiceReady) {
            work *w = workQueue.first();
            // Remove empty work
            // queue empty  recipients empty  action
            //  T              T              return
            //  T              F              shouldn't happen.  exit loop?
            //  F              T              advance to next work; remain in loop
            //  F              F              exit loop

            while (w->recipients.isEmpty()) {
                delete w;
                w = 0;
                workQueue.removeFirst();
                if (! workQueue.isEmpty())
                    w = workQueue.first();
                else
                    break;
            }

            if (w && (! w->recipients.isEmpty())) {
                QString phoneNumber = w->recipients.first().toString();
                w->recipients.removeFirst();
                QMessageAddress address = QMessageAddress(QMessageAddress::Phone, phoneNumber);
                QMessage message = QMessage();
                message.setType(QMessage::Sms);
                message.setBody(w->message); // second arg mime-type defaults to "text/plain".  Yay!
                message.setTo(address);

                if (w->recipients.isEmpty()) {
                    delete w;
                    w = 0;
                    workQueue.removeFirst();
                }

                qDebug() << "Sending to " << phoneNumber;
#if 0
                // DEBUG: Require permission before sending
                // DEBUG: This prevents sending to the same
                // DEBUG: phone number over and over again.

                int r = QMessageBox::warning(this, tr("Sanity Check"),
                                             QString("%1 %2").arg(tr("Send to ")).arg(phoneNumber),
                                             QMessageBox::Yes | QMessageBox::No);
                if (QMessageBox::Yes == r)
#endif
                service->send(message);
                // TALMAGE: Given the way that w->recipients is constructed,
                // must I delete each of its QStrings?
            }
        }
    } else {
        ui->panicButton->setStyleSheet("color:green");
        ui->panicButton->setText(tr("Don't Panic!"));
    }
}

// TODO: Display a yellow banner "Getting contacts from contact manager"
// until allContactsFetchRequestStateChanged() is called with
// QContactAbstractRequest::FinishedState
void PanicButtonMainWindow::setContacts()
{
    chooserDialog.clear();
    allContactsIndex = 0;
    allContactsFetchRequest.start();
}

void PanicButtonMainWindow::setGPSUpdateInterval()
{
    bool ok;
    int newInterval = QInputDialog::getInteger(this, tr("GPS Update Interval"),
                                               tr("Seconds between location reports from the GPS"),
                                               gpsUpdateInterval, 1, INT_MAX/1000, 1, &ok);
    if (ok) {
        gpsUpdateInterval = newInterval;
        // The arg is milliseconds, but gpsUpdateInterval is seconds,
        // so multiply by 1000.
        source->setUpdateInterval(gpsUpdateInterval * 1000);
        gpsUpdateIntervalSet = true;
    }
}

void PanicButtonMainWindow::setPanicMessage()
{
    QString newPanicMessage = QInputDialog::getText(this,
                                                    tr("Panic Message"),
                                                    tr("What do you want to say?"),
                                                    QLineEdit::Normal,
                                                    panicMessage);
    if (0 != newPanicMessage)
        panicMessage = newPanicMessage;
}
