/*
  PackageView - Show debian package details
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <fstream>
#include <iostream>

#include <Lum/OS/Main.h>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>

#include <Lum/Dlg/About.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Selection.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Main.h>

#include <Lum/Button.h>
#include <Lum/Dialog.h>
#include <Lum/Label.h>
#include <Lum/String.h>
#include <Lum/Table.h>
#include <Lum/Text.h>

#include "Package.h"
#include "PackageDetails.h"
#include "PackageList.h"

#include "config.h"

std::wstring             ossoVersion;

static Lum::Def::AppInfo info;

typedef Lum::Model::StdTable<Package*>     PackageModel;
typedef Lum::Base::Reference<PackageModel> PackageModelRef;

class PackageObject : public Lum::Object
{
private:
  Package           *package;
  Lum::OS::FontRef  font1;
  Lum::OS::FontRef  font2;

public:
  PackageObject()
  : font1(Lum::OS::display->GetFont()),
    font2(Lum::OS::display->GetFont(Lum::OS::Display::fontScaleFootnote))
  {
    // no code
  }

  void SetPackage(Package* package)
  {
    this->package=package;
  }

  void CalcSize()
  {
    minWidth=2*Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceObjectBorder)+
             10*font1->StringWidth(L"m",Lum::OS::Font::bold);
    minHeight=Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
              font1->height+
              Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
              font2->height+
              Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder);

    width=minWidth;
    height=minHeight;

    Object::CalcSize();
  }

  void Draw(int x, int y, size_t w, size_t h)
  {
    Object::Draw(x,y,w,h);

    if (!OIntersect(x,y,w,h)) {
      return;
    }

    /* --- */

    Lum::OS::DrawInfo *draw=GetDrawInfo();

    draw->PushForeground(parent->GetTextColor(draw));
    draw->PushFont(font1,Lum::OS::Font::bold);
    draw->DrawString(this->x,
                     this->y+Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                     font1->ascent,
                     package->GetName());

    std::wstring version=package->GetVersion();

    draw->DrawString(this->x+width-font1->StringWidth(version,Lum::OS::Font::bold)+1,
                     this->y+Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                     font1->ascent,
                     version);

    draw->PopFont();

    draw->PushFont(font2);
    draw->DrawString(this->x,
                     this->y+Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                     font1->height+
                     Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                     font2->ascent,
                     package->GetShortDescription());

    std::wstring packageSize=Lum::Base::ByteSizeToWString(package->GetInstalledSize());

    draw->DrawString(this->x+width-font2->StringWidth(packageSize)+1,
                     this->y+Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                     font1->height+
                     Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                     font2->ascent,
                     packageSize);

    draw->PopFont();

    draw->PopForeground();
  }
};

class PackageDataProvider : public PackageModel::DataProvider
{
private:
  PackageObject *object;

public:
  PackageDataProvider()
  : object(new PackageObject())
  {
    object->SetFlex(true,true);
    object->SetMinWidth(Lum::Base::Size::stdCharWidth,15);
  }

  virtual ~PackageDataProvider()
  {
    delete object;
  }

  Lum::Object* GetObject(const PackageModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 1:
      object->SetPackage(*iter);
      return object;
    default:
      return NULL;
    }
  }
};

class PackageTable : public PackageModel
{
public:
  class PackageComparator : public PackageModel::Comparator
  {
  public:
    bool operator()(const PackageModel::ElementType& a,
                    const PackageModel::ElementType& b,
                    size_t column, bool down) const
    {
      return a->GetName()<b->GetName();
    }
  };

public:
  PackageTable()
  : PackageModel(new PackageDataProvider(),
                 new PackageComparator())
  {
    // no code
  }
};

typedef Lum::Base::Reference<PackageTable> PackageTableRef;

class MainDialog : public Lum::Dialog
{
private:
  std::list<Package>                 packages;
  Lum::Model::ActionRef              showPackageAction;
  Lum::Model::ActionRef              aboutAction;
  std::wstring                       lastSearch;
  Lum::Model::StringRef              search;
  Lum::Model::ActionRef              searchTimerAction;
  PackageTableRef                    currentPackages;
  Lum::Model::SingleLineSelectionRef selection;
  PackageListTask                    packageListTask;

public:
  MainDialog()
  : showPackageAction(new Lum::Model::Action()),
    aboutAction(new Lum::Model::Action()),
    search(new Lum::Model::String(L"")),
    searchTimerAction(new Lum::Model::Action()),
    currentPackages(new PackageTable()),
    selection(new Lum::Model::SingleLineSelection()),
    packageListTask(packages)
  {
    Observe(packageListTask.GetFinishedAction());

    Observe(GetOpenedAction());
    Observe(GetClosedAction());

    Observe(search);
    Observe(searchTimerAction);

    Observe(selection);

    Observe(showPackageAction);
    Observe(aboutAction);

    DetectOssoVersion();
  }

  virtual void PreInit()
  {
    Lum::Panel            *vpanel;
    Lum::Label            *filter;
    Lum::Table            *table;
    Lum::Model::HeaderRef headerModel;

    filter=Lum::Label::Create(true,false);
    filter->AddLabel(L"Filter:",
                     Lum::String::Create(search,true,false)
                     ->ShowClearButton(true)
                     ->SetLabel(L"Filter"));

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Package",Lum::Base::Size::stdCharWidth,35,true);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetMinHeight(Lum::Base::Size::stdCharHeight,5);
    table->SetHeight(Lum::Base::Size::stdCharHeight,10);
    table->SetModel(currentPackages);
    table->SetHeaderModel(headerModel);
    table->SetSelection(selection);
    table->SetDoubleClickAction(showPackageAction);
    table->SetShowHeader(false);
    table->GetTableView()->SetAutoFitColumns(true);
    table->GetTableView()->SetAutoHSize(true);

    vpanel=Lum::VPanel::Create(true,true);

    switch (Lum::OS::display->GetTheme()->GetToolbarPosition()) {
    case Lum::OS::Theme::toolbarBottom:
      vpanel->Add(table);
      vpanel->AddSpace();
      vpanel->Add(filter);
      break;
    default:
      vpanel->Add(filter);
      vpanel->AddSpace();
      vpanel->Add(table);
      break;
    }

    SetMain(vpanel);

    Lum::Def::Menu *menu=Lum::Def::Menu::Create();

    menu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->GroupHelp()
        //->ActionHelp()
        ->ActionAbout(aboutAction)
      ->End();

    SetMenu(menu);

    packageListTask.SetParent(this);
    packageListTask.SetCaption(L"Reading package list...");

    Dialog::PreInit();
  }

  void DetectOssoVersion()
  {
    std::ifstream file;

    file.open("/etc/osso_software_version",std::ios::in);

    if (file) {
      std::string version;

      file >> version;
      file.close();

      if (version.find("2006")!=std::string::npos) {
        ossoVersion=L"OS 2006";
      }
      else if (version.find("2007")!=std::string::npos) {
        ossoVersion=L"OS 2007";
      }
      else if (version.find("2008")!=std::string::npos) {
        ossoVersion=L"OS 2008";
      }
    }

    if (ossoVersion.empty()) {
      ossoVersion=L"OS ????";
    }
  }

  void UpdateEntries()
  {
    currentPackages->Disable();
    currentPackages->Off();
    currentPackages->Clear();

    std::wstring search=this->search->Get();

    for (std::list<Package>::iterator iter=packages.begin();
         iter!=packages.end();
         ++iter) {
      if (search.empty() || iter->GetName().find(search)!=std::wstring::npos) {
        currentPackages->Append(&(*iter));
      }
    }

    currentPackages->Sort(1,true);
    currentPackages->On();
    currentPackages->Enable();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      currentPackages->Disable();
      currentPackages->Off();
      packageListTask.Start();
    }
    else if (model==showPackageAction && showPackageAction->IsFinished()) {
      Package*       package=currentPackages->GetEntry(selection->GetLine());
      PackageDetails *dialog;

      dialog=new PackageDetails(*package);
      dialog->SetParent(this);
      dialog->SetTitle(L"Details...");

      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();
      }

      delete dialog;
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      /*if (configurationChanged && !SaveConfig()) {
        std::cerr << "Error while storing configuration!" << std::endl;
      }*/
    }
    else if (model==packageListTask.GetFinishedAction() && packageListTask.GetFinishedAction()->IsFinished()) {
      UpdateEntries();
    }
    else if (model==search) {
      if (search->Empty()) {
        UpdateEntries();
        lastSearch=search->Get();
      }
      else {
        Lum::OS::display->RemoveTimer(searchTimerAction);
        Lum::OS::display->AddTimer(1,150000,searchTimerAction);
      }
    }
    else if (model==searchTimerAction && searchTimerAction->IsFinished()) {
      if (search->Get()!=lastSearch) {
        UpdateEntries();
        lastSearch=search->Get();
      }
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"Show installed applications..."));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2008, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    return Lum::OS::MainDialog<MainDialog>::Prepare();
  }
};


LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))
