# -*- coding: latin-1 -*-
#
#This file is part of Outlaw Solitaire.
#
#Copyright (c) 2010, Toni Knsl
#All rights reserved.
#
#Redistribution and use in source and binary forms, with or without 
#modification, are permitted provided that the following conditions are met:
#
#    * Redistributions of source code must retain the above copyright notice, 
#      this list of conditions and the following disclaimer.
#    * Redistributions in binary form must reproduce the above copyright notice, 
#      this list of conditions and the following disclaimer in the documentation 
#      and/or other materials provided with the distribution.
#    * The name of the authors may not be used to endorse or promote products 
#      derived from this software without specific prior written permission.
#
#THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
#ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
#WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
#DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE 
#FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
#DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
#SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
#CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
#OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
#OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
import pygame, fontlib

class Widget(pygame.sprite.DirtySprite):
    #Attributes controlled by Container
    parent = None
    sub_layer = None
    offset = None
    event_area = None
    event_area_offset = None

    #Attributes controlled by self
    rect = None
    image = None
    mouseover = None
    mousewasdown = None
    clicked = None
    
    def __init__(self):
        pygame.sprite.DirtySprite.__init__(self)

        self.mouseover = False
        self.mousewasdown = False
        self.clicked = False

    def _event_occured(self):
        pass

    def hide(self):
        self.visible = False
        self.dirty = True

    def show(self):
        self.visible = True
        self.dirty = True

class PictureBox(Widget):
    def __init__(self, image):
        self.image = image
        self.rect = self.image.get_rect()

        Widget.__init__(self)

    def change_image(self, image):
        self.image = image
        self.rect.size = self.image.get_size()
        self.dirty = True

class Label(Widget):
    bitmap_font = None
    color = None
    
    def __init__(self, text, bitmap_font, color=False):
        self.bitmap_font = bitmap_font
        self.color = color
        
        self.image = fontlib.CreateTextImage(text, bitmap_font, color)
        self.rect = self.image.get_rect()

        Widget.__init__(self)

    def change_text(self, text, justify='left'):
        fontlib.CreateTextImage(text, self.bitmap_font, self.color, self.image, justify)
        self.dirty = True

class MultilineLabel(Widget):
    bitmap_font = None
    pad_y = None
    justify = None
    color = None
    
    def __init__(self, text_lines, bitmap_font, pad_y=0, color=False, justify='left'):
        self.bitmap_font = bitmap_font
        self.pad_y = pad_y
        self.justify = justify
        self.color = color
        
        self.image = fontlib.CreateMultilineTextImage(text_lines, bitmap_font, pad_y, color, justify)
        self.rect = self.image.get_rect()

        Widget.__init__(self)

    def change_text(self, text_lines):
        fontlib.CreateMultilineTextImage(text_lines, self.bitmap_font, self.pad_y, self.color, self.justify, self.image)

class ImageButton(Widget):
    active_image = None
    inactive_image = None
    state = None
    STATE_INACTIVE = 0
    STATE_ACTIVE = 1
    
    def __init__(self, active_image, inactive_image):
        self.active_image = active_image
        self.inactive_image = inactive_image
        self.state = self.STATE_INACTIVE

        self.image = self.inactive_image        
        self.rect = self.image.get_rect()

        Widget.__init__(self)

    def _event_occured(self):
        if self.mousewasdown == True and self.state == self.STATE_INACTIVE:
            self.state = self.STATE_ACTIVE
            self.image = self.active_image
            self.dirty = True
        elif self.mousewasdown == False and self.state == self.STATE_ACTIVE:
            self.state = self.STATE_INACTIVE
            self.image = self.inactive_image
            self.dirty = True

class TextButton(ImageButton):
    def __init__(self, text, bitmap_font, active_color=False, inactive_color=False):
        active_image = fontlib.CreateTextImage(text, bitmap_font, active_color)
        inactive_image = fontlib.CreateTextImage(text, bitmap_font, inactive_color)

        ImageButton.__init__(self, active_image, inactive_image)

class TextButton2(ImageButton):
    def __init__(self, text, bitmap_font_active, bitmap_font_inactive):
        active_image = fontlib.CreateTextImage(text, bitmap_font_active)
        inactive_image = fontlib.CreateTextImage(text, bitmap_font_inactive)

        ImageButton.__init__(self, active_image, inactive_image)

class Container():
    #Attributes controlled by ContainerGroup
    parent = None
    layer = None

    #Attributes controlled by self   
    widgets = None
    rect = None
    event_handling = None

    def __init__(self):
        self.widgets = []
        self.rect = pygame.Rect((0,0), (0,0))
        self.event_handling = True

    def update(self):
        pass 

    def handle_events(self, event):
        if event.type == pygame.MOUSEMOTION or event.type == pygame.MOUSEBUTTONUP or event.type == pygame.MOUSEBUTTONDOWN:
            if self.rect.collidepoint(event.pos):
                if event.type == pygame.MOUSEMOTION:
                    for temp_widget in self.widgets:
                        if temp_widget.event_area.collidepoint(event.pos):
                            temp_widget.mouseover = True
                        else:
                            temp_widget.mouseover = False
                            temp_widget.clicked = False
                            temp_widget.mousewasdown = False
                        temp_widget._event_occured()
                elif event.type == pygame.MOUSEBUTTONDOWN:
                    for temp_widget in self.widgets:
                        if temp_widget.event_area.collidepoint(event.pos):
                            temp_widget.mouseover = True
                            temp_widget.clicked = False
                            temp_widget.mousewasdown = True
                        else:
                            temp_widget.mouseover = False
                            temp_widget.clicked = False
                            temp_widget.mousewasdown = False
                        temp_widget._event_occured()
                elif event.type == pygame.MOUSEBUTTONUP:
                    for temp_widget in self.widgets:
                        if temp_widget.event_area.collidepoint(event.pos) and temp_widget.mousewasdown == True:
                            temp_widget.mouseover = True
                            temp_widget.clicked = True
                            temp_widget.mousewasdown = False
                        else:
                            temp_widget.mouseover = False
                            temp_widget.clicked = False
                            temp_widget.mousewasdown = False
                        temp_widget._event_occured()
            else:
                for temp_widget in self.widgets:
                    temp_widget.mouseover = False
                    temp_widget.clicked = False
                    temp_widget.mousewasdown = False
                    temp_widget._event_occured()

    def post_update(self):
        for temp_widget in self.widgets:
            temp_widget.clicked = False

    def add_widget(self, widget, sub_layer=0, offset=(0,0), event_area_size=None, event_area_offset=(0,0)):
        if widget.parent:
            raise RuntimeError('Container: add_widget: Given Widget already belongs to some Container!')
        else:
            widget.parent = self
            widget.sub_layer = sub_layer
            widget.offset = offset

            widget.rect.move_ip(widget.offset)

            widget.event_area_offset = event_area_offset
            if event_area_size:
                widget.event_area = pygame.Rect((0,0), event_area_size)
                widget.event_area.center = widget.rect.center
            else:
                widget.event_area = pygame.Rect(widget.rect)

            widget.event_area.move_ip(widget.event_area_offset)
            
            self.widgets.append(widget)
            self.rect.union_ip(widget.event_area)  

    def disable_event_handling(self):
        self.event_handling = False

    def enable_event_handling(self):
        self.event_handling = True        

    def hide(self):
        for temp_widget in self.widgets:
            temp_widget.hide()

    def show(self):
        for temp_widget in self.widgets:
            temp_widget.show()             

    def dirty(self):
         for temp_widget in self.widgets:
            temp_widget.dirty = True 

    def move(self, offset):
        for temp_widget in self.widgets:
            temp_widget.rect.move_ip(offset)
            
            temp_widget.event_area.move_ip(offset)

        self.rect.move_ip(offset)

    def set_top_left(self, top_left):
        for temp_widget in self.widgets:
            temp_widget.rect.topleft = top_left
            temp_widget.rect.move_ip(temp_widget.offset)

            temp_widget.event_area.center = temp_widget.rect.center
            temp_widget.event_area.move_ip(temp_widget.event_area_offset)            

        self.rect.topleft = top_left

    def set_center(self, center):
        for temp_widget in self.widgets:
            temp_widget.rect.center = center
            temp_widget.rect.move_ip(temp_widget.offset)

            temp_widget.event_area.center = temp_widget.rect.center
            temp_widget.event_area.move_ip(temp_widget.event_area_offset) 

        self.rect.center = center

    def set_mid_bottom(self, mid_bottom):
        for temp_widget in self.widgets:
            temp_widget.rect.midbottom = mid_bottom
            temp_widget.rect.move_ip(temp_widget.offset)

            temp_widget.event_area.center = temp_widget.rect.center
            temp_widget.event_area.move_ip(temp_widget.event_area_offset)

        self.rect.midbottom = mid_bottom

class ContainerGroup():
    containers = None

    widget_group = None    
    layer_spacing = None
    
    def __init__(self, layer_spacing):
        self.layer_spacing = layer_spacing

        self.containers = []
        self.widget_group = pygame.sprite.LayeredDirty(_use_update=True)
        self.widget_group.set_timing_treshold(1000)

    def add_container(self, container, layer=0):
        if container.parent:
            raise RuntimeError('ContainerGroup: add_container: Given Container already belongs to some ContainerGroup!')
        else:        
            container.parent = self
            container.layer = layer

            self.containers.append(container)

            for temp_widget in container.widgets:
                temp_widget_layer = (layer*self.layer_spacing) + temp_widget.sub_layer
                if temp_widget.sub_layer > self.layer_spacing-1:
                    raise RuntimeError('ContainerGroup: add_container: Sub layer (%d) of Widget does not fit to layer spacing (%d)!' % (temp_widget_layer, self.layer_spacing))
                else:
                    self.widget_group.add(temp_widget, layer=temp_widget_layer)

    def disable_all_container_event_handlings(self):
        for temp_container in self.containers:
            temp_container.disable_event_handling()     

    def enable_all_container_event_handlings(self):
        for temp_container in self.containers:
            temp_container.enable_event_handling()  

    def update(self):
        for temp_container in self.containers:
            temp_container.update()
            temp_container.post_update()

    def handle_events(self, event):
        for temp_container in self.containers:
            if temp_container.event_handling:
                temp_container.handle_events(event)

    def draw(self, surface):
        return self.widget_group.draw(surface)

if __name__ == '__main__':
    pass    