/*
 * This file is part of NeHeGLES.
 *
 * NeHeGLES is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * NeHeGLES is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NeHeGLES.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <unistd.h>

#include "config.h"
#include "misc.h"
#include "icon.h"
#include "list.h"

typedef struct {
  char *shot;
  char *title;
  char *desc;
  char *link;
  char *exec;
  char *controls;
} lesson_t;

lesson_t lessons[] = {
  {
    "lesson01", "Lesson 01: Setup",
    "Setup is trivial assuming you already have a working "
    "scratchbox setup with extras-devel repository enabled. A simple "
    "'apt-get source nehegles' will install everything needed.",
    "http://wiki.maemo.org/Documentation/Maemo5_Final_Installation",
    NULL,
    NULL
  },

  {
    "lesson02", "Lesson 02: Your First Polygon",
    "This lesson shows how to place basic objects in the screen. A triangle "
    "and a simple rectangle may be boring bit they show you the basics "
    "of setting up a working GLES environment.",
    "http://nehe.gamedev.net/data/lessons/lesson.asp?lesson=02",
    "lesson02",
    "Tap screen to exit."
  },

  {
    "lesson03", "Lesson 03: Adding Color",
    "This lesson teaches you how to add colour to the objects of the "
    "previous lesson. The triangle will be using smooth/blended coloring "
    "while the square will use flat coloring.",
    "http://nehe.gamedev.net/data/lessons/lesson.asp?lesson=03",
    "lesson03",
    "Tap screen to exit"
  },

  {
    "lesson04", "Lesson 04: Rotation",
    "This lesson adds movement to the scene by rotating the triangle and "
    "square. It will teach you how to rotate an object around a point on "
    "the screen causing the object to move around the screen rather than "
    "spin on its axis.",
    "http://nehe.gamedev.net/data/lessons/lesson.asp?lesson=04",
    "lesson04",
    "Tap screen to exit\nPress CTRL-Backspace to move remove focus from the "
    "program to see how the animation is being stopped to save battery."
  },

  {
    "lesson05", "Lesson 05: 3D Shapes",
    "What would OpenGL/GLES be without 3D? This lesson will show you how "
    "to add depth to your objects.",
    "http://nehe.gamedev.net/data/lessons/lesson.asp?lesson=05",
    "lesson05",
    "Tap screen to exit\nPress CTRL-Backspace to move remove focus from the "
    "program. Unlike the previous lessons, this demo does not try to save "
    "battery and thus will continue to run in the background."
  },
};

static void on_link_clicked(GtkButton *button, gpointer user) {
  GtkWidget *parent = gtk_widget_get_toplevel(GTK_WIDGET(button));
  GtkWidget *root = g_object_get_data(G_OBJECT(parent), "root");

  lesson_t *lesson = (lesson_t *)user;
  browser_url(root, lesson->link);
}

static void on_run_clicked(GtkButton *button, gpointer user) {
  GtkWidget *parent = gtk_widget_get_toplevel(GTK_WIDGET(button));
  lesson_t *lesson = (lesson_t *)user;

  pid_t pid = fork();
  if (!pid) {                // child
    char *path = g_strdup_printf(DATADIR "%s", lesson->exec);
    printf("running %s\n", path);
    execl(path, path, NULL);
    g_free(path);
    _exit(0); // If exec fails then exit forked process.
  } else if (pid < 0)            // failed to fork
    errorf(parent, "Failed to fork process");
}

static void on_more_clicked(GtkButton *button, gpointer user) {
  lesson_t *lesson = (lesson_t *)user;

  GtkWidget *parent = gtk_widget_get_toplevel(GTK_WIDGET(button));

#ifdef MAEMO5
  GtkWidget *window = hildon_stackable_window_new();
  g_object_set_data(G_OBJECT(window), "root", parent);
#else
  GtkWidget *dialog = 
    gtk_dialog_new_with_buttons(lesson->title,
		GTK_WINDOW(parent), GTK_DIALOG_MODAL,
		GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, 
		NULL);

  gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 400);
#endif

  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);

  /* ------------------------------------------- */

  gtk_box_pack_start(GTK_BOX(vbox), label_big_new(lesson->title), 
		     FALSE, FALSE, 0);
  
  if(lesson->link) {
    GtkWidget *link = button_new_with_label("Open tutorial in browser");
    g_signal_connect(link, "clicked", G_CALLBACK(on_link_clicked), lesson);
    gtk_box_pack_start(GTK_BOX(vbox), link, FALSE, FALSE, 0);
  }

  if(lesson->exec) {
    GtkWidget *run = button_new_with_label("Run the Demo");
    g_signal_connect(run, "clicked", G_CALLBACK(on_run_clicked), lesson);
    gtk_box_pack_start(GTK_BOX(vbox), run, FALSE, FALSE, 0);
  }

  if(lesson->controls) {
    GtkWidget *frame = gtk_frame_new("Demo controls:");
    GtkWidget *ctrl = label_wrap_new(lesson->controls);
    gtk_container_add(GTK_CONTAINER(frame), ctrl);
    gtk_box_pack_start_defaults(GTK_BOX(vbox), frame);
  }

  /* ------------------------------------------- */
  
#ifdef MAEMO5
  gtk_container_add(GTK_CONTAINER(window), vbox);
  gtk_widget_show_all(window);
#else
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), vbox);
  gtk_widget_show_all(dialog);
  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
#endif
}

GtkWidget *list_new() {
  /* put a list of descriptions into a long scrollable vbox */
  GtkWidget *swin = 
    scrolled_window_new(GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);

  /* -------------------------- */
  
  int i;
  for(i=0;i<sizeof(lessons)/sizeof(lesson_t);i++) {
    if(i)
     gtk_box_pack_start(GTK_BOX(vbox), gtk_hseparator_new(), FALSE, FALSE, 0);

    GtkWidget *hbox = gtk_hbox_new(FALSE, 0);

    GtkWidget *ivbox = gtk_vbox_new(FALSE, 0);

    GtkWidget *button = gtk_button_new();
    g_signal_connect(button, "clicked", 
	     G_CALLBACK(on_more_clicked), lessons+i);
    GtkWidget *shot = icon_get_widget(vbox, lessons[i].shot);
    gtk_button_set_image(GTK_BUTTON(button), shot);
    gtk_box_pack_start(GTK_BOX(ivbox), button, FALSE, FALSE, 0);

    gtk_box_pack_start(GTK_BOX(hbox), ivbox, FALSE, FALSE, 0);

    GtkWidget *ivbox2 = gtk_vbox_new(FALSE, 0);

    GtkWidget *title = label_big_new(lessons[i].title);
    gtk_box_pack_start(GTK_BOX(ivbox2), title, FALSE, FALSE, 0);

    GtkWidget *desc = label_wrap_new(lessons[i].desc);
    gtk_box_pack_start_defaults(GTK_BOX(ivbox2), desc);

    gtk_box_pack_start_defaults(GTK_BOX(hbox), ivbox2);

    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  }
  
  /* -------------------------- */

  scrolled_window_add_with_viewport(swin, vbox);
  return swin;
}
