/*
 *  Nameday desktop applet (widget) for Maemo.
 *  Copyright (C) 2009 Roman Moravcik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <glib/gi18n-lib.h>

#include "nameday_calendar.h"
#include "nameday_dbparser.h"
#include "nameday_settings.h"

static void
nameday_calendar_on_window_destroy_event (GtkWidget *window,
					  GdkEvent  *event,
					  gpointer data);
static GtkWidget*
nameday_calendar_treeview_window (NamedayDesktopApplet *priv,
				  GtkTreeModel *tree_model);

static gboolean
nameday_calendar_filered_view_visible_func (GtkTreeModel *model,
					    GtkTreeIter *iter,
					    gpointer data)
{
	gchar *name = NULL, *ascii_name = NULL;
	const gchar *searched_name = NULL;
	gboolean found = FALSE;

	g_return_val_if_fail (data, FALSE);

	searched_name = g_ascii_strdown (data, strlen (data));

	gtk_tree_model_get (model, iter, COLUMN_NAME, &name, -1);

	if (name) {
		ascii_name = g_ascii_strdown (name,  strlen (name));
		g_free (name);
	}

	if (g_strstr_len (ascii_name, strlen (ascii_name), searched_name) != NULL)
		found = TRUE;

	if (ascii_name)
		g_free (ascii_name);

	return found;
}

static void
nameday_calendar_on_find_toolbar_search_event (HildonFindToolbar *toolbar,
						 gpointer data)
{
	NamedayDesktopApplet *priv;
	GtkWidget *window, *search_menu_item;
	GtkListStore *list_store;
	GtkTreeModel *filtered_model;
	gchar *searched_name = NULL;

	g_return_if_fail (data);

	priv = NAMEDAY_DESKTOP_APPLET (data);

	g_object_get (G_OBJECT (toolbar), "prefix", &searched_name, NULL);

	/* do nothing if searched name is empty */
        if ((searched_name == NULL) || (strcmp (searched_name, "") == 0)) {
		g_free (searched_name);
		return;
	}

	list_store = nameday_dbparser_liststore_create (priv);

	/* filtered view */
	filtered_model = gtk_tree_model_filter_new (GTK_TREE_MODEL (list_store), NULL);

	gtk_tree_model_filter_set_visible_func (GTK_TREE_MODEL_FILTER (filtered_model),
						nameday_calendar_filered_view_visible_func,
						(gpointer) searched_name,
						NULL);
	g_object_unref (list_store);

	/* create window with tree view */
	window = nameday_calendar_treeview_window (priv, filtered_model);
	g_object_unref (filtered_model);
	gtk_window_set_title(GTK_WINDOW(window), _("Search results"));

	g_signal_connect (window, "delete-event", G_CALLBACK (gtk_widget_hide_on_delete), NULL);

	/* set search menu item inactive */
	search_menu_item = g_object_get_data (G_OBJECT (priv->calendar_window), "search_menu_item");
	gtk_widget_set_sensitive (GTK_WIDGET (search_menu_item), TRUE);

	/* show window */
	gtk_widget_show_all (window);

	priv->result_window = window;

	if (searched_name)
		g_free (searched_name);
}

static void
nameday_calendar_on_find_toolbar_close_event (HildonFindToolbar *toolbar,
						gpointer data)
{
	NamedayDesktopApplet *priv;
	GtkWidget *search_menu_item;

	g_return_if_fail (data);

	priv = NAMEDAY_DESKTOP_APPLET (data);

	/* set search menu item inactive */
	search_menu_item = g_object_get_data (G_OBJECT (priv->calendar_window), "search_menu_item");
	gtk_widget_set_sensitive (GTK_WIDGET (search_menu_item), TRUE);

	/* hide find toolbar */
	gtk_widget_hide (GTK_WIDGET (toolbar));
}

static void
nameday_calendar_on_menu_search_event (GtkButton *button,
				       gpointer data)
{
	NamedayDesktopApplet *priv;
	GtkWidget *find_toolbar;

	g_return_if_fail (data);

	priv = NAMEDAY_DESKTOP_APPLET (data);

	find_toolbar = g_object_get_data (G_OBJECT (priv->calendar_window), "find_toolbar");

	/* set search menu item inactive */
	gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);

	/* show find toolbar */
	gtk_widget_show (find_toolbar);

	g_signal_connect (find_toolbar, "search",
			  G_CALLBACK (nameday_calendar_on_find_toolbar_search_event), priv);
	g_signal_connect (find_toolbar, "close",
			  G_CALLBACK (nameday_calendar_on_find_toolbar_close_event), priv);
}

static void
nameday_calendar_on_window_destroy_event (GtkWidget *window,
					  GdkEvent  *event,
					  gpointer data)
{
	NamedayDesktopApplet *priv;

	g_return_if_fail (data);

	priv = NAMEDAY_DESKTOP_APPLET (data);

	/* destory window if it was created */
	if (priv->calendar_window)
		gtk_widget_destroy (priv->calendar_window);

	priv->calendar_window = NULL;
}

GtkWidget*
nameday_calendar_window_create (NamedayDesktopApplet *priv)
{
	GtkWidget *window, *menu_item, *find_toolbar;
	GtkListStore *list_store;
	HildonAppMenu *menu = NULL;

	g_return_val_if_fail (priv, NULL);

	list_store = nameday_dbparser_liststore_create (priv);

	/* create window with tree view */
	window = nameday_calendar_treeview_window (priv, GTK_TREE_MODEL (list_store));
	g_object_unref (list_store);
	gtk_window_set_title (GTK_WINDOW (window), _("Nameday calendar"));
	hildon_program_add_window (hildon_program_get_instance(), HILDON_WINDOW (window));

	/* find toolbar */
	find_toolbar = hildon_find_toolbar_new (_("Search"));
	hildon_window_add_toolbar (HILDON_WINDOW (window), GTK_TOOLBAR (find_toolbar));
	g_object_set_data (G_OBJECT (window), "find_toolbar", find_toolbar);

	/* application menu */
	menu = HILDON_APP_MENU (hildon_app_menu_new ());
	hildon_window_set_app_menu (HILDON_WINDOW (window), menu);

	/* search menu item */
	menu_item = hildon_gtk_button_new (HILDON_SIZE_AUTO);
	gtk_button_set_label (GTK_BUTTON (menu_item), _("Search"));
	hildon_app_menu_append (menu, GTK_BUTTON (menu_item));
	g_object_set_data (G_OBJECT (window), "search_menu_item", menu_item);

	g_signal_connect (menu_item, "clicked",
			  G_CALLBACK (nameday_calendar_on_menu_search_event), priv);
	g_signal_connect (window, "destroy_event",
			  G_CALLBACK (nameday_calendar_on_window_destroy_event), priv);
	g_signal_connect (window, "delete_event",
			  G_CALLBACK (nameday_calendar_on_window_destroy_event), priv);

	/* do not show find toolbar */
	gtk_widget_set_no_show_all (find_toolbar, TRUE);

	/* show menu */
	gtk_widget_show_all (GTK_WIDGET (menu));

	/* show window */
	gtk_widget_show_all (window);

	return window;
}

static GtkWidget*
nameday_calendar_treeview_window (NamedayDesktopApplet *priv,
				  GtkTreeModel *tree_model)
{
	GtkWidget *window, *alignment, *scrolled_window, *treeview;
	GtkCellRenderer *renderer;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *path;
	gchar *name = NULL;
	gboolean iter_valid = TRUE;
	GdkColor style_color;

	g_return_val_if_fail (priv, NULL);

	/* main window */
	window = hildon_stackable_window_new();

	alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), HILDON_MARGIN_HALF, 0, HILDON_MARGIN_DOUBLE, HILDON_MARGIN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (window), alignment);

	/* scrolled window */
	scrolled_window = hildon_pannable_area_new ();
	gtk_container_add (GTK_CONTAINER (alignment), scrolled_window);

	/* nameday treeview */
	treeview = hildon_gtk_tree_view_new (HILDON_UI_MODE_EDIT);
	gtk_tree_view_set_model (GTK_TREE_VIEW (treeview), tree_model);
	gtk_container_add (GTK_CONTAINER (scrolled_window), treeview);

	/* date column */
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						     -1,
						     "Date", renderer,
						     "text", COLUMN_DATE,
						     NULL);

	if (!gtk_style_lookup_color (GTK_WIDGET (priv->label)->style, "SecondaryTextColor",
				     &style_color)) {
		gdk_color_parse ("grey", &style_color);
	}
	
	g_object_set (G_OBJECT (renderer),
		      "xalign", 1.0,
	              "width-chars", 6,
	              "foreground-gdk", &style_color,
	              "foreground-set", TRUE,
	              NULL);

	/* name column */
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						     -1,
						     "Name", renderer,
						     "text", COLUMN_NAME,
						     NULL);

	gtk_style_lookup_color (GTK_WIDGET (priv->label)->style, "ActiveTextColor",
				&style_color);
	
	g_object_set (G_OBJECT (renderer),
		      "ellipsize", PANGO_ELLIPSIZE_END,
		      "ellipsize-set", TRUE,
	              "foreground-gdk", &style_color,
	              "foreground-set", TRUE,
		      NULL);

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
	for (iter_valid = gtk_tree_model_get_iter_first (model, &iter); iter_valid;
	     iter_valid = gtk_tree_model_iter_next (model, &iter)) {
		gtk_tree_model_get (model, &iter, COLUMN_NAME, &name, -1);
		if (strcmp (name, gtk_label_get_text (GTK_LABEL (priv->label))) == 0)
		{
			model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
			path = gtk_tree_model_get_path (model, &iter);
			gtk_tree_view_set_cursor (GTK_TREE_VIEW (treeview), path, NULL, FALSE);
			gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview), path, NULL, TRUE,
						      0.5, 0);
			gtk_tree_path_free(path);
			break;
		}
	}

	return window;
}
