/*
 *  Nameday desktop applet (widget) for Maemo.
 *  Copyright (C) 2009 Roman Moravcik
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <glib/gi18n-lib.h>

#include "nameday_calendar.h"
#include "nameday_dbparser.h"
#include "nameday_settings.h"
#include "nameday_applet.h"

#define BACKGROUND_IMAGE DATADIR"/nameday/images/background.png"

HD_DEFINE_PLUGIN_MODULE (NamedayDesktopApplet, nameday_desktop_applet, HD_TYPE_HOME_PLUGIN_ITEM)

void
nameday_desktop_applet_show_notification (NamedayDesktopApplet *priv,
					  const gchar *text)
{
	GtkWidget *banner;

	banner = hildon_banner_show_information (GTK_WIDGET (priv), NULL, text);
	hildon_banner_set_timeout (HILDON_BANNER (banner), 9000);
}

static void
nameday_desktop_applet_on_show_settings (GtkWidget *widget,
					 gpointer data)
{
	NamedayDesktopApplet *priv = NAMEDAY_DESKTOP_APPLET (widget);

	if (!priv->settings_window) {
		priv->settings_window = nameday_settings_dialog_create (priv);
	}
	gtk_widget_show_all (priv->settings_window);
}

static void
nameday_desktop_applet_on_button_press_event (GtkWidget *widget,
					      gpointer data)
{
	NamedayDesktopApplet *priv = NAMEDAY_DESKTOP_APPLET (widget);

	if (priv->db == NULL) {
		nameday_desktop_applet_show_notification (priv, _("Nameday calendar not openned"));
		return;
	}

	if (!priv->calendar_window) {
		priv->calendar_window = nameday_calendar_window_create (priv);
	}
}

gboolean
nameday_desktop_applet_on_timeout (gpointer data)
{
	NamedayDesktopApplet *priv;
	const time_t current_time = time (NULL);
	struct tm *local_time;
	guint current_day = 0, current_month = 0;

	g_return_val_if_fail (data, FALSE);

	priv = NAMEDAY_DESKTOP_APPLET (data);

	g_return_val_if_fail (priv->db, FALSE);

	if (priv->db == NULL)
		return FALSE;

	local_time = localtime (&current_time);
	current_day = local_time->tm_mday;
	current_month = local_time->tm_mon + 1;

	if (priv->previous_day != current_day) {
		gchar *name = NULL;

		name = nameday_dbparser_get_name_from_date (priv, current_day, current_month);

		if (name == NULL)
			name = g_strdup_printf (_("No one"));

		gtk_label_set (GTK_LABEL (priv->label), name);

		g_free (name);

		priv->previous_day = current_day;
	}

	return TRUE;
}

static GtkWidget *
nameday_desktop_create_widget (NamedayDesktopApplet *priv)
{
	GtkWidget *alignment;
	GdkColor style_color;

	alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), HILDON_MARGIN_DEFAULT, HILDON_MARGIN_DEFAULT, 29, 29);

	priv->label = gtk_label_new (_("Unknow"));
	gtk_misc_set_alignment ( GTK_MISC (priv->label), 0.5, 0.55);
	gtk_label_set_ellipsize (GTK_LABEL (priv->label), PANGO_ELLIPSIZE_END);
	gtk_container_add (GTK_CONTAINER (alignment), priv->label);

	if (!gtk_style_lookup_color (GTK_WIDGET (priv->label)->style, "ReversedTextColor",
				    &style_color)) {
		gdk_color_parse ("black", &style_color);
	}

	gtk_widget_modify_fg (GTK_WIDGET (priv->label), GTK_STATE_NORMAL, &style_color);

	gtk_widget_show_all (GTK_WIDGET (alignment));
	return GTK_WIDGET (alignment);
}

static void
nameday_desktop_applet_realize (GtkWidget *widget)
{
	/* Use An RGBA colormap rather than RGB,
	 * so we can use transparency in our expose_event() implementation. 
	 */
	GdkScreen *screen = gtk_widget_get_screen (widget);
	gtk_widget_set_colormap (widget, gdk_screen_get_rgba_colormap (screen));

	gtk_widget_set_app_paintable (widget, TRUE);

	/* Call the base class's implementation: */
	GTK_WIDGET_CLASS (nameday_desktop_applet_parent_class)->realize (widget);
}

static gboolean
nameday_desktop_applet_expose_event (GtkWidget *widget,
				     GdkEventExpose *event)
{
	NamedayDesktopApplet *priv = NAMEDAY_DESKTOP_APPLET (widget);

	cairo_t *cr  = gdk_cairo_create (widget->window);

	gdk_cairo_region (cr, event->region);
	cairo_clip (cr);

	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);

	if (priv->background) {
		cairo_set_source_surface(cr, priv->background, 0.0, 0.0);
	} else {
		cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 0.9);
	}

	cairo_paint (cr);

	cairo_destroy (cr);

	return GTK_WIDGET_CLASS (nameday_desktop_applet_parent_class)->expose_event (widget, event);
}

static void
nameday_desktop_applet_dispose (GObject *object)
{
	NamedayDesktopApplet *priv = NAMEDAY_DESKTOP_APPLET (object);

	if (priv->settings_window) {
		gtk_widget_destroy (priv->settings_window);
		priv->settings_window = NULL;
	}

	if (priv->calendar_window) {
		gtk_widget_destroy (priv->calendar_window);
		priv->calendar_window = NULL;
	}

	if (priv->result_window) {
		gtk_widget_destroy (priv->result_window);
		priv->result_window = NULL;
	}

	/* Call the base class's implementation: */
	G_OBJECT_CLASS (nameday_desktop_applet_parent_class)->dispose (object);
}

static void
nameday_desktop_applet_finalize (GObject *object)
{
	NamedayDesktopApplet *priv = NAMEDAY_DESKTOP_APPLET (object);

	nameday_dbparser_finalize (priv);
	nameday_settings_finalize (priv);

	/* Remove the timout handler: */
	if (priv->timeout_handler) {
		g_source_remove (priv->timeout_handler);
	}

	/* Call the base class's implementation: */
	G_OBJECT_CLASS (nameday_desktop_applet_parent_class)->finalize (object);
}

static void
nameday_desktop_applet_init (NamedayDesktopApplet *priv)
{
	GtkWidget *widget;

	nameday_settings_init (priv);
	nameday_dbparser_init (priv);

	widget = nameday_desktop_create_widget (priv);
	gtk_container_add (GTK_CONTAINER (priv), widget);

	gtk_widget_set_double_buffered (GTK_WIDGET (priv), FALSE);
	priv->background = cairo_image_surface_create_from_png(BACKGROUND_IMAGE);
	if (cairo_surface_status(priv->background) != CAIRO_STATUS_SUCCESS) {
		cairo_surface_destroy(priv->background);
		priv->background = NULL;
	}

	if (priv->background) {
		gtk_widget_set_size_request(GTK_WIDGET(priv), cairo_image_surface_get_width(priv->background), cairo_image_surface_get_height(priv->background));
	} else {
		gtk_widget_set_size_request(GTK_WIDGET(priv), 338, 80);
	}

	/* force update desktop applet */
	nameday_desktop_applet_on_timeout (priv);

	priv->timeout_handler = g_timeout_add (60000, nameday_desktop_applet_on_timeout, (gpointer) priv);

	hd_home_plugin_item_set_settings (HD_HOME_PLUGIN_ITEM (priv), TRUE);

	g_signal_connect (priv, "show-settings", G_CALLBACK (nameday_desktop_applet_on_show_settings), NULL);
	g_signal_connect (priv, "button_press_event", G_CALLBACK(nameday_desktop_applet_on_button_press_event), NULL);
}

static void
nameday_desktop_applet_class_init (NamedayDesktopAppletClass *class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (class);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (class);

	object_class->dispose = nameday_desktop_applet_dispose;
	object_class->finalize = nameday_desktop_applet_finalize;

	widget_class->realize = nameday_desktop_applet_realize;
	widget_class->expose_event = nameday_desktop_applet_expose_event;
}

static void
nameday_desktop_applet_class_finalize (NamedayDesktopAppletClass *class)
{
}
