"""

    GTK Kinetic scroller
    (C) Copyright 2007 Anthony Maro
    http://tony.maro.net/
    
    Based significantly on GPL pygame code from Kagu:
    https://www.guardiani.us/projects/kagu
    Copyright (c) 2007 Jesse Guardiani <jesse@guardiani.us>
   
"""


import gtk, pango, pygtk, time, threading, gobject
from ..lgc.yentry import YEntry
from ..lgc import ystorage
from ..lgc.conf import Cfg

class Rect(object):
    def __init__(self, xywh):
        """
        xywh must be a 2 or 4 tuple or a rect instance.
        """
        if isinstance(xywh, Rect):
            self.left = xywh.left
            self.top = xywh.top
            self.width = xywh.width
            self.height = xywh.height
        else:
            if len(xywh) == 4:
                self.left, self.top, self.width, self.height = (i for i in xywh)
            elif len(xywh) == 2:
                self.left, self.top, self.width, self.height = (0.0,0.0) + tuple(i for i in xywh)
                
    def set_right(self, s):
        self.left = s - self.width
        
    def get_right(self):
        return self.left + self.width    
    
    right = property(get_right, set_right)
    
    def set_bottom(self, s):
        self.top = s - self.height
        
    def get_bottom(self):
        return self.top + self.height
    
    bottom = property(get_bottom, set_bottom)
        
    def collidepoint(self, xy):
        """
        Test if a point intersects with this rect.
        """
        x,y = xy
        return x >= self.left and x <= (self.left + self.width) and y >= self.top and y <= (self.top + self.height)

    def intersects(self, other):
        """
        Test if a rect intersects with this rect.
        """
        if self.left > other.left+other.width: return False
        if self.top > other.top + other.height: return False
        if self.left + self.width < other.left: return False
        if self.top + self.height < other.top: return False
        return True 
    
    def __str__(self):
        return ""+str(self.left)+","+str(self.top)+" "+str(self.width)+"x,"+str(self.height)
                

class BaseWidget(object):
  def __init__(self):    
    self.image = None
    self.rect  = None

  def handle_event(self,event):
    ''' override me '''

  def is_idle(self):
    ''' override me '''
    return 1

  def update(self):
    ''' override me '''

  def draw(self,widget):
    return

  def close(self):
    ''' override me '''
    
class ScrollYItem(BaseWidget):
    PIXBUF_STAR = None
    PIXBUF_STAR_W = 17
    PIXBUF_PREV = None
    PIXBUF_NEXT = None
    DETAILS_X = 70
    CACHE_WIDTH = 720
    CACHE_H = 65
    ANIM = ["-","\\","|","/","-","\\","|","/"]
    
    
    def __init__(self, obj, idx):
        """
        @param obj: yentry object
        @param idx: index position. used just as a hint
        """
        BaseWidget.__init__(self)
        self.__idx = idx
        
        if ScrollYItem.PIXBUF_STAR == None:
            ScrollYItem.PIXBUF_STAR = ystorage.loadPixbuf("myt-star.png", -1, -1)
        if ScrollYItem.PIXBUF_PREV == None:
            ScrollYItem.PIXBUF_PREV = ystorage.loadPixbuf("myt-left.png", -1, -1)
            ScrollYItem.PIXBUF_NEXT = ystorage.loadPixbuf("myt-right.png", -1, -1)

        
        self.rect = Rect((0,0,ScrollYItem.CACHE_WIDTH,ScrollYItem.CACHE_H))
        
        self.obj = obj
        self.focus = False
        self.dirty = True
        self.childDirty = False
                
    def draw(self, widget):
        #####################################
        xpad = 1
        xalign = 1
        ypad = 1
        yalign = 1
        obj = self.obj
        
        try:
            #some exception happens here while changing pages. i'm a python n00b therefore i'm allowed to fix
            #it like this
            picW = obj.thumbW
        except:
            print "exception here1"
            return
        
        x1 = 1+picW
        
        if obj.imgCache == None:
            obj.imgCache = gtk.gdk.Pixmap(widget.window, 
                                          ScrollYItem.CACHE_WIDTH, ScrollYItem.CACHE_H, -1)
            obj.bg_gc = widget.get_style().bg_gc[gtk.STATE_NORMAL]
            obj.fg_gc = widget.get_style().fg_gc[gtk.STATE_NORMAL]
            obj.imgCache.draw_rectangle(obj.bg_gc, True, 0, 0, 
                                        ScrollYItem.CACHE_WIDTH, 
                                        ScrollYItem.CACHE_H)
            
        if (self.focus):
            obj.bg_gc = widget.get_style().bg_gc[gtk.STATE_SELECTED]
            obj.fg_gc = widget.get_style().fg_gc[gtk.STATE_SELECTED]
        else:
            obj.bg_gc = widget.get_style().bg_gc[gtk.STATE_NORMAL]
            obj.fg_gc = widget.get_style().fg_gc[gtk.STATE_NORMAL]
        
        #render stars
        if obj.dirty:
            percent = obj.percentDl
            #print "dirty me !"
            # Fill the background        
            obj.imgCache.draw_rectangle(obj.bg_gc, True, 0, 0, 
                                        ScrollYItem.CACHE_WIDTH,
                                        ScrollYItem.CACHE_H)
            #draw rating stars
            obj.imgCache.draw_pixbuf(None,ScrollYItem.PIXBUF_STAR,
                          round(5-obj.rating)*17, 0,
                          xpad*xalign+x1, ypad * yalign + 15,
                          ScrollYItem.PIXBUF_STAR_W*5, -1,
                          gtk.gdk.RGB_DITHER_NONE,
                          0, 0)
        
            #set title
            context = widget.get_pango_context()
            layout = pango.Layout(context)
            layout.set_text(obj.title)
            layout.set_font_description(pango.FontDescription("Sans 11"))
            #gc = window.new_gc()
            obj.imgCache.draw_layout(obj.fg_gc, 
                                    xpad*xalign + picW+2,
                                    ypad * yalign, layout)
                                    
            
            layout.set_text("[%s]" %YEntry.formatDuration(obj.duration))
            if (int(percent) == 100):
                if obj.downloadedKb < 1024:
                    layout.set_text("%s [%dKB]" %(layout.get_text(), obj.downloadedKb))
                else:
                    layout.set_text("%s [%.1fMB]" %(layout.get_text(), float(obj.downloadedKb)/1024))
            elif (percent > 0):
                layout.set_text(layout.get_text() +" ["+str(percent)+"%]")
            elif obj.dispAnim >= 0:
                layout.set_text(layout.get_text() +" ["+ScrollYItem.ANIM[obj.dispAnim]+"]")
            #layout.set_width(cell.width)
            layout.set_font_description(pango.FontDescription("Sans Bold 10"))
            obj.imgCache.draw_layout(obj.fg_gc, xpad*xalign + picW+2, 
                                    ScrollYItem.CACHE_H-20, layout)
            #draw a marker for the first item
            if (self.__idx == 0):
                obj.imgCache.draw_rectangle(obj.fg_gc, True, 0, 0, 
                                        ScrollYItem.CACHE_WIDTH, 1)
                #obj.imgCache.draw_rectangle(obj.fg_gc, True, 0, ScrollYItem.CACHE_H-1, 
                #                        ScrollYItem.CACHE_WIDTH, 1)
            #draw a border on right
            #obj.imgCache.draw_rectangle(obj.fg_gc, True, ScrollYItem.CACHE_WIDTH-2, 0, 
            #                            2, ScrollYItem.CACHE_H)
            #draw entry thumb
            if (obj.thumbpix != None):
                obj.imgCache.draw_pixbuf(None, obj.thumbpix,
                               0, 0,
                               xpad*xalign, ypad * yalign,
                               -1, -1,
                               gtk.gdk.RGB_DITHER_NONE,
                               0, 0)
            
        
            obj.dirty = False
        #####################################
        buffer = widget.window
        buffer.draw_drawable(obj.fg_gc, obj.imgCache,
                             0, 0, 
                             self.rect.left, self.rect.top,
                             ScrollYItem.CACHE_WIDTH,ScrollYItem.CACHE_H)
        #------------
        #draw markers
        #------------
        if (self.focus and self.rect.width > 500):
            buffer.draw_pixbuf(None, ScrollYItem.PIXBUF_NEXT,
                               0, 0,
                               self.rect.left + xpad*xalign+widget.allocation.width - ScrollYItem.DETAILS_X,
                               self.rect.top + ypad*yalign + 17,
                               -1, -1, gtk.gdk.RGB_DITHER_NONE, 0, 0)
    
    def set_focus(self,focus):
        if focus != self.focus:
            self.obj.dirty = True
            if not focus:
                self.offset = 0
        self.focus = focus

class ScrollWidget(gtk.DrawingArea):
  ''' This widget wraps sprites with equal heights and 
  presents the user with an interface for scrolling and
  selecting these sprites'''
  def __init__(self):
    gtk.DrawingArea.__init__(self)
    #BaseWidget.__init__(self) # call Group initializer
    #self.rect       = Rect(rect) # this is where we draw our widget within the surface
    self.rect = None
    self.y_spacer   = 5 # height in pixels of whitespace between the top and bottom of sprites
    
    self.wrap       = True
    
    self.scrollable     = None
    self.sprites        = []
    self.sorted_sprites = []
    self.selected_index = None
    self.need_sort      = False
    self.selected_cb = None
    self.last_state     = "idle" # see self._change_state() for possibilities
    self.cur_state      = "idle"
    self.tap_event_threshold = 12
    self.tap_distance_threshold = 12
    self.visible_sprite_indexes = [] # optimization for is_mostly_visible()
    self.mousex = -1    
    self.mousey = -1
    self.smousex = -1
    self.smousey = -1   
    self.dirty = True   
    self.image = None # Stores my blit buffer
    self.colorgc = None
    self.backgroundcolor = '#FFFFFF'
    self.timeoutobj = -1
    self.timeoutfocus = -1
   
    self.clear()
    
    # Connect all my own personal events
    self.add_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.BUTTON_RELEASE_MASK | gtk.gdk.POINTER_MOTION_MASK | \
                    gtk.gdk.POINTER_MOTION_HINT_MASK | gtk.gdk.STRUCTURE_MASK | gtk.gdk.EXPOSURE_MASK)

    gtk.DrawingArea.connect(self, "button_press_event", self.clickbutton)
    gtk.DrawingArea.connect(self, "button_release_event", self.upbutton)
    gtk.DrawingArea.connect(self, "motion_notify_event", self.mousemove)
    gtk.DrawingArea.connect(self, "destroy",self.unrealized)
    gtk.DrawingArea.connect(self, "expose-event", self.expose)
    gtk.DrawingArea.connect(self, "realize", self.realized)
    gtk.DrawingArea.connect(self, "size-allocate", self.resized)
    
    self.timeoutwindowobj = -1
    

  def cb_timeoutFocus(self):
      if self.is_almost_idle():
          self.tap_select(True)
      else:
          pass #print "$$$ fast enough"
      self.timeoutfocus = -1
      return False #end
      
    
  def resized(self, widget, event=None, data=None):
      #print ">>>resized"
      self.rect = None
      self.stop_animation()
      
  def selecteditem(self):
      myitem = self.sorted_sprites[self.selected_index]
      return myitem
  
  def connect(self, eventtype, callback):
      # Wedge myself between...
      if eventtype == 'scrollitem_click_event':
          self.selected_cb = callback
          return
          # Pass through cause we don't do that...
      gtk.DrawingArea.connect(self, eventtype, callback)

  def unrealized(self, widget=None, data=None):
      ''' Call when we are unrealized to stop the drawing loop '''
      if self.timeoutobj <> -1:
          gobject.source_remove(self.timeoutobj)
          self.timeoutobj = -1
      self.sprites = []
      self.sorted_sprites = []
      self.visible_sprite_indexes = []
          

  def updateme(self):
      ''' Called once every .075 seconds to handle drawing '''
      gtk.gdk.threads_enter()
      #allow max  100 events to be handled first
      for i in range(0, 2):
          if gtk.events_pending():
              gtk.main_iteration(block=False)
          else:
              break
      gtk.gdk.threads_leave()
      
      if not self.is_idle():
          #print "updateme ... ",self.allocation.width
          self.update()
          self.expose()
          
      return True # continue using the timeout callback
      
  def _makerect(self):
      ''' Create the widget rectangle used for calculations '''
      w,h = [self.allocation.width, self.allocation.height]
      if (w <= 0 or h <=0):
        return
      self.rect = Rect((0,0,w,h))
      #print ">>>new rect ",self.rect

  def clear(self):
      self.offset     = 0
      self.distance   = 0
      self.dist_n_evs = [] #distance made in the last N events
      self.mousedown  = 0
      self.speed      = 0
      self.num_events = 0
      self.friction   = 3
      self.friction_c = 0
      self.viewport_index = 0
      self._makerect()
      self.image = None
      self.scrollable     = None
      self.sprites        = []
      self.sorted_sprites = []
      self.selected_index = None
      self.need_sort      = False
      #self.selected_cb = None
      self.last_state     = "idle" # see self._change_state() for possibilities
      self.cur_state      = "idle"
      self.tap_event_threshold = 12
      self.tap_distance_threshold = 12
      self.visible_sprite_indexes = [] # optimization for is_mostly_visible()
      self.mousex = -1    
      self.mousey = -1
      self.smousex = -1
      self.smousey = -1   
      self.dirty = True   
      
  def realized(self, widget=None, event=None, data=None):
      ''' Triggered when the widget is finally ready to draw '''
      self.update()
      self.expose()
      if (self.timeoutobj == -1):      
          self.timeoutobj = gobject.timeout_add(10,self.updateme) 
      
  def expose(self, widget=None, event=None, data=None):
    ###print ">>>expose ",event," ", data
    #is this the invalidate equivalent ?
    if event != None: 
        self.dirty = True
    if self.get_style().white_gc == None:
         return # See if we are usable...
    if (self.window == None):
        self.stop_animation()
        return
        #self.window.end_paint()
    ''' Main drawing routine '''
    if self.rect == None:
        self.update()
        self._makerect()    
    
    if self.dirty:
        self.window.begin_paint_rect((self.rect.top,self.rect.left,self.rect.width,self.rect.height))
    
        
    if not self.sorted_sprites:
        if self.dirty:
            self.window.end_paint()
        self.dirty = False
        self.dirtyChild = False
        return
    i = self.viewport_index
    for i in self.visible_sprite_indexes:
      #self.sorted_sprites[i].draw(self.image)
      #if self.sorted_sprites[i].dirty:
      if self.dirty:
          #print "drawing all ", i
          self.sorted_sprites[i].draw(self)
      elif self.sorted_sprites[i].dirty:
          lrect = self.sorted_sprites[i].rect
          ###print "drawing dirty ", i, " ", lrect
          #self.window.begin_paint_rect((lrect.top, lrect.left, lrect.width, lrect.height))
          self.sorted_sprites[i].draw(self)
          self.sorted_sprites[i].dirty = False
          #self.window.end_paint()
      else:
          pass###print "NOT drawing VISIBLE ", i
          
    if self.dirty:
         self.window.end_paint()
    
    #TODO: check friction
    if self.speed == 0:
        self.dirty = False
    self.dirtyChild = False
    return True
    
      
  def upbutton(self, widget, event, data=None):
      ''' User let go of the screen '''
      
      mousepoint = Rect((event.x,event.y))
      self.mousex = event.x
      self.mousey = event.y
      
    
      #compute the next speed
      self.speed = 0
      #move the widget at the last average speed
      if len(self.dist_n_evs) > 0:
          deltaT = max(1, time.clock()*1000 - self.dist_n_evs[0][1])
          
          for dist in self.dist_n_evs:
            self.speed = self.speed + dist[0]*10
          self.speed =(int)(self.speed/deltaT)
    
          if (self.speed > Cfg.KIN_MAX_SPEED):
            self.speed = Cfg.KIN_MAX_SPEED
          if (self.speed < -Cfg.KIN_MAX_SPEED):
            self.speed = -Cfg.KIN_MAX_SPEED
      #print "dbg releasing at speed = %d" %self.speed
      #self.speed = self.speed2

      if self.mousedown:
        self.mousedown  = 0
        self.friction_c = 0
        if self._is_tap():
          # This was a tap, not a drag
          self._change_state('tap')
          self.stop_animation()
       
      #TODO: MIA: check   
      if self.last_state == 'idle' and self.cur_state == 'tap':
          #print "aaaaaaaaaaaaaa"
          self.tap_select(False)
      self.dirty = True
      if len(self.sprites) == 0: 
          return
      self.update()
      self.expose()   
      
  def clickbutton(self, widget, event, data=None):
      ''' User clicking down on the mouse '''
      if self.rect == None:
          return
      
      mousepoint = (event.x,event.y)
      self.smousey = event.y
      
      if self.rect.collidepoint(mousepoint):
        if self.timeoutfocus != -1:
            gobject.source_remove(self.timeoutfocus)
            self.timeoutfocus = -1
        self.timeoutfocus = gobject.timeout_add(300, self.cb_timeoutFocus)
        #self.tap_select(False)
        self.speed      = 0
        self.distance   = 0
        self.num_events = 0
        self.mousedown = 1
      #self.dirty = True
      self.update()
      self.expose()   
      
    
  def mousemove(self, widget, event, data=None):
      ''' User dragging the widget around '''
      if event.is_hint:
          self.mousex, self.mousey, state = event.window.get_pointer()
      else:
          self.mousex = event.x
          self.mousey = event.y
          state = event.state
      
      if self.mousedown and self.sorted_sprites and self.is_scrollable():
        rel_y = int(self.mousey) - int(self.smousey)
        self.offset = self.offset + rel_y
        if cmp(rel_y,0) and cmp(rel_y,0) != cmp(self.distance,0): # changed direction
          self.num_events = 0
          self.distance   = 0
        self.num_events = self.num_events + 1
        self.distance   = self.distance   + rel_y
        self.speed      = 0 #self.distance / self.num_events
        self.dist_n_evs.append([rel_y, time.clock()*1000])
        if (len(self.dist_n_evs) == 3):
            self.dist_n_evs.pop(0)
                
        if not self._is_tap(): self._change_state('drag')        
        self.smousey = self.mousey
        self.dirty = True
      
  def _change_state(self,new_state):
    l = ['idle','tap','drag']
    if new_state not in l:
      print "WARNING: unknown state %s" % (new_state,)
      return
    self.last_state = self.cur_state
    self.cur_state  = new_state

  def _sort_sprites(self):
    if (self.need_sort): # cache sort order between list updates
      self.scrollable = None
      #print "sorting"
      self.need_sort = False
      self.sorted_sprites = self.sprites
      #self.sorted_sprites.sort()

  def add(self, *sprite):
    self.need_sort = True
    self.sprites.append(*sprite)

  def remove(self,*sprite):
    self.need_sort = True
    self.sprites.remove(*sprite)

  def has_selected(self):
    ''' The "selected" sprite is the one that we think the user has chosen '''
    if self.selected_index == None:
        return False
    return True

  def next(self):
    ''' Move selected_sprite to next sprite in list '''
    if self.is_idle():
      if self.has_selected():
        self.set_selected_index(self.selected_index + 1,True)
      else:
        self.new_selected_index()

  def prev(self):
    ''' Move selected_sprite to previous sprite in list '''
    if self.is_idle():
      if self.has_selected():
        self.set_selected_index(self.selected_index - 1,True)
      else:
        self.new_selected_index()

  def get_selected_sprite(self):
    if not self.has_selected(): 
        return None
    return self.sorted_sprites[self.selected_index]

  def get_selected_sprite_index(self):
    if not self.has_selected(): return None
    return self.selected_index

  def _calc_new_index(self,index,list):
    length = len(list)
    while index != 0: # FIXME: looping is silly for very large abs() values of index. maybe use divmod instead?
      if index < 0: 
        index = index + length
      elif index > length - 1:
        index = index - length
      else: 
          break # within the normal range, so return it
    return index

  def new_selected_index(self):
    self.selected_index = self.viewport_index
    self.offset = 0
    self.update()

  def set_selected_index(self,index,snap=False):
    if self.has_selected(): self.get_selected_sprite().set_focus(False)
    self.selected_index = self._calc_new_index(index,self.sorted_sprites)
    self.get_selected_sprite().set_focus(True)
    if snap and self.is_scrollable():
      if not self.is_index_mostly_visible(self.selected_index):
        self.viewport_index = self.selected_index
        self.offset = 0
    self.update()

  def set_selected_sprite(self,new_selected_sprite):
    c = 0
    for sprite in self.sorted_sprites: # make sure this new selected sprite exists in our list
      if new_selected_sprite == sprite:
        self.set_selected_index(c,True)
        return True
      c += 1
    print "invalid set_selected!" # this should never happen
    return False

#  def get_sprites(self):
#    self._sort_sprites()
#    return self.sorted_sprites

  def get_sprite(self, index):
    if index < 0 or index >= len(self.sprites):
        return None
    return self.sprites[index]

  def get_sprites_count(self):
      return len(self.sprites)
  
  def get_sprite_idx(self, video):
      for i in range(0, len(self.sprites)):
          if self.sprites[i].obj == video:
              return i
      return -1

  def _calc_viewport_index(self):
    while self.offset != 0:
      height = self.sorted_sprites[self.viewport_index].rect.height + self.y_spacer
      if self.offset > 0:
        if self.offset > height:
          self.offset = self.offset - height
          self.viewport_index = self._calc_new_index(self.viewport_index - 1,self.sorted_sprites)
        elif self.sorted_sprites[self.viewport_index].rect.top > self.rect.top:
          self.offset = 0 - height + self.offset
          self.viewport_index = self._calc_new_index(self.viewport_index - 1,self.sorted_sprites)
        else: return
      else:
        if abs(self.offset) > height:
          self.offset = self.offset + height
          self.viewport_index = self._calc_new_index(self.viewport_index + 1,self.sorted_sprites)
        else: return

  def _apply_friction(self):      
    self.friction_c = self.friction_c + float(Cfg.KIN_FRIC)/100
    slowed = False
    if self.friction_c >= self.friction:
        self.friction_c = 0
        self.speed = self.speed - cmp(self.speed,0)
        slowed = True
    
    snap_at = 2
    
    if abs(self.speed) <=snap_at: # Special handling to snap to a sprite when we get slow
        if abs(self.offset) > 5:
            # still have a ways to go, so don't slow down yet
            #if slowed and abs(self.speed) < snap_at:
            #    self.speed = self.speed + cmp(self.speed,0)
            pass
        else:
            # Just snap to the edge
            self.speed = 0
            self.offset = 0
              
    
  def _move(self):
    #print "dbg speed %d" %self.speed
    if not self.wrap: # prevent movement wrapping
      if ((self.speed < 0 or self.offset < 0) and self.viewport_index == len(self.sorted_sprites) - 1) or \
         ((self.speed > 0 or self.offset > 0) and self.viewport_index == 0):
        self.speed    = 0
        self.offset   = 0

    self.offset = self.offset + self.speed
    self._calc_viewport_index()
    if not self.speed:
        return
    self._apply_friction()

  def _update_sprite_rect(self,index,offset):      
    sprite = self.sorted_sprites[index]
    #if nothing changes, this does not need redrawing
    if not sprite.dirty and sprite.rect.top == self.rect.top + offset:
        sprite.dirty = False
    else:
        sprite.dirty = True

    sprite.rect.top  = self.rect.top + offset
    sprite.rect.left = self.rect.left
    #update this for the slide arrow to be shown
    sprite.rect.width = self.allocation.width
    #print "sprite width = ",sprite.rect.width
    
    if not self.rect.intersects(sprite.rect):
        sprite.dirty = False
        return False
    
    return True

  def is_index_mostly_visible(self,index):
    if not self.sorted_sprites: return False
    if index not in self.visible_sprite_indexes: return False
    if self.rect.collidepoint(self.sorted_sprites[index].rect.center): return True
    return False

  def update(self):    
    # self.props.visible() test to see if I'm really here    
    
    if self.rect == None:
        self._makerect()
    self._move()
    self._sort_sprites()
    self.visible_sprite_indexes = []
    if not self.sorted_sprites: return

    i      = self.viewport_index
    offset = self.offset
    first  = True
    
    while self._update_sprite_rect(i,offset) or first:
      self.visible_sprite_indexes.append(i)
      offset = self.sorted_sprites[i].rect.bottom + self.y_spacer - self.rect.top
      if not self.wrap and i == len(self.sorted_sprites) - 1: 
          break # prevent display wrapping
      i = self._calc_new_index(i+1,self.sorted_sprites)
      first = False
      if i == self.viewport_index: 
          break # handle lists that are shorter than the viewport

    if self.has_selected():
      self.get_selected_sprite().update()

  def _is_tap(self):
    if self.num_events <= \
        self.tap_event_threshold and abs(self.distance) <= \
        self.tap_distance_threshold: return True
    return False

  def tap_select(self,run_callback=True):
    if not self.sorted_sprites: return
    mouse_pos = (self.mousex, self.mousey)
    for i in self.visible_sprite_indexes:
      if self.sorted_sprites[i].rect.collidepoint(mouse_pos):
        self.set_selected_index(i)
        if run_callback:
          if self.selected_cb != None:              
              self.selected_cb(self,None)
          self._change_state('idle')
        break
  
  def get_sprite_at_y(self, y):
    if not self.sorted_sprites:
        return (None, -1)
    pos = (10, y)
    for i in range(0, len(self.sprites)):
      if self.sprites[i].rect.collidepoint(pos):
        return (self.sprites[i], i)
    return (None, -1)

  def is_scrollable(self):
    if not self.visible_sprite_indexes: self.update()
    
    if self.scrollable != None: return self.scrollable
    if not self.sorted_sprites: return False
    height = 0
    for i in self.sorted_sprites:
      height += i.rect.height
    spacer_height = (len(self.visible_sprite_indexes) - 1) * self.y_spacer
    height += spacer_height
    
    if height > self.rect.height: self.scrollable = True
    else: self.scrollable = False
    return self.scrollable

  def is_idle(self):
    #and not self.mousedown
    idle = (not self.speed and not self.need_sort and not self.dirty and not self.dirtyChild)
    if idle and self.cur_state != 'idle': 
        self._change_state('idle')
    #debug msg
    #if not idle:
    #    print "need sort = ",self.need_sort," dirty=",self.dirty, " speed =",self.speed
    return idle

  #animation is slow enough to be considered idle
  def is_almost_idle(self):
      import math
      if (math.fabs(self.speed) > 1):
          return False
      return True


  def stop_animation(self):
    self.speed      = 0
    self.distance   = 0
    self.num_events = 0
    self.mousedown  = 0
    self.friction_c = 0
    self.update()

  def close(self):
    for widget in self.sorted_sprites: widget.remove()
    
