/**
 * dispatcher.c
 * 
 * This file is part of MSA program
 * 
 * Copyright (C) 2009 Burlak I.L.
 * 
 * MSA program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *#include <drivers/vkontakte/driver.h>
#include <drivers/vkontakte/vkontakte.h>
#include <drivers/vkontakte/parser.h>
 * MSA program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MSA program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */

#include "dispatcher.h"

#define MODULE_PATH DATADIR"/msa/"

/** main init function in all external modules **/
#define MODULE_INIT "msa_module_init"
#define G_FILE_ATTRIBUTE_STANDARD_NAME "standard::name"            g_debug("!!!!!!");

/**
 * @brief initialization all programm external module 
 * @return 0 or status error.
 */
int msa_disp_init()
{
    g_debug("disp:msa_disp_init: dispatcher initialization START!");
                
    /** set startup dispatcher status OFF **/
    msa_disp_status = DISP_OFF;
    
    msa_driver_info* drv_list;
    msa_module_disp* module;

    GError* error;

    GList* list_dir;
    GList* node;

    GnomeVFSResult result;
    GnomeVFSFileInfo* info;

    gpointer obj;
    gpointer key;    
    
    int i = 0;
    char* name;
    char* path;

    /** see general.h **/    
    list_drivers = NULL;

    /** init gnomeVFS **/
    if (gnome_vfs_init() == FALSE) {
        g_debug("disp:msa_disp_init: gnome_vfs not init!");
        return FAILURE;
    }
    
    /** create module hash table **/
    table_mod = g_hash_table_new(g_str_hash, g_str_equal);

    #ifdef TEST
        /** connect test module **/
        if (test_send != NULL) {
            module = msa_disp_module_new();
            module->msa_mod->id = g_strdup_printf(TEST_ID);
            module->msa_mod->type = TEST_MODULE;
            module->msa_mod->send = test_send;
            module->msa_mod->status = ON;
            module->msa_mod->state = CONF;
            module->msa_mod->shutdown = NULL;
            
            /** save struct module in table_mod **/
            save_module(module);
            g_debug("disp:msa_disp_init:TEST SEND WORK!");    
        } else {
            g_debug("disp:msa_disp_init:TEST sorry test SEND no work!");
            return FAILURE;        
        }
    #endif    

    /** sign internal kernel module  **/
    module = msa_disp_module_new();
    module->msa_mod->id = g_strdup_printf(KERNEL_ID);
    module->msa_mod->type = SYSTEM;
    module->msa_mod->send = KERNEL_SEND;
    module->msa_mod->status = ON;
    module->msa_mod->state = CONF;
    module->msa_mod->shutdown = KERNEL_SHUTDOWN;
    module->msa_mod->msa_settings = msa_settings;
   
    /** save struct module in table_mod **/
    save_module(module);

    /** generate path to drivers folder **/
    path = g_strdup_printf("%s/drivers/", MODULE_PATH);

    /** open drivers folder, use gnomeVFS **/
    result = gnome_vfs_directory_list_load(&list_dir, path,
                                           GNOME_VFS_FILE_INFO_FOLLOW_LINKS);

    if(result != GNOME_VFS_OK) {
        g_debug("disp:msa_disp_init: error no file in such directory %s", MODULE_PATH);
        return FAILURE;
    }

    /** pass on directory files **/
    for (node = list_dir; node != NULL; node = node->next) {
        
        /** get folder name from GList **/
        info = (GnomeVFSFileInfo*)node->data;

        /** if it isn't ".." or "." **/
        if (strcmp(info->name, "..") != 0 && strcmp(info->name, ".") != 0) {

            /** create msa_module struct and sign driver **/
            module = msa_disp_module_new();
            module->msa_mod->id = g_strdup_printf("%s", info->name);
            module->msa_mod->type = DRIVER;

            /** connect curent driver **/            module->msa_mod->set_events = handler_put_data;

            if (msa_disp_driver_connect(module) == SUCCESS) {
                /** create info struct for external module **/
                msa_driver_info_new(module->msa_mod);
                /** save struct module in table_mod **/
                save_module(module);        
            } else {
                g_debug("disp:msa_disp_init: can't connect driver %s", module->msa_mod->id);            
            }
        }    
    }

    /** init DB **/
    /** init database **/ 
    module = msa_disp_module_new();
    module->msa_mod->id = g_strdup_printf(DB_ID);
    module->msa_mod->type = MODULE;
    module->msa_mod->shutdown = DB_DOWN;

    if (msa_disp_driver_connect(module) == SUCCESS) {
       save_module(module);
    } else {
       g_debug("disp:msa_disp_init: db can't work!");     
       return FAILURE;    
    }

    module = msa_disp_module_new();    
    module->msa_mod->id = g_strdup_printf(UI_ID);
    module->msa_mod->type = MODULE;
    module->msa_mod->shutdown = UI_DOWN;

    /** init ui **/
    if (msa_disp_driver_connect(module) == SUCCESS) {
        save_module(module);
    } else {
        g_debug("disp:msa_disp_init: ui can't work!");
        return FAILURE;     
    }

    msa_disp_status = DISP_ON;
    return SUCCESS;
}

/**
 * @brief create new msa_module stract and init var
 * @return msa_module or null.
 */
static msa_module_disp* msa_disp_module_new()
{
	msa_module_disp* module;
    if ((module = malloc(sizeof(msa_module_disp))) == NULL) {
        g_debug("disp:msa_disp_init: memmory allocation error");
        return NULL;
    }
    if ((module->msa_mod = malloc(sizeof(msa_module))) == NULL) {
        g_debug("disp:msa_disp_init: memmory allocation error");
        return NULL;
    }

	module->msa_mod->send = NULL;
	module->msa_mod->shutdown = NULL;
	module->handle = NULL;
	module->msa_mod->id = NULL;
	module->msa_mod->name = NULL;

	return module;
}


/**
 * @brief create new struct msa_driver_info for curen module
 * @param mod - msa_module struct
 * @return 0 or status error.
 */
static int msa_driver_info_new(msa_module* mod) 
{
	msa_driver_info* drv_list;
	if ((drv_list = malloc(sizeof(msa_driver_info))) == NULL) {
		  g_debug("disp:msa_disp_init: memmory allocation error");
		  return FAILURE;
	}

	drv_list->id = mod->id;
	drv_list->name = mod->name;
	drv_list->status = mod->status;
	drv_list->state = mod->state;
	drv_list->pic = mod->pic;
    drv_list->proxy = NULL;
    drv_list->port = 0;

    list_drivers = g_list_append(list_drivers, (gpointer)drv_list);
	return SUCCESS;	
}


/**
 * @brief save msa_module_disp in hash table: table_mod (see general.h)
 * @param mod - module for saving
 */
static void save_module(msa_module_disp* mod)
{
    gpointer obj;
    gpointer key;
    
    obj = (gpointer)mod;
    key = (gpointer)mod->msa_mod->id;
    g_hash_table_insert(table_mod, key, obj); 

}

/**
 * @brief conntect curent module
 * @param msa_module_disp struct 
 * @return 0 or status error.
 */
static int msa_disp_driver_connect(msa_module_disp* mod)
{
    void* handle;
    int (*driver_init_msa)(msa_module*);

    gchar* error; 
    gchar* path;
    gpointer value;
    msa_module_disp* kernel_mod;

    /** create path to module **/
    if (mod->msa_mod->type == DRIVER) {
        path = g_strdup_printf("%s/drivers/%s/lib%s.so", MODULE_PATH, 
								mod->msa_mod->id, mod->msa_mod->id);
    }
    else {
        path = g_strdup_printf("%s/%s/lib%s.so", MODULE_PATH,
								mod->msa_mod->id, mod->msa_mod->id);
    }
    
    /** open external library **/
    handle = dlopen(path, RTLD_LAZY);
    if (!handle) {
        g_debug("disp:msa_driver_connect:dlopen_error: %s", dlerror());
        mod->msa_mod->status = BROKEN;
        return FAILURE;
    }

    /** try execute init function **/
    driver_init_msa = dlsym(handle, MODULE_INIT);
    if ((error = dlerror()) != NULL) {
        g_debug("disp:msa_module_connect:dlsym_error: %s", error);
        mod->msa_mod->status = BROKEN;
        return FAILURE;
    }

    driver_init_msa(mod->msa_mod);
    mod->msa_mod->status = ON;
    mod->msa_mod->set_events = handler_put_data;

    return SUCCESS;
}


#ifdef TEST
/** !!!!!! TEST INTERFACE !!!!!! **/
msa_module_disp* test_msa_disp_module_new()
{
    msa_module_disp* mod;
    mod = msa_disp_module_new();
    return mod;
}

int test_msa_disp_driver_connect(msa_module_disp* mod)
{
    return msa_disp_driver_connect(mod);
}

int test_init(int (*_test_send)(xmlDocPtr request, xmlDocPtr* response, msa_driver_info* info))
{   
    /** set startup dispatcher status OFF **/
    msa_disp_status = DISP_OFF;

    int retval;

    test_send = _test_send;
    retval = msa_disp_init();
    
    return retval;    
}
#endif
