/*
 * db_logic.c - function for logic management for XML-DB. 
 * This file is part of Maemo-DB project.
 *
 * Copyright (C) 2009 - Alexander A. Lomov
 *
 * Maemo-DB project is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Maemo-DB project is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MSA program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */


// FIXME: write functions: load schrmas from config file; get data without binary;
// FIXME: rewrite get data by id, add data and add data with id 
// FIXME: check free xpath object
// FIXME: fix bin_id == null
// FIXME: xpath and "ID"
// FIXME: ' - symbol


 
#include "db.h"
#include "db_logic.h"
#include "db_functions.h"
#include "db_xml_functions.h"

#include <stdio.h>

#define USERNAME g_get_user_name()

#define FILE_PERMISSIONS 0750
#define DB_LINK_ID "ID"
#define DB_OR_LOGIC_OPERATOR "or" 

#define SLASH_AT_DELIMETER "/@"
#define AT_DELIMETER "@"


static GHashTable* g_tables = NULL;
static GHashTable* g_schemas_links = NULL;
static gchar* g_db_name = NULL;
static gchar* g_db_path = NULL;

static const gchar* const schemas_list[SCHEMAS_COUNT + 1] = {
//    SCHEMA_NAME_GROUPS, 
//    SCHEMA_NAME_PROFILES, 
    SCHEMA_NAME_METACONTACTS, 
    SCHEMA_NAME_CONTACTS,
    SCHEMA_NAME_CONTACTS_LIST,  
//    SCHEMA_NAME_SERVICES, 
    SCHEMA_NAME_MESSAGES,
    SCHEMA_NAME_OWNER_MESSAGES,  
    NULL
};


enum db_logic_errors {
    ERROR_CANT_LOAD_TABLE = 1,
    ERROR_TABLE_ALLREADY_LOAD,
    ERROR_INCORRECT_TABLE_INDEX,
    ERROR_CANT_GET_TABLE_SCHEMA
    
};



typedef struct _schema_links {
    gchar* schema_name;
    gchar* linked_schema;
    gchar** keys;
    gchar** links;
    gint* actions;
} schema_link;


typedef struct _delete_data {
    gchar* schema_name;
    GSList* id_list;
    gchar* id_path;
} delete_data;



#define DB_DEBUG_LEVEL 0
#define DB_DEBUG_OFF

#ifdef DB_DEBUG_ON
#define DB_DEBUG(level, message ) if (level <= DB_DEBUG_LEVEL) g_debug(message);
#define DB_PDEBUG(level, message, params) if (level <= DB_DEBUG_LEVEL) g_debug(message,params);
#define DB_P2DEBUG(level, message, par1, par2) if (level <= DB_DEBUG_LEVEL) g_debug(message,par1, par2);
#define DB_P3DEBUG(level, message, par1, par2, par3) if (level <= DB_DEBUG_LEVEL) g_debug(message, par1, par2, par3);
#else
#define DB_DEBUG(level, message ) //
#define DB_PDEBUG(level, message, parms) // 
#define DB_P2DEBUG(level, message, par1, par2) //
#define DB_P3DEBUG(level, message, par1, par2, par3) //
#endif 
 
   
   
static void init_tables_store(gboolean reload_tables);      
static void unload_tables(GHashTable* tables, const gchar* const schemas[]);      
static gint load_table(GHashTable* tables, const gchar* schema_name);      
//static xmlDocPtr get_table_from_list(const gchar* schema_name);    

static GList* get_hash_table_values(GHashTable* hash_table);

gchar* create_list_value_filter(const gchar* key, const GSList* values, const gchar* logic_op);


static GSList* convert_strv_to_gslist(gchar** strv, gboolean duplicate_values);
static gchar* build_binary_nodes_select_query();//(const gchar* schema_name);



gchar* db_get_binary_data_content(const gchar* binary_id)
{
    gchar* inner_binary_id = g_base64_encode((const guchar*)binary_id, strlen(binary_id));

    gchar* db_path = cm_get_db_path_by_name_or_default(g_db_name);
    
    gchar* dir_path = g_strconcat(db_path, G_DIR_SEPARATOR_S, 
                                  BINARY_DIR_NAME, G_DIR_SEPARATOR_S, NULL);

    gchar* filepath = g_strconcat(dir_path, G_DIR_SEPARATOR_S, inner_binary_id, NULL);
    
    gchar* content = NULL;
    
    if (filepath != NULL) {
        g_file_get_contents(filepath, &content, NULL, NULL);    
    }    
        
    g_free(filepath);
    g_free(inner_binary_id);
    g_free(dir_path);
    g_free(db_path);       
    return content;
}


gboolean db_set_binary_data_content(const gchar* binary_id, const gchar* content)
{
    gchar* inner_binary_id = g_base64_encode((const guchar*)binary_id, strlen(binary_id));
    gchar* db_path = cm_get_db_path_by_name_or_default(g_db_name);
    gchar* dir_path = g_strconcat(db_path, G_DIR_SEPARATOR_S, 
                                  BINARY_DIR_NAME, G_DIR_SEPARATOR_S, NULL);

    if (g_mkdir_with_parents(dir_path, FILE_PERMISSIONS) != 0 ) {
        return FALSE;
    }

    gchar* filepath = g_strconcat(dir_path, G_DIR_SEPARATOR_S, inner_binary_id, NULL);
    
    gboolean save_resilt = FALSE;

    if (filepath != NULL) {
        save_resilt = g_file_set_contents(filepath, content, -1, NULL);    
    }    
        
    g_free(filepath);
    g_free(inner_binary_id);
    g_free(dir_path);
    g_free(db_path);        
    return save_resilt;
}


static gboolean save_node_content_to_file(const xmlNodePtr node, const gchar* filepath, 
                                      gboolean remove_content)
{
    DB_DEBUG(3, "save_node_content_to_file START");
    xmlChar* content = xmlNodeGetContent(node);  

    // FIXME: change: is null or empty
    if (content == NULL ) {
        return FALSE;
    }
    if (xmlStrlen(content) == 0) {
        xmlFree(content);
        return FALSE;
    
    }
    if (xmlStrlen(content) == 0) {
        xmlFree(content);
        return FALSE;
    }

    gboolean save_result = g_file_set_contents(filepath, (const gchar*)content, -1, NULL);

    g_free(content);

    
    if (save_result == TRUE && remove_content == TRUE) {
        xmlNodeSetContent(node, NULL);	
    }
    DB_DEBUG(3, "save_node_content_to_file END");
    return save_result;
}

static gint move_files(const gchar* scr_folder, const gchar* dest_folder) {
    
    if (g_mkdir_with_parents(dest_folder, FILE_PERMISSIONS) != 0 ) {
        return -1;
    }
    
    GDir* src_dir = g_dir_open(scr_folder, 0, NULL);
    
    if (src_dir == NULL) {
        return -1;
    }
    
    const gchar* name =  g_dir_read_name(src_dir);
    
    gint error_move_count = 0;
    while (name != NULL) {
        gchar* scr_path = g_strconcat(scr_folder, G_DIR_SEPARATOR_S, name, NULL);
        gchar* dest_path = g_strconcat(dest_folder, G_DIR_SEPARATOR_S, name, NULL);
        
        if (rename(scr_path, dest_path) != 0) {
            ++error_move_count;
        }
    
        name = g_dir_read_name(src_dir);

        g_free(scr_path);
        g_free(dest_path);
    }
    
    g_dir_close(src_dir);
    
    return error_move_count;
} 

#define BINARY_NODE_TAG_FOR_FILENAME "uri"


static gint save_binary_data() 
{
    gchar* db_path = cm_get_db_path_by_name_or_default(g_db_name);
    gchar* tmp_path = g_strconcat(db_path, G_DIR_SEPARATOR_S, 
                                  TEMP_BINARY_DIR_NAME, G_DIR_SEPARATOR_S, NULL);

    gchar* bin_path = g_strconcat(db_path, G_DIR_SEPARATOR_S, 
                                  BINARY_DIR_NAME, G_DIR_SEPARATOR_S, NULL);

    gint save_result = move_files(tmp_path, bin_path);
                  
    g_free(tmp_path);                              
    g_free(bin_path);      
    g_free(db_path);         
    
    return save_result;

}

//#include "md5.h"

static gint prepare_binary_data_fo_save(const gchar* schema_name, xmlDocPtr doc)
{
    gchar* db_path = cm_get_db_path_by_name_or_default(g_db_name);
    
    if(db_path == NULL) {
        return FAILURE;
    }
    
    gchar* tmp_path = g_strconcat(db_path, G_DIR_SEPARATOR_S, 
                                  TEMP_BINARY_DIR_NAME, G_DIR_SEPARATOR_S, NULL);
    
    if (g_mkdir_with_parents(tmp_path, FILE_PERMISSIONS) != 0 ) {
        return FAILURE;
    }
    
    gchar* query = build_binary_nodes_select_query();
    xmlChar* ns = BAD_CAST cm_get_namespaces_key(schema_name);
        
    xmlXPathObjectPtr result = db_get_nodeset(doc, BAD_CAST query, ns);

//    struct cvs_MD5Context context;
//    unsigned char checksum[16];

    if (result != NULL) {
        xmlNodeSetPtr nodeset = result->nodesetval;

        gint size = (nodeset) ? nodeset->nodeNr - 1 : 0;

        gint i = 0;
        for (i= size; i >= 0; --i) {
            gchar* bin_id = (gchar*)xmlGetProp(nodeset->nodeTab[i], BAD_CAST BINARY_NODE_TAG_FOR_FILENAME);
            if (bin_id == NULL) {
                continue;
            }
            gchar* md5 =
//            		cvs_MD5Init (&context);
//		cvs_MD5Update (&context, bin_id, strlen(bin_id));
//		cvs_MD5Final (checksum, &context);
//		DB_P2DEBUG(3, "checksum = %s, bin_id = %s", checksum, bin_id);
        
             g_base64_encode((const guchar*)bin_id, strlen(bin_id));
//            md5_get_digest(bin_id, -1, md5);
//             = g_compute_checksum_for_string(G_CHECKSUM_MD5, bin_id, -1);
            gchar* filepath = g_strconcat(tmp_path, G_DIR_SEPARATOR_S, md5, NULL);
    

            if (g_file_test(filepath,  G_FILE_TEST_EXISTS) == FALSE) {
                save_node_content_to_file(nodeset->nodeTab[i], filepath, TRUE);
            }    
                                      
            if (nodeset->nodeTab[i]->type != XML_NAMESPACE_DECL) {
        	    nodeset->nodeTab[i] = NULL;
        	}    
        	g_free(filepath);
            g_free(bin_id);
            g_free(md5);  
        }
        xmlXPathFreeObject(result);    
    }
    
    xmlFree(ns);
    g_free(tmp_path);
    g_free(db_path);                     
    
    return 0;             
}


gint load_binary_data_to_doc(const gchar* schema_name, xmlDocPtr doc)
{
    DB_DEBUG(3, "load_binary_data_to_doc START");
    gchar* db_path = cm_get_db_path_by_name_or_default(g_db_name);
    
    DB_PDEBUG(3, "db_path = %s", db_path);
    if(db_path == NULL) {
        return FAILURE;
    }
    
    gchar* tmp_path = g_strconcat(db_path, G_DIR_SEPARATOR_S, 
                                  BINARY_DIR_NAME, G_DIR_SEPARATOR_S, NULL);
    
   
    
    gchar* query = build_binary_nodes_select_query();
    xmlChar* ns = BAD_CAST cm_get_namespaces_key(schema_name);
    DB_PDEBUG(3, "query = %s", query);      
    xmlXPathObjectPtr result = db_get_nodeset(doc, BAD_CAST query, ns);

    if (result != NULL) {
        xmlNodeSetPtr nodeset = result->nodesetval;

        gint size = (nodeset) ? nodeset->nodeNr - 1 : 0;

        gint i = 0;
        for(i = size; i >= 0; --i) {
            gchar* bin_id = (gchar*)xmlGetProp(nodeset->nodeTab[i], BAD_CAST BINARY_NODE_TAG_FOR_FILENAME);
              DB_PDEBUG(3, "bin_id %s", bin_id);
            gchar* md5 = g_base64_encode((const guchar*)bin_id, strlen(bin_id));
//            md5_get_digest(bin_id, -1, md5);
//             = g_compute_checksum_for_string(G_CHECKSUM_MD5, bin_id, -1);
            gchar* filepath = g_strconcat(tmp_path, G_DIR_SEPARATOR_S, md5, NULL);
    
            gchar* content = NULL;

            g_file_get_contents(filepath, &content, NULL, NULL);            

            xmlNodeSetContent(nodeset->nodeTab[i], BAD_CAST content);
            DB_P2DEBUG(3,"filepath = %s %s", filepath, content);                                      
                                      
            if (nodeset->nodeTab[i]->type != XML_NAMESPACE_DECL) {
        	    nodeset->nodeTab[i] = NULL;
        	}    
	    g_free(filepath);
	    g_free(content);
            g_free(bin_id);
            g_free(md5);  
        }
        xmlXPathFreeObject(result);    
    }
    
    xmlFree(ns);
    g_free(tmp_path);
    g_free(db_path);                     
    
    return 0;             
}


 
static GSList* convert_strv_to_gslist(gchar** strv, gboolean duplicate_values) {
    if (strv == NULL) {
        return NULL;
    }

    gint index = 0;
    GSList* list = NULL;
    
    if (duplicate_values == FALSE) {
        while (strv[index] != NULL) {
            list = g_slist_prepend(list, strv[index]);    
            ++index;
        } 
    } else {
        while (strv[index] != NULL)  {
            list = g_slist_prepend(list, g_strdup(strv[index]));    
            ++index;
        }
   }

    return list;
}

#define XPATH_NAME_FUNCTION "name()"

static gchar* build_binary_nodes_select_query() {//(const gchar* schema_name) {
    gchar** binary_tags = cm_get_all_binary_tags_names();    
    GSList* tags_list = convert_strv_to_gslist(binary_tags, FALSE);
    
    gchar* xpath_filter = create_list_value_filter(XPATH_NAME_FUNCTION, tags_list, DB_OR_LOGIC_OPERATOR);    
    gchar* id_xpath = g_strdup("//*");//cm_get_table_id_path_key(schema_name);    
    
    gchar* xpath = g_strconcat(id_xpath, xpath_filter, NULL);
    
    g_free(xpath_filter);
    g_free(id_xpath);
    g_strfreev(binary_tags);
    g_slist_free(tags_list);
    
    return xpath;
}





static void insert_hash_table_value_to_list(gpointer key, gpointer value, gpointer values_list) {
    key = key; // Just for cancel warning 
    GList* list = (GList*)values_list;
    list = g_list_append(list, value);
}

static GList* get_hash_table_values(GHashTable* hash_table) {
    GList* values = g_list_alloc();
    
    g_hash_table_foreach(hash_table, insert_hash_table_value_to_list, values);

    return g_list_delete_link(g_list_first(values), values);
}
    
static void print_schemas_links()
{
    DB_DEBUG(1, "print_schemas_links START");
    if (g_schemas_links == NULL) {
        return;
    }
    
    //FIXME: need check
    GList* links = get_hash_table_values(g_schemas_links); //g_hash_table_get_values(g_schemas_links);
    GList* keys = get_hash_table_values(g_schemas_links); //g_hash_table_get_keys(g_schemas_links);

    g_printf("\n\nKeys: ");
    GList* list_walker = keys; 
    
    while (list_walker != NULL) {
           DB_PDEBUG(1, "Key: %s", (gchar*)list_walker->data);
           list_walker = g_list_next(list_walker);
    }

     list_walker = links; 
    
    while (list_walker != NULL) {
        if (list_walker->data == NULL) {
                    list_walker = g_list_next(list_walker);
                continue; 
            }
        
            GList* list_walker1 = list_walker->data; 
    
            while (list_walker1 != NULL) {
        
                g_printf("\n\nLink");
                schema_link* link = list_walker1->data;        
                if (link == NULL) {
                    continue;
                }    
                g_printf("\nNamer: %s", link->schema_name);
                g_printf("\nNamerLink: %s", link->linked_schema);
                g_printf("\nLinks: %s", link->links[0]);
                g_printf("\nKeys: %s", link->keys[0]);
                list_walker1 = g_list_next(list_walker1);        
            }
        list_walker = g_list_next(list_walker);
    }
    
 //   g_list_free(links);
       
}    


static void print_schema_link(schema_link* link)
{
      DB_DEBUG(1, "DB: print_schema_link START");
    if (link == NULL) {
        return;
    }
    
       
    g_printf("\n\nLink");
    g_printf("\nNamer: %s", link->schema_name);
    g_printf("\nNamerLink: %s", link->linked_schema);
    g_printf("\nLinks: %s", link->links[0]);
    g_printf("\nKeys: %s", link->keys[0]);

   
   DB_DEBUG(1, "DB: print_schema_link END");
}    
    

static void free_schema_link(schema_link* link)
{ 
    g_free(link->schema_name);
    g_free(link->linked_schema);
    g_strfreev(link->links);
    g_strfreev(link->keys);
    g_free(link->actions);
 
    g_free(link);
}

static void free_linked_links(GSList* linked_links)
{
    if (linked_links == NULL) {
        return;
    } 

    GSList* list_walker = linked_links;
  
    while (list_walker != NULL) {      
        free_schema_link((schema_link*)list_walker->data);               
        list_walker = g_slist_next(list_walker);    
    }  

    g_slist_free(linked_links);
}
    
    
static GSList* get_linked_links(const gchar* schema_name)
{
     DB_DEBUG(1, "DB: get_linked_links START");
    gchar** linked_schemas = cm_get_links_names(schema_name);
    GSList* links_list = NULL;
    if (linked_schemas == NULL) {
        return NULL;
    }
    gint index = 0;

    while (linked_schemas[index] != NULL) {
         DB_DEBUG(5, "DB: get_linked_links; create link");
        schema_link* link = g_new0(schema_link, 1);
       
        link->schema_name = g_strdup(schema_name);
        link->linked_schema = g_strdup(linked_schemas[index]);   
        link->links = cm_get_links(schema_name, linked_schemas[index]);
        link->keys = cm_get_keys(schema_name, linked_schemas[index]);
        link->actions = NULL;
        
        links_list = g_slist_append(links_list, link);
        print_schema_link(link);
        ++index;
    }
    
    g_strfreev(linked_schemas);

    return links_list;
}      


gchar* get_id_text_notation(const gchar* schema_name)
{
    gchar* id_name = cm_get_table_id_name_key(schema_name);
    gchar* id_ns_prefix = cm_get_table_id_ns_prefix_key(schema_name);

    if (id_name == NULL) {
        g_free(id_ns_prefix);
        g_free(id_name);
 
        return NULL;
    } 
    
    if (id_ns_prefix == NULL) {
        id_ns_prefix = g_strdup("");
    } else {
        gchar* tmp = id_ns_prefix;
        id_ns_prefix = g_strconcat(id_ns_prefix, ":", NULL);
        g_free(tmp);                
    }
    
    gchar* id = g_strconcat("@", id_ns_prefix, id_name, NULL);
    
    g_free(id_ns_prefix);
    g_free(id_name);
    
    return id;
}


 GSList* get_nodes_ids(const gchar* schema_name, gchar* xpath)
{
    DB_DEBUG(1, "get_nodes_ids START");
    
    const xmlDocPtr doc = get_table_from_list(schema_name);       
    xmlChar* ns = BAD_CAST cm_get_namespaces_key(schema_name);
    
    xmlXPathObjectPtr result = db_get_nodeset(doc, BAD_CAST xpath, ns);
    
    GSList* id_list = NULL;
    
    if (result != NULL) {
        xmlNodeSetPtr nodeset = result->nodesetval;

        gint size = (nodeset) ? nodeset->nodeNr - 1 : 0;
		gint i;

        for(i = size; i >= 0; --i) {
            gchar* content = (gchar*)xmlNodeGetContent(nodeset->nodeTab[i]);
            id_list = g_slist_prepend(id_list, content);
        }   
    }    
    DB_DEBUG(1, "get_nodes_ids END ");

    return id_list;
}

static GSList* copy_text_slist(const GSList* list) {
    GSList* new_list = NULL;
    
    while (list != NULL) {
        new_list = g_slist_prepend(new_list, g_strdup((gchar*)list->data));
        list = g_slist_next(list);
    }
    
    return new_list;
}



static GSList* build_delete_ids_for_link(const gchar* schema_name,  gchar** key, const GSList* delete_ids) {
      g_debug("build_delete_ids_for_link START ");
    if (schema_name == NULL || key == NULL || delete_ids == NULL) {
        return NULL;
    }    
    

    gboolean is_nativ_ids_insert = FALSE;
    GSList* id_list = NULL;

    gchar* id = get_id_text_notation(schema_name);
    gchar* id_xpath = cm_get_table_id_path_key(schema_name);

    const GSList* list_walker = delete_ids;

    GString* tmp = g_string_new("[");              
    
    while (list_walker != NULL) {
        tmp = g_string_append(tmp, id);
        tmp = g_string_append(tmp, "='");
        tmp = g_string_append(tmp, (gchar*)list_walker->data);
        tmp = g_string_append(tmp, "'");
        list_walker = g_slist_next(list_walker);
        
        if (list_walker != NULL) {
            tmp = g_string_append(tmp, " or ");
        }    
    }
    tmp = g_string_append(tmp, "]");

    gchar* or_part = g_string_free(tmp, FALSE);

    gint index = 0;
    while (key[index] != NULL) {
        if (strcmp(key[index], DB_LINK_ID) == 0 && is_nativ_ids_insert == FALSE) {
            id_list = g_slist_concat(id_list, copy_text_slist(delete_ids));
            g_debug("DATA: %s", (gchar*)id_list->data);
            is_nativ_ids_insert = TRUE;
        } else {
        
            gchar* xpath_req = g_strconcat(id_xpath, or_part, "/", key[index], NULL);                    
            id_list = g_slist_concat(id_list, get_nodes_ids(schema_name, xpath_req));
            g_debug("xpath_req: %s", xpath_req);
        }
        
      
        ++index;
    }
      g_debug("build_delete_ids_for_link END ");    
    return id_list;
}


static GSList* build_delete_ids_for_schema_links(const GSList* schema_links_list, const GSList* delete_ids)
{
        g_debug("build_delete_ids_for_schema_links START ");
    if (/*schema_links_list == NULL ||*/ delete_ids == NULL) {
        return NULL;
    } 

    const GSList* list_walker = schema_links_list;
    
    GSList* delete_data_list = NULL;       
    while (list_walker != NULL) {
        schema_link* link = (schema_link*)list_walker->data;
        
        delete_data* data = g_new0(delete_data, 1);    
        data->schema_name = g_strdup(link->linked_schema);
        data->id_list = build_delete_ids_for_link(link->schema_name, link->keys, delete_ids);
        gchar** links = cm_get_links(link->schema_name, link->linked_schema);
        data->id_path = g_strdup(links[0]);
        
        g_strfreev(links);
        delete_data_list = g_slist_prepend(delete_data_list, data);
        
        list_walker = g_slist_next(list_walker);
    }

    g_debug("build_delete_ids_for_schema_links END");
    return delete_data_list;
}



static GSList* build_rev_link_select_xpath(const delete_data* del_data, const gchar* linked_schema_name)
{
    DB_PDEBUG(1, "build_rev_link_select_xpath START: %s ", del_data->schema_name);
    gchar** keys = cm_get_keys(linked_schema_name, del_data->schema_name);

    GSList* id_list = NULL;

    gint index = 0;    
    while (keys != NULL && keys[index] != NULL) {
        DB_PDEBUG(4, "Key: %s", keys[index]);
        gchar** split_key = g_strsplit(keys[index], SLASH_AT_DELIMETER, 2);

        ++index;
        if (g_strv_length(split_key) != 2) {
            g_strfreev(split_key);
            continue;      
        }
        
        gchar* id = g_strconcat(AT_DELIMETER, split_key[1], NULL);
        
        gchar* filter = create_list_value_filter(id, del_data->id_list, DB_OR_LOGIC_OPERATOR);
        gchar* id_xpath = cm_get_table_id_path_key(linked_schema_name);
        
        delete_data* data = g_new0(delete_data, 1);    
        data->schema_name = g_strdup(linked_schema_name);
        data->id_list = copy_text_slist(del_data->id_list);
        data->id_path = g_strconcat(id_xpath, "/", split_key[0], filter, NULL);
        
        id_list = g_slist_prepend(id_list, data);
        
        DB_PDEBUG(3, "Rev link filter: %s", data->id_path);

        g_free(id_xpath);
        g_free(filter);
        g_free(id);        
        g_strfreev(split_key);
    }
    
    g_strfreev(keys);
    DB_DEBUG(1, "build_rev_link_select_xpath END");
    return id_list;
}

static GSList* build_rev_delete_ids(const GSList* delete_ids)
{
    DB_DEBUG(1, "build_rev_delete_ids START");

    GSList* rev_id_list = NULL;

    
    while (delete_ids != NULL) {
        delete_data* data = (delete_data*)delete_ids->data;
        gchar** names = cm_get_revers_links_names(data->schema_name);        
        gint index = 0;
        DB_PDEBUG(3, "Schema: %s", data->schema_name);
        
        if(names == NULL) {
           // continue;
        }
        
        while (names != NULL && names[index] != NULL) {
            DB_PDEBUG(3, "Schema: %s", data->schema_name);
             rev_id_list = g_slist_concat(rev_id_list, build_rev_link_select_xpath(data, names[index]));
            
            ++index;
        }        
        
        delete_ids = g_slist_next(delete_ids);
        g_strfreev(names);
    }

    DB_DEBUG(1, "build_rev_delete_ids END");
    return rev_id_list;
}


static GSList* build_delete_ids(delete_data* delete_ids) {
    
    g_debug("build_delete_ids START");
    if (delete_ids == NULL) {
        return NULL;
    }
  
    
    GSList* delete_list = g_slist_append(NULL, delete_ids);
 //   delete_list = g_slist_append(delete_list, data);
  
    GSList* list_walker = delete_list;
    
    while (list_walker != NULL) {
        g_debug("SCHEMA: %s", ((delete_data*)list_walker->data)->schema_name);
        GSList* schema_links = (GSList*)g_hash_table_lookup(g_schemas_links, 
                                                      ((delete_data*)list_walker->data)->schema_name);
                                                      
        delete_list = g_slist_concat(delete_list, build_delete_ids_for_schema_links(schema_links, ((delete_data*)list_walker->data)->id_list));       
        
        list_walker = g_slist_next(list_walker);                                     
    }
    
    
    list_walker = delete_list;
    
//    while (list_walker != NULL) {
    
        
//    }
    

//    xmlXPathObjectPtr db_get_nodeset(const xmlDocPtr doc, const xmlChar* xpath, 
//                                     const xmlChar* ns_list);

    DB_DEBUG(1, "build_delete_ids END");
    return delete_list;
}



gchar* create_list_value_filter(const gchar* key, const GSList* values, const gchar* logic_op)
{
    DB_PDEBUG(1, "create_list_value_filter START: key = %s", key);
    if (key == NULL) {
        return NULL;
    }
    
    GString* filter_string = g_string_new("[");        

    
    if (values == NULL) {
        filter_string = g_string_append(filter_string, key);
        filter_string = g_string_append(filter_string, "]");        
        return g_string_free(filter_string, FALSE);
    }
    
    while (values != NULL) {
        gchar* tmp = g_strconcat(" ", key, "='", (gchar*)values->data, "' ", NULL);
        filter_string = g_string_append(filter_string, tmp);
        g_free(tmp);

        values = g_slist_next(values);
        
        if (values != NULL) {
            filter_string = g_string_append(filter_string, logic_op);
        }    
     }
     
    filter_string = g_string_append(filter_string, "]");     
    DB_DEBUG(1, "create_list_value_filter END");   
    return g_string_free(filter_string, FALSE);

}


gchar* create_select_xpath_request(const gchar* schema_name, 
                                   const gchar* id_path, const GSList* id_list)
{
    DB_DEBUG(1, "create_select_xpath_request START");
    if (schema_name == NULL || id_path == NULL ) {
        return NULL;
    }
    
    gchar* id = get_id_text_notation(schema_name);
    gchar* id_xpath = cm_get_table_id_path_key(schema_name);
    
    gchar* filter = NULL;
    
    if (strcmp(id_path, DB_LINK_ID) == 0) {
        filter = create_list_value_filter(id, id_list, DB_OR_LOGIC_OPERATOR);
    } else {
        filter = create_list_value_filter(id_path, id_list, DB_OR_LOGIC_OPERATOR);
    }
    DB_PDEBUG(1, "Filter %s", filter);    
    gchar* xpath_query = g_strconcat(id_xpath, filter, NULL); 
    
    g_free(id);
    g_free(id_xpath);
    g_free(filter);
    
    DB_PDEBUG(1, "create_select_xpath_request END %s", xpath_query);
    return xpath_query;
}


static void remove_nodes(const GSList* delete_data_list, gboolean generate_select_xpath)
{
   
    while (delete_data_list != NULL) {
        delete_data* delete_ids = (delete_data*)delete_data_list->data;
        
        const xmlDocPtr doc = get_table_from_list(delete_ids->schema_name);   
 
        gchar* ns = cm_get_namespaces_key(delete_ids->schema_name);     

        if (generate_select_xpath == TRUE) {
            gchar* del_xpath = create_select_xpath_request(delete_ids->schema_name, 
                                      delete_ids->id_path, delete_ids->id_list);
            db_remove_node_by_xpath(doc, del_xpath, ns);
            g_free(del_xpath);
        } else {
            db_remove_node_by_xpath(doc, delete_ids->id_path, ns);
        }
        
        xmlFree(ns);
//FIXME: Just for debug   
        gchar* db_path = cm_get_db_path_by_name_or_default(g_db_name);
        db_save_table(doc, delete_ids->schema_name, db_path);
        g_free(db_path);
   
         delete_data_list = g_slist_next( delete_data_list);
    }
}


/**
 * @brief Find nodes, links associated with given schema and remove it from db.
 *
 * @param schema_name name of schema.
 * @param id_list NULL-ended id list.
 * 
 * @return 0 on success or not 0 otherwise.
 */
 
gint db_remove_nodes_with_links(gchar* schema_name, GSList* id_list)
{
    DB_DEBUG(1, "db_remove_nodes_with_links START");
    delete_data* data = g_new0(delete_data, 1);
    data->schema_name = schema_name;
    data->id_list = id_list;
    data->id_path = g_strdup(DB_LINK_ID);

    GSList* delete_id_list = build_delete_ids(data);
    GSList* rev_delete_id_list = build_rev_delete_ids(delete_id_list);
//    build_rev_link_select_xpath(const delete_data* del_data, const gchar* linked_schema_name)

 //   GSList* list_walker = delete_id_list;
   
   remove_nodes(delete_id_list, TRUE);
   remove_nodes(rev_delete_id_list, FALSE);
   /* 
    while (list_walker != NULL) {
        delete_data* delete_ids = (delete_data*)list_walker->data;
        const xmlDocPtr doc = get_table_from_list(delete_ids->schema_name);   
 
        gchar* ns = cm_get_namespaces_key(delete_ids->schema_name);     
        gchar* del_xpath = create_select_xpath_request(delete_ids->schema_name, 
                                                       delete_ids->id_path, 
                                                       delete_ids->id_list);

       // db_remove_node_by_xpath(doc, del_xpath, ns);

        g_free(del_xpath);
        xmlFree(ns);
        gchar* db_path = cm_get_db_path_for_user(USERNAME);
        db_save_table(doc, delete_ids->schema_name, db_path);
        g_free(db_path);
        list_walker = g_slist_next(list_walker);
    }
    */
    DB_DEBUG(1, "db_remove_nodes_with_links END");
    return 0;
}



static void build_schemas_links()
{
    DB_DEBUG(1, "DB: get_links START");
    gchar** schemas_names = cm_get_all_schemas_names();
    
    if (schemas_names == NULL) {
        return;
    }
 
    g_schemas_links = g_hash_table_new_full(g_str_hash, g_str_equal, 
                            (GDestroyNotify)g_free, (GDestroyNotify)free_linked_links);
       
    gint index = 0;
    DB_DEBUG(1, "DB: get_links ");
   
    while (schemas_names[index] != NULL) {
        DB_DEBUG(1, "ssss");
         g_debug("\nName: %s", schemas_names[index]);
        GSList* links = get_linked_links(schemas_names[index]);
       
        g_hash_table_insert(g_schemas_links, g_strdup(schemas_names[index]), links);         
       g_printf("~~~~~~~~~~~~~~%i \n",   g_slist_length(links));

        if(links!=NULL)
         print_schema_link(links->data);
        // print_schema_link( ((GSList*)g_hash_table_lookup(g_schemas_links, schemas_names[index]))->data);   
                 g_printf("~~~~~~~~~~~~~~");
        ++index;       
    }
    
    g_strfreev(schemas_names);
    
    DB_DEBUG(1, "DB: get_links END");
    return;
}
      
// FIXME: write reload tables cod      
/**
 * @brief Initialization table store.
 *
 * @param reload_tables if equals TRUE all old table unload from store.
 * 
 */      
static void init_tables_store(gboolean reload_tables)
    {
    if (g_tables == NULL) {
        g_tables = g_hash_table_new_full(g_str_hash, g_str_equal, 
                                         g_free, (GDestroyNotify)xmlFreeDoc);
    } else  if (reload_tables == TRUE) {
        g_hash_table_remove_all(g_tables);
    }
}      
      
      
// FIXME; exclude function from this file
/**
 * @brief Theck string for NULL and zero length.
 *
 * @param str string for check.
 *
 * @return TRUE if pointer equals NULL or string zero-length and FALSE otherwise. 
 */      
static gboolean is_str_null_or_empty(const gchar* str) 
{
    if (str == NULL) {
        return TRUE;
    }
    
    if (strlen(str) == 0) {
        return TRUE;
    }
    
    return FALSE;
}

   
/**
 * @brief Create path to data base.
 *
 * @return 0 on success or not 0 otherwise
 */      
static gint create_path(const gchar* db_name)
{
    gchar* db_path = NULL;
    g_free(g_db_name);
    g_db_name = NULL;

    if (is_str_null_or_empty(db_name) == TRUE) {
        db_path = cm_get_default_db_path();
    } else {
        g_db_name = g_strdup(db_name);
        db_path = cm_get_db_path_by_name(db_name);
    }    

    if (db_path == NULL && is_str_null_or_empty(db_name) == TRUE) {
        db_path = g_strconcat(HOME_DIR, G_DIR_SEPARATOR_S, PROGRAM_DIR_NAME, 
                              G_DIR_SEPARATOR_S, PROGRAM_DB_NAME, NULL);
    } else {
       db_path = g_strconcat(HOME_DIR, G_DIR_SEPARATOR_S, PROGRAM_DIR_NAME, 
                             G_DIR_SEPARATOR_S, PROGRAM_DB_NAME, G_DIR_SEPARATOR_S, db_name, NULL);
    }

    gint result = g_mkdir_with_parents(db_path, FILE_PERMISSIONS);    

    if (result == SUCCESS && db_name != NULL) {
        cm_set_db_path_by_name(db_name, db_path);
        cm_save_config();
    }
    
    g_free(g_db_path);
    g_db_path = db_path;
    
    DB_P2DEBUG(3, "DB_LOGIC - create_path: DB name = %s; path = %s", g_db_name, g_db_path);
    
    return result;
}


/**
 * @brief Load tables associated with given schema.
 *
 * @param tables, store for xml-data.
 * @param schema_name name of schema.
 * 
 * @return 0 on success or not 0 otherwise
 */
static gint load_table(GHashTable* tables, const gchar* schema_name)
{
    init_tables_store(FALSE);

    if (g_hash_table_lookup(tables, schema_name) != NULL) {
        return ERROR_TABLE_ALLREADY_LOAD;
    }

    gchar* db_path = cm_get_db_path_by_name_or_default(g_db_name);

    xmlDocPtr table = db_get_table(schema_name, db_path);

    g_free(db_path);
        
    if (table == NULL) {
        return FAILURE;
    }
        
    g_hash_table_replace(tables, (gchar*)schema_name, table);
   
    return SUCCESS;
}


/**
 * @brief Load tables associated with given schemas.
 *
 * @param tables, store for xml-data.
 * @param schemas schemas of data.
 * 
 * @return 0 on success or not 0 otherwise
 */
static gint load_tables(GHashTable* tables, const gchar* const schemas[])
{
    gchar* db_path = cm_get_db_path_by_name_or_default(g_db_name);
    init_tables_store(TRUE);
 
    gint index = 0;
    while (schemas[index] != NULL) {
        xmlDocPtr table = db_get_table(schemas[index], db_path);
        
        if (table == NULL) {
            g_free(db_path);
            unload_tables(tables, schemas);
            return FAILURE;
        }
        
        g_hash_table_replace(tables, (gchar*)schemas[index], table);
 
        ++index;
    }
    
    g_free(db_path);
    db_path = NULL;
  
    return SUCCESS;     
}


/**
 * @brief Unload tables associated with given schemas.
 *
 * @param tables, xml-files with data.
 * @param schemas schemas of data.
 * 
 */
static void unload_tables(GHashTable* tables, const gchar* const schemas[])
{
    gint index = 0;
    while (schemas[index] != NULL) {
        gpointer data = g_hash_table_lookup(tables, (gchar*)schemas[index]);

        if (data != NULL) {
             xmlFreeDoc(data);
             data = NULL;
        }  
        ++index;
    }
} 


/**
 * @brief Unload table associated with given schema.
 *
 * @param tables, xml-files with data.
 * @param schema_name name of schema.
 * 
 */
static void unload_table(GHashTable* tables, const gchar* schema_name)
{
     gpointer data = g_hash_table_lookup(tables, schema_name);

     if (data != NULL) {
             xmlFreeDoc(data);
             data = NULL;
     }  
}


/**
 * @brief Get table associated with given schema.
 *
 * Look table in tables list, if table not load try load it.
 *
 * @param schema_name name of schema.
 *
 * @return table, xml-data, or NULL on error.
 */
 xmlDocPtr get_table_from_list(const gchar* schema_name)
{
    if (load_table(g_tables, schema_name) == FAILURE) {
        return NULL;
    }
    
    return g_hash_table_lookup(g_tables, schema_name);
}

/*
static xmlDocPtr get_table_from_list(table_indexes index)
{
    if (index >= TABLES_COUNT) {
        return NULL;
    }

    if (!tables_list[index]) {
        //load_table(index, tables_list);
    }
    
    return tables_list[index];
}

static const gchar* get_table_schema(table_indexes index)
{
    if (index >= TABLES_COUNT) {
        g_warning("DB logic func: get_table_schema - incorret table index.");
        return NULL;
    }
    
    return schemas_list[index];
}
*/


/**
 * @brief Add new parameter to parameter list.
 *
 * @param parameters list of parameters, NULL-terminated list of strings.
 * @param name name of new parameter.
 * @param value of new parameter.
 * 
 * @retunr new parameters list on success or NULL otherwise
 */
static gchar** add_to_parameters(const gchar** parameters, 
                                 const gchar* name, const gchar* value) 
{
    gulong size = 3;  // Add size: 1 - name, 2 - value and 3 - NULL
    
    if (name == NULL || value == NULL) {
        return (gchar**)parameters;
    }
    
    if (parameters != NULL) {
        size += sizeof(parameters) / sizeof(parameters[0]) - 1;
    }

    gchar** new_parameters = g_try_realloc(parameters, size);
    
    if (new_parameters == NULL) {
        return NULL;
    }

    new_parameters[size - 3] = g_strdup(name);//g_strdup(id_name);    
    new_parameters[size - 2] = g_strdup(value);//g_strdup(id_value);
    new_parameters[size - 1] = NULL;

    gint i = 0;
    while (new_parameters[i]) {
        g_message("params: %i - %s", i, new_parameters[i]);
        ++i;
    }
    
    return new_parameters;
}


/**
 * @brief Find node and remove it from xml document associated with given schema.
 *
 * @param schema_name name of schema.
 * @param id identifier of node.
 * 
 * @return 0 on success or not 0 otherwise.
 */
gint db_remove_node_by_id(const gchar* schema_name ,const gchar* id)
{
    gchar* id_xpath = cm_get_table_id_path_key(schema_name);
    gchar* id_name = cm_get_table_id_name_key(schema_name);
    gchar* id_ns_prefix = cm_get_table_id_ns_prefix_key(schema_name);

    xmlChar* ns = BAD_CAST cm_get_namespaces_key(schema_name);


    const xmlDocPtr doc = get_table_from_list(schema_name);

    //FIXME: id_ns_prefix maybe NULL
    if (!id_xpath || !id_name || /*!id_ns_prefix ||*/ doc == NULL) {
        g_free(id_ns_prefix);
        g_free(id_name);
        g_free(id_xpath);

        return FAILURE;
    }
    
   
    //FIXME: rewrite, set ns with ":"
    gchar* xpath_query = NULL;        
    if (id_ns_prefix == NULL){
        if (strlen(id) == 0 || strcmp("*", id) == 0) {
            xpath_query = g_strconcat(id_xpath, "[@",   id_name, "]", NULL );
        } else {   
            xpath_query = g_strconcat(id_xpath, "[@",  id_name, "=", id, "]", NULL );
        }    
    } else {
        if (strlen(id) == 0 || strcmp("*", id) == 0) {
            xpath_query = g_strconcat(id_xpath, "[@",  id_ns_prefix, ":", id_name, "]", NULL );
        } else {   
            xpath_query = g_strconcat(id_xpath, "[@",  id_ns_prefix, ":", id_name, "=", id, "]", NULL );
        }    
    }
    g_debug("REMOVE XPATH = %s", xpath_query);
    xmlXPathObjectPtr result = db_get_nodeset(doc, BAD_CAST xpath_query, ns);


     if (result != NULL) {
	    xmlNodeSetPtr nodeset = result->nodesetval;
		
        gint size = (nodeset) ? nodeset->nodeNr - 1 : 0;
		gint i;

        for(i = size; i >= 0; --i) {
		g_debug("Remove node %i", i);
		    xmlUnlinkNode(nodeset->nodeTab[i]);
		    xmlFreeNode(nodeset->nodeTab[i]);
		    
		    
//		    if (nodeset->nodeTab[i]->type != XML_NAMESPACE_DECL) {
    	        nodeset->nodeTab[i] = NULL;
//            }
            
                    
        }
    }
    
     gchar* db_path = cm_get_db_path_by_name_or_default(g_db_name);
     db_save_table(doc, schema_name, db_path);
     g_free(db_path);
    
     return 0;
}


/**
 * @brief Find node in xml document associated with given schema.
 *
 * @param schema_name name of schema.
 * @param id identifier of node.
 * 
 * @retunr 0 on success or not 0 otherwise.
 */
 //FIXME: need retorn node
GSList* db_get_node_by_id(const gchar* schema_name ,const gchar* id)
{
    gchar* id_xpath = cm_get_table_id_path_key(schema_name);
    gchar* id_name = cm_get_table_id_name_key(schema_name);
    gchar* id_ns_prefix = cm_get_table_id_ns_prefix_key(schema_name);

    xmlChar* ns = BAD_CAST cm_get_namespaces_key(schema_name);

    const xmlDocPtr doc = get_table_from_list(schema_name);

    //FIXME: id_ns_prefix maybe NULL
    if (!id_xpath || !id_name || /*!id_ns_prefix ||*/ doc == NULL) {
        g_free(id_ns_prefix);
        g_free(id_name);
        g_free(id_xpath);

        return NULL;
    }
    //FIXME: rewrite, set ns with ":"
    gchar* xpath_query = NULL;        
    if (id_ns_prefix == NULL){
        if (strlen(id) == 0 || strcmp("*", id) == 0) {
            xpath_query = g_strconcat(id_xpath, "[@",   id_name, "]", NULL );
        } else {   
            xpath_query = g_strconcat(id_xpath, "[@",  id_name, "=", id, "]", NULL );
        }    
    } else {
        if (strlen(id) == 0 || strcmp("*", id) == 0) {
            xpath_query = g_strconcat(id_xpath, "[@",  id_ns_prefix, ":", id_name, "]", NULL );
        } else {   
            xpath_query = g_strconcat(id_xpath, "[@",  id_ns_prefix, ":", id_name, "=", id, "]", NULL );
        }    
    }

    g_debug("XPATH = %s", xpath_query);
    
    
    g_debug("Find node, xpath: %s", xpath_query);
    g_free(id_ns_prefix);
    g_free(id_name);
    g_free(id_xpath);
    
    xmlXPathObjectPtr result = db_get_nodeset(doc, BAD_CAST xpath_query, ns);

    GSList* doc_list = NULL; 

     if (result != NULL) {
	    xmlNodeSetPtr nodeset = result->nodesetval;
		
        gint size = (nodeset) ? nodeset->nodeNr - 1 : 0;
		gint i;		

        for(i = size; i >= 0; --i) {
		g_debug("Find node %i", i);
//		    xmlUnlinkNode(nodeset->nodeTab[i]);
	//	    xmlFreeNode(nodeset->nodeTab[i]);
		    //xmlNodeGetContent

		     
		    
		    xmlNodePtr node = xmlCopyNode	( nodeset->nodeTab[i],1);
		    		    xmlBufferPtr buf = xmlBufferCreate();
		    xmlNodeDump(buf, 
					  NULL, node, //nodeset->nodeTab[i], 
					0,1);
	        xmlChar* content = (xmlChar*)xmlBufferContent(buf);//= xmlNodeListGetString(NULL , xmlDocGetRootElement	(doc),1); //(doc,nodeset->nodeTab[i],0 );
	        
	        
	        
	        g_printf("\n\nNODE Content: %s", content);
	        
	        xmlDocPtr elem = xmlParseDoc(content);
	        
            xmlDocPtr external_format = db_apply_revers_stylesheet(schema_name, elem, NULL);
	        
   	        g_printf("\n\nNODE Afte transform1: ");
   	        load_binary_data_to_doc(schema_name, external_format);
//DEBUG   	        xmlDocDump(stdout, external_format);
   	        	
	        
	        doc_list = g_slist_prepend(doc_list, xmlCopyDoc(external_format,1));  
	        xmlFreeDoc(external_format);
	        xmlFreeDoc(elem);
	        xmlFree(content);
//		    if (nodeset->nodeTab[i]->type != XML_NAMESPACE_DECL) {
//    	        nodeset->nodeTab[i] = NULL;
//            }
            
                    
        }
        }
         gchar* db_path = cm_get_db_path_by_name_or_default(g_db_name);
     db_save_table(doc, schema_name, db_path);
     
     return doc_list;
}


/**
 * @brief Add new node to data base.
 *
 * @param schema_name name of schema.
 * @param new_data new xml node.
 * @param params NULL-terminated list of strings. 
 *        format: name value name value ... NULL.
 * 
 * @retunr 0 on success or not 0 otherwise.
 */
gchar* db_add_data_to_db(const gchar* schema_name, const xmlDocPtr new_data,
        const gchar** parameters)
{
    
    if (schema_name == NULL) {
        return NULL;//ERROR_CANT_GET_TABLE_SCHEMA;
    }

    const xmlDocPtr table = get_table_from_list(schema_name);
    
    if (table == NULL) {
        return NULL;//ERROR_CANT_LOAD_TABLE;
    }
    
    gchar* id_value = db_generate_id(schema_name, table);
    gchar* id_name = cm_get_table_id_name_key(schema_name);

//    gchar* id_ns = cm_get_table_id_ns_key(schema_name);
//    gchar* id_ns_prefix = cm_get_table_id_ns_prefix_key(schema_name);
    
    gchar** new_parameters = NULL;
        
    g_debug("DB logic func: add_data_to_db - newId = %s", id_value);

    if (id_value && id_name) {
        new_parameters = add_to_parameters(parameters, id_name, id_value);
    } else {
        new_parameters = g_strdupv((gchar**)parameters);
    }

    gchar* xslt_path = cm_get_xslt_path_key(schema_name);

    xsltStylesheetPtr style = NULL;
    
    if (xslt_path) {
        style = xsltParseStylesheetFile(BAD_CAST xslt_path);
    }
    
    if (!style) {
        g_message("NO STYLE: %s", xslt_path);
    }


    prepare_binary_data_fo_save(schema_name, new_data);
    xmlDocPtr db_format_doc = db_apply_stylesheet(schema_name, new_data, 
                                                  (const gchar**)new_parameters);

 //   xmlChar* ns = BAD_CAST cm_get_namespaces_key(schema_name);
//    gchar* path = cm_get_table_id_path_key(schema_name); 
         
//    xmlChar* xpath = g_strconcat(path, "[@", id_ns_prefix,":", id_name, "=\"", id_value, "\"]", NULL);
    
//    g_message("PATH: %s %s", xpath, ns);     

//    db_get_nodes_count(db_format_doc, xpath, 
//                       ns);
   
//    if (db_check_unique(schema_name, table, db_format_doc) == 0 ) {
        db_add_doc_to_tree(table, db_format_doc, NULL, NULL);

        gchar* db_path = cm_get_db_path_by_name_or_default(g_db_name);//USERNAME);

        db_save_table(table, schema_name, db_path);
        save_binary_data();
//    }

    xmlFreeDoc(db_format_doc); 
//    g_free(id_value);
    g_free(id_name);
    g_strfreev(new_parameters);
    
    return id_value;
}


/**
 * @brief Add new node to data base.
 *
 * @param schema_name name of schema.
 * @param new_data new xml node.
 * @param params NULL-terminated list of strings. 
 *        format: name value name value ... NULL.
 * 
 * @retunr 0 on success or not 0 otherwise.
 */
gint db_add_data_to_db_with_id(const gchar* schema_name, const xmlDocPtr new_data,
                               const gchar** parameters)
{
    
    if (schema_name == NULL) {
        return ERROR_CANT_GET_TABLE_SCHEMA;
    }

    const xmlDocPtr table = get_table_from_list(schema_name);
    
    if (table == NULL) {
        return ERROR_CANT_LOAD_TABLE;
    }
    
    gchar** new_parameters = NULL;

    new_parameters = g_strdupv((gchar**)parameters);

    gchar* xslt_path = cm_get_xslt_path_key(schema_name);

    xsltStylesheetPtr style = NULL;
    
    if (xslt_path) {
        style = xsltParseStylesheetFile(BAD_CAST xslt_path);
    }
    
    if (style == NULL) {
        g_message("NO STYLE: %s", xslt_path);
    }
    xsltFreeStylesheet(style);
    g_free(xslt_path);
        
    prepare_binary_data_fo_save(schema_name, new_data);

    
    xmlDocPtr db_format_doc = db_apply_stylesheet(schema_name, new_data, 
                                                  (const gchar**)new_parameters);

   
//    if (db_check_unique(schema_name, table, db_format_doc) == 0 ) {
        db_add_doc_to_tree(table, db_format_doc, NULL, NULL);

        gchar* db_path = cm_get_db_path_by_name_or_default(g_db_name);//USERNAME);

        db_save_table(table, schema_name, db_path);
        save_binary_data();
//    }
    g_free(db_path);
    xmlFreeDoc(db_format_doc); 
    g_strfreev(new_parameters);
    
    return 0;
}



/**
 * @brief Prepare data base for work.
 * 
 */
void db_prepare(const gchar* db_name)
{
    
    create_path(db_name);
    build_schemas_links();
 
    //print_schemas_links();
    init_tables_store(TRUE);
    DB_DEBUG(1,"!!!!!!!!!!!!!!!!AAAAAAAAAAAAA");
}







