/*******************************************************************************
**
** vktransport.cpp - transport module for vkgallery
** Based on livkontakte API:  http://oss.fruct.org/wiki/Libvkontakte/xml-data
**
** Copyright (C) 2010 Sergey Zakharov
**
** vkgallery is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** vkgallery is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with QMF-Vkontakte plug-in; if not, write to the Free Software
** Foundation, Inc., 51 Franklin St, Fifth Floor,
** Boston, MA  02110-1301  USA
**
*******************************************************************************/

#include <QDebug>
#include <QDateTime>

#include "vktransport.h"


// Name of exported function
#define MODULE_INIT "msa_module_init"
#define MODULE_GET_INFO "msa_module_get_info"

#define VK_RESP_CODE "//string[@name='code']"
#define VK_RESP_TEXT "//string[@name='text']"
#define VK_FIRST_NAME "//string[@name='FirstName']"
#define VK_LAST_NAME "//string[@name='LastName']"
#define VK_TITLE "//string[@name='title']"
#define VK_DESCRIPTION "//string[@name='description']"
#define VK_OWNER_ID "//string[@name='ownerId']"
#define VK_ALBUM_ID "//string[@name='albumId']"
#define VK_THUMB_URL "//string[@name='urlSmall']"
#define VK_PHOTO_URL "//string[@name='urlBig']"
#define VK_FRIEND_IMG "//img[@name='FriendImg']/@uri"
#define VK_ALBUM_IMG "//img[@name='albumImg']/@uri"

#define VK_STATUS "/string[@name='FriendStatus']"
#define VK_STRUCT "//array[@name='messagetList']/struct["
#define VK_ID_R "/@id"
#define VK_SENDER_NAME "/string[@name='SenderName']"
#define VK_SENDER_ID "/string[@name='SenderId']"
#define VK_RECIPIENT_NAME "/string[@name='RecipientName']"
#define VK_RECIPIENT_ID "/string[@name='RecipientId']"
#define VK_TIME "/string[@name='Time']"
#define VK_TEXT "/string[@name='Text']"
#define VK_IMG "//img[@name='Img']"
#define VK_FCSID "//string[@name='fcsid']"
#define VK_FCCODE "//string[@name='fccode']"
#define VK_IMG "//img[@name='Img']"
#define VK_MODULE_NAME "/string[@name='moduleName']"
#define VK_OLD_REQUEST "/string[@name='oldRequest']"
#define VK_FUNCTION "//Response/@function"
#define VK_FROM "//array/@from"
#define VK_TO "//array/@to"
#define VK_QUANTITY "//array/@quantity"
#define VK_PARAMS "//TransitData/Content/Response/Params"
#define VK_PARAMS_ID "//Params/@id"

#define VK_ARRAY_FRIENDS "//array[@name='ContactList']"
#define VK_ARRAY_ALBUMS "//array[@name='albumList']"
#define VK_ARRAY_PHOTOS "//array[@name='photosList']"
#define VK_STRUCT_FRIEND "//array[@name='ContactList']/struct["
#define VK_STRUCT_ALBUM "//array[@name='albumList']/struct["
#define VK_STRUCT_PHOTO "//array[@name='photosList']/struct["
#define VK_TIME_CREATE "//number[@name='created']"
#define VK_TIME_UPDATE "//number[@name='updated']"
#define VK_SIZE "//number[@name='size']"



VkTransport::VkTransport(QString libraryname, QString accountName)
    : msa(0)
{
    activeRequests = 0;
    needShutdown = false;

    this->libraryName = libraryname;
    // Add dynamic library (libvkontakte.so)
    if ((msa = new QLibrary(libraryname)) == 0)
        qFatal("Problems with initiation QLibrary object");

    msa->setLoadHints(QLibrary::ResolveAllSymbolsHint); // analog RTLD_LAZY
    // Check driver existance
    if (!msa->isLoaded())
        if (!msa->load())
            qFatal((QString("can't load driver library: ") + msa->errorString()).toUtf8());

    if ((driverModule = new struct msa_module) == 0)
        qFatal("can't init msa_module");

    // fill important properties
    driverModule->proxy = 0;
    driverModule->port = 0;

    if (accountName.isNull() || accountName.isEmpty())
        driverModule->id = NULL;
    else
        driverModule->id = g_strdup((gchar*) accountName.toStdString().data());

    gint (*driver_init)(msa_module *);

    *(void **) (&driver_init) = msa->resolve(MODULE_INIT);
    (*driver_init)(driverModule);

    if (driverModule == 0)
        qFatal((QString("can't resolve funtion of  library: ")+msa->errorString()).toUtf8());

    this->accountId = QString(driverModule->id);

    // Decoder of "russian" messages from libvkontakte
    decoder =  QTextCodec::codecForName("utf8");
}

VkTransport::~VkTransport()
{
    if (msa != 0) {
        if (msa->unload())
            qWarning((QString("can't unload  library: ")+msa->errorString()).toUtf8());
        delete msa;
        delete driverModule;
        qDebug() << "vktransport deleted";
    }

}

void VkTransport::vkGetPhotos(QString userId, QString albumId)
{
    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='photos' function='getPhotos'>")
              + QString("<Params id='%1'>").arg(userId)
              + QString("<string name='albumId'>%1</string>").arg(albumId)
              + QString("</Params></Request></Content></TransitData>");
    xmlDocPtr respXml = vkSendRequest(reqStr.toUtf8(), VkTransport::getPhotosAction);
    if(respXml != NULL)
        xmlFreeDoc(respXml);
}


void VkTransport::vkGetAlbums(QString userId)
{
    qDebug() << "Prepare request getAlbums for " << userId;
    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='photos' function='getAlbums'>")
              + QString("<Params id='") + userId
              + QString("'/></Request></Content></TransitData>");
    xmlDocPtr respXml = vkSendRequest(reqStr.toUtf8(), VkTransport::getAlbumsAction);
    if(respXml != NULL)
        xmlFreeDoc(respXml);
}

void VkTransport::vkGetFriends()
{
    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='friends' function='updateListFriends'>")
              + QString("<Params /></Request></Content></TransitData>");
    xmlDocPtr respXml = vkSendRequest(reqStr.toUtf8(), VkTransport::updateListFriendsAction);
    if(respXml != NULL)
        xmlFreeDoc(respXml);
}

bool VkTransport::vkDownloadPhoto(QString url, QString file_name)
{
    qDebug() << "Prepare request getPhoto";
    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='photos' function='getPhoto'>")
              + QString("<Params>")
              + QString("<string name='url'>%1</string>").arg(url)
              + QString("<string name='path'>%1</string>").arg(file_name)
              + QString("</Params></Request></Content></TransitData>");

    xmlDocPtr respXml = vkSendRequest(reqStr.toUtf8(), VkTransport::getPhotoAction);
    if(respXml != NULL){
        xmlXPathContextPtr respContext = xmlXPathNewContext(respXml);

        VkSystemMessage msg = vkCheckMessage(respContext, VkTransport::getPhotoAction);
        xmlXPathFreeContext(respContext);
        xmlFreeDoc(respXml);
        if(!msg.code.isEmpty())
            return !msg.isError;
        else 
            return false;
    } else
            return false;
}

void VkTransport::vkGetProfile()
{
    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='profile' function='updateProfile'>")
              + QString("<Params /></Request></Content></TransitData>");
    xmlDocPtr respXml = vkSendRequest(reqStr.toUtf8(), VkTransport::updateProfileAction);
    if(respXml != NULL)
        xmlFreeDoc(respXml);
}

void VkTransport::vkGetSend(int reqFrom = 0,
                             int reqTo = VK_MESSAGES_AT_ONCE)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__
             << ": Called vkGetSend("<< reqFrom << ", " << reqTo <<");";
    // Add XML data to "request" string
    QByteArray reqStr;
    QTextStream ts(&reqStr);
    ts << "<?xml version='1.0'?>"
       << "<TransitData id='1' type='data'>"
       << "<SourceID>" << CLIENT_NAME << "</SourceID>"
       << "<TargetID>" << this->driverModule->id << "</TargetID>"
       << "<Content><Request class='messages' function = 'updateOutboxMessages'>"
       << "<Params>"
       << "<number name='from'>" << reqFrom << "</number>"
       << "<number name='to'>" << reqTo << "</number>"
       << "</Params></Request></Content></TransitData>";
    ts.flush();
    xmlDocPtr respXml = vkSendRequest(reqStr, VkTransport::updateOutboxMessagesAction);
    if(respXml != NULL)
        xmlFreeDoc(respXml);
}

void VkTransport::vkGetInbox(int reqFrom = 0,
                             int reqTo = VK_MESSAGES_AT_ONCE)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__
             << ": Called vkGetInbox("<< reqFrom << ", " << reqTo <<");";
    // Add XML data to "request" string
    QByteArray reqStr;
    QTextStream ts(&reqStr);
    ts << "<?xml version='1.0'?>"
       << "<TransitData id='1' type='data'>"
       << "<SourceID>" << CLIENT_NAME << "</SourceID>"
       << "<TargetID>" << this->driverModule->id << "</TargetID>"
       << "<Content><Request class='messages' function = 'updateInboxMessages'>"
       << "<Params>"
       << "<number name='from'>" << reqFrom << "</number>"
       << "<number name='to'>" << reqTo << "</number>"
       << "</Params></Request></Content></TransitData>";
    ts.flush();
    xmlDocPtr respXml = vkSendRequest(reqStr, VkTransport::updateInboxMessagesAction);
    if(respXml != NULL)
        xmlFreeDoc(respXml);
    //qDebug() << "vkGetInbox(): OK" << endl;
}

void VkTransport::vkGetSend() {
    // response only one message for checking of new outgoing messages.
    vkGetSend(0, VK_MIN_MESSAGES_AT_ONCE);
}

void VkTransport::vkGetInbox() {
    // response only one message for checking of new incoming messages.
    vkGetInbox(0, VK_MIN_MESSAGES_AT_ONCE);
}

/** vkInit()
  * @param strProxyHost
  * @param strProxyPort
  */
void VkTransport::vkInit(QString proxyHost, uint proxyPort)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__<<": Call vkInit()";

    if (proxyHost.isNull()) {
        driverModule->proxy = (gchar*) NULL;
    } else {
        char *host = new char[proxyHost.toLocal8Bit().size()+1];
        strcpy(host,proxyHost.toLocal8Bit().data());
        driverModule->proxy = (gchar*) host;
    }
    driverModule->port = (gint) proxyPort;

    // Send data to driver module
    qDebug() << "vkInit(): OK" << endl;
}

/** vkGetSettings()
  */
void VkTransport::vkGetSettings()
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__<<": Call vkGetSettings()";

    QString reqStr = QString("<?xml version='1.0'?>")
                     + QString("<TransitData id='1' type='data'>")
                     + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
                     + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
                     + QString("<Content>")
                     + QString("<Request class='settings' function='getSettings'>")
                     + QString("<Params/>")
                     + QString("</Request></Content></TransitData>");
    //Q_UNUSED(reqStr);
    xmlDocPtr respXml = vkSendRequest(reqStr.toUtf8(), VkTransport::getSettingsAction);
    if(respXml != NULL)
        xmlFreeDoc(respXml);
}

/** vkSetSettings()
  * @param strLogin
  * @param strPassword
  */
void VkTransport::vkSetSettings(QString settings)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__<<": Call vkSetSettings()";

    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='settings' function='setSettings'>")
              + settings
              + QString("</Request></Content></TransitData>");
    xmlDocPtr respXml = vkSendRequest(reqStr.toUtf8(), VkTransport::setSettingsAction);
    if(respXml != NULL)
        xmlFreeDoc(respXml);
}

void VkTransport::vkSendMessage(QByteArray strRecipientId, QByteArray strText="")
{
    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='messages' function='sendMessage'>")
              + QString("<Params id='vk%1'>").arg(QString(strRecipientId))
              + QString("<string name='text'>%1</string>").arg(QString(strText))
              + QString("</Params></Request></Content></TransitData>");
    xmlDocPtr respXml = vkSendRequest(reqStr.toUtf8(), VkTransport::sendMessageAction);
    if(respXml != NULL)
        xmlFreeDoc(respXml);
}

void VkTransport::vkReadMessage(QString messageId)
{
    qDebug() << "init VkTransport::vkReadMessage";
    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='messages' function='readMessage'>")
              + QString("<Params>")
              + QString("<string name='messageId'>vk%1</string>").arg(messageId)
              + QString("</Params></Request></Content></TransitData>");
    xmlDocPtr respXml = vkSendRequest(reqStr.toUtf8(), VkTransport::readMessageAction);
    if(respXml != NULL)
        xmlFreeDoc(respXml);
}

void VkTransport::vkDeleteMessage(QString messageId)
{
    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='messages' function='deleteMessage'>")
              + QString("<Params>")
              + QString("<string name='messageId'>vk%1</string>").arg(messageId)
              + QString("</Params></Request></Content></TransitData>");
    xmlDocPtr respXml = vkSendRequest(reqStr.toUtf8(), VkTransport::deleteMessageAction);
    if(respXml != NULL)
        xmlFreeDoc(respXml);
}

char* VkTransport::parseString(QByteArray charRequest,
                                       xmlXPathContextPtr &content,
                                       Action acc,
                                       bool important=false) {
    xmlChar *xrequest = xmlCharStrdup(charRequest.data());
    char* xresponse = parseString(xrequest, content, acc, important);
    free(xrequest);
    return xresponse;
}

char* VkTransport::parseString(xmlChar *xpathRequest,
                                       xmlXPathContextPtr &content,
                                       Action acc,
                                       bool important = false) {
    xmlXPathObjectPtr object = 0;
    if (!(object = xmlXPathEvalExpression(xpathRequest, content))) {
        qDebug() << __FILE__<< ":" << "error in xmlXPathEvalExpression";
        return 0;
    }
    if ((object->type != XPATH_NODESET)
        || ( xmlXPathNodeSetIsEmpty(object->nodesetval))
        || (xmlXPathNodeSetGetLength(object->nodesetval) != 1)
        || (!object->nodesetval->nodeTab[0])
        || (!object->nodesetval->nodeTab[0]->children)
        || (!object->nodesetval->nodeTab[0]->children->content)
        ){
        xmlXPathFreeObject(object);
        if (important) {
            qDebug() << "Requested: " << (char *) xpathRequest;
            qDebug() << __FILE__<< ":" << __LINE__
                    << " incorrect response or XPath request type";
            emit errorOccurred(this->accountId, -1,QString("incorrext response or XPath request"), acc);
        }
        return 0;
    }
    char *data = new char[strlen((char*) object->nodesetval->nodeTab[0]->
                                 children->content) + 1];
    if (!data) {
        qWarning() << "Not Enough Memory";
        return 0;
    }
    strcpy(data, (char*) object->nodesetval->nodeTab[0]->children->content);
    xmlXPathFreeObject(object);
    return data;
}

void VkTransport::vkClose()
{

    needShutdown = true;
    while (activeRequests != 0)
    {
        sleep(1);
    }

    if (activeRequests == 0)
        if (driverModule != NULL) {
            qDebug() << "Shutdown..."<< endl;
                driverModule->shutdown(driverModule);
        }

}

void VkTransport::vkGetNewInbox()
{
}

xmlDocPtr VkTransport::vkSendRequest(QByteArray xmlRequest, Action acc)
{

    if(needShutdown){
        return NULL;
    } else 
        activeRequests++;

    xmlChar* request = xmlCharStrdup(xmlRequest.constData());
    xmlDocPtr reqXml = xmlParseDoc(request);
    xmlDocPtr respXml = NULL;

    xmlFree(request);
    // Process request
    if (driverModule != NULL) {
        qDebug() << "Sending: " << xmlRequest;
        try {
            driverModule->send(reqXml, &respXml, driverModule);
        } catch (...) {
            qDebug() << "Caught an exception.";
            emit errorOccurred(this->accountId, -1, tr("Caught an exception."), acc);
        }
    } else {
        qDebug() << "driverModule == NULL" << endl;
    }
    qDebug() << "Receive: ";
    xmlDocDump(stderr, respXml);

    activeRequests--;

    //xmlFreeDoc(reqXml);
    vkGenerateSignal(respXml, acc);
    return respXml;
}
void VkTransport::vkGenerateSignal(xmlDocPtr respXml, Action acc)
{
    // Parsing from DOM to XPath Content (Need free memory)
    xmlXPathContextPtr respContext = xmlXPathNewContext(respXml);
    if (respContext == NULL) {
        qDebug() << __FILE__ << ":" << "error in getting xmlXPathNewContext";
        return;
    }
    char xpathRequestBuffer[255]; // Buffer for generating xpath requests
    char *xpathRequestBufferPointer = xpathRequestBuffer;
    char *xpathRequestPattern = stpcpy (xpathRequestBuffer, VK_STRUCT_FRIEND);
    char *parsedString;

    // Getting type of response
    char *functionName  = parseString(VK_FUNCTION, respContext, acc, true);
    if (functionName == NULL) {
        qDebug() << "functionName not found" << endl;
        return;
    }

    qDebug() << "Processing " << functionName << endl;

    xmlNodePtr node = NULL;
    xmlNodePtr tmp_node = NULL;

    if (strcmp(functionName, "getSettings") == 0) {
        xmlBufferPtr buff = xmlBufferCreate();
        QString settings;

        node = xpath_get_node(VK_PARAMS, respXml);
        if (node != NULL) {
            if(xmlNodeDump(buff, node->doc, node, 0, 0) != -1) {
                settings = decoder->toUnicode((char*)buff->content);
                xmlBufferFree(buff);
            } 
        }
        emit settingsReceived(this->accountId, settings);
    } else
    if (strcmp(functionName, "errorMessage") == 0) {
        QString code;
        QString text;

        // code
        if ((parsedString = parseString(VK_RESP_CODE, respContext, acc)) != 0) {
            code = decoder->toUnicode(parsedString);
            delete [] parsedString;
        }

        // text
        if ((parsedString = parseString(VK_RESP_TEXT, respContext, acc)) != 0) {
            text = decoder->toUnicode(parsedString);
            delete [] parsedString;
        }

        emit this->errorOccurred(this->accountId, code.toInt(), text, acc);
    } else
    if (strcmp(functionName, "getProfile") == 0) {
        Friend profile;
        profile.accountId = this->accountId;

        // Id
        if ((parsedString = parseString(VK_PARAMS_ID, respContext, acc)) != 0) {
            profile.id = decoder->toUnicode(parsedString);
            delete [] parsedString;
        } 

        // Name
        if ((parsedString = parseString(VK_FIRST_NAME, respContext, acc)) != 0) {
            profile.name = decoder->toUnicode(parsedString);
            delete [] parsedString;
        } 

        if ((parsedString = parseString(VK_LAST_NAME, respContext, acc)) != 0) {
            if (!profile.name.isEmpty())
                profile.name.append(QChar(' '));
            profile.name.append(decoder->toUnicode(parsedString));
            delete [] parsedString;
        } 

        // Icon url
        if ((parsedString = parseString(VK_IMG, respContext, acc)) != 0) {
            profile.icon_url = decoder->toUnicode(parsedString);
            delete [] parsedString;
        } 

        // Icon
        QDir dir;
        if (!dir.exists(Utils::getFriendsIconDir(this->driverModule->name)))
            dir.mkpath(Utils::getFriendsIconDir(this->driverModule->name));
        QString icon;
        if (!profile.icon_url.isEmpty()){
            icon = vkGenerateFileName(Utils::getFriendsIconDir(this->driverModule->name), profile.icon_url);
            QFile file(icon);
            if(file.exists())
                profile.icon = icon;
        }
        emit profileReceived(this->accountId, profile);

        if (!icon.isEmpty() && !profile.icon_url.isEmpty()){
            QFile file(icon);
            if(!file.exists()){
                if(vkDownloadPhoto(profile.icon_url, icon)) {
                    profile.icon = icon;
                    emit profileReceived(this->accountId, profile);
                }
            }
        }
    } else
    // Friends 
    if (strcmp(functionName,"updateListFriends") == 0) {
        FriendList friends_list;// = new QList();

        int index = 0;
        node = xpath_get_node(VK_ARRAY_FRIENDS, respXml);
        for (tmp_node = node->children; tmp_node != NULL; tmp_node = tmp_node->next){
            index++;
            Friend fr = Friend();


            // Friend id
            char indexString[20]; // Only for message index;
            sprintf(indexString,"%d]", index); // Bad hack with ]
            xpathRequestBufferPointer = stpcpy (xpathRequestPattern, indexString);
            char *xpathRequestBufferCurrentPointer = xpathRequestBufferPointer;
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferPointer, VK_ID_R);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {
                qDebug() << "Friend id " << index << " have uid = " << parsedString;
                fr.id = decoder->toUnicode(parsedString);
                delete [] parsedString;
            } else {
                qDebug() << "Friend uid not found." << endl;
                continue;
            }

            // Name
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_FIRST_NAME);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {
                fr.name = decoder->toUnicode(parsedString);

                delete [] parsedString;
            }

            // Name
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_LAST_NAME);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {
                if (!fr.name.isEmpty())
                    fr.name.append(QChar(' '));
                fr.name.append(decoder->toUnicode(parsedString));
                delete [] parsedString;
            }



            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_STATUS);
            if ((parsedString = parseString(xpathRequestBuffer, respContext, acc)) != 0) {
                if(strcmp(parsedString, "1") == 0){
                    fr.online = true;
                } else {
                    fr.online = false;
                }
                delete [] parsedString;
            }

            // Icon url
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_FRIEND_IMG);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {

                        fr.icon_url = decoder->toUnicode(parsedString);
                delete [] parsedString;
            }

            // Icon
            QDir dir;
            if (!dir.exists(Utils::getFriendsIconDir(this->driverModule->name)))
                dir.mkpath(Utils::getFriendsIconDir(this->driverModule->name));
            QString icon = NULL;
            if (!fr.icon_url.isEmpty()){
                icon = vkGenerateFileName(Utils::getFriendsIconDir(this->driverModule->name), fr.icon_url);
                QFile file(icon);
                if(file.exists())
                    fr.icon = icon;
            }

            fr.accountId = this->accountId;
            friends_list.append(fr);
        }
        qDebug() << "Total friends:" + friends_list.length();
        emit friendsReceived(this->accountId, friends_list, false);
        updateFriendsIcons(friends_list);
    } else 
    if (strcmp(functionName,"getAlbums") == 0) {
        xpathRequestPattern = stpcpy (xpathRequestBuffer, VK_STRUCT_ALBUM);
        AlbumList albums_list;
        QString album_owner = NULL;

        int index = 0;
        node = xpath_get_node(VK_ARRAY_ALBUMS, respXml);
        for (tmp_node = node->children; tmp_node != NULL; tmp_node = tmp_node->next){
            index++;
            Album al = Album();


            // Album id
            char indexString[20]; // Only for message index;
            sprintf(indexString,"%d]", index); // Bad hack with ]
            xpathRequestBufferPointer = stpcpy (xpathRequestPattern, indexString);
            char *xpathRequestBufferCurrentPointer = xpathRequestBufferPointer;
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferPointer, VK_ID_R);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {
                qDebug() << "Album id " << index << " have uid = " << parsedString;
                al.albumId = decoder->toUnicode(parsedString);
                delete [] parsedString;
            } else {
                qDebug() << "Album id not found." << endl;
                continue;
            }

            // Title
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_TITLE);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {
                al.title = decoder->toUnicode(parsedString);

                delete [] parsedString;
            }

            // Description
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_DESCRIPTION);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {
                al.description = decoder->toUnicode(parsedString);

                delete [] parsedString;
            }

            // OwnerId
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_OWNER_ID);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {
                al.ownerId = decoder->toUnicode(parsedString);
                if (album_owner == NULL)
                    album_owner = al.ownerId;
                delete [] parsedString;
            }

            // Time create
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_TIME_CREATE);
            if ((parsedString = parseString(xpathRequestBuffer, respContext, acc)) != 0) {
                QDateTime mdf;
                mdf.setTime_t(QString(parsedString).toUInt());
                al.time_create = mdf.toString("dd.MM.yyyy hh:mm:ss");
                delete [] parsedString;
            }

            // Time update
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_TIME_UPDATE);
            if ((parsedString = parseString(xpathRequestBuffer, respContext, acc)) != 0) {
                QDateTime mdf;
                mdf.setTime_t(QString(parsedString).toUInt());
                al.time_update = mdf.toString("dd.MM.yyyy hh:mm:ss");
                delete [] parsedString;
            }

            // Size
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_SIZE);
            if ((parsedString = parseString(xpathRequestBuffer, respContext, acc)) != 0) {
                al.size = QString(parsedString).toUInt();
                delete [] parsedString;
            } else 
                al.size = 0;

            // Icon url
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_ALBUM_IMG);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {

                        al.icon_url = decoder->toUnicode(parsedString);
                delete [] parsedString;
            }

            // Icon
            QDir dir;
            if (!dir.exists(Utils::getAlbumsIconDir(this->driverModule->name)))
                dir.mkpath(Utils::getAlbumsIconDir(this->driverModule->name));
            QString icon = NULL;
            if (!al.icon_url.isEmpty()){
                icon = vkGenerateFileName(Utils::getAlbumsIconDir(this->driverModule->name), al.icon_url);
                QFile file(icon);
                if(file.exists())
                    al.icon = icon;
            }

            al.accountId = this->accountId;
            albums_list.append(al);
        }

        emit albumsReceived(this->accountId, album_owner, albums_list, false);
        updateAlbumsIcons(albums_list, album_owner);
    } else
    if (strcmp(functionName,"getPhotos") == 0) {
        xpathRequestPattern = stpcpy (xpathRequestBuffer, VK_STRUCT_PHOTO);
        PhotoList photos_list;
        QString photo_owner;
        QString album_owner;

        int index = 0;
        node = xpath_get_node(VK_ARRAY_PHOTOS, respXml);
        if (node != NULL)
        for (tmp_node = node->children; tmp_node != NULL; tmp_node = tmp_node->next){
            index++;
            Photo ph = Photo();


            // Photo id
            char indexString[20]; // Only for message index;
            sprintf(indexString,"%d]", index); // Bad hack with ]
            xpathRequestBufferPointer = stpcpy (xpathRequestPattern, indexString);
            char *xpathRequestBufferCurrentPointer = xpathRequestBufferPointer;
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferPointer, VK_ID_R);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {
                qDebug() << "Photo id " << index << " have id = " << parsedString;
                ph.photoId = decoder->toUnicode(parsedString);
                delete [] parsedString;
            } else {
                qDebug() << "Photo id not found." << endl;
                continue;
            }

            // AlbumId
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_ALBUM_ID);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {
                ph.albumId = decoder->toUnicode(parsedString);
                if (album_owner.isEmpty())
                    album_owner = ph.albumId;
                delete [] parsedString;
            }

            // OwnerId
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_OWNER_ID);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {
                ph.ownerId = decoder->toUnicode(parsedString);
                if (photo_owner.isEmpty())
                        photo_owner = ph.ownerId;
                delete [] parsedString;
            }

            // Thumb url
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_THUMB_URL);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {
                ph.icon_url = decoder->toUnicode(parsedString);

                delete [] parsedString;
            }

            // Photo url
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_PHOTO_URL);
            if ((parsedString = parseString(xpathRequestBuffer,respContext, acc)) != 0) {
                ph.photo_url = decoder->toUnicode(parsedString);
                delete [] parsedString;
            }

            // Time create
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_TIME_CREATE);
            if ((parsedString = parseString(xpathRequestBuffer, respContext, acc)) != 0) {
                QDateTime mdf;
                mdf.setTime_t(QString(parsedString).toUInt());
                ph.time_create = mdf.toString("dd.MM.yyyy hh:mm:ss");
                delete [] parsedString;
            }

            ph.accountId = this->accountId;
            photos_list.append(ph);
        }

        emit photosReceived(this->accountId, photo_owner, album_owner, photos_list, false);
        updatePhotoIcons(photos_list, photo_owner, album_owner);
    }

/*
    bool error=false; // true if was error message
    bool sent=false;  // true if incoming sent messages (not inbox)

    // Info/Error message (Errors it problems with network or account)
    // Avoid duplication of code. Combine errors and info in one.
    if ((error = (strcmp(functionName,"errorMessage") == 0))
        || (strcmp(functionName,"infoMessage") == 0)) {

        // error == true if was error message
        if (error) {
            response->action = VkTransport::errorMessageAction;
            qDebug("Error message:");
        } else {
            response->action = VkTransport::infoMessageAction;
            qDebug("Info message:");
        }

        if ((parsedString = parseString(VK_RESP_CODE,respContext)) != 0) {
            qDebug() << "Code: " << parsedString;
            response->code = QByteArray(parsedString);
            delete [] parsedString;
        } else {
            response->code = "-1";
            qWarning() << "Info/Error code not found.";
        }

        if ((parsedString = parseString(VK_RESP_TEXT,respContext)) != 0) {
            qDebug() << "Text: " << decoder->toUnicode(parsedString);
            response->text = new QString(parsedString);
            delete [] parsedString;
        } else {
            response->text = new QString("Some message");
            qWarning() << "Info/Error text not found.";
        }

        error = false;
        emit responseReceived(response);
        response = 0;
    } else
    // Profile message
    if (strcmp(functionName,"updateProfile") == 0) {
        userName.clear();
        if ((parsedString = parseString(VK_FIRST_NAME,respContext)) != 0) {
            userName = decoder->toUnicode(parsedString);
            delete [] parsedString;
        } else {
            qWarning() << "FirstName not found.";
        }

        if ((parsedString = parseString(VK_LAST_NAME,respContext)) != 0) {
            if (!userName.isEmpty())
                userName.append(QChar(' '));
            userName.append(decoder->toUnicode(parsedString));
            delete [] parsedString;
        } else {
            qWarning() << "LastName not found.";
        }

        response->text = new QString(userName);
        response->action = VkTransport::updateProfileAction;
        emit responseReceived(response);
    } else
    // Inbox/outbox message
    if ((sent = (strcmp(functionName,"updateOutboxMessages") == 0))
        || (strcmp(functionName,"updateInboxMessages") == 0)) {

        response->action = sent
                           ? VkTransport::updateOutboxMessagesAction
                           : VkTransport::updateInboxMessagesAction;

        // Parse From String
        if ((parsedString = parseString(VK_FROM,respContext)) == 0) {
            qDebug() << __FILE__<< ":" << "\"From\" not found" << endl;
            return;
        }
        int inboxFrom = QString(parsedString).toInt();
        delete [] parsedString;

        // Parse To String
        if ((parsedString = parseString(VK_TO,respContext)) == 0) {
            qDebug() << __FILE__<< ":" << "\"To\" not found" << endl;
            return;
        }
        int inboxTo = QString(parsedString).toInt();
        delete [] parsedString;

        // Parse To String
        if ((parsedString = parseString(VK_QUANTITY,respContext)) == 0) {
            qDebug() << __FILE__<< ":" << "\"Quantity\" not found" << endl;
            return;
        }
        int inboxQuantity = QString(parsedString).toInt();
        delete [] parsedString;

        qDebug() << "From =" << inboxFrom
                 << "To ="   << inboxTo
                 << "Quantity ="   << inboxQuantity;

        // returned 0 messages
        if (inboxFrom >= inboxTo) {
            qDebug() << "Retrivied none messages." << endl;
            response->text = new QString("inboxFrom >= inboxTo");
        }
        response->midFrom = QByteArray(QByteArray::number(inboxTo));
        response->midTo = QByteArray(QByteArray::number(inboxQuantity));

        if (inboxTo == inboxQuantity) {
            qDebug() << "Retrivied last message." << endl;
            response->text = new QString("inboxTo == inboxQuantity");
        }

        emit responseReceived(response);
    } else {
        qDebug() << "Impossible response: " << functionName << ". Passed.";
        delete response;
    }
*/
    delete [] functionName;
    // Clear memory
    xmlXPathFreeContext(respContext);


}
void VkTransport::updateAlbumsIcons(AlbumList list, QString album_owner){
    int count = 0;

    QDir dir;
    if (!dir.exists(Utils::getAlbumsIconDir(this->driverModule->name)))
        dir.mkpath(Utils::getAlbumsIconDir(this->driverModule->name));
    qDebug()<< "Length AlbumList = "<<list.length()<<endl;
    for(int i = 0; i < list.length(); i++){
        if (list.at(i).icon_url.isEmpty())
            continue;

        QString icon = NULL;
        icon = vkGenerateFileName(Utils::getAlbumsIconDir(this->driverModule->name), list.at(i).icon_url);
        QFile file(icon);
        if(!file.exists()){
            if(vkDownloadPhoto(list.at(i).icon_url, icon))
                list[i].icon = icon;
            count++;
            if(count > 0 && count%10 == 0){
                qDebug() << "emit AlbumsReceived signal" << endl;
                emit albumsReceived(this->accountId, album_owner, list, false);
            }

        } else 
            list[i].icon = icon;
    }
    emit albumsReceived(this->accountId, album_owner, list, true);
}

void VkTransport::updatePhotoIcons(PhotoList list, QString friendId, QString albumId) {
    int count = 0;

    QDir dir;
    if (!dir.exists(Utils::getAlbumsIconDir(this->driverModule->name)))
        dir.mkpath(Utils::getAlbumsIconDir(this->driverModule->name));
    qDebug()<< "Length AlbumList = "<<list.length()<<endl;
    for (int i = 0; i < list.length(); i++) {
        if (list.at(i).icon_url.isEmpty())
            continue;

        QString icon = NULL;
        icon = vkGenerateFileName(Utils::getAlbumsIconDir(this->driverModule->name), list.at(i).icon_url);
        QFile file(icon);
        if (!file.exists()) {
            count++;
            if (vkDownloadPhoto(list.at(i).icon_url, icon))
                list[i].icon = icon;
            if (count > 0 && count%10 == 0) {
                qDebug() << "emit PhotosReceived signal";
                emit this->photosReceived(this->accountId, friendId, albumId, list, false);
            }
        } else
            list[i].icon = icon;

    }
    emit this->photosReceived(this->accountId, friendId, albumId, list, true);
}

void VkTransport::updateFriendsIcons(FriendList list){
    int count = 0;

    QDir dir;
    if (!dir.exists(Utils::getFriendsIconDir(this->driverModule->name)))
        dir.mkpath(Utils::getFriendsIconDir(this->driverModule->name));
    qDebug()<< "Length friendList = "<<list.length()<<endl;
    for(int i = 0; i < list.length(); i++){
        if (list.at(i).icon_url.isEmpty())
            continue;

        QString icon = NULL;
        icon = vkGenerateFileName(Utils::getFriendsIconDir(this->driverModule->name), list[i].icon_url);
        QFile file(icon);
        if(!file.exists()){
            count++;
            if(vkDownloadPhoto(list[i].icon_url, icon))
                list[i].icon = icon;
            if(count > 0 && count%10 == 0){
                qDebug() << "emit friendsReceived signal" << endl;
                emit friendsReceived(this->accountId, list, false);
            }
        } else
            list[i].icon = icon;
    }

    emit friendsReceived(this->accountId, list, true);
}


VkSystemMessage VkTransport::vkCheckMessage(xmlXPathContextPtr respContext, Action acc)
{
    VkSystemMessage msg;

    bool error;
    char *functionName  = parseString(VK_FUNCTION, respContext, acc);
    char* parsedString = NULL;

    if ((error = (strcmp(functionName,"errorMessage") == 0))
        || (strcmp(functionName,"infoMessage") == 0)) {

        // error == true if was error message
        if (error) {
            msg.isError = true;
            qDebug("Error message:");
        } else {
            msg.isError = false;
            qDebug("Info message:");
        }

        if ((parsedString = parseString(VK_RESP_CODE,respContext, acc)) != 0) {
            qDebug() << "Code: " << parsedString;
            msg.code = QString(parsedString);
            delete [] parsedString;
        } else {
            msg.code = QString();
            qWarning() << "Info/Error code not found.";
        }

        if ((parsedString = parseString(VK_RESP_TEXT,respContext, acc)) != 0) {
            qDebug() << "Text: " << decoder->toUnicode(parsedString);
            msg.text = QString(parsedString);
            delete [] parsedString;
        } else {
            msg.text = QString();
            qWarning() << "Info/Error text not found.";
        }
        delete [] functionName;
        return msg;
    } else { 
        delete [] functionName;
        return msg;
    }
}
QString VkTransport::vkGenerateFileName(QString path, QString url)
{
    QString icon = NULL;
    if (url != NULL) {
        QFileInfo fi(url);
        QString ext = fi.suffix();
        icon = path + QString(url.toUtf8().toBase64()) + (ext.isEmpty() ? QString() : QString(".") + ext);
    }
    return icon;
}

xmlNodePtr VkTransport::xpath_get_node(char* path, xmlDocPtr doc)
{
    xmlXPathObject *Obj = NULL;
    xmlNodePtr node = NULL;
    Obj = xpath(path, doc);
    if(Obj == NULL)
        return NULL;
    if(!xmlXPathNodeSetIsEmpty(Obj->nodesetval))
        node = xmlXPathNodeSetItem(Obj->nodesetval, 0);
    xmlXPathFreeObject(Obj);
    return node;
}

/**
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
xmlXPathObject* VkTransport::xpath(gchar* req, xmlDocPtr doc)
{
	xmlXPathObject *Obj;
	xmlXPathContextPtr Ctx;
	
	g_debug("vk_xpath: %s", req);

	Ctx = xmlXPathNewContext(doc);
	
	if (Ctx == NULL) {
		g_debug("vk_xpath: error wrong request format!");
		return NULL;	
	}

	Obj = xmlXPathEvalExpression(BAD_CAST req, Ctx);

	g_debug("vk_xpath: end");

        xmlXPathFreeContext(Ctx);

	return Obj;

}

VkResponse::VkResponse()
    : text(QString())
{
}

VkResponse::~VkResponse()
{
}

VkSystemMessage::VkSystemMessage()
    :text(QString()),
    code(QString())
{
}
VkSystemMessage::~VkSystemMessage()
{
}

DriverInfo *VkTransport::getDriverInfo()
{
    DriverInfo *ret = new DriverInfo(this->parent());

    // name
    ret->name = QString::fromUtf8(this->driverModule->name);
    // library
    ret->library = this->libraryName;
    // icon
    QImage image;
    if (!image.loadFromData(QByteArray::fromBase64(this->driverModule->pic)))
        qDebug() << "Error loading image for " << this->accountId << ": " << this->driverModule->pic;
    QPixmap pix = QPixmap::fromImage(image);
    ret->icon = QIcon::QIcon(pix);

    return ret;
}

DriverInfo *VkTransport::getDriverInfo(QString libraryPath)
{
    QLibrary *lib;
    // Add dynamic library (libvkontakte.so)
    if ((lib = new QLibrary(libraryPath)) == 0) {
        qWarning("Problems with initiation QLibrary object");
        return NULL;
    }

    lib->setLoadHints(QLibrary::ResolveAllSymbolsHint); // analog RTLD_LAZY
    // Check driver existance
    if (!lib->isLoaded())
        if (!lib->load()) {
            qWarning() << QString("can't load driver library: ") + lib->errorString();
            delete lib;
            return NULL;
        }

    struct msa_module * dm;

    if ((dm = new struct msa_module) == 0) {
        qWarning() << "can't init msa_module";
        lib->unload();
        delete lib;
        return NULL;
    }

    void (*driver_init)(msa_module *);

    *(void **) (&driver_init) = lib->resolve(MODULE_GET_INFO);
    if (driver_init == NULL) {
        qWarning() << QString("Cann't find function ") + QString(MODULE_GET_INFO) +
                QString(". Probably old library type");
        delete dm;
        lib->unload();
        delete lib;
        return NULL;
    }

    (*driver_init)(dm);

    if (dm == 0) {
        qWarning() << QString("can't resolve funtion of  library: ") + lib->errorString();
        lib->unload();
        delete lib;
        return NULL;
    }
    DriverInfo *ret = new DriverInfo();
    ret->name = QString::fromUtf8(dm->name);
    // library
    ret->library = libraryPath;
    // icon
    QImage image;
    if (!image.loadFromData(QByteArray::fromBase64(dm->pic)))
        qDebug() << "Error loading image: " << dm->pic;
    QPixmap pix = QPixmap::fromImage(image);
    ret->icon = QIcon::QIcon(pix);

    delete dm;
    lib->unload();
    delete lib;

    return ret;
}

void VkTransport::vkDownloadPhotoList(PhotoList list, Photo curPhoto, int nearest) {

    if (nearest < 0)
        return;

    if(list.isEmpty())
        return;

    int count = 0;

    QDir dir;
    if (!dir.exists(Utils::getPhotoDir(this->driverModule->name)))
        dir.mkpath(Utils::getPhotoDir(this->driverModule->name));

    int len = list.length();
    int startItem = -1;
    for (int i = 0; i < len; i++) {
        if (list.at(i).photoId == curPhoto.photoId) {
            startItem = i;
            break;
        }
    }

    for(int i = 0; i <= nearest; i++)
    {
        for (int j = -1; j < 2; j += 2)
        {
            if (i == 0 && j > 0)
                continue;

            int cp = startItem + i * j;
            qDebug() << "download image: start=" << startItem << " i=" << i << " cp=" << cp;
            while (cp < 0)
                cp += len;

///FIXME
            while(cp >= list.size())
                cp -= len;

            qDebug()<< "photo_url = "<< list.at(cp).photo_url << endl;

            QString photo_path = vkGenerateFileName(Utils::getPhotoDir(this->driverModule->name), list.at(cp).photo_url);
            QFile file(photo_path);
            if(!file.exists()){
                vkDownloadPhoto(list.at(cp).photo_url, file.fileName());
                count++;
                list[cp].photo = photo_path;
                qDebug() << "emit PhotosDownload signal" << endl;
                emit photosReceived(curPhoto.accountId, curPhoto.ownerId, curPhoto.albumId, list, false);
            }else{
                list[cp].photo = photo_path;
            }
        }
    }
    qDebug() << "Finish download image for album " << curPhoto.albumId;
    emit photosReceived(curPhoto.accountId, curPhoto.ownerId, curPhoto.albumId, list, true);
}
