#ifndef ACCOUNT_H
#define ACCOUNT_H

#include <QObject>
#include <QDomDocument>
#include <QFile>
#include <QDebug>
#include <QDir>
#include <QMetaType>
#include <QtConcurrentRun>

#include "vktransport.h"
#include "friend.h"
#include "utils.h"

// the name of file which profile was stored
#define FILE_ACCOUNT_DATA "/profile.xml"

// XML root node for profile
#define NODE_ACCOUNT_ROOT "profile"

// XML node for storing library name
#define NODE_ACCOUNT_LIBRARY "library"

// XML node for storing driver settings
#define NODE_ACCOUNT_SETTINGS "settings"

#define FILE_FRIENDS_DATA "/friends.xml"

#define NODE_FRIENDS_ROOT "friendsList"

/*
 * class for account information storage.
 */
class Account: public QObject
{
    Q_OBJECT

private:
    FriendList friends; // list of loaded friends.
    AlbumList albums; // list of my albums.
    QString settings; // account settings. Uses in driver.
    QString oldSettings; // old account settings. Used if user didn't work with application

    // user account data
    Friend profile;

public:
    Account(QString libraryName);
    Account(QString libraryName, QString accountId);
    ~Account();

    VkTransport *transport; // interface of used driver.

    void saveAccount(); // store account settings.
    static Account* loadAccount(QString accountId); // load account from storage.

    // get friend list or load it from cache file.
    FriendList getFriendList();

    // set friend list and store it to cache file.
    void setFriendList(FriendList);

    /**
     * @brief get list of my albums or load it from cache
     *
     * @return AlbumList list of my albums
    */
    AlbumList getAlbumList();

    /**
     * @brief set list of my albums and store it to cache file
     *
     * @param AlbumList
    */
    void setAlbumList(AlbumList);

    Friend getProfile(bool isNeedUpdate);

    /**
     * @brief setup proxy settings for current driver
     *
     * @param proxyHost proxy host name or QString() if proxy not use
     * @param proxyPort proxy port number or 0 if proxy not use
    */
    void setProxy(QString proxyHost, int proxyPort);

    // checks that it is a first request
    bool isFirstRequest;

    // true if first request is success, otherwise false
    bool isNetworkEnabled;

signals:
    void updateProfile(QString accountId, Friend profile);

private slots:
    void gotSettings(QString accountId, QString settings);
    void gotProfile(QString accountId, Friend);
};

typedef QList<Account *> AccountList;

Q_DECLARE_METATYPE(Account *)

#endif // ACCOUNT_H
