/*
 * This file is part of mydicto
 *
 * Copyright (C) 2010 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <gst/gst.h>
#include <gconf/gconf-client.h>
#include <hildon/hildon.h>
#include <libosso.h>

#ifdef WITH_SHARING

#ifdef WITH_SHARING_H
#include <sharingdialog/sharing-dialog.h>
#endif

#else
void sharing_dialog_with_file (osso_context_t * osso, GtkWindow * parent, const gchar * uri);
#endif

#include <dicto/dicto.h>
#include <dicto/dicto-ui.h>
#include "config.h"

HildonProgram *program;
HildonWindow *window;
osso_context_t *osso;

GtkWidget *dw;
GtkWidget *source_button, *sharing_button, *format_button;
Dicto *d;
GtkTreeModel *formats;

#define MYDICTO_GCONF_PATH "/apps/mydicto"
#define MYDICTO_GCONF_AUDIO_FORMAT MYDICTO_GCONF_PATH  "/audio-format"

static gboolean
get_selected_id_from_selector(HildonTouchSelector *selector, gint *id)
{
GtkTreePath *path;
GtkTreeModel *model;
GtkTreeIter iter;

path=hildon_touch_selector_get_last_activated_row(selector, 0);
if (path!=NULL) {
	model=hildon_touch_selector_get_model(selector, 0);
	gtk_tree_model_get_iter(model, &iter, path);
	gtk_tree_model_get(model, &iter, 0, id, -1);
}
gtk_tree_path_free(path);
return path ? TRUE : FALSE;
}

static gint
mydicto_dbus_req_handler(const gchar *interface, const gchar *method, GArray *arguments, gpointer data, osso_rpc_t *retval)
{
return OSSO_OK;
}

static gboolean
dicto_error_cb(Dicto *d, gpointer err, gpointer data)
{
GError *e=(GError *)err;

g_debug("Dicto reported error: (%d) %s", e->code, e->message);
hildon_banner_show_information(GTK_WIDGET(window), "", e->message);
return FALSE;
}

static gboolean
dicto_delete_query_cb(GtkWidget *dw, const gchar *file, gpointer data)
{
return dicto_widget_delete_query_dialog(GTK_WINDOW(window), file);
}

static gboolean
dicto_stop_cb(Dicto *d, gpointer data)
{
gtk_widget_show(sharing_button);
gtk_widget_show(format_button);
hildon_banner_show_information(GTK_WIDGET(window), "", "Stopped");
return FALSE;
}

static gboolean
dicto_playing_cb(Dicto *d, gpointer data)
{
gtk_widget_hide(sharing_button);
gtk_widget_hide(format_button);
hildon_banner_show_information(GTK_WIDGET(window), "", "Playing");
return FALSE;
}

static gboolean
dicto_recording_cb(Dicto *d, gpointer data)
{
gtk_widget_hide(sharing_button);
gtk_widget_hide(format_button);
hildon_banner_show_information(GTK_WIDGET(window), "", "Recording");
return FALSE;
}

static void
mydicto_source_cb(GtkWidget *w, gpointer data)
{

}

static void
mydicto_format_cb(HildonPickerButton *button, gpointer data)
{
HildonTouchSelector *selector=(HildonTouchSelector *)data;
gint id;

if (!get_selected_id_from_selector(selector, &id))
	return;
if (!dicto_set_format(d, (DictoFormats)id))
	hildon_banner_show_information(GTK_WIDGET(window), "", "Failed to set new format");
}

#if HILDON_CHECK_VERSION(2,2,0)
static void
mydicto_menu_button_share_cb(GtkWidget *w, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
gchar *file;
#ifdef WITH_SHARING
g_object_get(G_OBJECT(d), "current-file", &file, NULL);
g_debug("File: %s", file);
if (file && strlen(file)>0 && g_file_test(file, G_FILE_TEST_IS_REGULAR)) {
	sharing_dialog_with_file(osso, GTK_WINDOW(window), file);
} else {
	hildon_banner_show_information(GTK_WIDGET(window), "", "No file selected");
}
g_free(file);
#else
hildon_banner_show_information(GTK_WIDGET(window), "", "Sharing disabled");
#endif
}
#else
static void
mydicto_menu_share_email_cb(GtkWidget *w, gpointer data)
{
}

static void
mydicto_menu_share_bt_cb(GtkWidget *w, gpointer data)
{
}
#endif

#if HILDON_CHECK_VERSION(2,2,0)
static HildonAppMenu *
mydicto_menu(void)
{
HildonAppMenu *menu;
HildonTouchSelector *s;
HildonTouchSelectorColumn *column;

menu=HILDON_APP_MENU(hildon_app_menu_new());

/* Sharing menu button */
sharing_button=gtk_button_new_with_label("Share");
g_signal_connect_after(sharing_button, "clicked", G_CALLBACK(mydicto_menu_button_share_cb), dw);
hildon_app_menu_append(menu, GTK_BUTTON(sharing_button));

#if 0
source_button=gtk_button_new_with_label("Source");
g_signal_connect_after(source_button, "clicked", G_CALLBACK(mydicto_source_cb), dw);
hildon_app_menu_append(menu, GTK_BUTTON(source_button));
#endif

format_button=hildon_picker_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
hildon_button_set_title(HILDON_BUTTON(format_button), "Format");

s=hildon_touch_selector_new();
column=hildon_touch_selector_append_column(s, GTK_TREE_MODEL(formats), gtk_cell_renderer_text_new(), "text", 1, NULL);
hildon_touch_selector_column_set_text_column(column, 1);
hildon_touch_selector_set_hildon_ui_mode(s, HILDON_UI_MODE_NORMAL);
hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(format_button), HILDON_TOUCH_SELECTOR(s));
hildon_app_menu_append(menu, GTK_BUTTON(format_button));

g_signal_connect(G_OBJECT(format_button), "value-changed", G_CALLBACK(mydicto_format_cb), s);

gtk_widget_show_all(GTK_WIDGET(menu));
return menu;
}
#else
static GtkMenu *
mydicto_menu(void)
{
GtkWidget *menu, *email, *bt;

menu=gtk_menu_new();
email=gtk_menu_item_new_with_label("Send via E-mail");
bt=gtk_menu_item_new_with_label("Send via Bluetooth");
gtk_menu_append(menu, email);
gtk_menu_append(menu, bt);

g_signal_connect(G_OBJECT(email), "activate", GTK_SIGNAL_FUNC(mydicto_menu_share_email_cb), dw);
g_signal_connect(G_OBJECT(bt), "activate", GTK_SIGNAL_FUNC(mydicto_menu_share_bt_cb), dw);

return menu;
}
#endif

static void
dicto_format_select_cb(GtkComboBox *widget, gpointer data)
{
GConfClient *client;
GtkTreeIter iter;
GtkTreeModel *model;
gint value=-1;

model=gtk_combo_box_get_model(widget);
if (!gtk_combo_box_get_active_iter(widget, &iter))
	return;

gtk_tree_model_get(model, &iter, 0, &value,  -1);
if (!dicto_set_format(d, value)) {
	hildon_banner_show_information(GTK_WIDGET(window), "", "Failed to set format");
	return;
}
client=gconf_client_get_default();
gconf_client_set_int(client, MYDICTO_GCONF_AUDIO_FORMAT, value, NULL);
}

static gboolean
mydicto_init(DictoFormats f, gchar *src, gchar *sink)
{
osso_return_t result;

g_set_application_name("myDicto");
gtk_window_set_default_icon_name("mydicto");

program=HILDON_PROGRAM(hildon_program_get_instance());

d=dicto_new_full(NOTES_BASEDIR, f, sink, src);
if (!d)
	return FALSE;
dw=dicto_widget_new(d);
formats=dicto_widget_get_formats(DICTO_WIDGET(dw));

#if HILDON_CHECK_VERSION(2,2,0)
window=HILDON_WINDOW(hildon_stackable_window_new());
hildon_window_set_app_menu(HILDON_WINDOW(window), mydicto_menu());
#else
window=HILDON_WINDOW(hildon_window_new());
hildon_window_set_menu(HILDON_WINDOW(window), mydicto_menu());
#endif

hildon_program_add_window(program, window);

gtk_window_set_title(GTK_WINDOW(window), "myDicto");
g_signal_connect(G_OBJECT(window), "destroy", G_CALLBACK(gtk_main_quit), NULL);

osso=osso_initialize("org.tal.mydicto", VERSION, FALSE, NULL);
if (!osso)
	return FALSE;

result=osso_rpc_set_cb_f(osso, "org.tal.mydicto", "/org/tal/mydicto", "org.tal.mydicto", mydicto_dbus_req_handler, NULL);
if (result!=OSSO_OK)
	return FALSE;

g_signal_connect(G_OBJECT(d), "error", G_CALLBACK(dicto_error_cb), dw);
g_signal_connect(G_OBJECT(d), "playing", G_CALLBACK(dicto_playing_cb), NULL);
g_signal_connect(G_OBJECT(d), "recording", G_CALLBACK(dicto_recording_cb), NULL);
g_signal_connect(G_OBJECT(d), "stopped-play", G_CALLBACK(dicto_stop_cb), NULL);
g_signal_connect(G_OBJECT(d), "stopped-record", G_CALLBACK(dicto_stop_cb), NULL);
g_signal_connect(G_OBJECT(dw), "delete-query", G_CALLBACK(dicto_delete_query_cb), dw);
g_object_set(dw, "enable-delete", TRUE, NULL);

gtk_container_add(GTK_CONTAINER(window), dw);
gtk_widget_show_all(GTK_WIDGET(window));

return TRUE;
}

gint 
main(gint argc, gchar * argv[])
{
gchar *sink, *src;
GConfClient *client;
DictoFormats f=FORMAT_WAV_8K_16B_M;

#if HILDON_CHECK_VERSION(2,2,0)
hildon_gtk_init(&argc, &argv);
#else
gtk_init(&argc, &argv);
#endif
gst_init(&argc, &argv);

sink=AUDIO_SINK;
src=AUDIO_SRC;

/* Veeery simple argument passing, mainly for getting working sinks under sdk */
if (argc>1) {
	f=atoi(argv[1]);
	if (argc>2)
		src=argv[2];
	if (argc>3)
		sink=argv[3];
} else {
	GError *error=NULL;
	gint v;

	client=gconf_client_get_default();
	v=gconf_client_get_int(client, MYDICTO_GCONF_AUDIO_FORMAT, &error);
	if (error!=NULL) {
		g_debug("Failed to get format from gconf.");
		g_error_free(error);
	} else {
		f=v;
	}
}

g_debug("Using format %d, source %s and sink %s", f, src, sink);

if (argc>4) {
	g_setenv("PULSE_PROP_media.role", argv[4], TRUE);
	g_debug("Pulseaudio role set to: %s", argv[4]);
}


if (!mydicto_init(f, src, sink)) {
	g_warning("myDicto init failed");
	return 1;
}

gtk_main();

return 0;
}

