#include "tubestatusdelegate.h"

#include <QHash>
#include <qdebug.h>
#include <qapplication.h>
#include <qfontmetrics.h>
#include <QStyleOptionViewItem>

TubeStatusDelegate::TubeStatusDelegate(QObject *parent) :
        QStyledItemDelegate(parent)
{
}


void TubeStatusDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const
{
    QVariant data = index.data();

    if ( data.canConvert<TubeStatus>() )
    {
        // TODO: highlight if selected
        //            if (option.state().isSet(QStyle.StateFlag.State_Selected)) {
        //                painter.fillRect(option.rect(), option.palette().highlight());
        //            }
        //initStyleOption(&option, index);
        painter->save();
        doPaint(painter, option, *(TubeStatus *)data.data());
        painter->restore();
    } else
        QStyledItemDelegate::paint(painter, option, index);
}

QSize TubeStatusDelegate::sizeHint(const QStyleOptionViewItem & option, const QModelIndex & index) const
{
    QSize ret;
    QVariant data = index.data();

    if( data.canConvert<TubeStatus>() ) {
        TubeStatus * ts = (TubeStatus *)data.data();

        QFont smallFont = option.font;
        smallFont.setPointSize(smallFont.pointSize()-2);

        QFontMetrics normalFM(option.font);
        QFontMetrics smallFM(smallFont);

        int height = normalFM.height() + smallFM.height() + 10;

        int minWidth = normalFM.width(ts->getName());
        if( smallFM.width(ts->getStatus()) > minWidth )
            minWidth = smallFM.width(ts->getStatus());

        ret.setHeight(height);
        ret.setWidth(minWidth + QApplication::style()->PM_ButtonIconSize + 10);
    }
    else
        ret = QStyledItemDelegate::sizeHint(option, index);
//    qDebug() << "Inside TubeStatusDelegate::sizeHint() : canvas size ("
//            << option.rect.height() << " x " << option.rect.width()
//            << ") item size(" << ret.height() << " x " << ret.width() << ")";

    return ret;
}


void TubeStatusDelegate::doPaint( QPainter *painter, const QStyleOptionViewItem &option, const TubeStatus & tubeStatus ) const
{
    const QRect &rect = option.rect;
    const QPalette & palette = option.palette;

    // QPainter::RenderHint
    painter->setRenderHint(QPainter::Antialiasing, true);
    // Qt::PenStyle
    //painter->setPen(Qt::NoPen);

    //         if (mode == ReadWrite)
    //             painter.setBrush(palette.highlight());
    //         else

    // +----------+-----------------------------------+
    // |P         |          CENTRAL LINE             |
    // |   I      |  +----------------------------+   |
    // |      C   |  |       Good Service         |   |
    // |         Y|  +----------------------------+   |
    // +----------+-----------------------------------+
    // rect = w * h
    // Tube line text height = 1/2 height
    // Tube status text height = 1/3 height
    // Rest is 3 times padding = (height - height/2 - height/3) * 3
    //
    // Layout 2:
    // +-----------------------------------+----------+
    // |  CENTRAL LINE                     | I      N |
    // |  +----------------------------+   |   C  O   |
    // |  | Good Service               |   |   C  O   |
    // |  +----------------------------+   | I      N |
    // +-----------------------------------+----------+


    // Calculate font sizes
    QFont fName = painter->font();
    fName.setBold(true);
    QFont fStatus = fName;
    fStatus.setPointSize(fName.pointSize()-2);

    QFontMetricsF fmName(fName);
    QFontMetricsF fmStatus(fStatus);

    float vPadding = (rect.height() - fmName.height() - fmStatus.height()) / 2.5;

    // Calculate icon size
    QIcon icon;
    if( tubeStatus.isGoodService() ) {
        if( tubeStatus.numMessages() == 0 ) {
            icon = QApplication::style()->standardIcon(QStyle::SP_DialogYesButton);
        } else {
            icon = QApplication::style()->standardIcon(QStyle::SP_MessageBoxInformation);
        }
    } else {
        icon = QApplication::style()->standardIcon(QStyle::SP_MessageBoxWarning);
    }

    QSize iconSize;
    iconSize.setHeight(rect.height()-10);
    iconSize.setWidth(iconSize.height());

    iconSize = icon.actualSize(iconSize);

    QRectF rName = rect;
    rName.setTop(rect.top() + vPadding);
    rName.setLeft(rect.left() + 5);
    rName.setHeight(fmName.height());
    rName.setWidth(rect.width() - iconSize.width() - 15);

    QRectF rStatus = rect;
    rStatus.setTop(rect.bottom() - vPadding - fmStatus.height());
    rStatus.setLeft(rect.left() + 5);
    rStatus.setHeight(fmStatus.height());
    rStatus.setWidth(rect.width() - iconSize.width() - 15);

    // Draw background
    if (option.state & QStyle::State_Selected) {
        QPen pen = QPen(Qt::red);
        pen.setWidth(3);
        painter->setPen(pen);
    } else {
        painter->setPen(Qt::white);
    }
    painter->setBrush(QBrush(getBackground(tubeStatus)));
    painter->drawRoundedRect(rect, 10, 10, Qt::AbsoluteSize);

    // Draw Tube line name text
    painter->setFont(fName);
    painter->setPen(getForeground(tubeStatus));
    painter->setBrush(palette.text());
    painter->drawText(rName, Qt::AlignLeft, tubeStatus.getName());

    // Draw Tube line status background
    QPen pen = QPen(Qt::white);
    pen.setWidth(2);
    painter->setPen(pen);
    if( tubeStatus.isGoodService() )
        painter->setBrush(Qt::white);
    else
        painter->setBrush(Qt::blue);
    painter->drawRect(rStatus);

    // Draw Tube line status text
    rStatus.setLeft(rStatus.left() + 3);
    rStatus.setWidth(rStatus.width() - 6);
    painter->setFont(fStatus);
    if( tubeStatus.isGoodService() )
        painter->setPen(Qt::black);
    else
        painter->setPen(Qt::white);
    //painter->setBrush(Qt::blue);
    painter->setBrush(palette.text());
    painter->drawText(rStatus, Qt::AlignLeft, tubeStatus.getStatus());


    // Draw icon
    QRect rIcon = rect;
#if 1
    // 5 pixel border, otherwise make icon as big as possible
    rIcon.setTop(rect.top()+(rect.height()-iconSize.height())/2);
    rIcon.setLeft(rect.right()-5-iconSize.width());
    rIcon.setHeight(iconSize.height());
    rIcon.setWidth(iconSize.width());
#else
    // Use the style's icon size and adjust border
    rIcon.setHeight(QStyle::PM_ListViewIconSize);
    rIcon.setWidth(QStyle::PM_ListViewIconSize);
    rIcon.setTop(rect.top() + (rect.height()-rIcon.height()) / 2 );
    rIcon.setLeft(rect.left() + (rect.height()-rIcon.height()) / 2 );
#endif

    icon.paint(painter, rIcon);
}


class TubeColour {
public:
    QColor foreground;
    QColor background;

    TubeColour(QString bg = "#000", QString fg = "#FFF")
    {
        foreground = QColor(fg);
        background = QColor(bg);
    }
};

static class TubeColours
{
private:
    QHash<QString, TubeColour> tubeColours;
public:
    TubeColours() {
        if( tubeColours.count() == 0 ) {
            tubeColours.insert("bakerloo", TubeColour("#AE6118","#FFF"));
            tubeColours.insert("central", TubeColour("#E41F1F","#FFF"));
            tubeColours.insert("circle", TubeColour("#F8D42D","#113B92"));
            //'circle' : ( gtk.gdk.color_parse('#F8D42D'), gtk.gdk.color_parse('#113B92') ),
            tubeColours.insert("district", TubeColour("#00A575","#FFF"));
            //'district' : ( gtk.gdk.color_parse('#00A575'), gtk.gdk.color_parse('#FFF') ),
            tubeColours.insert("hammersmithcity", TubeColour("#E899A8","#113B92"));
            //'hammersmithcity' : ( gtk.gdk.color_parse('#E899A8'), gtk.gdk.color_parse('#113B92') ),
            tubeColours.insert("jubilee", TubeColour("#8F989E","#FFF"));
            //'jubilee' : ( gtk.gdk.color_parse('#8F989E'), gtk.gdk.color_parse('#FFF') ),
            tubeColours.insert("metropolitan", TubeColour("#893267","#FFF"));
            //'metropolitan' : ( gtk.gdk.color_parse('#893267'), gtk.gdk.color_parse('#FFF') ),
            tubeColours.insert("northern", TubeColour("#000000","#FFF"));
            //'northern' : ( gtk.gdk.color_parse('#000000'), gtk.gdk.color_parse('#FFF') ),
            tubeColours.insert("piccadilly", TubeColour("#0450A1","#FFF"));
            //'piccadilly' : ( gtk.gdk.color_parse('#0450A1'), gtk.gdk.color_parse('#FFF') ),
            tubeColours.insert("victoria", TubeColour("lightblue","#FFF"));
            //'victoria' : ( gtk.gdk.color_parse('lightblue'), gtk.gdk.color_parse('#FFF') ),
            tubeColours.insert("waterloocity", TubeColour("#70C3CE","#113B92"));
            //'waterloocity' : ( gtk.gdk.color_parse('#70C3CE'), gtk.gdk.color_parse('#113B92') ),
            tubeColours.insert("docklands", TubeColour("#00BBB4","#FFF"));
            //'docklands' :  ( gtk.gdk.color_parse('#00BBB4'), gtk.gdk.color_parse('#FFF') ),
            tubeColours.insert("overground", TubeColour("#F47F25","#FFF"));
        }
    }

    const TubeColour & get(const QString & key) {
        // NB: Use iterators rather than syntactic sugar overloaded operator []
        //     in order to return a const reference instead of a copy.
        static const TubeColour defaultColour;
        QHash<QString, TubeColour>::const_iterator it;

        it = tubeColours.find(key);
        if( it == tubeColours.end() )
            return defaultColour;
        return it.value();
    }
} myTubeColours;

QColor TubeStatusDelegate::getForeground(const TubeStatus &tubeStatus) const
{
    return myTubeColours.get(tubeStatus.getKey()).foreground;
}

QColor TubeStatusDelegate::getBackground(const TubeStatus &tubeStatus) const
{
        return myTubeColours.get(tubeStatus.getKey()).background;
}
