#include "albumselectdialog.h"
#include "ui_albumselectdialog.h"

#include <QtGui/QIcon>
#include <QtCore/QString>
#include <QtCore/QVariant>
#include <QtCore/QDebug>

#include "datatypes/account.h"
#include "datatypes/friend.h"
#include "servicemgr.h"

AlbumSelectDialog::AlbumSelectDialog(ServiceMgr* mgr, QWidget* parent) :
    QDialog(parent),
    ui(new Ui::AlbumSelectDialog),
    mServiceMgr(mgr), mLastSelectedItem(0)
{
    ui->setupUi(this);

    fillAccountList();
    connect(ui->accountListWidget, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onItemActivated(QListWidgetItem*)));
    connect(ui->albumListWidget, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(setSelectedAlbumItem(QListWidgetItem*)));
}

AlbumSelectDialog::~AlbumSelectDialog()
{
    clearList();

    foreach (QList<QListWidgetItem*> list, mAccountAlbums) {
        qDeleteAll(list);
    }

    delete ui;
}

void AlbumSelectDialog::fillAccountList()
{
    AccountList accountList = mServiceMgr->getAccounts();

    foreach (Account* account, accountList) {
        addAccountToList(account);
    }
}

void AlbumSelectDialog::addAccountToList(Account* account)
{
    const Friend& fr = account->getProfile(false);

    // Resize friend icon
    QImage icon = Utils::adjustImage(QImage(fr.icon()), QSize(64, 64));
    QListWidgetItem* widgetItem = new QListWidgetItem(QIcon(QPixmap::fromImage(icon)), fr.name());
    ui->accountListWidget->addItem(widgetItem);
    widgetItem->setData(Qt::UserRole, QVariant::fromValue(account));
}

void AlbumSelectDialog::onItemActivated(QListWidgetItem* item)
{
    Account* account = item->data(Qt::UserRole).value<Account*>();

    // Create album list for selected account
    if (!mAccountAlbums.contains(item)) {
        QList<QListWidgetItem*> itemList;
        AlbumList albumList = account->getAlbumList();

        foreach (const Album& album, albumList) {
            QIcon icon;
            QImage img = QImage(album.icon());
            if (img.isNull())
                icon = QIcon(":/res/tasklaunch_photos.png");
            else
                icon = QIcon(QPixmap::fromImage(Utils::adjustImage(img, QSize(64, 64))));

            QListWidgetItem* albumItem = new QListWidgetItem(icon, album.title());
            albumItem->setData(Qt::UserRole, QVariant::fromValue(album));
            itemList.append(albumItem);
        }
        mAccountAlbums.insert(item, itemList);
    }

    clearList();
    QList<QListWidgetItem*> itemList = mAccountAlbums.value(item);
    foreach (QListWidgetItem* item, itemList) {
        ui->albumListWidget->addItem(item);
    }
}

void AlbumSelectDialog::clearList()
{
    mLastSelectedItem = 0;
    while (ui->albumListWidget->count() > 0)
        ui->albumListWidget->takeItem(0);
}

void AlbumSelectDialog::setSelectedAlbumItem(QListWidgetItem* item)
{
    mLastSelectedItem = item;
}

Album AlbumSelectDialog::selectedAlbum() const
{
    if (mLastSelectedItem != 0)
        return mLastSelectedItem->data(Qt::UserRole).value<Album>();
    else
        return Album();
}

void AlbumSelectDialog::on_buttonBox_accepted()
{
    if (mLastSelectedItem != 0)
        accept();
    else
        reject();
}
