/*
 * This file is a variation of he-fullscreen-button.c, which is part of the Hildon-Extras library
 */
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <gdk/gdkx.h>
#include <X11/Xatom.h>
#include <string.h>
#include "imgparam-window.h"

#define IMGPARAM_WINDOW_HIDE_DELAY 5000
#define IMGPARAM_WINDOW_CORNER_RADIUS 20
#define IMGPARAM_WINDOW_PADDING 10
#define OFFSET 20

typedef struct _ImgparamWindowPrivate ImgparamWindowPrivate;

struct _ImgparamWindowPrivate
{
	gchar *markup;
	gboolean enabled;

	GtkWidget *overlay;
};

#define	IMGPARAM_WINDOW_GET_PRIVATE(object) \
		(G_TYPE_INSTANCE_GET_PRIVATE((object), \
		TYPE_IMGPARAM_WINDOW, ImgparamWindowPrivate))

G_DEFINE_TYPE(ImgparamWindow, imgparam_window, G_TYPE_OBJECT)

/**
 * Hides the image parameters window.
 */
void
imgparam_window_hide (ImgparamWindow *self)
{
    g_return_if_fail (IS_IMGPARAM_WINDOW (self));

    ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
    g_return_if_fail (priv != NULL);

    if (priv->overlay != NULL && GTK_IS_WIDGET (priv->overlay)) {
    	gtk_widget_hide (priv->overlay);
    }
}


/**
 * Changes the position of the image parameters window.
 */
static void
imgparam_window_set_position (ImgparamWindow *self)
{
	GtkWidget *parent = GTK_WIDGET (self->parent_window);
	GtkWidget *overlay = NULL;

	ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
	g_return_if_fail (priv != NULL);

	overlay = GTK_WIDGET (priv->overlay);

	/* For some reason I have to call hide/show to make it appear at the new position */
	gint x = OFFSET;
	gint y = OFFSET;
	if (!(gdk_window_get_state(gtk_widget_get_window(parent)) & GDK_WINDOW_STATE_FULLSCREEN)) y += HILDON_WINDOW_TITLEBAR_HEIGHT;

	gtk_widget_hide (overlay);
	gtk_window_move (GTK_WINDOW (overlay), x, y);
	gtk_widget_show (overlay);
}

/**
 * Shows the image parameters window.
 */
void
imgparam_window_show (ImgparamWindow *self)
{
    g_return_if_fail (self != NULL);

    ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
    g_return_if_fail (priv != NULL);

    g_return_if_fail (GTK_IS_WIDGET (priv->overlay));

    if (priv->enabled) {
    	imgparam_window_set_position (self);
    	gtk_widget_show (priv->overlay);
    }
}

/**
 * Everytime the button is clicked, emit the "clicked" signal.
 */
static gboolean
imgparam_window_on_clicked (GtkWidget *widget, GdkEventButton *event G_GNUC_UNUSED, gpointer data)
{
	ImgparamWindow *self = IMGPARAM_WINDOW (data);
	g_signal_emit_by_name (self, "clicked");

	return TRUE;
}


/**
 * Force compositing to stay on
 */
static void
imgparam_window_force_compositing (GtkWidget *widget)
{
	int zero = 0;
	Atom atom;
    GdkDisplay *display = NULL;

    display = gtk_widget_get_display(widget);
    /* atom = XInternAtom(display,"_HILDON_NON_COMPOSITED_WINDOW",False); */
    atom = gdk_x11_get_xatom_by_name_for_display(display,"_HILDON_NON_COMPOSITED_WINDOW");
    XChangeProperty(GDK_DISPLAY_XDISPLAY (display),
				    GDK_WINDOW_XID (GDK_WINDOW (widget->window)),atom,XA_INTEGER,32,
					PropModeReplace,(unsigned char *)&zero,1);
}


/**
 * Creates a rounded rectangle.
 */
static void
imgparam_window_create_rectangle (cairo_t *ctx, double x, double y, double w, double h, double r)
{
	cairo_move_to(ctx,x + r,y);
	cairo_line_to(ctx,x + w - r,y);
	cairo_curve_to(ctx,x + w,y,x + w,y,x + w,y + r);
	cairo_line_to(ctx,x + w,y + h - r);
	cairo_curve_to(ctx,x + w,y + h,x + w,y + h,x + w - r,y + h);
	cairo_line_to(ctx,x + r,y + h);
	cairo_curve_to(ctx,x,y + h,x,y + h,x,y + h - r);
	cairo_line_to(ctx,x,y + r);
	cairo_curve_to(ctx,x,y,x,y,x + r,y);
}


/**
 * Does the actual drawing of the semi transparent text and background
 */
static gboolean
imgparam_window_on_expose_event (GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
	g_return_val_if_fail(IS_IMGPARAM_WINDOW(data), FALSE);

	ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (data);
	g_return_val_if_fail((priv != NULL), FALSE);

	cairo_t *ctx;
    gint txtwidth = 0,txtheight = 0;
	PangoLayout *textbuff = NULL;

    /* Create context */
    ctx = gdk_cairo_create (widget->window);
    cairo_rectangle(ctx,event->area.x,event->area.y,event->area.width,event->area.height);
    cairo_clip(ctx);

    /* Clear surface */
    cairo_set_operator (ctx, CAIRO_OPERATOR_CLEAR);
    cairo_paint (ctx);

	/* Create text layout */
	textbuff = pango_cairo_create_layout(ctx);
	if (priv->markup != NULL) pango_layout_set_markup(textbuff,priv->markup,-1);
	else pango_layout_set_markup(textbuff,"<span size=\"medium\" face=\"sans\">No EXIF data</span>",-1);
	pango_layout_get_pixel_size(textbuff,&txtwidth,&txtheight);

	/* Set widget size according to layout */
	gtk_widget_set_size_request(GTK_WIDGET (priv->overlay),txtwidth + (IMGPARAM_WINDOW_PADDING * 2),txtheight + (IMGPARAM_WINDOW_PADDING * 2));
	gtk_window_resize(GTK_WINDOW (priv->overlay),txtwidth + (IMGPARAM_WINDOW_PADDING * 2),txtheight + (IMGPARAM_WINDOW_PADDING * 2));

    /* Add rectangle */
    cairo_set_operator (ctx, CAIRO_OPERATOR_OVER);
    cairo_set_source_rgba (ctx, 0, 0, 0, 0.4);
    imgparam_window_create_rectangle (ctx, 0, 0, txtwidth + (IMGPARAM_WINDOW_PADDING * 2),txtheight + (IMGPARAM_WINDOW_PADDING * 2), IMGPARAM_WINDOW_CORNER_RADIUS);
    cairo_fill (ctx);

    /* Add text */
	cairo_new_path(ctx);
	cairo_move_to(ctx,IMGPARAM_WINDOW_PADDING,IMGPARAM_WINDOW_PADDING);
	cairo_set_line_width(ctx,1);
	cairo_set_source_rgb(ctx,0.8,0.8,0.8);
	pango_cairo_update_layout(ctx,textbuff);
	pango_cairo_layout_path(ctx,textbuff);
	cairo_fill_preserve(ctx);
	/*cairo_set_line_width(ctx,0.5);
	cairo_set_source_rgb(ctx,0,0,0);
	cairo_stroke(ctx);*/

    /* Destroy context and free layout */
    cairo_destroy (ctx);
	g_object_unref(textbuff);

	return TRUE;
}


/**
 * Creates the semi transparent window graphic.
 */
static GtkWidget*
imgparam_window_create_gfx (ImgparamWindow *self)
{
    g_return_val_if_fail(IS_IMGPARAM_WINDOW(self), NULL);

    ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
	g_return_val_if_fail((priv != NULL), NULL);

    GtkWidget *img = gtk_image_new ();
    gtk_widget_show (img);
    g_signal_connect (img, "expose_event", G_CALLBACK (imgparam_window_on_expose_event), self);

    GtkWidget *box = gtk_event_box_new ();
    gtk_event_box_set_visible_window (GTK_EVENT_BOX(box), FALSE);
    gtk_widget_show (box);
    gtk_container_add (GTK_CONTAINER(box), img);
    g_signal_connect (box, "button-release-event", G_CALLBACK (imgparam_window_on_clicked), self);

    GtkWidget *overlay = gtk_window_new (GTK_WINDOW_POPUP);
    gtk_window_set_decorated (GTK_WINDOW (overlay), FALSE);
    gtk_window_set_resizable (GTK_WINDOW (overlay), FALSE);
    gtk_window_set_transient_for (GTK_WINDOW (overlay), self->parent_window);
    gtk_window_set_destroy_with_parent (GTK_WINDOW (overlay), TRUE);
    gtk_container_add (GTK_CONTAINER (overlay), box);

    GdkScreen *screen = gtk_widget_get_screen (overlay);
    gtk_widget_set_colormap (overlay, gdk_screen_get_rgba_colormap (screen));

    gtk_widget_realize (overlay);

    imgparam_window_force_compositing (overlay);

    return overlay;
}

/**
 * Called when the parent_window's is on the screen/not on the screen.
 * Only called if parent_window is a HildonWindow (or derived from it).
 *
 * We check if the window is not on the screen.
 * If not, we disable the button.
 */
static void
imgparam_window_on_is_topmost_changed (GObject *object G_GNUC_UNUSED,
		                                 GParamSpec *property G_GNUC_UNUSED,
		                                 gpointer data)
{
	ImgparamWindow *self = IMGPARAM_WINDOW (data);

	ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
	g_return_if_fail (priv != NULL);

	if (hildon_window_get_is_topmost (HILDON_WINDOW(self->parent_window))) {
		if (!(gdk_window_get_state (GTK_WIDGET (self->parent_window)->window) & GDK_WINDOW_STATE_FULLSCREEN)) {
			imgparam_window_show (self);
		}
	}
	else {
		imgparam_window_hide (self);
	}
}

/**
 * Called when the "markup" property of the image parameters window has changed.
 *
 * We force a redraw of the window.
 */
static void
imgparam_window_on_markup_changed (GObject *object G_GNUC_UNUSED,
								   GParamSpec *property G_GNUC_UNUSED,
								   gpointer data)
{
	ImgparamWindow *self = IMGPARAM_WINDOW (object);

	ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
	g_return_if_fail (priv != NULL);

	if (hildon_window_get_is_topmost (HILDON_WINDOW(self->parent_window))) {
		imgparam_window_show (self);
	}
}


/**
 * Destroys the image parameters window and disconnects itself from the parent window.
 */
static void
imgparam_window_destroy (GtkWidget *parent_window G_GNUC_UNUSED, ImgparamWindow *self)
{
	g_return_if_fail (self != NULL);

	ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
	g_return_if_fail (priv != NULL);

	if (self->parent_window != NULL) {
		g_signal_handlers_disconnect_by_func (self->parent_window, imgparam_window_destroy, self);
		if (HILDON_IS_WINDOW(parent_window)) {
			g_signal_handlers_disconnect_by_func (self->parent_window, imgparam_window_on_is_topmost_changed, self);
		}
	}

	imgparam_window_hide (self);

	if (priv->overlay != NULL && GTK_IS_WIDGET(priv->overlay)) {
		gtk_widget_destroy (GTK_WIDGET(priv->overlay));
		priv->overlay = NULL;
	}

	if (priv->markup != NULL) {
		g_free(priv->markup);
	}
}


/**
 * Called when the size allocation of the parent window changes.
 * We change the position of the image parameters window to always be in
 * the middle of the left or right edge.
 */
static void
imgparam_window_on_parent_size_changed (GtkWidget     *widget,
                                     GtkAllocation *allocation,
                                     gpointer       user_data)
{
    g_return_if_fail (widget != NULL);
    g_return_if_fail (allocation != NULL);

    ImgparamWindow *self = IMGPARAM_WINDOW(user_data);
    g_return_if_fail (self != NULL);

    ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
    g_return_if_fail (priv != NULL);

    GtkWidget *ui_win = GTK_WIDGET(priv->overlay);
    g_return_if_fail (ui_win != NULL);

    if (gdk_window_is_visible(priv->overlay->window)) {
    	imgparam_window_set_position(self);
    }
}

/**
* Called, whenever the state of the parents window's GdkWindow changes.
* We check if the new state is fullscreen or non-fullscreen.
* If it is not fullscreen, we show the image parameters window. If not, we don't.
*/
static gboolean
imgparam_window_on_window_state_changed (GtkWidget *widget G_GNUC_UNUSED,
										 GdkEventWindowState *event,
										 gpointer data)
{
	ImgparamWindow *self = IMGPARAM_WINDOW (data);

	ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
	g_return_val_if_fail(priv != NULL, FALSE);

	/* Only run if this window is topmost. */
	if (HILDON_IS_WINDOW (self->parent_window)) {
		if (!hildon_window_get_is_topmost(HILDON_WINDOW(self->parent_window))) {
			return FALSE;
		}
	}

	/*if (event->changed_mask & GDK_WINDOW_STATE_FULLSCREEN) {
		if (event->new_window_state == GDK_WINDOW_STATE_FULLSCREEN) {
			imgparam_window_hide(self);
		} else {
			imgparam_window_show(self);
		}
	}*/

	return FALSE;
}

/**
 * Default handler for the "clicked" signal. If no user handler is
 * defined we just print a debug message
 * Otherwise only the user handler is executed.
 */
static void
imgparam_window_clicked_default_handler (ImgparamWindow *self)
{
	guint signal_id = g_signal_lookup ("clicked", TYPE_IMGPARAM_WINDOW);
	gulong handler_id = g_signal_handler_find (self, G_SIGNAL_MATCH_ID, signal_id, 0, NULL, NULL, NULL);

	/* Run only if no signal handler is connected */
	if (handler_id == 0) {
		g_print("DEBUG: Image parameters window clicked.\n");
	}
}

/**
 * Create new image parameters window instance.
 * This function attaches the image parameters window to the given parent window.
 *
 * If you destroy the parent window, this ImgparamWindow instance get
 * destroyed as well.
 *
 * Pass it a HildonWindow (or one of its deriatives) to ensure the widget disables/
 * enables itself on focus-out/focus-in respectively.
 *
 * @param parent_window A GtkWindow instance.
 * @return New ImgparamWindow instance.
 */
ImgparamWindow *
imgparam_window_new (GtkWindow *parent_window, const gchar *markup)
{
	g_return_val_if_fail (parent_window != NULL, NULL);
	g_return_val_if_fail (GTK_IS_WINDOW (parent_window), NULL);

    ImgparamWindow *self = g_object_new (TYPE_IMGPARAM_WINDOW, NULL);

    ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
    g_return_val_if_fail (priv != NULL, NULL);

    self->parent_window = parent_window;
    priv->markup = g_strdup(markup);
    priv->overlay = imgparam_window_create_gfx (self);

    g_signal_connect (parent_window, "destroy",
    		G_CALLBACK(imgparam_window_destroy), self);

    g_signal_connect_after (parent_window, "size-allocate",
    		G_CALLBACK(imgparam_window_on_parent_size_changed), self);

    /* g_signal_connect (parent_window, "window-state-event",
			G_CALLBACK(imgparam_window_on_window_state_changed), self); */

    if (HILDON_IS_WINDOW(parent_window)) {
        g_signal_connect (parent_window, "notify::is-topmost",
            G_CALLBACK(imgparam_window_on_is_topmost_changed), self);
    }

    g_signal_connect (self, "notify::markup",
        G_CALLBACK(imgparam_window_on_markup_changed), NULL);

    return self;
}

/**
 * Returns the GtkWidget displaying the actual overlaid button.
 *
 * @param self An instance of ImgparamWindow
 * @return The GtkWidget of the overlaid button. This widget belongs to ImgparamWindow and must not be destroyed or modified.
 */
GtkWidget *
imgparam_window_get_overlay (ImgparamWindow *self)
{
    ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
    g_return_val_if_fail (priv != NULL, NULL);

    return priv->overlay;
}

/**
 * Returns the GtkWindow that this ImgparamWindow
 * is attached to.
 *
 * @param self An instance of ImgparamWindow
 * @return The GtkWindow to which this button is attached to
 */
GtkWindow *
imgparam_window_get_window (ImgparamWindow *self)
{
	return self->parent_window;
}


/*
 * GObject stuff
 */

enum {
	CLICKED,
	LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

enum
{
  PROP_0,
  PROP_MARKUP,
  PROP_ENABLED
};

static void
imgparam_window_set_property (GObject      *object,
							  guint         property_id,
							  const GValue *value,
							  GParamSpec   *pspec)
{
    ImgparamWindow *self = IMGPARAM_WINDOW(object);
    g_return_if_fail (self != NULL);

	ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
	g_return_if_fail (priv != NULL);

	switch (property_id) {
		case PROP_MARKUP:
			g_free (priv->markup);
			priv->markup = g_value_dup_string (value);
			break;

		case PROP_ENABLED:
			priv->enabled = g_value_get_boolean (value);
			break;

		default:
			/* We don't have any other property... */
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
			break;
	}
}

static void
imgparam_window_get_property (GObject    *object,
                        guint       property_id,
                        GValue     *value,
                        GParamSpec *pspec)
{
    ImgparamWindow *self = IMGPARAM_WINDOW(object);
    g_return_if_fail (self != NULL);

	ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
	g_return_if_fail (priv != NULL);

	switch (property_id) {
		case PROP_MARKUP:
			g_value_set_string (value, priv->markup);
			break;

		case PROP_ENABLED:
			g_value_set_boolean (value, priv->enabled);
			break;

		default:
			/* We don't have any other property... */
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
			break;
	}
}

static void
imgparam_window_class_init (ImgparamWindowClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->set_property = imgparam_window_set_property;
	object_class->get_property = imgparam_window_get_property;

	klass->clicked = imgparam_window_clicked_default_handler;

	/**
	 * ImgparamWindow::clicked
	 *
	 * Emitted when the #ImgparamWindow was clicked by the user.
	 */
	signals[CLICKED] =
		g_signal_new(
				"clicked",
				TYPE_IMGPARAM_WINDOW,
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET(ImgparamWindowClass, clicked),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID,
				G_TYPE_NONE,
				0);

	g_object_class_install_property(
				object_class, PROP_MARKUP,
				g_param_spec_string ("markup",
				"Set markup to be displayed",
				"Text which will be displayed using pango markup language",
				"",
				G_PARAM_READWRITE));

	g_object_class_install_property(
				object_class, PROP_ENABLED,
				g_param_spec_boolean ("enabled",
				"Enable or disable the image parameters window",
				"When disabled, it will never be displayed",
				FALSE,
				G_PARAM_READWRITE));

	g_type_class_add_private (klass, sizeof (ImgparamWindowPrivate));
}

static void
imgparam_window_init (ImgparamWindow *self)
{
    g_return_if_fail (self != NULL);

    ImgparamWindowPrivate *priv = IMGPARAM_WINDOW_GET_PRIVATE (self);
    g_return_if_fail (priv != NULL);

    memset (priv, 0, sizeof (ImgparamWindowPrivate));

    self->parent_window = NULL;
    priv->overlay = NULL;

    priv->markup = NULL;
    priv->enabled = FALSE;
}
