/*
 * This file is a variation of he-fullscreen-button.c, which is part of the Hildon-Extras library
 */
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <string.h>
#include "arrow-button.h"

#define ARROW_BUTTON_WIDTH          80
#define ARROW_BUTTON_HEIGHT         70
#define ARROW_BUTTON_HIDE_DELAY		5000
#define ARROW_BUTTON_CORNER_RADIUS  20
#define ARROW_BUTTON_NEXT_ICON      "general_forward"
#define ARROW_BUTTON_PREVIOUS_ICON  "general_back"
#define ARROW_BUTTON_ICON_SIZE      48
#define OFFSET 20

typedef struct _ArrowButtonPrivate ArrowButtonPrivate;

struct _ArrowButtonPrivate
{
        gboolean release_event;
        guint32 last_event_time;

        guint button_press_signal_id;
        guint button_release_signal_id;

        gulong button_press_hook_id;
        gulong button_release_hook_id;

        arrow_direction_t direction;

        GtkWidget *overlay;
};

#define	ARROW_BUTTON_GET_PRIVATE(object) \
		(G_TYPE_INSTANCE_GET_PRIVATE((object), \
		TYPE_ARROW_BUTTON, ArrowButtonPrivate))

G_DEFINE_TYPE(ArrowButton, arrow_button, G_TYPE_OBJECT)

/**
 * Hides the arrow button.
 */
static void
arrow_button_hide (ArrowButton *self)
{
    g_return_if_fail (IS_ARROW_BUTTON (self));

    ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
    g_return_if_fail (priv != NULL);

    /* Reset timer */
    g_source_remove_by_user_data ((gpointer) self);

    if (priv->overlay != NULL && GTK_IS_WIDGET (priv->overlay)) {
    	gtk_widget_hide (priv->overlay);
    }
}


/**
 * Changes the position of the arrow button.
 */
static void
arrow_button_set_position (ArrowButton *self)
{
	GtkWidget *parent = GTK_WIDGET (self->parent_window);
	GtkWidget *overlay = NULL;

	ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
	g_return_if_fail (priv != NULL);

	overlay = GTK_WIDGET (priv->overlay);

	/* For some reason I have to call hide/show to make it appear at the new position */
	gint x = ((priv->direction == ARROW_RIGHT) ? (parent->allocation.width - overlay->allocation.width - OFFSET) : OFFSET);
	gint y = ((parent->allocation.height - overlay->allocation.height) / 2);
	if (!(gdk_window_get_state(gtk_widget_get_window(parent)) & GDK_WINDOW_STATE_FULLSCREEN)) y += HILDON_WINDOW_TITLEBAR_HEIGHT;

	gtk_widget_hide (overlay);
	gtk_window_move (GTK_WINDOW (overlay), x, y);
	gtk_widget_show (overlay);
}


/**
 * Everytime the timer runs out, we hide the arrow button.
 */
static gboolean
arrow_button_on_hide_timer (gpointer data)
{
    g_return_val_if_fail (data != NULL, FALSE);
    arrow_button_hide (ARROW_BUTTON (data));
    return FALSE;
}


/**
 * Shows the arrow button.
 */
static void
arrow_button_show (ArrowButton *self)
{
    g_return_if_fail (self != NULL);

    ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
    g_return_if_fail (priv != NULL);

    g_return_if_fail (GTK_IS_WIDGET (priv->overlay));

    /* Stop return button hide timeout */
    g_source_remove_by_user_data ((gpointer) self);

    /* Only show overlay if we come here through a button release event, not a button press event */
    if (priv->release_event) {

    	arrow_button_set_position (self);
    	gtk_widget_show (priv->overlay);

        /* Set the return button hide timeout */
        g_timeout_add (ARROW_BUTTON_HIDE_DELAY,
        		arrow_button_on_hide_timer, (gpointer) self);
    }
}


/**
 * This hook function is called for each mouse button press or
 * button release on the parent window.
 */
static gboolean
arrow_button_input_activity_hook (GSignalInvocationHint *ihint G_GNUC_UNUSED,
                                       guint n_param_values,
                                       const GValue *param_values,
                                       gpointer data)
{
    ArrowButton *self = ARROW_BUTTON (data);
    g_return_val_if_fail (self, FALSE);

    ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
    g_return_val_if_fail (priv != NULL, FALSE);

    GdkEventAny *event = NULL;

    if (n_param_values >= 2)
        event = (GdkEventAny*) g_value_peek_pointer (&(param_values[1]));

    g_return_val_if_fail (event, TRUE);

    guint32 time = 0;
    switch (event->type) {
        case GDK_BUTTON_PRESS:
        case GDK_BUTTON_RELEASE:
            time = ((GdkEventButton*) event)->time;
            break;
        case GDK_KEY_PRESS:
        case GDK_KEY_RELEASE:
            time = ((GdkEventKey*) event)->time;
            break;
        default:
            /* Filter out unexpected messages */
            return TRUE;
    }

    /* We're likely to get events multiple times as they're propagated, so
       filter out events that we've already seen. */
    if (time == priv->last_event_time) {
        return TRUE;
    }
    priv->last_event_time = time;

    if (event && (event->type == GDK_BUTTON_PRESS)) {
        priv->release_event = FALSE;
    } else {
        priv->release_event = TRUE;
    }

    arrow_button_show (self);

    return TRUE;
}


/**
 * This function makes the arrow button visible and hooks mouse and
 * key event signal emissions. The button is hidden after some time and
 * is reshown when ever one of the signal hooks are activated.
 *
 * @param self A ArrowButton instance.
 */
void
arrow_button_enable (ArrowButton *self)
{
    g_return_if_fail(IS_ARROW_BUTTON(self));

    ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
    g_return_if_fail (priv != NULL);

    if (priv->button_press_hook_id == 0) {
        priv->button_press_signal_id =
        	g_signal_lookup ("button-press-event", GTK_TYPE_WIDGET);
        priv->button_press_hook_id =
            g_signal_add_emission_hook (priv->button_press_signal_id, 0,
            		arrow_button_input_activity_hook,
            		(gpointer) self, NULL);
    }

    if (priv->button_release_hook_id == 0) {
        priv->button_release_signal_id =
        	g_signal_lookup ("button-release-event", GTK_TYPE_WIDGET);
        priv->button_release_hook_id =
            g_signal_add_emission_hook (priv->button_release_signal_id, 0,
            		arrow_button_input_activity_hook,
            		(gpointer) self, NULL);
    }

    arrow_button_show(self);
}


/**
 * Hides the arrow button and releases mouse and
 * key event signal emission hooks.
 *
 * @param self An ArrowButton instance.
 */
void
arrow_button_disable (ArrowButton *self)
{
    g_return_if_fail (IS_ARROW_BUTTON (self));

    ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
    g_return_if_fail (priv != NULL);

    arrow_button_hide (self);

    if (priv->button_release_hook_id > 0) {
        g_signal_remove_emission_hook (priv->button_release_signal_id,
                                       priv->button_release_hook_id);
        priv->button_release_hook_id = 0;
    }

    if (priv->button_press_hook_id > 0) {
        g_signal_remove_emission_hook (priv->button_press_signal_id,
                                       priv->button_press_hook_id);
        priv->button_press_hook_id = 0;
    }
}

/**
 * Check wether the arrow button is enabled or not.
 *
 * @param self An ArrowButton instance.
 * @return gboolean
 */
gboolean
arrow_button_is_enabled (ArrowButton *self)
{
    g_return_val_if_fail (IS_ARROW_BUTTON (self),FALSE);

    ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
    g_return_val_if_fail (priv != NULL,FALSE);

    return ((priv->button_release_hook_id > 0) && (priv->button_press_hook_id > 0));
}

/**
 * Everytime the button is clicked, emit the "clicked" signal.
 */
static gboolean
arrow_button_on_clicked (GtkWidget *widget, GdkEventButton *event G_GNUC_UNUSED, gpointer data)
{
	ArrowButton *self = ARROW_BUTTON (data);
	g_signal_emit_by_name (self, "clicked");

	return TRUE;
}


/**
 * Creates a rounded rectangle.
 */
static void
arrow_button_create_rectangle (cairo_t *ctx, double x, double y, double w, double h, double r)
{
	cairo_move_to(ctx,x + r,y);
	cairo_line_to(ctx,x + w - r,y);
	cairo_curve_to(ctx,x + w,y,x + w,y,x + w,y + r);
	cairo_line_to(ctx,x + w,y + h - r);
	cairo_curve_to(ctx,x + w,y + h,x + w,y + h,x + w - r,y + h);
	cairo_line_to(ctx,x + r,y + h);
	cairo_curve_to(ctx,x,y + h,x,y + h,x,y + h - r);
	cairo_line_to(ctx,x,y + r);
	cairo_curve_to(ctx,x,y,x,y,x + r,y);
}


/**
 * Does the actuall drawing of the semi transparent button graphic.
 */
static gboolean
arrow_button_on_expose_event (GtkWidget *widget, GdkEventExpose *event G_GNUC_UNUSED, gpointer data)
{
    cairo_t *ctx;
    GdkPixbuf *pixbuf = GDK_PIXBUF (data);

    /* Create context */
    ctx = gdk_cairo_create (widget->window);

    /* Clear surface */
    cairo_set_operator (ctx, CAIRO_OPERATOR_CLEAR);
    cairo_paint (ctx);

    /* Add rectangle */
    cairo_set_operator (ctx, CAIRO_OPERATOR_OVER);
    cairo_set_source_rgba (ctx, 0, 0, 0, 0.60);
    arrow_button_create_rectangle (ctx, 0, 0, ARROW_BUTTON_WIDTH, ARROW_BUTTON_HEIGHT, ARROW_BUTTON_CORNER_RADIUS);
    cairo_fill (ctx);

    /* Add icon */
    gdk_cairo_set_source_pixbuf (ctx, pixbuf, 15, 10);
    cairo_paint (ctx);

    /* Destroy context */
    cairo_destroy (ctx);
    return TRUE;
}


/**
 * Creates the semi transparent button graphic.
 */
static GtkWidget*
arrow_button_create_gfx (ArrowButton *self)
{
    g_return_val_if_fail(IS_ARROW_BUTTON(self), NULL);

    ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
	g_return_val_if_fail((priv != NULL), NULL);

    GdkPixbuf *pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (), ((priv->direction == ARROW_LEFT) ? ARROW_BUTTON_PREVIOUS_ICON : ARROW_BUTTON_NEXT_ICON), ARROW_BUTTON_ICON_SIZE, 0, NULL);
    GtkWidget *img = gtk_image_new_from_pixbuf (pixbuf);
    gtk_widget_show (img);
    g_object_unref (pixbuf);
    g_signal_connect (img, "expose_event", G_CALLBACK (arrow_button_on_expose_event), pixbuf);

    GtkWidget *box = gtk_event_box_new ();
    gtk_event_box_set_visible_window (GTK_EVENT_BOX(box), FALSE);
    gtk_widget_show (box);
    gtk_container_add (GTK_CONTAINER(box), img);
    g_signal_connect (box, "button-release-event", G_CALLBACK (arrow_button_on_clicked), self);

    GtkWidget *overlay = gtk_window_new (GTK_WINDOW_POPUP);
    gtk_window_set_decorated (GTK_WINDOW (overlay), FALSE);
    gtk_widget_set_size_request (overlay, ARROW_BUTTON_WIDTH, ARROW_BUTTON_HEIGHT);
    gtk_window_set_resizable (GTK_WINDOW (overlay), FALSE);
    gtk_window_set_transient_for (GTK_WINDOW (overlay), self->parent_window);
    gtk_window_set_destroy_with_parent (GTK_WINDOW (overlay), TRUE);
    gtk_container_add (GTK_CONTAINER (overlay), box);

    GdkScreen *screen = gtk_widget_get_screen (overlay);
    gtk_widget_set_colormap (overlay, gdk_screen_get_rgba_colormap (screen));

    gtk_widget_realize (overlay);

    return overlay;
}

/**
 * Called when the parent_window's is on the screen/not on the screen.
 * Only called if parent_window is a HildonWindow (or derived from it).
 *
 * We check if the window is not on the screen.
 * If not, we disable the button.
 */
static void
arrow_button_on_is_topmost_changed (GObject *object G_GNUC_UNUSED,
		                                 GParamSpec *property G_GNUC_UNUSED,
		                                 gpointer data)
{
	ArrowButton *self = ARROW_BUTTON (data);

	ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
	g_return_if_fail (priv != NULL);

	if (hildon_window_get_is_topmost (HILDON_WINDOW(self->parent_window))) {
		arrow_button_enable (self);
	}
	else {
		arrow_button_disable (self);
	}
}


/**
 * Destroys the arrow button and disconnects itself from the parent window.
 */
static void
arrow_button_destroy (GtkWidget *parent_window G_GNUC_UNUSED, ArrowButton *self)
{
	g_return_if_fail (self != NULL);

	ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
	g_return_if_fail (priv != NULL);

	if (self->parent_window != NULL) {
		g_signal_handlers_disconnect_by_func (self->parent_window, arrow_button_destroy, self);
		if (HILDON_IS_WINDOW(parent_window)) {
			g_signal_handlers_disconnect_by_func (self->parent_window, arrow_button_on_is_topmost_changed, self);
		}
	}

	arrow_button_disable (self);

	if (priv->overlay != NULL && GTK_IS_WIDGET(priv->overlay)) {
		gtk_widget_destroy (GTK_WIDGET(priv->overlay));
		priv->overlay = NULL;
	}
}


/**
 * Called when the size allocation of the parent window changes.
 * We change the position of the arrow button to always be in
 * the middle of the left or right edge.
 */
static void
arrow_button_on_parent_size_changed (GtkWidget     *widget,
                                     GtkAllocation *allocation,
                                     gpointer       user_data)
{
    g_return_if_fail (widget != NULL);
    g_return_if_fail (allocation != NULL);

    ArrowButton *self = ARROW_BUTTON(user_data);
    g_return_if_fail (self != NULL);

    ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
    g_return_if_fail (priv != NULL);

    GtkWidget *ui_win = GTK_WIDGET(priv->overlay);
    g_return_if_fail (ui_win != NULL);

    if (gdk_window_is_visible(priv->overlay->window)) {
    	arrow_button_set_position(self);
    }
}


/**
 * Default handler for the "clicked" signal. If no user handler is
 * defined we just print a debug message
 * Otherwise only the user handler is executed.
 */
static void
arrow_button_clicked_default_handler (ArrowButton *self)
{
	guint signal_id = g_signal_lookup ("clicked", TYPE_ARROW_BUTTON);
	gulong handler_id = g_signal_handler_find (self, G_SIGNAL_MATCH_ID, signal_id, 0, NULL, NULL, NULL);

	/* Run only, if no signal handler is connected */
	if (handler_id == 0) {
		g_print("DEBUG: Arrow button clicked.\n");
	}
}

/**
 * Create new arrow button instance.
 * This function attaches the arrow button to the given parent window.
 *
 * If you destroy the parent window, this ArrowButton instance get
 * destroyed as well.
 *
 * Pass it a HildonWindow (or one of its deriatives) to ensure the widget disables/
 * enables itself on focus-out/focus-in respectively.
 *
 * @param parent_window A GtkWindow instance.
 * @return New ArrowButton instance.
 */
ArrowButton *
arrow_button_new (GtkWindow *parent_window, arrow_direction_t arrow_direction)
{
	g_return_val_if_fail (parent_window != NULL, NULL);
	g_return_val_if_fail (GTK_IS_WINDOW (parent_window), NULL);

    ArrowButton *self = g_object_new (TYPE_ARROW_BUTTON, NULL);

    ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
    g_return_val_if_fail (priv != NULL, NULL);

    self->parent_window = parent_window;
    priv->direction = arrow_direction;
    priv->overlay = arrow_button_create_gfx (self);

    g_signal_connect (parent_window, "destroy",
    		G_CALLBACK(arrow_button_destroy), self);

    g_signal_connect_after (parent_window, "size-allocate",
    		G_CALLBACK(arrow_button_on_parent_size_changed), self);

    if (HILDON_IS_WINDOW(parent_window)) {
        g_signal_connect (parent_window, "notify::is-topmost",
            G_CALLBACK(arrow_button_on_is_topmost_changed), self);
    }

    arrow_button_enable(self);
    return self;
}

/**
 * Returns the GtkWidget displaying the actual overlaid button.
 *
 * @param self An instance of ArrowButton
 * @return The GtkWidget of the overlaid button. This widget belongs to ArrowButton and must not be destroyed or modified.
 */
GtkWidget *
arrow_button_get_overlay (ArrowButton *self)
{
    ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
    g_return_val_if_fail (priv != NULL, NULL);

    return priv->overlay;
}

/**
 * Returns the GtkWindow that this ArrowButton
 * is attached to.
 *
 * @param self An instance of ArrowButton
 * @return The GtkWindow to which this button is attached to
 */
GtkWindow *
arrow_button_get_window (ArrowButton *self)
{
	return self->parent_window;
}


/*
 * GObject stuff
 */

enum {
	CLICKED,
	LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

static void
arrow_button_class_init (ArrowButtonClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	klass->clicked = arrow_button_clicked_default_handler;

	/**
	 * ArrowButton::clicked
	 *
	 * Emitted when the #ArrowButton was clicked by the user.
	 */
	signals[CLICKED] =
		g_signal_new(
				"clicked",
				TYPE_ARROW_BUTTON,
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET(ArrowButtonClass, clicked),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID,
				G_TYPE_NONE,
				0);

	g_type_class_add_private (klass, sizeof (ArrowButtonPrivate));
}


static void
arrow_button_init (ArrowButton *self)
{
    g_return_if_fail (self != NULL);

    ArrowButtonPrivate *priv = ARROW_BUTTON_GET_PRIVATE (self);
    g_return_if_fail (priv != NULL);

    memset (priv, 0, sizeof (ArrowButtonPrivate));

    self->parent_window = NULL;
    priv->overlay = NULL;
    priv->release_event = TRUE;
    priv->last_event_time = 0;

    priv->direction = ARROW_RIGHT;

    priv->button_press_signal_id = 0;
    priv->button_release_signal_id = 0;

    priv->button_press_hook_id = 0;
    priv->button_release_hook_id = 0;
}

