/*******************
 * Interface builder
 *******************/
/*
 * Include declarations
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include <dbus/dbus.h>
#include <mce/mode-names.h>
#include <mce/dbus-names.h>
#include <libosso.h>
#include <hildon/hildon.h>
#include <hildon/hildon-gtk.h>
#include <gtk/gtk.h>
#include <pango/pango.h>

#include <libraw/libraw.h>

#include "appdata.h"
#include "interface.h"
#include "callbacks.h"
#include "helpers.h"
#include "he-fullscreen-button.h"
#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

/*
 * Main application's window
 */
void create_app(app_data_t *myapp) {
	HildonAppMenu *appmenu;
	GtkWidget *openitem;
	GtkWidget *imgdetitem;
	GtkWidget *fullscreenitem;
	GtkWidget *aboutitem;
	GtkWidget *previtem;
	GtkWidget *nextitem;
	GtkWidget *panarea;
    GtkWidget *eventbox;
    HeFullscreenButton *fsbutton;

    /* Enable accelerometers */
    enable_accelerometers(myapp,TRUE);

    /* Create the hildon program and setup the title */
    myapp->app = HILDON_PROGRAM(hildon_program_get_instance());
    g_set_application_name("RAW Viewer");

    /* Create HildonWindow and set it to HildonProgram */
    myapp->mainwin = HILDON_WINDOW (hildon_window_new());
    gtk_window_set_title(GTK_WINDOW (myapp->mainwin),g_get_application_name());
    hildon_gtk_window_set_portrait_flags(GTK_WINDOW (myapp->mainwin), HILDON_PORTRAIT_MODE_SUPPORT);
    hildon_program_add_window(myapp->app, myapp->mainwin);

    /* Create the main menu */
    appmenu = HILDON_APP_MENU(hildon_app_menu_new());

	/* Add menu buttons */
    previtem = gtk_button_new_with_label("Previous image");
    gtk_widget_set_sensitive(previtem,FALSE);
    g_signal_connect_after(G_OBJECT (previtem),"clicked",G_CALLBACK (load_prev_image),myapp);
    hildon_app_menu_insert(appmenu,GTK_BUTTON(previtem),0);

    nextitem = gtk_button_new_with_label("Next image");
    gtk_widget_set_sensitive(nextitem,FALSE);
    g_signal_connect_after(G_OBJECT (nextitem),"clicked",G_CALLBACK (load_next_image),myapp);
    hildon_app_menu_insert(appmenu,GTK_BUTTON(nextitem),1);

    openitem = gtk_button_new_with_label("Open image");
    g_signal_connect_after(G_OBJECT (openitem),"clicked",G_CALLBACK (open_image_file),myapp);
    hildon_app_menu_insert(appmenu,GTK_BUTTON(openitem),2);

    imgdetitem = gtk_button_new_with_label("View details");
    g_signal_connect_after(G_OBJECT (imgdetitem),"clicked",G_CALLBACK (create_image_details),myapp);
    hildon_app_menu_insert(appmenu,GTK_BUTTON (imgdetitem),3);

    fullscreenitem = gtk_button_new_with_label("Go fullscreen");
    g_signal_connect_after(G_OBJECT (fullscreenitem),"clicked",G_CALLBACK (toggle_fullscreen_mode),myapp);
    hildon_app_menu_insert(appmenu,GTK_BUTTON (fullscreenitem),4);

    aboutitem = gtk_button_new_with_label("About");
    g_signal_connect_after(G_OBJECT (aboutitem),"clicked",G_CALLBACK (create_about),myapp);
    hildon_app_menu_insert(appmenu,GTK_BUTTON(aboutitem),5);

    /* Add the main manu to the app itself */
    gtk_widget_show_all(GTK_WIDGET (appmenu));
    hildon_window_set_app_menu(HILDON_WINDOW (myapp->mainwin),appmenu);

	/* Add a pannable area with the image container inside */
    panarea = GTK_WIDGET (hildon_pannable_area_new());
    g_object_set(G_OBJECT (panarea),"mov-mode",HILDON_MOVEMENT_MODE_BOTH,NULL);
    gtk_widget_show(GTK_WIDGET (panarea));
    gtk_container_add(GTK_CONTAINER (myapp->mainwin),panarea);
    eventbox = gtk_event_box_new();
    gtk_widget_show(eventbox);
    gtk_widget_set_events(eventbox,GDK_BUTTON_PRESS | GDK_BUTTON_RELEASE | GDK_KEY_PRESS);
    myapp->image = GTK_WIDGET (gtk_image_new());
    gtk_widget_show(GTK_WIDGET (myapp->image));
    gtk_container_add(GTK_CONTAINER (eventbox),myapp->image);
    hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA (panarea),eventbox);

    /* Show everything */
    gtk_widget_show_all(GTK_WIDGET (myapp->mainwin));

    /* Add Fullscreen button */
    fsbutton = he_fullscreen_button_new(GTK_WINDOW (myapp->mainwin));
    g_signal_connect(G_OBJECT (fsbutton),"clicked",G_CALLBACK (toggle_fullscreen_mode),myapp);

    /* Load configuration from GConf */
    load_configuration(myapp);

    /* Ungrab volume keys */
    ungrab_volume_keys(GTK_WIDGET (myapp->mainwin));

    /* Connect some signals */
    g_signal_connect(G_OBJECT (myapp->mainwin),"destroy",
    		G_CALLBACK (destroy_app),myapp);
    g_signal_connect(G_OBJECT (myapp->mainwin),"configure_event",
    		G_CALLBACK (detect_main_window_changes),myapp);
    g_signal_connect(G_OBJECT (eventbox),"button_press_event",
    		G_CALLBACK (detect_button_press_on_image),myapp);
    g_signal_connect(G_OBJECT (eventbox),"button_release_event",
        	G_CALLBACK (detect_button_press_on_image),myapp);
    g_signal_connect(G_OBJECT(myapp->mainwin),"key_press_event",
			G_CALLBACK (detect_hardware_key_press),myapp);

    /* Connect DBus events handler */
	myapp->sysbus = dbus_bus_get (DBUS_BUS_SYSTEM,NULL);
	if (myapp->sysbus) {
		dbus_bus_add_match (myapp->sysbus,MCE_SIGNAL_MATCH,NULL);
		dbus_connection_add_filter(myapp->sysbus,(DBusHandleMessageFunction)detect_mce_signal,myapp,NULL);
	}

    /* Automatically show the "Open ..." dialog if there's no filename provided at start time or load last image opened */
    if (!myapp->currfname || !g_file_test(g_strconcat(myapp->currdir,"/",myapp->currfname,NULL),G_FILE_TEST_EXISTS)) open_image_file(GTK_WIDGET (myapp->mainwin),myapp);
    else load_image(myapp,g_strconcat(myapp->currdir,"/",myapp->currfname,NULL));

    /* Finally take a screenshot: say cheese :-) */
    hildon_gtk_window_take_screenshot(GTK_WINDOW (myapp->mainwin),TRUE);
}

/*
 * Create the About dialog
 */
void create_about(GtkWidget *callerobj,app_data_t *myapp) {
	GtkWidget *abdialog;
	GtkWidget *appicon;
	GtkWidget *copyright;
	GtkWidget *separator;
	GtkWidget *panarea;
	GtkWidget *dispmessage;
	gchar *message;
	const char **cameralist;
	gint i;

	abdialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW (abdialog),g_strconcat("About ",PACKAGE," ",VERSION,NULL));
	gtk_window_set_default_size(GTK_WINDOW (abdialog),-1,400);
	appicon = GTK_WIDGET (gtk_image_new_from_file(g_strconcat(HILDONPIXMAPDIR,"/",PACKAGE,".png",NULL)));
	gtk_box_pack_start(GTK_BOX (GTK_DIALOG (abdialog)->vbox),appicon,FALSE,FALSE,10);
	copyright = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL (copyright),"<span size=\"small\" style=\"italic\">(C) 2009 Luca Donaggio (donaggio@gmail.com)</span>");
	gtk_box_pack_start(GTK_BOX (GTK_DIALOG (abdialog)->vbox),copyright,FALSE,FALSE,10);
	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX (GTK_DIALOG (abdialog)->vbox),separator,FALSE,FALSE,10);
	panarea = GTK_WIDGET (hildon_pannable_area_new());
	dispmessage = gtk_label_new(NULL);
	gtk_label_set_line_wrap(GTK_LABEL (dispmessage),TRUE);
	message = g_strconcat("<b>",PACKAGE,"</b> is an image viewer for the Maemo platform supporting RAW image file formats as those produced by modern DSLRs.\n\n"
						  "For an exahustive list of supported cameras please see below.\n\n",
						  "<b>",PACKAGE,"</b> is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License as published by the Free Software Foundation; either version 2.1 of the License, or (at your option) any later version.\n\n",
						  "<b>",PACKAGE,"</b> is based on LibRaw ",libraw_version()," which in turn is (C) 2008-2009 LibRaw LLC (info@libraw.org)\n\n",
						  "Supported cameras:\n\n<small>",NULL);
	cameralist = libraw_cameraList();
	for(i = 0;i < libraw_cameraCount();i++) message = g_strconcat(message,"\t",cameralist[i],"\n",NULL);
	message = g_strconcat(message,"</small>",NULL);
	gtk_label_set_markup(GTK_LABEL (dispmessage),message);
	hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA (panarea),dispmessage);
	gtk_box_pack_end_defaults(GTK_BOX (GTK_DIALOG (abdialog)->vbox),panarea);
	gtk_widget_show_all(abdialog);
	gtk_dialog_run(GTK_DIALOG (abdialog));
	gtk_widget_destroy(abdialog);
}

/*
 * Create the image details widget
 */
void create_image_details(GtkWidget *callerobj,app_data_t *myapp) {
	GdkScreen *screen;
	PangoLayout *textbuff;
	cairo_t *cr;
	gint offsetx, offsety, padding, txtwidth, txtheight;

	/* For the widget itself let's use a GtkWindow */
	if ((myapp->imgparamwin != NULL) & GTK_IS_WINDOW(myapp->imgparamwin)) draw_image_details(myapp->imgparamwin,NULL,myapp);
	else {
		myapp->imgparamwin = gtk_window_new(GTK_WINDOW_POPUP);
		gtk_window_set_decorated(GTK_WINDOW (myapp->imgparamwin),FALSE);
		gtk_widget_set_app_paintable(myapp->imgparamwin,TRUE);
		screen = gtk_widget_get_screen(myapp->imgparamwin);
		gtk_widget_set_colormap(myapp->imgparamwin,gdk_screen_get_rgba_colormap(screen));
		gtk_window_set_transient_for(GTK_WINDOW (myapp->imgparamwin),GTK_WINDOW (myapp->mainwin));
		gtk_window_set_destroy_with_parent(GTK_WINDOW (myapp->imgparamwin),TRUE);
		gtk_window_set_modal(GTK_WINDOW (myapp->imgparamwin),TRUE);
		gtk_widget_realize(myapp->imgparamwin);
		/* Get the Cairo context and the overall dimensions of the text to be displayed */
		cr = gdk_cairo_create(GDK_DRAWABLE (myapp->imgparamwin->window));
		textbuff = pango_cairo_create_layout(cr);
		pango_layout_set_markup(textbuff,myapp->imgparam,-1);
		pango_layout_get_pixel_size(textbuff,&txtwidth,&txtheight);
		cairo_destroy(cr);
		g_object_unref(textbuff);
		/* Show the widget */
		padding = 20;
		offsetx = (gint)round((double)(gdk_screen_get_width(screen) - (txtwidth + padding)) / 2);
		offsety = (gint)round((double)(gdk_screen_get_height(screen) - (txtheight + padding)) / 2);
		gtk_widget_set_size_request(myapp->imgparamwin,txtwidth + padding,txtheight + padding);
		gtk_window_set_resizable(GTK_WINDOW (myapp->imgparamwin),FALSE);
		/* gdk_window_set_override_redirect(myapp->imgparamwin->window,TRUE); */
		/* gdk_window_reparent(myapp->imgparamwin->window,gtk_widget_get_window(GTK_WIDGET (myapp->mainwin)),offsetx,offsety); */
		gtk_window_move(GTK_WINDOW (myapp->imgparamwin),offsetx,offsety);
		gtk_grab_add(myapp->imgparamwin);
		gtk_widget_show(myapp->imgparamwin);
		/* Actual drawing needs to take place when the widget is exposed */
		g_signal_connect_after(myapp->imgparamwin,"expose-event",
				G_CALLBACK (draw_image_details),myapp);
		g_signal_connect_after(myapp->imgparamwin,"button-press-event",
				G_CALLBACK (gtk_widget_destroy),NULL);
	}
}
