/********************
 * Callback functions
 ********************/
/*
 * Include declarations
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <glob.h>

#include <dbus/dbus.h>
#include <mce/mode-names.h>
#include <mce/dbus-names.h>
#include <libosso.h>
#include <hildon/hildon.h>
#include <hildon/hildon-defines.h>
#include <gtk/gtk.h>
#include <gdk/gdkpixbuf.h>
#include <gdk/gdkx.h>
#include <X11/Xatom.h>
#include <glib/gprintf.h>
#include <glib/gfileutils.h>
#include <gconf/gconf-client.h>

#include <libraw/libraw.h>

#include "appdata.h"
#include "interface.h"
#include "callbacks.h"
#include "helpers.h"
#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

/*
 * Load image from filename
 */
void load_image(app_data_t *myapp, gchar *imgfname) {
	GtkWidget *errdialog;
	GdkPixbufLoader *imgloader;
	libraw_data_t *rawdata;
	gboolean imgerr = FALSE;
	gint shutternum = 0;
	gchar *shutter;
	const gchar *paramformat = "<span size=\"medium\" face=\"arial\">Filename: <b>%s</b>\nCamera model: <b>%s %s</b>\nShutter speed: <b>%s s</b>\nAperture: <b>f/%-.1f</b>\nISO: <b>%-.0f</b>\nFocal length: <b>%-.0f mm</b></span>";

	g_free(myapp->currfname);
	myapp->currfname = g_strndup(g_path_get_basename(imgfname),MAXFNLEN);
	gtk_window_set_title(GTK_WINDOW (myapp->mainwin),g_strconcat(g_get_application_name()," - ",myapp->currfname,NULL));
	hildon_gtk_window_set_progress_indicator(GTK_WINDOW (myapp->mainwin),1);
	while (gtk_events_pending()) gtk_main_iteration();
	rawdata = libraw_init(0);
	/* DEBUG */
	/* g_print("Supported RAW formats: %s\n",g_strjoinv("\n",(gchar**) libraw_cameraList())); */
	if (libraw_open_file(rawdata,imgfname) == LIBRAW_SUCCESS) {
		if (libraw_unpack_thumb(rawdata) == LIBRAW_SUCCESS) {
			imgloader = gdk_pixbuf_loader_new();
			if (gdk_pixbuf_loader_write(imgloader,(guchar *) rawdata->thumbnail.thumb,(gsize) rawdata->thumbnail.tlength,NULL)) {
				switch(rawdata->sizes.flip) {
					case 3:
						myapp->imgbuffer = gdk_pixbuf_rotate_simple(gdk_pixbuf_loader_get_pixbuf(imgloader),GDK_PIXBUF_ROTATE_UPSIDEDOWN);
						break;
					case 5:
						myapp->imgbuffer = gdk_pixbuf_rotate_simple(gdk_pixbuf_loader_get_pixbuf(imgloader),GDK_PIXBUF_ROTATE_COUNTERCLOCKWISE);
						break;
					case 6:
						myapp->imgbuffer = gdk_pixbuf_rotate_simple(gdk_pixbuf_loader_get_pixbuf(imgloader),GDK_PIXBUF_ROTATE_CLOCKWISE);
						break;
					case 0:
					default:
						myapp->imgbuffer = gdk_pixbuf_loader_get_pixbuf(imgloader);
						break;
				}
			} else imgerr = TRUE;
			gdk_pixbuf_loader_close(imgloader,NULL);
		} else imgerr = TRUE;
	} else imgerr = TRUE;
	while (gtk_events_pending()) gtk_main_iteration();
	hildon_gtk_window_set_progress_indicator(GTK_WINDOW (myapp->mainwin),0);
	g_free(myapp->imgparam);
	if (!imgerr) {
		shutternum = (gint) ((rawdata->other.shutter < 1) ? round(1 / rawdata->other.shutter) : round(rawdata->other.shutter));
		shutter = g_strdup_printf(((rawdata->other.shutter < 1) ? "1/%d" : "%d"),shutternum);
		myapp->imgparam = g_strdup_printf(paramformat,myapp->currfname,rawdata->idata.make,rawdata->idata.model,shutter,rawdata->other.aperture,rawdata->other.iso_speed,rawdata->other.focal_len);
		g_free(shutter);
		scale_image(GTK_WIDGET (myapp->mainwin),myapp);
	} else {
		g_object_unref(myapp->imgbuffer);
		gtk_image_set_from_pixbuf(GTK_IMAGE (myapp->image),NULL);
		errdialog = gtk_message_dialog_new(GTK_WINDOW (myapp->mainwin),
											GTK_DIALOG_DESTROY_WITH_PARENT,
											GTK_MESSAGE_ERROR,
											GTK_BUTTONS_CLOSE,
											"Error loading image '%s'",
											g_path_get_basename(imgfname));
		 gtk_dialog_run(GTK_DIALOG (errdialog));
		 gtk_widget_destroy(errdialog);
	}
	libraw_close(rawdata);
}

/*
 * Scale current image according to selected zoom factor
 */
gboolean scale_image(GtkWidget *callerobj,app_data_t *myapp) {
	const gchar *zoomtext;
	GdkPixbuf *tmpbuffer = NULL;
	gint dispwidth, dispheight, imgwidth, imgheight;
	guint scaledwidth = 0, scaledheight = 0;
	gdouble dispratio, imgratio;
	GdkInterpType scalingmode = GDK_INTERP_BILINEAR;

	/* DEBUG */
	/* g_print("Scale image with buffer: %d (%d)\n",myapp->imgbuffer,G_IS_OBJECT (G_OBJECT (myapp->imgbuffer))); */
	if (myapp->imgbuffer != NULL) {
		hildon_gtk_window_set_progress_indicator(GTK_WINDOW (myapp->mainwin),1);
		while (gtk_events_pending()) gtk_main_iteration();
		/* Get actual image widget dimensions */
		gtk_window_get_size(GTK_WINDOW (myapp->mainwin),&dispwidth,&dispheight);
		dispratio = (gdouble) dispwidth / (gdouble) dispheight;
		imgwidth = gdk_pixbuf_get_width(myapp->imgbuffer);
		imgheight = gdk_pixbuf_get_height(myapp->imgbuffer);
		imgratio = (gdouble) imgwidth / (gdouble) imgheight;
		scaledwidth = scaledheight = 0;
		/* Eventually scale image */
		if ((imgwidth > dispwidth) || (imgheight > dispheight)) {
			switch (myapp->zoomlevel) {
				case FIT_SCREEN:
					if (imgratio >= dispratio) {
						scaledwidth = dispwidth;
						scaledheight = (guint) round((double) dispwidth * (1 / imgratio));
					} else {
						scaledwidth = (guint) round((double) (dispheight * imgratio));
						scaledheight = dispheight;
					}
					scalingmode = GDK_INTERP_NEAREST;
					zoomtext = "";
					break;
				case ONE_THIRD_RESOLUTION:
					scaledwidth = (guint) round((double) imgwidth * 0.33);
					scaledheight = (guint) round((double) imgheight * 0.33);
					zoomtext = " (33%)";
					break;
				case HALF_RESOLUTION:
					scaledwidth = (guint) round((double) imgwidth * 0.5);
					scaledheight = (guint) round((double) imgheight * 0.5);
					zoomtext = " (50%)";
					break;
				case FULL_RESOLUTION:
				default:
					zoomtext = " (Full Res.)";
					break;
			}
			if ((scaledwidth > 0) & (scaledheight > 0)) {
				tmpbuffer = gdk_pixbuf_scale_simple(myapp->imgbuffer,scaledwidth,scaledheight,scalingmode);
			}
		} else zoomtext = "";
		gtk_image_set_from_pixbuf(GTK_IMAGE (myapp->image), (tmpbuffer ? tmpbuffer : myapp->imgbuffer));
		while (gtk_events_pending()) gtk_main_iteration();
		hildon_gtk_window_set_progress_indicator(GTK_WINDOW (myapp->mainwin),0);
		gtk_window_set_title(GTK_WINDOW (myapp->mainwin),g_strconcat(g_get_application_name()," - ",myapp->currfname,zoomtext,NULL));
		if (tmpbuffer) g_object_unref(tmpbuffer);
		/* DEBUG */
		/* g_print("\n\nOriginal dim.: %dx%d (%f) - Window dim.: %dx%d (%f) - Scaled dim.: %dx%d\n\n",imgwidth,imgheight,imgratio,dispwidth,dispheight,dispratio,scaledwidth,scaledheight); */
	}
	return TRUE;
}

/*
 * Draw left / right arrows
 */
gboolean draw_prev_next_arrow(app_data_t *myapp) {
	GtkImageType imgtype;
	cairo_t *cr;
	GdkPixmap *tmppm;
	GdkBitmap *tmpbm;
	gint winwidth, winheight, pointerx, pointery, x0, y0, r, m, w, h;
	GTimeVal now;

	g_get_current_time(&now);
	gdk_window_get_pointer(gtk_widget_get_root_window(GTK_WIDGET (myapp->mainwin)),&pointerx,&pointery,NULL);
	if (myapp->actioninprog != NONE) {
		if (((now.tv_sec - myapp->pressbuttontime) >= MINBUTPRESSTIME) &
			(abs(round((double)(pointerx - myapp->pressbuttonx))) <= MAXBUTPRESSDISTANCE) &
			(abs(round((double)(pointery - myapp->pressbuttony))) <= MAXBUTPRESSDISTANCE))
		{
			imgtype = gtk_image_get_storage_type(GTK_IMAGE (myapp->image));
			if (imgtype != GTK_IMAGE_EMPTY) {
				/* If the image is zoomed in, zoom out to FIT_SCREEN before drawing the arrow */
				if (myapp->zoomlevel != FIT_SCREEN) {
					myapp->zoomlevel = FIT_SCREEN;
					scale_image(GTK_WIDGET (myapp->mainwin),myapp);
				}
				if (imgtype == GTK_IMAGE_PIXBUF) gdk_pixbuf_render_pixmap_and_mask(gtk_image_get_pixbuf(GTK_IMAGE (myapp->image)),&tmppm,&tmpbm,0);
				else if (imgtype == GTK_IMAGE_PIXMAP) {
					gtk_image_get_pixmap(GTK_IMAGE (myapp->image),&tmppm,&tmpbm);
					tmppm = g_object_ref(tmppm);
					tmpbm = g_object_ref(tmpbm);
				} else return FALSE;
				gdk_drawable_get_size(GDK_DRAWABLE (tmppm),&winwidth,&winheight);
				r = 5;
				m = 15;
				w = h = 40;
				if (myapp->actioninprog == NEXT) x0 = (winwidth - w - m);
				else x0 = m;
				y0 = round((double)((winheight - h) / 2));
				cr = gdk_cairo_create(GDK_DRAWABLE (tmppm));

				/* Draw a rounded rectangle */
				cairo_set_source_rgba(cr,0,0,0,0.4);
				cairo_move_to(cr,x0,y0 - r);
				cairo_line_to(cr,x0 + w,y0 -r);
				cairo_curve_to(cr,x0 + w + r,y0 -r,x0 + w + r,y0 -r,x0 + w + r,y0);
				cairo_line_to(cr,x0 + w + r,y0 + h);
				cairo_curve_to(cr,x0 + w + r,y0 + h + r,x0 + w + r,y0 + h + r,x0 + w,y0 + h + r);
				cairo_line_to(cr,x0,y0 + h + r);
				cairo_curve_to(cr,x0 -r,y0 + h + r,x0 -r,y0 + h + r,x0 -r,y0 + h);
				cairo_line_to(cr,x0 - r,y0);
				cairo_curve_to(cr,x0 - r,y0 -r,x0 -r,y0 -r,x0,y0 - r);
				cairo_close_path(cr);
				cairo_fill(cr);
				/* Draw an arrow according to the action being performed*/
				cairo_new_path(cr);
				cairo_set_source_rgba(cr,1,1,1,0.6);
				if (myapp->actioninprog == NEXT) {
					cairo_move_to(cr,x0,y0);
					cairo_line_to(cr,x0 + w,y0 + (h / 2));
					cairo_line_to(cr,x0,y0 + h);
					cairo_line_to(cr,x0,y0);
				} else {
					cairo_move_to(cr,x0,y0 + (h / 2));
					cairo_line_to(cr,x0 + w,y0);
					cairo_line_to(cr,x0 + w,y0 + h);
					cairo_line_to(cr,x0,y0 + (h / 2));
				}
				cairo_close_path(cr);
				cairo_fill(cr);

				cairo_destroy(cr);
				gtk_image_set_from_pixmap(GTK_IMAGE (myapp->image),tmppm,tmpbm);
				g_object_unref(tmppm);
				/*
				 * Un-referencing tmpbm is not needed
				 * g_object_unref(tmpbm);
				 *
				 */
			}
			return FALSE;
		} else return TRUE;
	} else return FALSE;
}

/*
 * Store image file names in current directory into application's data structure
 */
void store_filenames_in_dir(app_data_t *myapp) {
	gchar *tmpstr;
	glob_t allimages;
	guint i;
	GList *li = NULL;

	/* First free the allocated memory in the list */
	if (myapp->allimgindir != NULL) {
		for(li = myapp->allimgindir;li != NULL;li = g_list_next(li))
			g_free(li->data);
		g_list_free(li);
		g_list_free(myapp->allimgindir);
		myapp->allimgindir = NULL;
	}
	/* Then search for other image files */
	if (myapp->currdir != NULL) {
		tmpstr = g_strconcat(myapp->currdir,"/*.{",g_strjoinv(",",(gchar**) myapp->rawext),"}",NULL);
		if (glob(tmpstr,GLOB_ERR | GLOB_BRACE,NULL,&allimages) == 0) {
			for(i = 0;i < allimages.gl_pathc;i++) {
				myapp->allimgindir = g_list_prepend(myapp->allimgindir,g_strndup(g_path_get_basename((gchar*) allimages.gl_pathv[i]),MAXFNLEN));
			}
			if (myapp->allimgindir != NULL) myapp->allimgindir = g_list_reverse(myapp->allimgindir);
		}
		if (allimages.gl_pathc <= 1) toggle_prev_next_menu_button("disable",myapp);
		else toggle_prev_next_menu_button("enable",myapp);
		globfree(&allimages);
		g_free(tmpstr);
	}
	/* DEBUG */
	/* for(li = myapp->allimgindir;li != NULL;li = g_list_next(li))
		g_print("File in list: %s\n",li->data);
	g_list_free(li);*/
}

/*
 * Ungrab volume keys
 */
void ungrab_volume_keys(GtkWidget *widget) {
    /* Tell maemo-status-volume daemon to ungrab keys */
    unsigned long val = 1; /* ungrab, use 0 to grab */
    Atom atom;
    GdkDisplay *display = NULL;
    display = gtk_widget_get_display(widget);
    atom = gdk_x11_get_xatom_by_name_for_display(display,"_HILDON_ZOOM_KEY_ATOM");
    XChangeProperty (GDK_DISPLAY_XDISPLAY (display),
                     GDK_WINDOW_XID (GDK_WINDOW (widget->window)),atom,XA_INTEGER,32,
                     PropModeReplace,(unsigned char *)&val,1);
}

/*
 * Store configuration values into GConf
 */
void save_configuration(app_data_t *myapp) {
	GConfClient* gcClient = NULL;

	gcClient = gconf_client_get_default();
	/* Store the values. */
	if (GCONF_IS_CLIENT (gcClient)) {
		if (!gconf_client_set_string(gcClient,g_strconcat(GC_ROOT,"currdir",NULL),(const gchar *)myapp->currdir,NULL)) {
			g_printerr("WARNING: Failed to set %scurrdir to %s\n",GC_ROOT,myapp->currdir);
		}
		if (!gconf_client_set_string(gcClient,g_strconcat(GC_ROOT,"currfname",NULL),(const gchar *)myapp->currfname,NULL)) {
			g_printerr("WARNING: Failed to set %scurrfname to %s\n",GC_ROOT,myapp->currfname);
		}
		if (!gconf_client_set_int(gcClient,g_strconcat(GC_ROOT,"currfindex",NULL),myapp->currfindex,NULL)) {
			g_printerr("WARNING: Failed to set %scurrfindex to %d\n",GC_ROOT,myapp->currfindex);
		}
		if (!gconf_client_set_int(gcClient,g_strconcat(GC_ROOT,"zoomlevel",NULL),myapp->zoomlevel,NULL)) {
			g_printerr("WARNING: Failed to set %szoomlevel to %d\n",GC_ROOT,myapp->zoomlevel);
		}
		g_object_unref(gcClient);
	} else g_printerr("WARNING: Failed to get default GConf client.\n");
}

/*
 * Restore configuration from GConf values
 */
void load_configuration(app_data_t *myapp) {
	GConfClient* gcClient = NULL;

	gcClient = gconf_client_get_default();
	/* Load the values. */
	if (GCONF_IS_CLIENT (gcClient)) {
		myapp->currdir = gconf_client_get_string(gcClient,g_strconcat(GC_ROOT,"currdir",NULL),NULL);
		myapp->currfname = gconf_client_get_string(gcClient,g_strconcat(GC_ROOT,"currfname",NULL),NULL);
		myapp->currfindex = gconf_client_get_int(gcClient,g_strconcat(GC_ROOT,"currfindex",NULL),NULL);
		myapp->zoomlevel = gconf_client_get_int(gcClient,g_strconcat(GC_ROOT,"zoomlevel",NULL),NULL);
		g_object_unref(gcClient);
		store_filenames_in_dir(myapp);
	} else g_printerr("WARNING: Failed to get default GConf client.\n");
}

/*
 * Enable or disable accelerometers
 */
void enable_accelerometers(app_data_t *myapp,gboolean mode) {
	if (osso_rpc_run_system(myapp->ossocont,MCE_SERVICE,MCE_REQUEST_PATH,MCE_REQUEST_IF,
			(mode ? MCE_ACCELEROMETER_ENABLE_REQ : MCE_ACCELEROMETER_DISABLE_REQ),NULL,DBUS_TYPE_INVALID) != OSSO_OK)
		g_printerr("WARNING: Cannot %s accelerometers\n",(mode ? "enable" : "disable"));
}
