// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "settingscontroller.h"

#include "data/settings.h"
#include "ui/mainwindow.h"
#include "ui/optionsdialog.h"

#include <QFile>
#include <QXmlStreamReader>
#include <QXmlStreamWriter>
#include <QDir>
#include <iostream>

SettingsController::SettingsController(MainWindow *main_window, Settings *settings)
    : _main_window(main_window),
    _settings(settings)
{
}

enum State {
    STATE_DOCUMENT,
    STATE_SETTINGS,
    STATE_LOCATION,
    STATE_LOCATION_NAME,
    STATE_PREVIOUS_LOCATIONS,
    STATE_PREVIOUS_LOCATION,
    STATE_PREVIOUS_LOCATION_NAME,
    STATE_GPS,
    STATE_SEARCH,
    STATE_ORIENTATION
};

void SettingsController::Load()
{
    QFile file(GetSettingsFileName());
    if (file.open(QIODevice::ReadOnly)) {
        QXmlStreamReader xml(&file);
        State state = STATE_DOCUMENT;
        Location location;
        Settings::Locations previous_locations;
        Location previous_location;
        while (!xml.atEnd()) {
            QXmlStreamReader::TokenType token = xml.readNext();
            if (token == QXmlStreamReader::StartElement) {
                if (state == STATE_DOCUMENT && xml.name() == "settings") {
                    state = STATE_SETTINGS;
                } else if (state == STATE_SETTINGS && xml.name() == "location") {
                    state = STATE_LOCATION;
                } else if (state == STATE_LOCATION && xml.name() == "name") {
                    state = STATE_LOCATION_NAME;
                } else if (state == STATE_SETTINGS && xml.name() == "previous-locations") {
                    state = STATE_PREVIOUS_LOCATIONS;
                } else if (state == STATE_PREVIOUS_LOCATIONS && xml.name() == "location") {
                    state = STATE_PREVIOUS_LOCATION;
                } else if (state == STATE_PREVIOUS_LOCATION && xml.name() == "name") {
                    state = STATE_PREVIOUS_LOCATION_NAME;
                } else if (state == STATE_SETTINGS && xml.name() == "gps") {
                    QString enabled = xml.attributes().value("enabled").toString();
                    _settings->SetGPSEnabled(enabled == "yes");
                    state = STATE_GPS;
                } else if (state == STATE_SETTINGS && xml.name() == "search") {
                    state = STATE_SEARCH;
                } else if (state == STATE_SETTINGS && xml.name() == "orientation") {
                    state = STATE_ORIENTATION;
                } else {
                    state = STATE_DOCUMENT;
                }
            } else if (token == QXmlStreamReader::EndElement) {
                if (state == STATE_ORIENTATION) {
                    state = STATE_SETTINGS;
                } else if (state == STATE_SEARCH) {
                    state = STATE_SETTINGS;
                } else if (state == STATE_GPS) {
                    state = STATE_SETTINGS;
                } else if (state == STATE_PREVIOUS_LOCATION_NAME) {
                    state = STATE_PREVIOUS_LOCATION;
                } else if (state == STATE_PREVIOUS_LOCATION) {
                    if (previous_location.IsValid()) {
                        previous_locations.append(previous_location);
                    }
                    state = STATE_PREVIOUS_LOCATIONS;
                } else if (state == STATE_PREVIOUS_LOCATIONS) {
                    state = STATE_SETTINGS;
                } else if (state == STATE_LOCATION_NAME) {
                    state = STATE_LOCATION;
                } else if (state == STATE_LOCATION) {
                    state = STATE_SETTINGS;
                } else if (state == STATE_SETTINGS) {
                    state = STATE_DOCUMENT;
                }
            } else if (token == QXmlStreamReader::Characters) {
                if (state == STATE_LOCATION_NAME) {
                    location.SetLocationName(xml.text().toString());
                } else if (state == STATE_PREVIOUS_LOCATION_NAME) {
                    previous_location.SetLocationName(xml.text().toString());
                } else if (state == STATE_SEARCH) {
                    if (xml.text() == "theaters") {
                        _settings->SetSearchObjectsType(Settings::THEATERS);
                    } else if (xml.text() == "movies") {
                        _settings->SetSearchObjectsType(Settings::MOVIES);
                    }
                } else if (state == STATE_ORIENTATION) {
                    if (xml.text() == "landscape") {
                        _settings->SetOrientationMode(Settings::LANDSCAPE);
                    } else if (xml.text() == "portrait") {
                        _settings->SetOrientationMode(Settings::PORTRAIT);
                    } else if (xml.text() == "autorotation") {
                        _settings->SetOrientationMode(Settings::AUTOROTATION);
                    }
                }
            }
        }
        if (xml.hasError()) {
            // TODO error message unable to read setting files?
            file.close();
            QDir().remove(GetSettingsFileName());
        } else {
            _settings->SetLocation(location);
            _settings->SetPreviousLocations(previous_locations);
        }
    }
}

void SettingsController::Save()
{
    QFile file(GetSettingsFileName());
    if (file.open(QIODevice::WriteOnly)) {
        QXmlStreamWriter xml(&file);
        xml.setAutoFormatting(true);
        xml.writeStartDocument();
        xml.writeStartElement("settings");
        xml.writeStartElement("location");
        xml.writeTextElement("name", _settings->GetLocation().GetLocationName());
        xml.writeEndElement(); // </location>
        xml.writeStartElement("previous-locations");
        Q_FOREACH(Location location, _settings->GetPreviousLocations()) {
            xml.writeStartElement("location");
            xml.writeTextElement("name", location.GetLocationName());
            xml.writeEndElement();
        }
        xml.writeEndElement(); // </previous-locations>
        QString v;
        xml.writeStartElement("gps");
        xml.writeAttribute("enabled", _settings->IsGPSEnabled() ? "yes" : "no");
        xml.writeEndElement(); // </gps>
        switch (_settings->GetSearchObjectsType()) {
        case Settings::THEATERS:
            v = "theaters";
            break;
        case Settings::MOVIES:
            v = "movies";
            break;
        }
        xml.writeTextElement("search", v);
        switch (_settings->GetOrientationMode()) {
        case Settings::LANDSCAPE:
            v = "landscape";
            break;
        case Settings::PORTRAIT:
            v = "portrait";
            break;
        case Settings::AUTOROTATION:
            v = "autorotation";
            break;
        }
        xml.writeTextElement("orientation", v);
        xml.writeEndElement(); // </settings>
        xml.writeEndDocument();
    } else {
        // TODO error message unable to create file
        std::cout << "Failed to open file " << qPrintable(GetSettingsFileName()) << std::endl;
    }
}

void SettingsController::EmitInitialSettings()
{
    emit SettingsChanged(*_settings);
}

void SettingsController::OpenSettingsDialog()
{
#ifdef Q_WS_MAEMO_5
    OptionsDialog *dialog = new OptionsDialog(_main_window);
    connect(dialog, SIGNAL(Accept(Settings)), this, SLOT(AcceptSettingsDialog(Settings)));
    connect(dialog, SIGNAL(Cancel(Settings)), this, SLOT(CancelSettingsDialog(Settings)));
    dialog->SetSettings(*_settings);
    dialog->show();
    // dialog deletes itself
#endif
}

void SettingsController::AcceptSettingsDialog(const Settings &settings)
{
    *_settings = settings;
    emit SettingsChanged(settings);
}

void SettingsController::CancelSettingsDialog(const Settings &settings)
{
    Q_UNUSED(settings);
    // do nothing
}

QString SettingsController::GetSettingsFileName()
{
    QString dirpath = QDir::homePath() + "/.movie-schedule";
    QDir().mkpath(dirpath);
    return dirpath + "/settings.xml";
}
