// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "theatercontroller.h"

#include "data/cinema.h"
#include "data/cinemaschedule.h"
#include "control/actioncontroller.h"
#include "control/itemmodelsortclient.h"
#include "ui/theaterwindow.h"
#include "ui/theaterschedulemodel.h"
#include "ui/contextdialog.h"
#include "ui/uiutils.h"
#include "searchclients/theaterschedulesearchclient.h"
#include "utils/assertedlocker.h"
#include "utils/asynccall.h"

#include <QSortFilterProxyModel>
#include <iostream>

static const char *MSG_NO_THEATER_SCHEDULE_FOUND = QT_TRANSLATE_NOOP("TheaterController", "No schedule found for %1.");
static const char *MSG_THEATER_SCHEDULE_ERROR = QT_TRANSLATE_NOOP("TheaterController", "Error on fetching theater schedule.");

TheaterController::TheaterController(TheaterWindow *theater_window, CinemaSchedule *cinema_schedule, ActionController *action_controller,
                                     ItemModelSortController *sort_controller, QThread *search_worker)
                                         : QObject(0),
                                         _theater_window(theater_window),
                                         _cinema_schedule(cinema_schedule),
                                         _action_controller(action_controller),
                                         _sort_controller(sort_controller),
                                         _search_worker(search_worker),
                                         _current_search_task_id(TheaterScheduleSearchClient::INVALID_SEARCH_TASK_ID),
                                         _theater_schedule_model(0),
                                         _theater_schedule_proxy_model(new QSortFilterProxyModel(this))
{
    connect(_theater_window, SIGNAL(ScheduleEntrySelected(ScheduleEntryKey)), this, SLOT(ScheduleEntrySelected(ScheduleEntryKey)));
}

void TheaterController::ShowTheater(CinemaKey cinema_key)
{
    CancelSearch();
    AssertedReadLocker locker(_cinema_schedule->GetLock());
    _cinema_key = cinema_key;
    const Cinema *cinema = ((const CinemaSchedule *) _cinema_schedule)->FindCinema(cinema_key);
    if (cinema != 0) {
        SetModel(0);
        _theater_window->SetTheaterName(cinema->GetName());
        _theater_window->SetTheaterScheduleModel(_theater_schedule_proxy_model);
        _theater_window->show();
        TheaterScheduleSearchClient *client = new TheaterScheduleSearchClient(_cinema_schedule);
        _current_search_task_id = client->GetSearchTaskId();
        connect(client, SIGNAL(SearchStarted(int)), this, SLOT(SearchStarted(int)));
        connect(client, SIGNAL(Reply(int,bool)), this, SLOT(Reply(int,bool)));
        connect(client, SIGNAL(SearchFinished(int,bool)), this, SLOT(SearchFinished(int,bool)));
        connect(client, SIGNAL(Error(int)), this, SLOT(Error(int)));
        client->moveToThread(_search_worker);
        CallAsync(client, &TheaterScheduleSearchClient::SearchSchedule, cinema->GetKey(), cinema->GetMoviesUrl());
        // client deletes itself
    }
}

void TheaterController::Cancel()
{
    _theater_window->hide();
    CancelSearch();
}

void TheaterController::CancelSearch()
{
    SetModel(0);
    AssertedWriteLocker locker(_cinema_schedule->GetLock());
    _current_search_task_id = TheaterScheduleSearchClient::INVALID_SEARCH_TASK_ID;
    TheaterScheduleSearchClient::CancelAllRunningSearchs();
}

void TheaterController::ScheduleEntrySelected(ScheduleEntryKey schedule_entry_key)
{
    ContextDialog *dialog = new ContextDialog(_cinema_schedule, _theater_window);
    connect(dialog, SIGNAL(AddToCalendar(ScheduleEntryKey)), _action_controller, SLOT(AddToCalendar(ScheduleEntryKey)));
    connect(dialog, SIGNAL(CallTheaterByPhone(CinemaKey)), _action_controller, SLOT(CallTheaterByPhone(CinemaKey)));
    connect(dialog, SIGNAL(FindRouteToTheater(CinemaKey)), _action_controller, SLOT(FindRouteToTheater(CinemaKey)));
    connect(dialog, SIGNAL(SearchMovieInWeb(MovieKey)), _action_controller, SLOT(SearchMovieInWeb(MovieKey)));
    connect(dialog, SIGNAL(SearchTheaterInWeb(CinemaKey)), _action_controller, SLOT(SearchTheaterInWeb(CinemaKey)));
    dialog->Show(schedule_entry_key);
    // dialog deletes itself
}

void TheaterController::SearchStarted(int search_task_id)
{
    if (search_task_id != _current_search_task_id) {
        return;
    }
    _theater_window->SetBusy(true);
}

void TheaterController::Reply(int search_task_id, bool intermediate)
{
    if (search_task_id != _current_search_task_id) {
        return;
    }
    Sort(intermediate, SLOT(SortFinished(QAbstractItemModel*,int,bool)));
}

void TheaterController::Error(int search_task_id)
{
    if (search_task_id != _current_search_task_id) {
        return;
    }
    Sort(false, SLOT(SortErrorFinished(QAbstractItemModel*,int,bool)));
}

void TheaterController::SearchFinished(int search_task_id, bool success)
{
    Q_UNUSED(success);
    if (search_task_id != _current_search_task_id) {
        return;
    }
    _theater_window->SetBusy(false);
}

void TheaterController::Sort(bool intermediate, const char *slot)
{
    TheaterScheduleModel *theater_schedule_model = new TheaterScheduleModel(_cinema_schedule, _cinema_key, this);
    theater_schedule_model->Update();
    QSortFilterProxyModel *sort_model = new QSortFilterProxyModel(this);
    sort_model->setSortCaseSensitivity(Qt::CaseInsensitive);
    sort_model->setSortRole(TheaterScheduleModel::SortRole);
    sort_model->setDynamicSortFilter(false);
    sort_model->setSourceModel(theater_schedule_model);
    ItemModelSortClient *sort_client = new ItemModelSortClient(_sort_controller, this);
    connect(sort_client, SIGNAL(SortFinished(QAbstractItemModel*,int,bool)), this, slot);
    sort_client->Sort(sort_model, _current_search_task_id, intermediate);
    // proxy deletes itself
}

void TheaterController::SortFinished(QAbstractItemModel *model, int search_task_id, bool intermediate)
{
    if (search_task_id != _current_search_task_id) {
        return;
    }
    SetModel(model);
    if (!intermediate) {
        if (_theater_schedule_model->rowCount() == 0) {
            _theater_window->SetError(tr(MSG_NO_THEATER_SCHEDULE_FOUND).arg(_cinema_key.GetName()));
        }
    }
}

void TheaterController::SortErrorFinished(QAbstractItemModel *model, int search_task_id, bool intermediate)
{
    Q_UNUSED(intermediate);
    if (search_task_id != _current_search_task_id) {
        return;
    }
    SetModel(model);
    if (_theater_schedule_model->rowCount() == 0) {
        _theater_window->SetError(tr(MSG_THEATER_SCHEDULE_ERROR));
    } else {
        UiUtils::ShowError(tr(MSG_THEATER_SCHEDULE_ERROR));
    }
}

void TheaterController::SetModel(QAbstractItemModel *model)
{
    delete _theater_schedule_proxy_model->sourceModel();
    _theater_schedule_proxy_model->setSourceModel(model);
    delete _theater_schedule_model;
    if (model != 0) {
        _theater_schedule_model = (TheaterScheduleModel *) ((QSortFilterProxyModel *) model)->sourceModel();
    } else {
        _theater_schedule_model = 0;
    }
}
