// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "locationcontroller.h"

#include "ui/locationdialog.h"
#include "searchclients/gpsclient.h"
#include "data/settings.h"
#include "ui/mainwindow.h"

static const char *MSG_LOCATION_NOT_FOUND = QT_TRANSLATE_NOOP("LocationController", "Unable to detect current city.");

LocationController::LocationController(MainWindow *main_window, Settings *settings) :
    QObject(0),
    _main_window(main_window),
    _settings(settings),
    _active_search_task_id(GpsClient::INVALID_SEARCH_TASK_ID)
{
}

void LocationController::OpenLocationDialog()
{
    LocationDialog *location_dialog = new LocationDialog(_main_window);
    connect(location_dialog, SIGNAL(Search(Location)), this, SLOT(SearchLocationFromCityDatabase(Location)));
    connect(location_dialog, SIGNAL(SearchPreviousLocation(Location)), this, SLOT(SearchLocation(Location)));
    connect(location_dialog, SIGNAL(SearchGPS()), this, SLOT(SearchGPS()));
    connect(location_dialog, SIGNAL(Cancel()), this, SLOT(CancelLocation()));
    location_dialog->SetLocation(_settings->GetLocation());
    location_dialog->SetPreviousLocations(_settings->GetPreviousLocations());
    location_dialog->SetGPSEnabled(true || _settings->IsGPSEnabled());
    location_dialog->show();
    // location_dialog deletes itself
}

void LocationController::CancelAllGpsSearchs()
{
    _active_search_task_id = GpsClient::INVALID_SEARCH_TASK_ID;
    GpsClient::CancelAllRunningSearchs();
}

void LocationController::SearchLocationFromCityDatabase(Location location)
{
    // TODO check city database
    SearchLocation(location);
}

void LocationController::SearchLocation(Location location)
{
    location = CleanLocation(location);
    RememberLocation(location);
    emit Search(location);
}

void LocationController::SearchGPS()
{
    GpsClient *client = new GpsClient();
    connect(client, SIGNAL(SearchStarted(int)), this, SLOT(GpsSearchStarted(int)));
    connect(client, SIGNAL(PositionUpdate(int)), this, SLOT(GpsPositionUpdate(int)));
    connect(client, SIGNAL(SearchForTownStarted(int)), this, SLOT(GpsSearchForTownStarted(int)));
    connect(client, SIGNAL(TownUpdate(int,QString)), this, SLOT(GpsTownUpdate(int,QString)));
    connect(client, SIGNAL(SearchError(int)), this, SLOT(GpsSearchError(int)));
    connect(client, SIGNAL(SearchFinished(int,bool)), this, SLOT(GpsSearchFinished(int,bool)));
    _active_search_task_id = client->GetSearchTaskId();
    client->SearchLocation();
    // gps client deletes itself
}

void LocationController::CancelLocation()
{
    // nothing to do
}

Location LocationController::CleanLocation(const Location &location)
{
    // TODO clash with city database?
    QString name = location.GetLocationName().trimmed();
    if (name.size() > 0) {
        if (name[0].isLower()) {
            name[0] = name[0].toUpper();
        }
    }
    Location cleaned_location;
    cleaned_location.SetLocationName(name);
    return cleaned_location;
}

void LocationController::RememberLocation(const Location &location)
{
    if (!location.IsValid()) {
        return;
    }
    Settings::Locations locations = _settings->GetPreviousLocations();
    QMutableListIterator<Location> it(locations);
    while (it.hasNext()) {
        const Location &prev_location = it.next();
        if (prev_location == location) {
            it.remove();
        }
    }
    locations.push_front(location);
    while (locations.size() > 5) {
        locations.pop_back();
    }
    _settings->SetPreviousLocations(locations);
}

void LocationController::GpsSearchStarted(int search_task_id)
{
    if (_active_search_task_id != search_task_id) {
        return;
    }
    _main_window->SetBusy(true);
}

void LocationController::GpsPositionUpdate(int search_task_id)
{
    if (_active_search_task_id != search_task_id) {
        return;
    }
}

void LocationController::GpsSearchForTownStarted(int search_task_id)
{
    if (_active_search_task_id != search_task_id) {
        return;
    }
}

void LocationController::GpsTownUpdate(int search_task_id, QString town)
{
    if (_active_search_task_id != search_task_id) {
        return;
    }
    Location location;
    location.SetLocationName(town);
    SearchLocation(location);
}

void LocationController::GpsSearchError(int search_task_id)
{
    if (_active_search_task_id != search_task_id) {
        return;
    }
    _main_window->SetError(tr(MSG_LOCATION_NOT_FOUND));
}

void LocationController::GpsSearchFinished(int search_task_id, bool success)
{
    if (_active_search_task_id != search_task_id) {
        return;
    }
    // In case of success SearchLocation() is started which will
    // unset busy state on its end
    if (!success) {
        _main_window->SetBusy(false);
    }
}
