#!/usr/bin/run-standalone.sh python2.5

# Import global modules
import dbus, gobject, time, helpers
from dbus.mainloop.glib import DBusGMainLoop
DBusGMainLoop(set_as_default=True)

# Try to load one gconf implementation, preferring python-gconf over gnome-python
for gconf_module in ('gconf', 'gnome.gconf'):
    try:
        gconf = __import__(gconf_module)
        break
    except:
        pass

# Try to load one pickle implementation, preferring cPickle
for pickle_module in ('cPickle', 'pickle'):
    try:
        pickle = __import__(pickle_module)
        break
    except:
        pass

class hotspot_backend:
    """Backend handler responsible for loading/unloading modules, iptables rulesets etc"""
    def __init__(self):
        self.loop = loop = gobject.MainLoop()
        self.gc = gconf.Client()
        self.bus = dbus.SystemBus()
        self.icd2_obj = self.bus.get_object('com.nokia.icd2', '/com/nokia/icd2')
        self.icd2 = dbus.Interface(self.icd2_obj, 'com.nokia.icd2')

        # Default properties
        # Kernel modules we need to work
        self.needed_modules = [ 
            'nf_conntrack', 
            'nf_defrag_ipv4',
            'nf_conntrack_ipv4',
            'x_tables',
            'ip_tables',
            'nf_nat',
            'iptable_nat',
            'ipt_MASQUERADE', # PONDER: Do we need to insert this module or will iptables take care of it ?
        ]
        self.wlanif = 'wlan0'
        self.usbif = 'usb0'
        self.gprsif = 'gprs0'

        # Modules directory
        self.module_directory = '/lib/modules/' + helpers.exec_cmd('uname -r') 

        # These are read from configuration but initialize *some* values anyway
        self.network = '10.10.10.' # Last byte missing on purpose
        self.ssid = 'MobileHotSpot'
        # TODO: Figure out if something else than WEP is supported (if not, the algo is useless)
        self.encryption_algo = None
        self.encryption_key = None
        self.connection = None

        # Store configuration/other state values before we mucked with them
        self.state = {}
        self.state['kernel'] = {}
        self.state['kernel']['loaded_modules'] = {}

        # Actual construct logic
        for modulename in self.needed_modules:
            self.state['kernel']['loaded_modules'][modulename] = False

        self.read_config_values()
        self.read_previous_state()
        self.old_connection_request_active = False

    def read_config_values(self):
        """Reads current config from GConf and updates values to our properties. Called by constructor"""
        # Stuff stored in config and accessed via members
        self.connection = self.gc.get_string('/apps/mobilehotspot/connection_name')
        if self.connection == None:
            raise Exception("No connection configured, run the frontend first")
        self.ssid = self.gc.get_string('/apps/mobilehotspot/ssid')
        if self.ssid == None:
            self.ssid = 'MobileHotSpot'
        self.network = self.gc.get_string('/apps/mobilehotspot/network_address')
        if self.network == None:
            try:
                self.gc.set_string('/apps/mobilehotspot/network_address', helpers.get_default_network())
            except:
                # Have atleast somewhat sane value here
                self.network = '10.10.10.'

        self.interface = self.gc.get_string('/apps/mobilehotspot/interface_name')
        if self.interface == 'USB':
            self.interface_name = self.usbif
        elif self.interface == 'WLAN':
            self.interface_name = self.wlanif
        else:
            self.interface = 'WLAN'
            self.interface_name = self.wlanif

        # Currently only WEP is supported, so we use this as encryption enabled/disabled indicator
        self.encryption_algo = self.gc.get_string('/apps/mobilehotspot/encryption_algo')
        self.encryption_key = self.gc.get_string('/apps/mobilehotspot/encryption_key')
        if self.encryption_key != None:
            # Pad the key to suitable size
            self.encryption_key = helpers.verify_wep_key(self.encryption_key)

        # Stuff stored in state, read the current values (some or most will be overwritten in read_previous_state())
        self.state['gconf'] = {}
        self.state['gconf']['wlan_search_interval'] = self.gc.get_int('/system/osso/connectivity/network_type/search_interval')
        self.state['icd'] = {}

    def read_previous_state(self):
        """Reads a state stored previously by store_current_state"""
        state_pickle = self.gc.get_string('/apps/mobilehotspot/state_pickle')
        #print "DEBUG: Got state_pickle=" + str(state_pickle)
        if state_pickle == None:
            return
        try:
            state_unpickled = pickle.loads(state_pickle)
            for key, value in state_unpickled.items():
                print "DEBUG-read_previous_state: setting state key '%s' to value: %s" % (key,value)
                self.state[key] = value
        except Exception, e_inst:
            print "Problem with reading pickle, error:\n";
            print e_inst

    def store_current_state(self):
        """stores the current state variable(s) somewhere we can read them back on next run"""
        state_pickle = pickle.dumps(self.state)
        #print "DEBUG: Writing state_pickle=" + str(state_pickle)
        status = self.gc.set_string('/apps/mobilehotspot/state_pickle', state_pickle)
        #print "DEBUG: Write status=" + str(status)

    def status(self):
        """check the status of the hotspot backend"""
        #raise Exception('Not implemented')


    def start(self):
        """start the hotspot backend"""
        #self.start_stop_iphbd()
        self.start_insert_modules()
        if self.interface == 'USB':
            #return self.check_usb_mode()
            self.check_usb_mode()
            self.start_configure_usb()
        else:
            self.start_configure_wlan()
        self.start_open_connection()
        self.start_run_dnsmasq(self.interface_name)
        self.start_setup_NAT()
        self.store_current_state()

    def check_usb_mode(self):
        usb_mode = helpers.exec_cmd('cat /proc/driver/musb_hdrc | grep "Gadget driver:" | cut -d" " -f 3')
        if usb_mode != 'g_nokia':
            note = 'USB must be in PC Suite Mode'
            helpers.exec_cmd('dbus-send --type=method_call --dest=org.freedesktop.Notifications /org/freedesktop/Notifications org.freedesktop.Notifications.SystemNoteDialog string:"' + note + '" uint32:0 string:"OK"')

    def start_configure_usb(self):
        helpers.exec_cmd('ifconfig ' + self.usbif + ' ' + self.network + '1 netmask 255.255.255.0 up')
        # TODO: Verify that the interface is fully up, now we just sleep a moment
        time.sleep(1)

    def stop_configure_usb(self):
        """Takes down interface"""
        # It seems we need to take the wlan interface back to managed mode or there will be trouble
        helpers.exec_cmd('ifconfig ' + self.usbif + ' down')
        # TODO: Verify that device is actually configured and properly down, for now we just sleep a moment
        time.sleep(1)

    def start_insert_modules(self):
        """Calls test_insert_module() for each needed module"""
        for modulename in self.needed_modules:
            self.state['kernel']['loaded_modules'][modulename] = self.test_insert_module(modulename)

    def icd2_open_connection(self, connection_name, connection_type):
        """Does IcD2 DBUS call to request open named connection of type"""
        # TODO: It should probably be possible to determine the connection type etc from the IAP info in gconf
        connection_bytearray = []
        for char in list(connection_name):
            connection_bytearray.append(dbus.Byte(char))
        connection_bytearray.append(dbus.Byte(0))
        self.icd2.connect_req( 
            #dbus.UInt32(1), # ICD_CONNECTION_FLAG_USER_EVENT 0x1
            dbus.UInt32(32768), # ICD_CONNECTION_FLAG_UI_EVENT 0x8000
            dbus.Array( # API Documentation seems to be wrong, dbus-monitor shows array of structs and the earlier approach gave exceptions
                [
                    dbus.Struct(
                        [
                            dbus.String(""),
                            dbus.UInt32(0),
                            dbus.String(""),
                            dbus.String(connection_type),
                            dbus.UInt32(83886080), # CD_NW_ATTR_AUTOCONNECT   0x04000000 & ICD_NW_ATTR_IAPNAME   0x01000000 
                            dbus.Array(connection_bytearray) 
                        ]
                    )
                ] 
            ) 
        )
        # TODO: Make a (separate) DBUS listener to wait untill the selected connection is up
        time.sleep(5)

    def icd2_state_listener(self, *args):
        print "icd2_state_listener got args:\n";
        print args

        # Handle the weird corner cases of reduced argument counts
        if (    len(args) == 1
            and int(args[0]) == 0):
            # no connections
            pass
        elif len(args) == 2:
            # Scan active
            pass
        elif (      self.old_connection_request_active == True # This is request made by us
                and str(args[6]) == '' # There was no error
                and int(args[7]) == 2 # the state of the connection is ICD_STATE_CONNECTED
              ): 
                self.state['icd']['old_connection_type'] = str(args[3]).rstrip()
                self.state['icd']['old_connection_name'] = str(args[5]).rstrip(chr(0x00)) # These bytearrays are C-style "strings": null-terminated 

        print "state['icd']:\n"
        print self.state['icd']
        self.expect_state_signals = self.expect_state_signals-1
        if self.expect_state_signals <= 0:
            # TODO: How to disconnect the signal ?
            self.loop.quit()
            return

    def alarm_handler(self):
        raise Exception('Caught SIGALARM, likely spending too much time waiting for DBUS mainloop to quit')

    def get_icd2_state(self):
        """Registers signal lister and sends a request for state, runs mainloop until the state listener quits it"""
        self.icd2_obj.connect_to_signal('state_sig', self.icd2_state_listener, 'com.nokia.icd2', byte_arrays=True)
        self.old_connection_request_active = True
        self.expect_state_signals = self.icd2.state_req()
        if self.expect_state_signals > 0: 
            # Setup 5 second timeout for the mainloop in case we do not manage to quit it via the signal listerner
            import signal
            signal.signal(signal.SIGALRM, self.alarm_handler)
            signal.alarm(5)
            self.loop.run()
            signal.alarm(0)
        self.old_connection_request_active = False

    def start_open_connection(self):
        """Stores the current connection (if any) and opens the selected one"""
        self.get_icd2_state()
        # Request the GPRS connection
        try:
            if (    self.connection == self.state['icd']['old_connection_name']
                and self.state['icd']['old_connection_type'] == 'GPRS'):
                    # Previous connection is same as the one we want, save some time and cpu-cycles
                    return
        except KeyError, e:
            pass
        self.icd2_open_connection(self.connection, 'GPRS')

    def start_configure_wlan(self):
        """Configures the WLAN: set scan interval to never then raise the interface"""
        # Disable WLAN scan (old interval was stored to state earlier)
        self.gc.set_int('/system/osso/connectivity/network_type/search_interval', 0) 
        # PONDER: Create temporary IAP and use IcD2 instead ??? (<- it might be impossible to use two IAPs at the same time)
        helpers.exec_cmd('ifconfig ' + self.wlanif + ' down')
        # TODO: Verify that the interface is fully down so we can configure it, now we just sleep a moment
        time.sleep(3)
        helpers.exec_cmd('iwconfig ' + self.wlanif + ' mode ad-hoc')
        helpers.exec_cmd('ifconfig ' + self.wlanif + ' up')
        # TODO: Verify that the interface is fully up so we can configure it, now we just sleep a moment
        time.sleep(3)
        if (    self.encryption_algo
            and self.encryption_key):
            helpers.exec_cmd('iwconfig ' + self.wlanif + ' key ' + helpers.hex_encode(helpers.verify_wep_key(self.encryption_key)) + ' restricted')
            #helpers.exec_cmd('iwconfig ' + self.wlanif + ' key ' + keyhexenc)
        helpers.exec_cmd('iwconfig ' + self.wlanif + ' essid "' + self.ssid + '"')
        helpers.exec_cmd('ifconfig ' + self.wlanif + ' ' + self.network + '1 netmask 255.255.255.0 up')
        # TODO: Verify that the interface is fully up, now we just sleep a moment
        time.sleep(3)

    def start_run_dnsmasq(self, interface_name):
        """Starts the dnsmasq daemon with dhcp enabled"""
        dnsmasq_command = '/usr/sbin/dnsmasq'
        dnsmasq_options = ' -i ' + interface_name
        dnsmasq_options += ' -a ' + self.network + '1'
        dnsmasq_options += ' -I lo -z -x /var/run/dnsmasq.' + interface_name + '.pid'
        dnsmasq_options += ' --dhcp-range=' + self.network + '10,' + self.network + '100,6h'
        dnsmasq_options += ' --dhcp-option=3,' + self.network + '1'
        dnsmasq_options += ' --dhcp-option=6,' + self.network + '1'
        helpers.exec_cmd(dnsmasq_command + dnsmasq_options)

    def start_stop_iphbd(self):
        """Stop iphbd, it causes device to crash when we're doing NAT"""
        helpers.exec_cmd('/etc/init.d/iphbd stop')

    def stop_start_iphbd(self):
        """Restart iphbd, it exists for a purpose"""
        helpers.exec_cmd('/etc/init.d/iphbd start')

    def start_setup_NAT(self):
        """Configures iptables for NAT"""

        # TODO read current forward state and remember it in state

        import distutils.file_util
        distutils.file_util.write_file('/proc/sys/net/ipv4/ip_forward', [ '1', ])

        # PONDER store current iptbales setup to a temp file ?
        helpers.exec_cmd('iptables --flush')
        helpers.exec_cmd('iptables --flush -t nat')
        
        # PONDER: Use more secure FW setup ?
        helpers.exec_cmd('iptables -P FORWARD ACCEPT')
        helpers.exec_cmd('iptables -P INPUT ACCEPT')
        helpers.exec_cmd('iptables -P OUTPUT ACCEPT')

        helpers.exec_cmd('iptables -t nat -A POSTROUTING -j MASQUERADE -s ' + self.network + '0/24')

    def stop(self):
        """stop the hotspot backend"""
        try:
            self.stop_teardown_NAT()
            self.stop_stop_dnsmasq(self.interface_name)
            if self.interface == 'USB':
                self.stop_configure_usb()
            else:
                self.stop_configure_wlan()
            self.stop_close_connection()
            self.stop_remove_modules()
            #self.stop_start_iphbd()
        except Exception, e_inst:
            # In case of failure just skip remaining operations and clear state
            print "Problem when stopping: \n"
            print e_inst
            print 'Clearing state and exiting normally anyway'
        self.state = {}
        self.store_current_state()

    def stop_teardown_NAT(self):
        """Restores old iptables configs"""
        # FIXME: Check command exit codes
        helpers.exec_cmd('iptables --flush')
        helpers.exec_cmd('iptables --flush -t nat')
        # TODO: If old iptables config is stored restore it now

        # TODO: If old forward state is stored restore it (now default to 0)
        import distutils.file_util
        distutils.file_util.write_file('/proc/sys/net/ipv4/ip_forward', [ '0', ])

    def stop_stop_dnsmasq(self, interface_name):
        """Stops the dnsmasq daemon previously started"""
        pidfile = '/var/run/dnsmasq.' + interface_name + '.pid'
        import os
        if (os.path.exists(pidfile)):
            import signal
            pid = open(pidfile, 'r').read()
            try:
                # PONDER: Ping the process first ??
                print "DEBUG: Sending SIGTERM to: " + str(pid)
                os.kill(int(pid), signal.SIGTERM)
                os.unlink(pidfile)
            except:
                # Failed to kill, probably process does not exist
                print "DEBUG: FAILED sending SIGTERM to: " + str(pid)
                try:
                    # try to remove the stale pidfile
                    os.unlink(pidfile)
                except:
                    # Remove failed, too bad but we don't want to die just yet
                    pass
        return

    def stop_configure_wlan(self):
        """Takes down interface"""
        # It seems we need to take the wlan interface back to managed mode or there will be trouble
        helpers.exec_cmd('ifconfig ' + self.wlanif + ' down')
        # TODO: Verify that device is actually configured and properly down, for now we just sleep a moment
        time.sleep(3)
        helpers.exec_cmd('iwconfig ' + self.wlanif + ' mode managed')
        helpers.exec_cmd('iwconfig ' + self.wlanif + ' key open')
        # TODO: Verify that device is actually configured and properly down, for now we just sleep a moment
        time.sleep(3)

    def stop_close_connection(self):
        """Reopens the old connection if there was any, then restores WLAN scan interval"""
        try:
            self.icd2_open_connection(self.state['icd']['old_connection_name'], self.state['icd']['old_connection_type'])
        except KeyError:
            # No old connection active was active, disconnect the current one
            self.icd2.disconnect_req(dbus.UInt32(32768)) # ICD_CONNECTION_FLAG_UI_EVENT 0x8000
        self.gc.set_int('/system/osso/connectivity/network_type/search_interval', self.state['gconf']['wlan_search_interval'])

    def stop_remove_modules(self):
        """Calls remove_module() for each module *we* inserted"""
        self.needed_modules.reverse()
        for modulename in self.needed_modules:
            if self.state['kernel']['loaded_modules'][modulename]:
                self.state['kernel']['loaded_modules'][modulename] = self.remove_module(modulename)

    def test_insert_module(self, module):
        """Inser given module to kernel if not already inserted, return True if insert was done, False if not
        and raises exception on errors"""
        lsmod_output = helpers.exec_cmd('lsmod')
        if (lsmod_output.find(module) != -1):
            return False
        helpers.exec_cmd('insmod ' + self.module_directory + '/' + module + '.ko')
        return True

    def remove_module(self, module):
        """Tries to remove a module from kernel returns status of module in kernel, ie False if remove is successfull and True if module is still inserted"""
        try:
            helpers.exec_cmd('rmmod ' + module)
            return False
        except:
            # TODO: Better error checking (requires better error reporting in the exec_cmd method) ?
            return True        

def cli():
    import sys
    backend = hotspot_backend()
    try:
        method = sys.argv[1]
    except IndexError:
        # No arguments, print help ?
        print "Try 'start'"
        return 0
    # TODO: dynamic calling
    #backend.method()
    if method == "start":
        backend.start()
    elif (method == "end" or method == "stop"):
        backend.stop()
    else:
        print "Try 'start'"
        return 1
    return 0

