/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "loginctrl.h"

/*! 
 * \brief LoginCtrl constructor.
 */
LoginCtrl::LoginCtrl(QGraphicsScene *scene)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _scene = scene;
    _loginView = new LoginView();

    showView();
    loadUsername();

    connect(_loginView,SIGNAL(done()),this,SLOT(deleteLater()));
    connect(_loginView, SIGNAL(exitLogin()), this, SLOT(_exitButtonPressed()));
    connect(_loginView, SIGNAL(loginAccount()), this, SLOT(_loginTwitter()));
    connect(Backend::getInstance(),SIGNAL(signedOn()),this,SLOT(_saveUsername()));
    connect(Backend::getInstance(),SIGNAL(signedOn()),_loginView,SLOT(connected()));
    connect(Backend::getInstance(), SIGNAL(connectionError(QString)), this, SLOT(_connectionError(QString)));
    connect(_loginView, SIGNAL(abort()), this, SLOT(_abort()));
}

/*! 
 * \brief LoginCtrl destructor.
 */
LoginCtrl::~LoginCtrl(void)
{
    qDebug() << "~ " << __PRETTY_FUNCTION__;
    if (_loginView) {
        delete _loginView;
    }

    emit done();
}

/*! 
 * \brief Show a dialog window with a error message
 * \param message the message in the dialog
 */
void LoginCtrl::_connectionError(QString message)
{
    _loginView->_cancelButton->setOpacity(0);
    _dialog = new DialogView(DialogView::Error,message);
    _scene->addItem(_dialog);
    _loginView->connectionError();
}

/*! 
 * \brief Creates the view and adds the scene
 */
void LoginCtrl::showView(void)
{
    _loginView->createUi();
    _scene->addItem(_loginView);
}

/*! 
 * \brief Try to login on the twitter.
 */
void LoginCtrl::_loginTwitter(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    if(_loginView->getUsername() != "" && _loginView->getPassword() != "")
        Backend::getInstance()->setUsernamePassword(_loginView->getUsername(),_loginView->getPassword());
    else {
        _connectionError("Login Failed");
        _loginView->setPassword("");
    }
}

/*! 
 * \brief This slot is called when the exit button is pressed
 * quits the application
 */
void LoginCtrl::_exitButtonPressed(void)
{
    _dialog = new DialogView(DialogView::Warning, QString("Are you sure you want to quit?"));
    connect(_dialog, SIGNAL(yesPressed()), this, SLOT(_exit()));
    _scene->addItem(_dialog);
}

/*!
 * \brief Exits the application.
 */
void LoginCtrl::_exit(void)
{
    QApplication::quit();
}

/*!
 * \brief Saves username in a file.
 */
void LoginCtrl::_saveUsername(void)
{

    QFile userInfo(QString("%1/.info.xml").arg(getenv("HOME")));
    QXmlStreamWriter writer(&userInfo);

    if (!userInfo.open(QIODevice::WriteOnly | QIODevice::Text))
        return;

    writer.setAutoFormatting(true);
    writer.setAutoFormattingIndent(4);
    writer.writeStartDocument("1.0");
    writer.writeStartElement("USERINFO");
    if(!_loginView->getRememberMe()) {
        writer.writeTextElement("username", "");
        writer.writeTextElement("password", "");
    } else {
        writer.writeTextElement("username", _loginView->getUsername());
        writer.writeTextElement("password", _loginView->getPassword());
    }

    writer.writeEndElement();

    userInfo.close();
}

/*!
 * \brief Loads username from a file if it exist.
 */
void LoginCtrl::loadUsername(void)
{
    QFile userInfo(QString("%1/.info.xml").arg(getenv("HOME")));
    QXmlStreamReader reader(&userInfo);

    if (!userInfo.open(QIODevice::ReadOnly | QIODevice::Text))
        return;

    while (!reader.atEnd()) {
        reader.readNext();
        if (reader.isStartElement()) {
            if (reader.name() == "username")
                _loginView->setUsername(reader.readElementText());
            if (reader.name() == "password")
                _loginView->setPassword(reader.readElementText());
        }
    }

    if(_loginView->getUsername() != "")
        _loginView->checkbox->setChecked(true);

}

/*!
 * \brief Aborts any connections in progress.
 */
void LoginCtrl::_abort()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    Backend::getInstance()->cancelNotPressed = false;
    Backend::getInstance()->m_twitLib->Abort();
}
