/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "twittertweetinfo.h"

/*! \brief TwitterTweetInfo constructor.
 *  \param m_username Twitter username
 *  \param m_text Text of respective username
 *  \param m_imageUrl Profile image of respective username
 */
TwitterTweetInfo::TwitterTweetInfo(Tweet *tweet, bool atHomeTimeline, QGraphicsWidget *parent)
                : TwitterApplicationItem(parent),
                _tweet(tweet),
                m_atHomeTimeline(atHomeTimeline)
{
    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setWeight(QFont::Normal);
    font.setPixelSize(18);

    setAttribute(Qt::WA_OpaquePaintEvent, true);

    setGeometry(0, 0, 796, 360);

    m_tweetMessage = _tweet->getMessage();
    m_username = _tweet->getUsername();
    m_tweetId = _tweet->getStatusId();
    m_favorite = _tweet->getFavorite();
    m_coordinates = _tweet->getCoordinates();
    m_tweetUrls = _createUrls(m_tweetMessage);
    m_isMyTweet = _tweet->getIsMyTweet();
    m_isFollowing = _tweet->getFollowing();

    _paintTweet();
    _createButtons();
    _paintUrlsAndMaps(m_tweetUrls, m_coordinates.split(" "));

    connect(_tweet, SIGNAL(imageDownloaded()), this, SLOT(_paintAvatar()));
}

/*!
 * \brief Overridden.
 * \param painter Painter object.
 * \param option Styles to drawing.
 */
void TwitterTweetInfo::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *)
{
    Q_UNUSED(option);

    /* Draw the white rect */
    painter->setPen(Qt::NoPen);
    painter->setBrush(Qt::white);
    painter->drawRect(0, 0, 796, 360);

    /* Draw Avatar */
    painter->drawImage(QRect(13,13,48,48),m_image);
}

/*!
 * \brief This function parses the message and puts all URLs that were found in a list.
 * \return URL list.
 */
QStringList TwitterTweetInfo::_createUrls(QString text)
{
    QStringList textSplitted = text.split(" ");
    QStringList listUrls;

    for(int i=0; i<textSplitted.count(); i++)
        if(textSplitted.at(i).contains(QRegExp("((https?|ftp)://[^ ]+)( ?)", Qt::CaseInsensitive)))
            listUrls << textSplitted.at(i);

    return listUrls;
}

/*!
 * \brief This function creates all buttons on the detailed tweet screen.
 */
void TwitterTweetInfo::_createButtons()
{
    if(m_favorite)
        m_favoriteButton = new TwitterButton(TwitterButton::Favorite, NULL, this);
    else
        m_favoriteButton = new TwitterButton(TwitterButton::NoFavorite, NULL, this);
    m_favoriteButton->setPos(794 - m_favoriteButton->boundingRect().width() - 10, m_sep1->line().y1()+12);

    m_replyButton = new TwitterButton(TwitterButton::Reply  , NULL, this);
    m_replyButton->setPos(m_favoriteButton->pos().x() - m_replyButton->boundingRect().width() - 10, m_sep1->line().y1()+12);

    m_retweetButton = new TwitterButton(TwitterButton::Retweet, NULL, this);
    m_retweetButton->setPos(m_replyButton->pos().x() - m_retweetButton->boundingRect().width() - 10, m_sep1->line().y1()+12);

    if(m_isMyTweet) {
        m_deleteButton = new TwitterButton(TwitterButton::Delete, NULL , this);
        m_deleteButton->setPos(m_retweetButton->pos().x() - m_deleteButton->boundingRect().width() - 10, m_sep1->line().y1()+12);
        connect(m_deleteButton,SIGNAL(released()), this, SLOT(_deleteTweet()));
    }
    else if (m_atHomeTimeline) {
        if(m_isFollowing)
            m_unfollowButton = new TwitterButton(TwitterButton::Generic, "Unfollow", this);
        else
            m_unfollowButton = new TwitterButton(TwitterButton::Generic, "Follow", this);

        m_unfollowButton->setPos(14 + 10, m_sep1->line().y1()+12);
        connect(m_unfollowButton, SIGNAL(released()), this, SLOT(_unfollow()));
    }

    if(haveTwitPic(m_tweetMessage)) {
        m_twitPicButton = new TwitterButton(TwitterButton::Generic, "Open \n image", this);
        m_twitPicButton->setPos(128, m_sep1->line().y1()+12);
        connect(m_twitPicButton, SIGNAL(released()), this, SIGNAL(twitpic()));
    }

    connect(m_favoriteButton, SIGNAL(released()), this, SLOT(_favoriteInfo()));
    connect(m_retweetButton, SIGNAL(released()), this, SLOT(_retweet()));
    connect(m_replyButton, SIGNAL(released()), this, SLOT(_reply()));
}

void TwitterTweetInfo::_paintTweet(void)
{
    m_tweet = new QGraphicsTextItem(this);
    m_tweet->setDocument(_tweet->getTweetDocument());
    m_tweet->setPos(74,0);

    m_sep1 = new QGraphicsLineItem(this);
    m_sep1->setLine(7,m_tweet->boundingRect().height()+1,787,m_tweet->boundingRect().height()+1);
    m_sep1->setPen(QPen(QColor(216,216,216)));

    m_sep2 = new QGraphicsLineItem(this);
    m_sep2->setLine(m_sep1->line().p1().x(), m_sep1->line().p1().y()+75,
                    m_sep1->line().p2().x(), m_sep1->line().p2().y()+75);
    m_sep2->setPen(QPen(QColor(216,216,216)));

    this->_paintAvatar();

}

void TwitterTweetInfo::_reply(void)
{
    QString replyMessage = "@" + m_username;
    emit reply(replyMessage, m_tweetId);
}

void TwitterTweetInfo::_retweet(void)
{
    QString retweetMessage = "RT: @" + m_username + " " + m_tweetMessage;
    emit retweet(retweetMessage);
}

void TwitterTweetInfo::_paintAvatar(void)
{
    m_image = _tweet->getImage();
    update();
}

void TwitterTweetInfo::_paintUrlsAndMaps(QStringList urls, QStringList coordinates)
{
    font.setPixelSize(18);

    if(coordinates.count() < 2 && urls.isEmpty())
        m_message = new QGraphicsTextItem("No links or maps avaiable in this tweet.",this);
    else
        m_message = new QGraphicsTextItem(this);
    m_message->setFont(font);
    m_message->setPos(15, m_sep2->line().y1() + 10);

    if(!urls.isEmpty()) {
        m_message->setPlainText("Links found in this tweet: ");

        for(int i=0;i<urls.count();i++) {
            QGraphicsTextItem *link = new QGraphicsTextItem(this);
            link->setHtml("<a href=\"" + urls.at(i) + "\">" + _shortUrl(urls.at(i)) +  "</a><br>");
            link->setTextInteractionFlags(Qt::TextBrowserInteraction);
            link->setFlag(QGraphicsItem::ItemIsFocusable,false);
            link->setOpenExternalLinks(true);
            link->setPos(15,m_message->pos().y()+30 + 35*i);
            link->setFont(font);
        }
    }
}

void TwitterTweetInfo::_favoriteInfo()
{
    emit favorite(this->m_tweetId, this->m_favorite);
}

void TwitterTweetInfo::_deleteTweet()
{
    emit del(m_tweetId, _tweet->getPosInList());
}

void TwitterTweetInfo::_unfollow()
{
    emit unfollow(m_username, m_isFollowing);
}

QString TwitterTweetInfo::_shortUrl(QString url)
{
    if(url.length() < 30)
        return url;

    QString newUrl = url.left(13) + "..." + url.right(15);
    return newUrl;
}

void TwitterTweetInfo::updateTweetInfo(QString type, bool flag)
{
    if(type=="Favorite") {
        _tweet->setFavorite(flag);
        m_favorite = flag;

        if(flag)
            m_favoriteButton->setTypeButton(TwitterButton::Favorite);
        else
            m_favoriteButton->setTypeButton(TwitterButton::NoFavorite);
        m_favoriteButton->setPos(794 - m_favoriteButton->boundingRect().width() - 10, m_sep1->line().y1()+12);
    }
    else if(type=="Follow") {
        m_isFollowing = flag;
        _tweet->setFollowing(m_username, m_isFollowing);

        if(flag)
            m_unfollowButton->setLabel(tr("Unfollow"));
        else
            m_unfollowButton->setLabel(tr("Follow"));
    }

}

bool TwitterTweetInfo::haveTwitPic(QString m_tweetMessage)
{
    QList<QString> temp = m_tweetMessage.split(" ");

        QListIterator<QString> iteratorUrl(temp);

    while (iteratorUrl.hasNext()) {
        QString nextWord = iteratorUrl.next();

        if ( nextWord.contains( QRegExp("http://twitpic*") ) ) {
            m_twitPicUrl = nextWord;
            return true;
        }
    }
    return false;
}

QString TwitterTweetInfo::getTwitPicUrl()
{
    return m_twitPicUrl;
}

void TwitterTweetInfo::_loadMap()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QStringList coordinates = m_coordinates.split(" ");
    QStringList urls = m_tweetUrls;

    if(coordinates.count()>1) {
        qreal latitude = coordinates.at(0).toDouble();
        qreal longitude = coordinates.at(1).toDouble();

        m_map = new GMaps();
        m_map->triggerLoading(latitude, longitude);
        m_map->setCenter(latitude, longitude);

        QGraphicsProxyWidget *map = new QGraphicsProxyWidget(this);
        map->setWidget(m_map);
        map->show();

        map->setGeometry(QRectF(5,155,this->boundingRect().width()-10,this->boundingRect().height() - m_sep2->line().y1() - 10));
        if (!urls.isEmpty())
            map->setGeometry(QRectF((this->boundingRect().width() -10)/2,155,(this->boundingRect().width() -10)/2,200));
    }
}
