/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "twittertopbar.h"

/*! \brief TwitterTopBar constructor.
 */
TwitterTopBar::TwitterTopBar(QGraphicsWidget *parent)
        : TwitterApplicationItem(parent)
{
//    QNetworkProxy proxy;
//    proxy.setType(QNetworkProxy::HttpProxy);
//    proxy.setHostName("172.19.244.1");
//    proxy.setPort(3128);
//    proxy.setUser("indt.rodrigo.ramos");
//    proxy.setPassword("indt.rodrigo.ramos");
//    QNetworkProxy::setApplicationProxy(proxy);

    bgImage = QImage(":images/resources/topbar/top_bar_bg.png");
    followingNameImage = QImage(":images/resources/topbar/following.png");
    followersNameImage = QImage(":images/resources/topbar/followers.png");
    tweetsNameImage = QImage(":images/resources/topbar/tweets.png");

    createTimeline = true;
    setCacheMode(DeviceCoordinateCache);
    setGeometry(0,0,311,55);

    connect(&m_manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(_handleNetworkRequest(QNetworkReply*)));

    this->updateTopBar(true);
}

/*!
 * \brief TwitterTimeline destructor.
 */
TwitterTopBar::~TwitterTopBar(void)
{
    qDebug() << "~  " << __PRETTY_FUNCTION__;
}

/*!
 * \brief Updates the topbar with newer information about the user.
 */
void TwitterTopBar::updateTopBar(bool timeline)
{
    createTimeline = timeline;

    list = Backend::getInstance()->userInfo();
    followingText = list[1];
    followersText = list[2];
    tweetsText = list[3];
    paintImage = false;

    this->_positions();
    this->_download(list[4]);
}

/*!
 * \brief This function handles the reply and loads the image.
 */
void TwitterTopBar::_handleNetworkRequest(QNetworkReply *reply)
{
    if (!m_image.load(reply, 0))
        m_image = QImage();
    reply->deleteLater();

    paintImage = true;
    update();

    if(createTimeline)
        emit finished();
}

/*!
 * \brief This function sends the request to download the profile image.
 */
void TwitterTopBar::_download(QString Url)
{
    QNetworkRequest request;
    request.setUrl(Url);
    m_manager.get(request);
}

/*!
 * \brief Overridden.
 * \param painter Painter object.
 * \param option Styles to drawing.
 */
void TwitterTopBar::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *)
{
    Q_UNUSED(option);

    /* Draw the white rect */
    painter->drawImage(QRect(0,0,311,55), bgImage);
    
    /* Draw the user Image */
    if (paintImage){
        painter->drawImage(QRect(4, 3, 48, 48), m_image);
    } else {
        painter->setBrush(QColor(238,252,255,200));
        painter->drawRect(4,3,48,48);
    }
    
    /* Draw the username */
    QFont font;
    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setPixelSize(18);
    painter->setPen(QColor(31,31,31));
    painter->setFont(font);
    painter->drawText(60,6,338,21, Qt::TextWrapAnywhere,list[0]);

    /* Draw the informations */
    font.setPixelSize(16);
    painter->setFont(font);
    painter->drawText(listPositions[1],list[1]);
    painter->drawText(listPositions[2],list[3]);
    painter->drawText(listPositions[5],list[2]);

    painter->setFont(font);
    painter->setPen(QColor(15,149,218));
    painter->drawImage(listPositions[0],followingNameImage);
    painter->drawImage(listPositions[3],tweetsNameImage);
    painter->drawImage(listPositions[4],followersNameImage);

}

/*!
 * \brief This function calculates each label's position inside the top bar.
 */
void TwitterTopBar::_positions(void)
{
    int ImageWidth;

    QFont font;
    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setPixelSize(15);
    QFontMetrics fm(font);

    listPositions << QPointF(60, 31);

    ImageWidth = followingNameImage.width();
    listPositions << QPointF(listPositions.last().x() + ImageWidth +6,47);

    ImageWidth = fm.width(list[3]);
    listPositions << QPointF(bgImage.width() - ImageWidth -8,47);

    ImageWidth = tweetsNameImage.width();
    listPositions << QPointF(listPositions.last().x() - ImageWidth -6,31);


    ImageWidth = followersNameImage.width();
    int followersSize = ImageWidth + fm.width(list[2]) + 6;
    int middleSpace = listPositions[3].x() - listPositions[1].x() - fm.width(list[1]);
    listPositions << QPointF((middleSpace - followersSize)/2 + listPositions[1].x() + fm.width(list[1]),31);

    ImageWidth = followersNameImage.width();
    listPositions << QPointF(listPositions.last().x() + ImageWidth + 6, 47);
}

/*!
 * \brief This function return the user's name.
 * \return The user's name.
 */
QString TwitterTopBar::getUsername()
{
    return list[0];
}

/*!
 * \brief Reimplements mouse press event.
 * This method is used to open a dialog so the user can choose an image to update its avatar.
 */
void TwitterTopBar::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    QPointF pos = event->pos();

    if(pos.x() > 4 && pos.x() < 52 && pos.y() > 3 && pos.y() < 51) {
        QFileDialog dialog;
        QString fileName = dialog.getOpenFileName(0, tr("Open File"), "/home", tr("Images (*.png *.gif *.jpg)"));
        if(fileName != NULL)
            emit changeAvatar(fileName);
    }
}
