//  Copyright  2011    Karl Wintermmann
//  E-Mail:             karl@rx-51.de
//  Sourcecode is distributed under the GNU GENERAL PUBLIC LICENSE (GPL)

#include "minidlna.h"
#include <QDebug>

MiniDLNA::MiniDLNA(QObject *parent) :
    QObject(parent)
{
    this->settings_changed = false;
    this->port = "8200";
    this->pathToConfFile = "/etc/minidlna.conf";
    this->network_interface = "wlan0";
    this->friendly_name = "N900";
    this->db_dir = "";
    this->log_dir = "";
    this->album_art_names = "Cover.jpg/cover.jpg/AlbumArtSmall.jpg/albumartsmall.jpg/AlbumArt.jpg/albumart.jpg/Album.jpg/album.jpg/Folder.jpg/folder.jpg/Thumb.jpg/thumb.jpg";
    this->inotify = "yes";
    this->enable_tivo = "no";
    this->strict_dlna = "no";
    this->presentation_url = "";
    this->notify_interval = "900";
    this->serial = "12345678";
    this->model_number = "1";
}

void MiniDLNA::SetPathToConfFile(QString path)
{
        pathToConfFile = path;
}

void MiniDLNA::AddShare(MiniDLNAShare share)
{
    QString tmp = "";
    if(share.Type == MiniDLNAShare::IMAGES)
        tmp = "P," + share.Path;
    else if(share.Type == MiniDLNAShare::AUDIO)
        tmp = "A," + share.Path;
    else if(share.Type == MiniDLNAShare::VIDEO)
        tmp = "V," + share.Path;
    else
        tmp = share.Path;
    if(!media_dir.contains(tmp)) media_dir.append(tmp);
    settings_changed = true;
    emit settingsChanged(settings_changed);
}
void MiniDLNA::AddShare(QList<MiniDLNAShare> shares)
{
    for(int i=0;i<shares.size();i++)
    {
        AddShare(shares[i]);
    }
    settings_changed = true;
    emit settingsChanged(settings_changed);
}
void MiniDLNA::SetShares(QList<MiniDLNAShare> shares)
{
    if(!media_dir.isEmpty()) media_dir.clear();
    AddShare(shares);
}

QList<MiniDLNAShare> MiniDLNA::GetShares()
{
    QList<MiniDLNAShare> tmplist;
    MiniDLNAShare tmpshare;
    QList<QString> media_dir_tmp = media_dir;
    for(int i=0;i<media_dir.size();i++)
    {
        if(media_dir_tmp[i].startsWith("P,"))
        {
            tmpshare.Type = MiniDLNAShare::IMAGES;
            tmpshare.Path = media_dir_tmp[i].remove(0,2);
            tmpshare.Name = media_dir_tmp[i].remove(0,media_dir_tmp[i].lastIndexOf('/')+1);
            if(!tmplist.contains(tmpshare)) tmplist.append(tmpshare);
        }
        else if(media_dir_tmp[i].startsWith("A,"))
        {
            tmpshare.Type = MiniDLNAShare::AUDIO;
            tmpshare.Path = media_dir_tmp[i].remove(0,2);
            tmpshare.Name = media_dir_tmp[i].remove(0,media_dir_tmp[i].lastIndexOf('/')+1);
            if(!tmplist.contains(tmpshare)) tmplist.append(tmpshare);
        }
        else if(media_dir_tmp[i].startsWith("V,"))
        {
            tmpshare.Type = MiniDLNAShare::VIDEO;
            tmpshare.Path = media_dir_tmp[i].remove(0,2);
            tmpshare.Name = media_dir_tmp[i].remove(0,media_dir_tmp[i].lastIndexOf('/')+1);
            if(!tmplist.contains(tmpshare)) tmplist.append(tmpshare);
        }
        else
        {
            tmpshare.Type = MiniDLNAShare::NONE;
            tmpshare.Path = media_dir_tmp[i];
            tmpshare.Name = media_dir_tmp[i].remove(0,media_dir_tmp[i].lastIndexOf('/')+1);
            if(!tmplist.contains(tmpshare)) tmplist.append(tmpshare);
        }
    }
    return tmplist;
}

bool MiniDLNA::GetServerStatus()
{
    QString tmp="";
    QProcess server;
    QByteArray output;
    server.setProcessChannelMode(QProcess::MergedChannels);
    server.start("pgrep minidlna");
    while(server.waitForReadyRead()){
        output+=server.readAll();
    }
    tmp=output;
    //qDebug()<<tmp;
    if(tmp!="") {
        emit ServerStatus(true);
        return true;
    }
    emit ServerStatus(false);
    return false;
}

void MiniDLNA::StartServer()
{
    ServerCommand("start");
    GetServerStatus();
}

void MiniDLNA::StopServer()
{
    ServerCommand("stop");
    emit ServerStatus(false);
}

void MiniDLNA::RestartServer()
{
    ServerCommand("restart");
    GetServerStatus();
}

bool MiniDLNA::SettingsChanged()
{
    return settings_changed;
}

void MiniDLNA::AcceptChanges()
{
    if(settings_changed) WriteConfFile();
    ServerCommand("clear-cache");
}

void MiniDLNA::RollbackChanges()
{
    if(settings_changed) ReadConfFile();
}

void MiniDLNA::UpdateFromConfFile()
{
    ReadConfFile();
}

bool MiniDLNA::ReadConfFile()
{
    QString tmp="";
    QFile data(this->pathToConfFile);
    if(!this->media_dir.isEmpty()) this->media_dir.clear();
    if (data.open(QIODevice::ReadOnly | QIODevice::Text)){
        QTextStream tstream(&data);
        do {
            tmp=tstream.readLine();
            tmp=tmp.simplified();
            if((tmp[0]=='#') || (tmp=="")) continue;

            // read config parameters
            if(tmp.contains("port=")) {
                tmp.remove("port=");
                this->port = tmp;
                continue;
            }
            if(tmp.contains("network_interface=")) {
                tmp.remove("network_interface=");
                this->network_interface = tmp;
                continue;
            }
            if(tmp.contains("media_dir=")) {
                tmp.remove("media_dir=");
                this->media_dir.append(tmp);
                continue;
            }
            if(tmp.contains("friendly_name=")) {
                tmp.remove("friendly_name=");
                this->friendly_name = tmp;
                continue;
            }
            if(tmp.contains("db_dir=")) {
                tmp.remove("db_dir=");
                this->db_dir = tmp;
                continue;
            }
            if(tmp.contains("log_dir=")) {
                tmp.remove("log_dir=");
                this->log_dir = tmp;
                continue;
            }
            if(tmp.contains("album_art_names=")) {
                tmp.remove("album_art_names=");
                this->album_art_names = tmp;
                continue;
            }
            if(tmp.contains("inotify=")) {
                tmp.remove("inotify=");
                this->inotify = tmp;
                continue;
            }
            if(tmp.contains("enable_tivo=")) {
                tmp.remove("enable_tivo=");
                this->enable_tivo = tmp;
                continue;
            }
            if(tmp.contains("strict_dlna=")) {
                tmp.remove("strict_dlna=");
                this->strict_dlna = tmp;
                continue;
            }
            if(tmp.contains("presentation_url=")) {
                tmp.remove("presentation_url=");
                this->presentation_url = tmp;
                continue;
            }
            if(tmp.contains("notify_interval=")) {
                tmp.remove("notify_interval=");
                this->notify_interval = tmp;
                continue;
            }
            if(tmp.contains("serial=")) {
                tmp.remove("serial=");
                this->serial = tmp;
                continue;
            }
            if(tmp.contains("model_number=")) {
                tmp.remove("model_number=");
                this->model_number = tmp;
            }

        } while(!tmp.isNull());
        data.close();
        settings_changed = false;
        emit settingsChanged(settings_changed);
        return true;
    }
    data.close();
    return false;
}

bool MiniDLNA::WriteConfFile()
{
    QFile data(this->pathToConfFile);
    if (data.open(QIODevice::WriteOnly | QIODevice::Text)){
        QTextStream tstream(&data);
        tstream<<"# port for HTTP (descriptions, SOAP, media transfer) traffic"<<endl;
        tstream<<"port="<<this->port<<endl;
        tstream<<"\n";

        tstream<<"# network interface to bind to (this is the only interface that will serve files)"<<endl;
        if(this->network_interface != "")
            tstream<<"network_interface="<<network_interface<<endl;
        else
            tstream<<"#network_interface=eth0"<<endl;
        tstream<<"\n";

        tstream<<"# set this to the directory you want scanned."<<endl;
        tstream<<"# * if have multiple directories, you can have multiple media_dir= lines"<<endl;
        tstream<<"# * if you want to restrict a media_dir to a specific content type, you"<<endl;
        tstream<<"#   can prepend the type, followed by a comma, to the directory:"<<endl;
        tstream<<"#   + \"A\" for audio  (eg. media_dir=A,/home/jmaggard/Music)"<<endl;
        tstream<<"#   + \"V\" for video  (eg. media_dir=V,/home/jmaggard/Videos)"<<endl;
        tstream<<"#   + \"P\" for images (eg. media_dir=P,/home/jmaggard/Pictures)"<<endl;
        if(!media_dir.isEmpty())
        {
            for(int i=0;i<media_dir.size();i++)
            {
                tstream<<"media_dir="<<media_dir[i]<<endl;
            }
        }
        else
            tstream<<"#media_dir=/home/user/MyDocs"<<endl;
        tstream<<"\n";

        tstream<<"# set this if you want to customize the name that shows up on your clients"<<endl;
        if(friendly_name!="")
            tstream<<"friendly_name="<<friendly_name<<endl;
        else
            tstream<<"#friendly_name=My DLNA Server"<<endl;
        tstream<<"\n";

        tstream<<"# set this if you would like to specify the directory where you want MiniDLNA to store its database and album art cache"<<endl;
        if(db_dir!="")
            tstream<<"db_dir="<<db_dir<<endl;
        else
            tstream<<"#db_dir=/var/cache/minidlna"<<endl;
        tstream<<"\n";

        tstream<<"# set this if you would like to specify the directory where you want MiniDLNA to store its log file"<<endl;
        if(log_dir!="")
            tstream<<"log_dir="<<log_dir<<endl;
        else
            tstream<<"#log_dir=/var/log"<<endl;
        tstream<<"\n";

        tstream<<"# this should be a list of file names to check for when searching for album art"<<endl;
        tstream<<"# note: names should be delimited with a forward slash (\"/\")"<<endl;
        if(album_art_names!="")
            tstream<<"album_art_names="<<album_art_names<<endl;
        else
            tstream<<"album_art_names=Cover.jpg/cover.jpg/AlbumArtSmall.jpg/albumartsmall.jpg/AlbumArt.jpg/albumart.jpg/Album.jpg/album.jpg/Folder.jpg/folder.jpg/Thumb.jpg/thumb.jpg"<<endl;
        tstream<<"\n";

        tstream<<"# set this to no to disable inotify monitoring to automatically discover new files"<<endl;
        tstream<<"# note: the default is yes"<<endl;
        if(inotify!="")
            tstream<<"inotify="<<inotify<<endl;
        else
            tstream<<"inotify=yes"<<endl;
        tstream<<"\n";

        tstream<<"# set this to yes to enable support for streaming .jpg and .mp3 files to a TiVo supporting HMO"<<endl;
        if(enable_tivo!="")
            tstream<<"enable_tivo="<<enable_tivo<<endl;
        else
            tstream<<"enable_tivo=no"<<endl;
        tstream<<"\n";

        tstream<<"# set this to strictly adhere to DLNA standards."<<endl;
        tstream<<"# * This will allow server-side downscaling of very large JPEG images,"<<endl;
        tstream<<"#   which may hurt JPEG serving performance on (at least) Sony DLNA products."<<endl;
        if(strict_dlna!="")
            tstream<<"strict_dlna="<<strict_dlna<<endl;
        else
            tstream<<"strict_dlna=no"<<endl;
        tstream<<"\n";

        tstream<<"# default presentation url is http address on port 80"<<endl;
        if(presentation_url!="")
            tstream<<"presentation_url="<<presentation_url<<endl;
        else
            tstream<<"#presentation_url=http://www.mylan/index.php"<<endl;
        tstream<<"\n";

        tstream<<"# notify interval in seconds. default is 895 seconds."<<endl;
        if(notify_interval!="")
            tstream<<"notify_interval="<<notify_interval<<endl;
        else
            tstream<<"notify_interval=900"<<endl;
        tstream<<"\n";

        tstream<<"# serial and model number the daemon will report to clients"<<endl;
        tstream<<"# in its XML description"<<endl;
        if(serial!="")
            tstream<<"serial="<<serial<<endl;
        else
            tstream<<"serial=12345678"<<endl;
        if(model_number!="")
            tstream<<"model_number="<<model_number<<endl;
        else
            tstream<<"model_number=1"<<endl;
        tstream<<"\n";
        data.close();
        settings_changed = false;
        emit settingsChanged(settings_changed);
        return true;
    }
    data.close();
    return false;
}

void MiniDLNA::ServerCommand(QString cmd)
{
    QProcess server;
    QByteArray output;
    server.setProcessChannelMode(QProcess::MergedChannels);
    server.start("sudo /etc/init.d/mdlna " + cmd);
    while(server.waitForReadyRead()){
        output+=server.readAll();
    }
    //qDebug()<<output;
}
