/*
 *  message blocker
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <gtk/gtk.h>
#include <hildon/hildon.h>

#include <telepathy-glib/dbus.h>

#include <telepathy-glib/svc-generic.h>
#include <telepathy-glib/svc-client.h>
#include <telepathy-glib/defs.h>
#include <gconf/gconf-client.h>

#include "message_blocker_sp.h"
#include "message_blocker_approver.h"
#include <rtcom-eventlogger/eventlogger.h>


#define MESSAGE_BLOCKER_STATUS_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, TYPE_MESSAGE_BLOCKER_STATUS_PUGIN, MessageBlockerStatusPluginPrivate))

#define GC_ROOT "/apps/maemo/message-blocker"
#define BLOCK_LIST_GCONF_PATH GC_ROOT 
#define ACCOUNTS_BLOCK_LIST_GCONF_PATH GC_ROOT "/accounts" 
#define BLOCKER_ACTIVE_GCONF_PATH GC_ROOT "/active"
#define DELETE_MSG_GCONF_PATH GC_ROOT "/delete-messages"

struct _MessageBlockerStatusPluginPrivate
{
  MessageBlockerApprover* approver;
  DBusGConnection* dbus;
  TpDBusDaemon* bus_daemon;
  guint gconf_notify_handler;
  gboolean delete_msg;
  RTComEl* rtcomel;
};

HD_DEFINE_PLUGIN_MODULE(MessageBlockerStatusPlugin, message_blocker_status_plugin, HD_TYPE_STATUS_MENU_ITEM);

static gchar*
_account_name_from_gconf_key(const gchar* key)
{

  gchar** elements = g_strsplit(key, "/", 7);
  gchar* account_name = g_strdup(elements[5]);
  g_strfreev(elements);
  return account_name;
}

static gboolean
_read_blocker_active_setting()
{
  GConfClient* client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
    return TRUE;

  gboolean active = gconf_client_get_bool(client, BLOCKER_ACTIVE_GCONF_PATH, NULL);
  g_object_unref(client);
  return active;
}

static gboolean
_read_delete_msg_setting()
{
  GConfClient* client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
    return TRUE;

  gboolean delete_msg = gconf_client_get_bool(client, DELETE_MSG_GCONF_PATH, NULL);
  g_object_unref(client);
  return delete_msg;
}

static void
message_blocker_status_plugin_show_icon(MessageBlockerStatusPlugin* plugin, gboolean show)
{
  if(!show)
  {
    hd_status_plugin_item_set_status_area_icon(HD_STATUS_PLUGIN_ITEM(plugin), NULL);
    return;
  }
  GdkPixbuf* pixbuf = gdk_pixbuf_new_from_file("/usr/share/pixmaps/message_blocker/message_blocker.png", NULL);

  if(pixbuf)
  {
    hd_status_plugin_item_set_status_area_icon(HD_STATUS_PLUGIN_ITEM(plugin), pixbuf);
    g_object_unref(pixbuf);
  }
}

static void
_gconf_block_list_changed(GConfClient* client,
			  guint cnxn_id,
			  GConfEntry* entry,
			  gpointer user_data)
{
  MessageBlockerStatusPlugin* plugin = MESSAGE_BLOCKER_STATUS_PLUGIN(user_data);
  if(g_strcmp0(entry->key, BLOCKER_ACTIVE_GCONF_PATH) == 0)
  {
    gboolean active = _read_blocker_active_setting();
    if(active)
    {
      message_blocker_status_plugin_show_icon(plugin, TRUE);
      message_blocker_approver_set_active(plugin->priv->approver,
					  TRUE);
    }
    else
    {
      message_blocker_status_plugin_show_icon(plugin, FALSE);
      message_blocker_approver_set_active(plugin->priv->approver,
					  FALSE);
    }
  }
  else if(g_strcmp0(entry->key, DELETE_MSG_GCONF_PATH) == 0)
  {
    plugin->priv->delete_msg = _read_delete_msg_setting();
  }
  else
  {
    gchar* account_name = _account_name_from_gconf_key(entry->key);
    gchar* account_name_unescaped = gconf_unescape_key(account_name, -1);
    GSList* block_list = gconf_client_get_list(client, entry->key, GCONF_VALUE_STRING, NULL);
    message_blocker_approver_set_block_list(plugin->priv->approver, account_name_unescaped, block_list);
    g_free(account_name_unescaped);
    g_free(account_name);
  }
}

static void
_unregister_gconf_changes(MessageBlockerStatusPlugin* plugin)
{
  GConfClient* client = NULL;
  client = gconf_client_get_default();
  if(GCONF_IS_CLIENT(client))
  {
    gconf_client_notify_remove(client, 
			       plugin->priv->gconf_notify_handler);
    plugin->priv->gconf_notify_handler = 0;
    gconf_client_remove_dir(client, BLOCK_LIST_GCONF_PATH,
			    NULL);
    g_object_unref(client);
  }
}

static void
_read_gconf_settings(MessageBlockerStatusPlugin* plugin)
{
  GConfClient* client = NULL;
  GError* error = NULL;
  client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
  {
    return;
  }

  GSList* entries = gconf_client_all_dirs(client,
					  ACCOUNTS_BLOCK_LIST_GCONF_PATH,
					  &error);
  if(error)
  {
    g_warning("no blocklist entries in gconf %s\n", error->message);
    g_error_free(error);
    error = NULL;
  }
  else
  {
    GSList* entry = entries;
    while(entry)
    {
      gchar* gconf_dir = g_strdup_printf("%s/block-list", entry->data);
      gchar* account_name = _account_name_from_gconf_key(gconf_dir);
      gchar* account_name_unescaped = gconf_unescape_key(account_name, -1);
      GSList* block_list = gconf_client_get_list(client, gconf_dir, GCONF_VALUE_STRING, NULL);
      message_blocker_approver_set_block_list(plugin->priv->approver, account_name_unescaped, block_list);
      g_free(account_name_unescaped);
      g_free(account_name);
      g_free(gconf_dir);
      entry = entry->next;
    }
    g_slist_free(entries);
  }
  
  g_object_unref(client);
}

static void
_register_gconf_changes(MessageBlockerStatusPlugin* plugin)
{
  GConfClient* client = NULL;
  GError* error = NULL;
  client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
  {
    return;
  }
  
  gconf_client_add_dir(client,
		       BLOCK_LIST_GCONF_PATH,	       
		       GCONF_CLIENT_PRELOAD_NONE,
		       &error);
  if(error != NULL)
  {
    g_warning("can not listen on gconf client changes %s\n", error->message);
    g_error_free(error);
    error = NULL;
    return;
  }
  
  plugin->priv->gconf_notify_handler = 
    gconf_client_notify_add(client, BLOCK_LIST_GCONF_PATH, _gconf_block_list_changed, plugin, NULL, &error);
  if(error != NULL)
  {
    g_warning("can not add notify on gconf client %s\n", error->message);
    g_error_free(error);
    error = NULL;
    return;
  }
  g_object_unref(client);
}

static void
message_blocker_status_plugin_finalize(GObject* object)
{
  MessageBlockerStatusPlugin* plugin = MESSAGE_BLOCKER_STATUS_PLUGIN(object);

  tp_dbus_daemon_release_name(plugin->priv->bus_daemon,
			      TP_CLIENT_BUS_NAME_BASE MESSAGE_BLOCKER_APPROVER_NAME, NULL);
  dbus_g_connection_unregister_g_object(plugin->priv->dbus, G_OBJECT(plugin->priv->approver));
  if(plugin->priv->rtcomel)
  {
    g_object_unref(plugin->priv->rtcomel);
    plugin->priv->rtcomel = NULL;
  }
  if(plugin->priv->gconf_notify_handler != 0)
    _unregister_gconf_changes(plugin);
  plugin->priv->gconf_notify_handler = 0;

  g_object_unref(plugin->priv->approver);
  g_object_unref(plugin->priv->bus_daemon);
  dbus_g_connection_unref(plugin->priv->dbus);

  //  G_OBJECT_CLASS(message_blocker_status_plugin_parent_class)->finalize(object);
}

static void
message_blocker_status_plugin_class_finalize(MessageBlockerStatusPluginClass* klass)
{
}

static gboolean
_init_dbus_daemon(gpointer user_data)
{
  MessageBlockerStatusPlugin* plugin = MESSAGE_BLOCKER_STATUS_PLUGIN(user_data);
  GError* error = NULL;
  plugin->priv->approver = message_blocker_approver_new();
  plugin->priv->bus_daemon = tp_dbus_daemon_dup(&error);

  if(error)
  {
    g_warning("error %s\n", error->message);
    g_error_free(error);
    error = NULL;
  }

  tp_dbus_daemon_request_name(plugin->priv->bus_daemon,
			      TP_CLIENT_BUS_NAME_BASE MESSAGE_BLOCKER_APPROVER_NAME,
			      TRUE, &error);

  if(error)
  {
    g_warning("error %s\n", error->message);
    g_error_free(error);
    error = NULL;
  }

  plugin->priv->dbus = hd_status_plugin_item_get_dbus_g_connection(HD_STATUS_PLUGIN_ITEM(&plugin->parent),
								   DBUS_BUS_SESSION,NULL);
  
  dbus_g_connection_register_g_object(plugin->priv->dbus, TP_CLIENT_OBJECT_PATH_BASE MESSAGE_BLOCKER_APPROVER_NAME, G_OBJECT(plugin->priv->approver));
  return FALSE;
}

static void
check_new_rtcom_event_cb(RTComEl* rtcomel,
			 gint event_id,
			 const gchar* local_uid,
			 const gchar* remote_uid,
			 const gchar* remote_ebook_uid,
			 const gchar* group_uid,
			 const gchar* service,
			 MessageBlockerStatusPlugin* plugin)
{
  if(plugin->priv->delete_msg &&
     _read_blocker_active_setting()  && 
     block_from_account(plugin->priv->approver, local_uid, remote_uid))
  {
      GError* error = NULL;
      rtcom_el_delete_event(plugin->priv->rtcomel,
			    event_id,
			    &error);
      if(error)
      {
	hildon_banner_show_information(NULL,
				       NULL,
				       error->message);
	
      }
      g_clear_error(&error);
  }
}

static void
_init_eventlogger_listener(MessageBlockerStatusPlugin* plugin)
{
  plugin->priv->rtcomel = rtcom_el_new();
  g_signal_connect(plugin->priv->rtcomel,
		   "new-event",
		   G_CALLBACK(check_new_rtcom_event_cb),
		   plugin);
}

static void
message_blocker_status_plugin_init(MessageBlockerStatusPlugin* plugin)
{
  plugin->priv = MESSAGE_BLOCKER_STATUS_PLUGIN_GET_PRIVATE(plugin);
  plugin->priv->gconf_notify_handler = 0;

  _init_dbus_daemon(plugin);

  _read_gconf_settings(plugin);
  _init_eventlogger_listener(plugin);
  plugin->priv->delete_msg = _read_delete_msg_setting();
  if(_read_blocker_active_setting())
  {
    message_blocker_approver_set_active(plugin->priv->approver, TRUE);
    message_blocker_status_plugin_show_icon(plugin, TRUE);
  }
  else
  {
    message_blocker_approver_set_active(plugin->priv->approver, FALSE);
    message_blocker_status_plugin_show_icon(plugin, FALSE);
  }
  _register_gconf_changes(plugin);
}

static void
message_blocker_status_plugin_class_init(MessageBlockerStatusPluginClass* klass)
{
  g_type_class_add_private(klass, sizeof(MessageBlockerStatusPluginPrivate));
  G_OBJECT_CLASS(klass)->finalize = (GObjectFinalizeFunc)message_blocker_status_plugin_finalize;
}
