/*
 *  media player widget for the maemo desktop.
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <libintl.h>
#include <locale.h>
#include <gconf/gconf-client.h>	
#include <gdk/gdkx.h>
#include <libmafw/mafw.h>
#include <libmafw/mafw-log.h>
#include <libmafw/mafw-registry.h>
#include <libmafw-shared/mafw-shared.h>
#include <gconf/gconf-client.h>	

#include "media-player-widget.h"

#define MEDIA_PLAYER_WIDGET_HOME_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MEDIA_PLAYER_WIDGET_TYPE_HOME_PLUGIN, MediaPlayerWidgetHomePluginPrivate))

HD_DEFINE_PLUGIN_MODULE(MediaPlayerWidgetHomePlugin, media_player_widget_home_plugin, HD_TYPE_HOME_PLUGIN_ITEM)
#define DEFAULT_WIDGET_WIDTH 256
#define DEFAULT_WIDGET_HEIGHT 64
#define MEDIAPLAYER_RENDERER "Mafw-Gst-Renderer"
#define MEDIAPLAYER_SOURCE "Mafw-Tracker-Source"
#define PAUSE_ICON "/usr/share/pixmaps/media-player-widget/pause.png"
#define PLAY_ICON "/usr/share/pixmaps/media-player-widget/play.png"
#define BACKWARD_ICON "/usr/share/pixmaps/media-player-widget/backward.png"
#define FORWARD_ICON "/usr/share/pixmaps/media-player-widget/forward.png"
#define LABEL_BACKGROUND_ICON "/usr/share/pixmaps/media-player-widget/label_background.png"
#define DEFAULT_ALBUM_ICON "/usr/share/pixmaps/media-player-widget/player.png"
#define MEDIAPLAYER_LAST_PLAYING_SONG_PATH "/apps/mediaplayer/last_playing_song"

#define HD_APP_MGR_DBUS_NAME "com.nokia.HildonDesktop.AppMgr"
#define HD_APP_MGR_DBUS_PATH "/com/nokia/HildonDesktop/AppMgr"
#define HD_APP_MGR_DBUS_IFACE "com.nokia.HildonDesktop.AppMgr"
#define HD_APP_MGR_LAUNCH_APPLICATION "LaunchApplication"
#define MEDIAPLAYER_APP_NAME "mediaplayer"

struct _MediaPlayerWidgetHomePluginPrivate
{
  MafwRegistry* mafw_registry;
  MafwRenderer* mafw_renderer;
  MafwPlaylist* mafw_playlist;
  MafwPlaylistManager* mafw_playlist_manager;
  GtkWidget* play_pause_icon;
  GtkWidget* album_icon;
  GtkWidget* label_background;
  GdkPixbuf* background_pixbuf;
  GdkPixbuf* album_pixbuf;
  gchar* title;
  gchar* artist;
  gchar* album;
  MafwPlayState state;
  gulong on_media_changed_handler_id;
  gulong on_state_changed_handler_id;
  gulong on_metadata_changed_handler_id;
  gulong on_playlist_changed_handler_id;
  gulong on_error_handler_id;
  gulong on_renderer_added_id;
  gulong on_renderer_removed_id;
};


static gboolean
media_player_widget_expose(GtkWidget* widget, GdkEventExpose* event)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(widget);
  cairo_t *cr;
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_region(cr, event->region);
  cairo_clip(cr);
  cairo_destroy(cr);
  return GTK_WIDGET_CLASS(media_player_widget_home_plugin_parent_class)->expose_event(widget, event);
}

static int
_mediaplayer_last_playlist_index()
{
  GConfClient* client = NULL;
  GError* error = NULL;
  client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
  {
    return 0;
  }
  int last_playing_song = gconf_client_get_int(client, MEDIAPLAYER_LAST_PLAYING_SONG_PATH, &error);
  if(error)
  {
    g_warning("can not get current view (%s)\n", error->message);
    g_error_free(error);
    error = NULL;
    last_playing_song = 0;
  }
  g_object_unref(client);
  return last_playing_song;
}

static void
_set_image_for_state(MafwPlayState state, MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  if(state == Playing)
  {
    gtk_image_set_from_file(GTK_IMAGE(desktop_plugin->priv->play_pause_icon),
			    PAUSE_ICON);
  }
  else if(state == -1 || state == Stopped || state == Paused)
  {
    gtk_image_set_from_file(GTK_IMAGE(desktop_plugin->priv->play_pause_icon),
			    PLAY_ICON);
  }
}

static void
_playlist_item_metadata_cb(MafwPlaylist *pls,
			   guint index,
			   const gchar *object_id,
			   GHashTable *metadata,
			   gpointer user_data)
{
  GError* error = NULL;
  GValue* value;
  const gchar* title;
  value = mafw_metadata_first(metadata, MAFW_METADATA_KEY_TITLE);
  if(value == NULL)
    return;

  title = g_value_get_string(value);

  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  if(title != NULL)
  {
    if(desktop_plugin->priv->title)
    {
      g_free(desktop_plugin->priv->title);
    }
    desktop_plugin->priv->title = g_strdup(title);
  }
  value = mafw_metadata_first(metadata, MAFW_METADATA_KEY_ARTIST);

  if(G_VALUE_HOLDS_STRING(value) && (title = g_value_get_string(value)) != NULL)
  {
    if(desktop_plugin->priv->artist)
    {
      g_free(desktop_plugin->priv->artist);
    }
    desktop_plugin->priv->artist = g_strdup(title);
  }
  value = mafw_metadata_first(metadata, MAFW_METADATA_KEY_ALBUM);
  if(G_VALUE_HOLDS_STRING(value) && (title = g_value_get_string(value)) != NULL)
  {
    if(desktop_plugin->priv->album)
    {
      g_free(desktop_plugin->priv->album);
    }
    desktop_plugin->priv->album = g_strdup(title);
  }

  value = mafw_metadata_first(metadata, MAFW_METADATA_KEY_ALBUM_ART_SMALL_URI);
  if(value != NULL)
  {
    title = g_value_get_string(value);
    gchar* filename = NULL;
    if(title != NULL && (filename = g_filename_from_uri(title,
						       NULL,
							NULL)) != NULL)
    {
      GdkPixbuf* pixbuf = gdk_pixbuf_new_from_file_at_size(filename,
							   64, 64, NULL);
      if(pixbuf)
      {
	
	gtk_image_set_from_pixbuf(GTK_IMAGE(desktop_plugin->priv->album_icon),
				  pixbuf);
	g_object_unref(pixbuf);
	if(desktop_plugin->priv->album_pixbuf)
	{
	  g_object_unref(desktop_plugin->priv->album_pixbuf);
	  desktop_plugin->priv->album_pixbuf = NULL;
	}
      }
      g_free(filename);
    }
  }
  else
  {
    if(desktop_plugin->priv->album_pixbuf == NULL)
    {
      desktop_plugin->priv->album_pixbuf = gdk_pixbuf_new_from_file_at_size(DEFAULT_ALBUM_ICON,
									    64, 64, NULL);
      gtk_image_set_from_pixbuf(GTK_IMAGE(desktop_plugin->priv->album_icon),
				desktop_plugin->priv->album_pixbuf);
    }
  }
  gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
}

static void
_on_renderer_get_status_cb(MafwRenderer* mafw_renderer,
			   MafwPlaylist* playlist,
			   guint index,
			   MafwPlayState state,
			   const gchar* object_id,
			   gpointer user_data,
			   const GError* error)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  if(error)
  {
    g_warning("error on playbackstate cb %s\n", error->message);
  }
  const gchar *const *keys;
  keys = MAFW_SOURCE_LIST(MAFW_METADATA_KEY_TITLE,
			  MAFW_METADATA_KEY_ARTIST,
			  MAFW_METADATA_KEY_ALBUM,
			  MAFW_METADATA_KEY_ALBUM_ART_SMALL_URI);
  mafw_playlist_get_items_md(playlist,
			     index, index,
			     keys,
			     _playlist_item_metadata_cb,
			     desktop_plugin, 
			     NULL);
  desktop_plugin->priv->state = state;
  _set_image_for_state(state, MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data));
}

static void
_on_media_changed_cb(MafwRenderer* mafw_renderer, gint index, gchar* object_id, gpointer user_data)
{
  g_warning("on media changed\n");
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  mafw_renderer_get_status(desktop_plugin->priv->mafw_renderer,
			   _on_renderer_get_status_cb, 
			   desktop_plugin);
}

static void
_on_state_changed_cb(MafwRenderer* mafw_renderer, gint state, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  desktop_plugin->priv->state = state;
  _set_image_for_state(state, MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data));
}

static void
_on_metadata_changed_cb(MafwRenderer* mafw_renderer, gchar *name, GValueArray *value, gpointer user_data)
{
  /* 
  if(value->n_values == 1)
  {
    GValue* v;
    v = g_value_array_get_nth(value, 0);
    if(v && G_VALUE_HOLDS_STRING(v))
    {
      const gchar* value_str = g_value_get_string(v);
      g_warning("value %s\n", value_str);
    }
  }
  */
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  mafw_renderer_get_status(desktop_plugin->priv->mafw_renderer,
			   _on_renderer_get_status_cb, 
			   desktop_plugin);
}

static void
_on_playlist_changed_cb(MafwRenderer* mafw_renderer, GObject* playlist, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  mafw_renderer_get_status(desktop_plugin->priv->mafw_renderer,
			   _on_renderer_get_status_cb, 
			   desktop_plugin);
}

static void
_on_error_cb(MafwExtension* mafw_extension, guint domain, gint code,
	     gchar *message,
	     gpointer user_data)
{
  g_warning("error message %s\n", message);
}

static void
_disconnect_renderer_signal_handler(MafwRenderer* mafw_renderer, MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  if(desktop_plugin->priv->on_media_changed_handler_id)
  {
    g_signal_handler_disconnect(mafw_renderer,
				desktop_plugin->priv->on_media_changed_handler_id);
    desktop_plugin->priv->on_media_changed_handler_id = 0;
  }
  if(desktop_plugin->priv->on_state_changed_handler_id)
  {
    g_signal_handler_disconnect(mafw_renderer,
				 desktop_plugin->priv->on_state_changed_handler_id);
    desktop_plugin->priv->on_state_changed_handler_id = 0;
  }
  if(desktop_plugin->priv->on_metadata_changed_handler_id)
  {
    g_signal_handler_disconnect(mafw_renderer,
				 desktop_plugin->priv->on_metadata_changed_handler_id);
    desktop_plugin->priv->on_metadata_changed_handler_id = 0;
  }
  if(desktop_plugin->priv->on_playlist_changed_handler_id)
  {
    g_signal_handler_disconnect(mafw_renderer,
				desktop_plugin->priv->on_playlist_changed_handler_id);
    desktop_plugin->priv->on_playlist_changed_handler_id = 0;
  }
  /*
  if(desktop_plugin->priv->on_error_handler_id)
  {
    g_signal_handler_disconnect(mafw_renderer,
				 desktop_plugin->priv->on_error_handler_id);
    desktop_plugin->priv->on_error_handler_id = 0;
  }
  */
}

static void
_connect_renderer_signal_handler(MafwRenderer* mafw_renderer, MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  // renderer signals
  desktop_plugin->priv->on_media_changed_handler_id = g_signal_connect(mafw_renderer,
								       "media-changed",
								       G_CALLBACK (_on_media_changed_cb),
								       desktop_plugin);

  desktop_plugin->priv->on_state_changed_handler_id = g_signal_connect(mafw_renderer,
								       "state-changed",
								       G_CALLBACK (_on_state_changed_cb),
								       desktop_plugin);

  desktop_plugin->priv->on_metadata_changed_handler_id = g_signal_connect(mafw_renderer,
									  "metadata-changed",
									  G_CALLBACK (_on_metadata_changed_cb),
									  desktop_plugin);

  desktop_plugin->priv->on_playlist_changed_handler_id = g_signal_connect(mafw_renderer,
									  "playlist-changed",
									  G_CALLBACK (_on_playlist_changed_cb),
									  desktop_plugin);
  /*
  // extension signals
  desktop_plugin->priv->on_error_handler_id = g_signal_connect(mafw_renderer,
							       "error",
							       G_CALLBACK (_on_error_cb),
							       desktop_plugin);
  */
}

static gboolean 
_init_renderer_playlist(gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  desktop_plugin->priv->mafw_playlist_manager = mafw_playlist_manager_get();
  MafwProxyPlaylist* playlist = mafw_playlist_manager_create_playlist(desktop_plugin->priv->mafw_playlist_manager,
								      "FmpAudioPlaylist", NULL);
  const gchar *const *keys;
  keys = MAFW_SOURCE_LIST(MAFW_METADATA_KEY_TITLE,
			  MAFW_METADATA_KEY_ARTIST,
			  MAFW_METADATA_KEY_ALBUM,
			  MAFW_METADATA_KEY_ALBUM_ART_SMALL_URI);
  int last_playing_song = _mediaplayer_last_playlist_index();
  mafw_playlist_get_items_md(MAFW_PLAYLIST(playlist),
			     last_playing_song, last_playing_song,
			     keys,
			     _playlist_item_metadata_cb,
			     desktop_plugin, 
			     NULL);
  return FALSE;
}

static void
_on_renderer_get_status_cb_init(MafwRenderer* mafw_renderer,
				MafwPlaylist* playlist,
				guint index,
				MafwPlayState state,
				const gchar* object_id,
				gpointer user_data,
				const GError* error)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  if(error)
  {
    g_warning("error on playbackstate cb %s\n", error->message);
  }
  if(playlist == NULL)
  {
    g_timeout_add(2000,
		  _init_renderer_playlist,
		  desktop_plugin);
    return;
  }
  const gchar *const *keys;
  keys = MAFW_SOURCE_LIST(MAFW_METADATA_KEY_TITLE,
			  MAFW_METADATA_KEY_ARTIST,
			  MAFW_METADATA_KEY_ALBUM,
			  MAFW_METADATA_KEY_ALBUM_ART_SMALL_URI);
  mafw_playlist_get_items_md(playlist,
			     index, index,
			     keys,
			     _playlist_item_metadata_cb,
			     desktop_plugin, 
			     NULL);
  desktop_plugin->priv->state = state;
  _set_image_for_state(state, MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data));
}

static void
_on_renderer_added(MafwRegistry* mafw_registry,
		   GObject* renderer,
		   gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  if(g_strcmp0(mafw_extension_get_name(MAFW_EXTENSION(renderer)), MEDIAPLAYER_RENDERER) == 0)
  {
    MafwRenderer* mafw_renderer = MAFW_RENDERER(renderer);
    desktop_plugin->priv->mafw_renderer = mafw_renderer;
    g_object_ref(mafw_renderer);
    _connect_renderer_signal_handler(mafw_renderer, desktop_plugin);
    mafw_renderer_get_status(desktop_plugin->priv->mafw_renderer,
			     _on_renderer_get_status_cb_init, 
			     desktop_plugin);
  }
}

static void
_on_renderer_removed(MafwRegistry* mafw_registry,
		     GObject* renderer,
		     gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  if(g_strcmp0(mafw_extension_get_name(MAFW_EXTENSION(renderer)), MEDIAPLAYER_RENDERER) == 0)
  {
    if(desktop_plugin->priv->mafw_renderer)
    {
      _disconnect_renderer_signal_handler(desktop_plugin->priv->mafw_renderer, desktop_plugin);
      g_object_unref(desktop_plugin->priv->mafw_renderer);
      desktop_plugin->priv->mafw_renderer = NULL;
    }
  }
}

static void
_connect_registry_signal_handler(MafwRegistry* mafw_registry, MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  desktop_plugin->priv->on_renderer_added_id = g_signal_connect(mafw_registry, 
								"renderer_added", 
								G_CALLBACK(_on_renderer_added), 
								desktop_plugin);

  desktop_plugin->priv->on_renderer_removed_id = g_signal_connect(mafw_registry, 
								  "renderer_removed", 
								  G_CALLBACK(_on_renderer_removed), 
								  desktop_plugin);
}

static void
_disconnect_registry_signal_handler(MafwRegistry* mafw_registry, MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  if(desktop_plugin->priv->on_renderer_added_id != 0)
  {
    g_signal_handler_disconnect(mafw_registry,
				desktop_plugin->priv->on_renderer_added_id);
    desktop_plugin->priv->on_renderer_added_id = 0;
  }
  if(desktop_plugin->priv->on_renderer_removed_id != 0)
  {
    g_signal_handler_disconnect(mafw_registry, desktop_plugin->priv->on_renderer_removed_id);
    desktop_plugin->priv->on_renderer_removed_id = 0;
  }
}

static void
_init_mafw(MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  MafwRegistry* mafw_registry = NULL;
  GError* error = NULL;
  mafw_log_init("mafw-example:ALL");
  mafw_registry = MAFW_REGISTRY(mafw_registry_get_instance());
  if(mafw_registry == NULL)
  {
    g_warning("can not init mafw registry\n");
    return;
  }
  desktop_plugin->priv->mafw_registry = mafw_registry;
  mafw_shared_init(mafw_registry, &error);
  if(error)
  {
    g_warning("mafw out-of-process discorvery failed: %s\n", error->message);
    g_error_free(error);
    error = NULL;
  }
  GList* renderers = mafw_registry_get_renderers(mafw_registry);
  if(renderers)
  {
    GList* renderer = renderers;
    while(renderer)
    {
      MafwRenderer* mafw_renderer = MAFW_RENDERER(renderer->data);
      if(g_strcmp0(mafw_extension_get_name(MAFW_EXTENSION(mafw_renderer)), MEDIAPLAYER_RENDERER) == 0)
      {
	desktop_plugin->priv->mafw_renderer = mafw_renderer;
	g_object_ref(mafw_renderer);
	_connect_renderer_signal_handler(mafw_renderer, desktop_plugin);
	mafw_renderer_get_status(desktop_plugin->priv->mafw_renderer,
				 _on_renderer_get_status_cb, 
				 desktop_plugin);
      }
      renderer = renderer->next;
    }
  }
  _connect_registry_signal_handler(mafw_registry, desktop_plugin);
  //  g_signal_connect(mafw_registry, "source_added", G_CALLBACK(source_added_cb), desktop_plugin);
  //  g_signal_connect(mafw_registry, "source_removed", G_CALLBACK(source_removed_cb), NULL);
}

static void
media_player_widget_realize(GtkWidget* widget)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(widget);
  GdkScreen *screen = gtk_widget_get_screen(widget);
  gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
  gtk_widget_set_app_paintable(widget, TRUE);
  _init_mafw(desktop_plugin);
  GTK_WIDGET_CLASS(media_player_widget_home_plugin_parent_class)->realize(widget);
}

static void
_on_current_desktop(GtkWidget* widget,
		    gpointer data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(widget);
  gboolean on;
  g_object_get(widget, "is-on-current-desktop", &on, NULL);
  if(on)
  {
    gtk_widget_queue_draw(widget);
  }
}

static void
_on_playback_state_changed(MafwRenderer* renderer,
			   gpointer user_data,
			   const GError* error)
{
  if(error)
  {
    g_warning("error on playbackstate cb %s\n", error->message);
  }
  else
  {
    if(MAFW_IS_RENDERER(renderer))
    {
      /* MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data); */
      /* mafw_renderer_get_status(renderer, */
      /* 			       _on_renderer_get_status_cb, */
      /* 			       desktop_plugin); */
    }
  }
}

static void
_on_backward_button_press(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  if(desktop_plugin->priv->mafw_renderer)
  {
    mafw_renderer_previous(desktop_plugin->priv->mafw_renderer,
			   _on_playback_state_changed, desktop_plugin);
  }
}

static void
_on_forward_button_press(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  mafw_renderer_next(desktop_plugin->priv->mafw_renderer,
		     _on_playback_state_changed, desktop_plugin);
}

static void
_on_play_pause_button_press(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  if(desktop_plugin->priv->state == Playing)
  {
    mafw_renderer_pause(desktop_plugin->priv->mafw_renderer,
			_on_playback_state_changed, desktop_plugin);
  }
  else if(desktop_plugin->priv->state == Stopped)
  {
    mafw_renderer_play(desktop_plugin->priv->mafw_renderer,
		       _on_playback_state_changed, desktop_plugin);

  }
  else if(desktop_plugin->priv->state == Paused)
  {
    mafw_renderer_resume(desktop_plugin->priv->mafw_renderer,
			 _on_playback_state_changed, desktop_plugin);

  }
  else if(desktop_plugin->priv->state == -1)
  {
    desktop_plugin->priv->mafw_playlist_manager = mafw_playlist_manager_get();
    MafwProxyPlaylist* playlist = mafw_playlist_manager_create_playlist(desktop_plugin->priv->mafw_playlist_manager,
									"FmpAudioPlaylist", NULL);
    if(mafw_renderer_assign_playlist(desktop_plugin->priv->mafw_renderer,
				     MAFW_PLAYLIST(playlist),
				     NULL))
    {
      int index = _mediaplayer_last_playlist_index();
      mafw_renderer_play(desktop_plugin->priv->mafw_renderer,
			 _on_playback_state_changed, desktop_plugin);
      mafw_renderer_goto_index(desktop_plugin->priv->mafw_renderer,
			       index,
			       NULL, NULL);
    }
  }
}

typedef struct _playlist_selection_data_t
{
  GtkWidget* dialog;
  MediaPlayerWidgetHomePlugin* desktop_plugin;
  GtkListStore* playlist_store;
} playlist_selection_data_t;

static void
_on_playlist_dialog_response(GtkDialog* dialog,
			     gint response_id,
			     gpointer user_data)
{
  playlist_selection_data_t* playlist_selection_data = (playlist_selection_data_t*)user_data;
  gtk_widget_hide(GTK_WIDGET(dialog));
  gtk_widget_destroy(playlist_selection_data->dialog);
  g_object_unref(playlist_selection_data->playlist_store);
  g_free(playlist_selection_data);
}

static void
_playlist_item_request_cb(MafwSource *source,
			  guint browse_id,
			  gint remaining,
			  guint index,
			  const gchar *object_id,
			  GHashTable *metadata,
			  gpointer user_data,
			  const GError *error)
{
  MafwPlaylist* playlist = MAFW_PLAYLIST(user_data);
  if(object_id != NULL) 
  {
    mafw_playlist_append_item(playlist,
			      object_id,
			      NULL);
  }
}

static MafwSource*
_get_mafw_source(MafwRegistry* mafw_registry)
{
  GList* sources = mafw_registry_get_sources(mafw_registry);
  GList* source = sources;
  while(source)
  {
    MafwSource* mafw_source = MAFW_SOURCE(source->data);
    if(g_strcmp0(mafw_extension_get_name(MAFW_EXTENSION(mafw_source)), MEDIAPLAYER_SOURCE) == 0)
    {
      return mafw_source;
    }
    else
    {
      source = source->next;
    }
  }
  return NULL;
}

static void
_on_playlist_item_selected(HildonTouchSelector* selector, 
			   gint column,
			   gpointer user_data)
{
  playlist_selection_data_t* playlist_selection_data = (playlist_selection_data_t*)user_data;
  GtkTreeIter iter;
  hildon_touch_selector_get_selected(HILDON_TOUCH_SELECTOR(selector),
				     0, &iter);
  gchar* playlist_name = NULL;
  gchar* playlist_oid = NULL;
  if(gtk_list_store_iter_is_valid(playlist_selection_data->playlist_store, &iter))
  {
    gint id = 0;
    gtk_tree_model_get(GTK_TREE_MODEL(playlist_selection_data->playlist_store),
		       &iter, 
		       0, &playlist_name,
		       1, &playlist_oid,
		       -1);
  }

  MafwPlaylistManager* mafw_playlist_manager = mafw_playlist_manager_get();
  MafwProxyPlaylist* old_playlist = mafw_playlist_manager_create_playlist(mafw_playlist_manager, 
									  "FmpAudioPlaylist",
									  NULL);

  mafw_renderer_pause(playlist_selection_data->desktop_plugin->priv->mafw_renderer,
		      NULL, NULL);
  
  if(g_strcmp0(playlist_oid, "intern") == 0)
  {
    MafwProxyPlaylist* new_playlist = NULL;
    new_playlist = mafw_playlist_manager_create_playlist(mafw_playlist_manager, 
							 playlist_name,
							 NULL);
    g_free(playlist_name);
      
    if(mafw_playlist_clear(MAFW_PLAYLIST(old_playlist),
			   NULL))
    {
      guint playlist_size = mafw_playlist_get_size(MAFW_PLAYLIST(new_playlist), NULL);
      int i;
      gchar** items = mafw_playlist_get_items(MAFW_PLAYLIST(new_playlist),
					      0,
					      playlist_size-1, 
					      NULL);
      gchar** items2 = items;
      for(i=0;i<playlist_size;++i)
      {
	GError* error = NULL;
	mafw_playlist_append_item(MAFW_PLAYLIST(old_playlist),
				  (*items2),
				  &error);
	items2++;
	if(error)
	{
	  g_error_free(error);
	}
      }
      g_strfreev(items);
    }
    if(mafw_renderer_assign_playlist(playlist_selection_data->desktop_plugin->priv->mafw_renderer,
				     MAFW_PLAYLIST(old_playlist),
				     NULL))
    {
      mafw_renderer_play(playlist_selection_data->desktop_plugin->priv->mafw_renderer,
			 _on_playback_state_changed, playlist_selection_data->desktop_plugin);
    }
  }
  else
  {
    MafwSource* mafw_source = _get_mafw_source(playlist_selection_data->desktop_plugin->priv->mafw_registry);
    if(mafw_source)
    {
      const gchar *const *keys;
      keys = MAFW_SOURCE_LIST(MAFW_METADATA_KEY_TITLE,
			      MAFW_METADATA_KEY_ARTIST,
			      MAFW_METADATA_KEY_ALBUM,
			      MAFW_METADATA_KEY_ALBUM_ART_SMALL_URI);

      mafw_playlist_clear(MAFW_PLAYLIST(old_playlist), NULL);
      mafw_source_browse(mafw_source,
			 playlist_oid,
			 FALSE,
			 NULL,
			 NULL,
			 keys,
			 0, MAFW_SOURCE_BROWSE_ALL,
			 _playlist_item_request_cb,
			 old_playlist);
    }
  }
  gtk_dialog_response(GTK_DIALOG(playlist_selection_data->dialog), GTK_RESPONSE_OK);
}

static void
_browse_playlists_cb(MafwSource *source,
		     guint browse_id,
		     gint remaining,
		     guint index,
		     const gchar *object_id,
		     GHashTable *metadata,
		     gpointer user_data,
		     const GError *error)
{
  playlist_selection_data_t* playlist_selection_data = (playlist_selection_data_t*)user_data;
  if(remaining == 0)
  {
    /*
    hildon_gtk_window_set_progress_indicator(GTK_WINDOW(playlist_selection_data->dialog),
					     0);
    */
  }

  if(object_id != NULL) 
  {
    if(metadata != NULL) 
    {
      const gchar *playlist_name;
      GValue *value;
      value = mafw_metadata_first (metadata,
				   MAFW_METADATA_KEY_TITLE);
      if(value)
      {
	playlist_name = g_value_get_string(value);
	GtkTreeIter iter;
	gtk_list_store_append(playlist_selection_data->playlist_store, &iter);
	gtk_list_store_set(playlist_selection_data->playlist_store, &iter,
			   0, playlist_name,
			   1, object_id,
			   -1);
      }
    }
  }
}
 
static void
_do_playlist_request(MafwSource* mafw_source, playlist_selection_data_t* playlist_selection_data)
{
  const gchar *const *keys;
  gchar *object_id = "localtagfs::music/playlists";
  
  keys = MAFW_SOURCE_LIST(MAFW_METADATA_KEY_TITLE);
  mafw_source_browse(mafw_source,
		     object_id,         
		     FALSE,             
		     NULL,              
		     NULL,              
		     keys,              
		     0, 100,             
		     _browse_playlists_cb,
		     playlist_selection_data);    
}

static void
_fill_playlist_list_from_manager(GtkListStore* playlist_store)
{
  MafwPlaylistManager* mafw_playlist_manager = mafw_playlist_manager_get();
  if(mafw_playlist_manager)
  {
    GArray* playlists = mafw_playlist_manager_list_playlists(mafw_playlist_manager, NULL);
    int i = 0;
    GtkTreeIter iter;
    for(i = 0; i< playlists->len;++i)
    {
      MafwPlaylistManagerItem* item = &g_array_index(playlists, MafwPlaylistManagerItem, i);
      
      if(g_strstr_len(item->name, 3, "Fmp") == NULL)
      {
	gtk_list_store_append(playlist_store, &iter);
	gtk_list_store_set(playlist_store, &iter,
			   0, item->name,
			   1, "intern",
			   -1);
      }
    }
    mafw_playlist_manager_free_list_of_playlists(playlists);
  }
}

static void
_fill_playlist_list_from_mafw_source(playlist_selection_data_t* playlist_selection_data, MafwRegistry* mafw_registry)
{
  MafwSource* mafw_source = _get_mafw_source(mafw_registry);
  if(mafw_source)
  {
    _do_playlist_request(mafw_source, playlist_selection_data);
  }
  else
  {
    /*
    hildon_gtk_window_set_progress_indicator(GTK_WINDOW(playlist_selection_data->dialog),
					     0);
    */
  }
}

static void
_on_label_button_press(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  GtkListStore* playlist_store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);

  GtkWidget* dialog = hildon_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), "Playlist");
  gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
  gtk_window_set_default_size(GTK_WINDOW(dialog), -1, 330);
  
  GtkWidget* selector = hildon_touch_selector_new_text();
  hildon_touch_selector_set_column_selection_mode(HILDON_TOUCH_SELECTOR(selector),
						  HILDON_TOUCH_SELECTOR_SELECTION_MODE_SINGLE);
  hildon_touch_selector_set_model(HILDON_TOUCH_SELECTOR(selector),
				  0, GTK_TREE_MODEL(playlist_store));

  desktop_plugin->priv->mafw_playlist_manager = mafw_playlist_manager_get();

  playlist_selection_data_t* playlist_selection_data = g_new0(playlist_selection_data_t, 1);
  playlist_selection_data->dialog = dialog;
  playlist_selection_data->desktop_plugin = desktop_plugin;
  playlist_selection_data->playlist_store = playlist_store;
  /*
  hildon_gtk_window_set_progress_indicator(GTK_WINDOW(dialog),
					   1);
  */
  _fill_playlist_list_from_manager(playlist_store);
  _fill_playlist_list_from_mafw_source(playlist_selection_data, desktop_plugin->priv->mafw_registry);

  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(selector),
				   0, -1);
  
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), selector, TRUE, TRUE, 0);
  g_signal_connect(selector, "changed", G_CALLBACK(_on_playlist_item_selected), playlist_selection_data);
  g_signal_connect(dialog, "response", G_CALLBACK(_on_playlist_dialog_response), playlist_selection_data);
  gtk_widget_show_all(dialog);
  gtk_dialog_run(GTK_DIALOG(dialog));
}

static gboolean
_on_label_expose(GtkWidget* widget, GdkEventExpose* event, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  cairo_t *cr;
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_region(cr, event->region);
  cairo_clip(cr);
  cairo_translate(cr,event->area.x,event->area.y);
  cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
  gdk_cairo_set_source_pixbuf(cr, desktop_plugin->priv->background_pixbuf, 0, 0);
  cairo_paint(cr);
  PangoLayout* layout;
  PangoFontDescription* desc;
  layout = pango_cairo_create_layout(cr);
  desc = pango_font_description_from_string("Sans Bold 12");
  pango_layout_set_font_description(layout ,desc);
  pango_layout_set_width(layout, (event->area.width-5)*1024);
  pango_layout_set_ellipsize(layout, PANGO_ELLIPSIZE_END);

  pango_font_description_free(desc);
  cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0); 
  cairo_set_operator(cr, CAIRO_OPERATOR_OVER);

  if(desktop_plugin->priv->title)
    pango_layout_set_text(layout, desktop_plugin->priv->title, -1);
  else
    pango_layout_set_text(layout, "", -1);
  cairo_move_to(cr, 5, 8);
  pango_cairo_show_layout(cr, layout);
  if(desktop_plugin->priv->artist)
    pango_layout_set_text(layout, desktop_plugin->priv->artist, -1);
  else
    pango_layout_set_text(layout, "", -1);
  cairo_move_to(cr, 5, 24);
  pango_cairo_show_layout(cr, layout);
  if(desktop_plugin->priv->album)
    pango_layout_set_text(layout, desktop_plugin->priv->album, -1);
  else
    pango_layout_set_text(layout, "", -1);
  cairo_move_to(cr, 5, 40);
  pango_cairo_show_layout(cr, layout);
  g_object_unref(layout);
  cairo_destroy(cr);
  //  return GTK_WIDGET_CLASS(media_player_widget_home_plugin_parent_class)->expose_event(widget, event);
  return TRUE;
}

static void
_on_album_button_press(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  DBusGConnection* dbus_conn = hd_home_plugin_item_get_dbus_g_connection(&desktop_plugin->hitem,
                                                                         DBUS_BUS_SESSION,
                                                                         NULL);
  DBusGProxy* dbus_proxy = dbus_g_proxy_new_for_name(dbus_conn,
                                                     HD_APP_MGR_DBUS_NAME,
                                                     HD_APP_MGR_DBUS_PATH,
                                                     HD_APP_MGR_DBUS_IFACE);
  if(dbus_proxy)
  {
    dbus_g_proxy_call_no_reply(dbus_proxy,
                               HD_APP_MGR_LAUNCH_APPLICATION,
                               G_TYPE_STRING,
                               MEDIAPLAYER_APP_NAME,
                               G_TYPE_INVALID,
                               G_TYPE_INVALID);

    g_object_unref(dbus_proxy);
  }
}

static void
_init_ui(MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  GtkWidget* backward = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(backward), FALSE); 
  gtk_container_set_border_width(GTK_CONTAINER(backward), 0);
  g_signal_connect(GTK_CONTAINER(backward), "button-release-event", G_CALLBACK(_on_backward_button_press), 
		   desktop_plugin);
  GtkWidget* forward = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(forward), FALSE); 
  gtk_container_set_border_width(GTK_CONTAINER(forward), 0);
  g_signal_connect(GTK_CONTAINER(forward), "button-release-event", G_CALLBACK(_on_forward_button_press), 
		   desktop_plugin);
  GtkWidget* play_pause = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(play_pause), FALSE); 
  gtk_container_set_border_width(GTK_CONTAINER(play_pause), 0);
  g_signal_connect(GTK_CONTAINER(play_pause), "button-release-event", G_CALLBACK(_on_play_pause_button_press), 
		   desktop_plugin);
  GtkWidget* album = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(album), FALSE); 
  gtk_container_set_border_width(GTK_CONTAINER(album), 0);
  g_signal_connect(GTK_CONTAINER(album), "button-release-event", G_CALLBACK(_on_album_button_press), 
		   desktop_plugin);
  GtkWidget* label = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(label), FALSE); 
  gtk_container_set_border_width(GTK_CONTAINER(label), 0);
  g_signal_connect(GTK_CONTAINER(label), "button-release-event", G_CALLBACK(_on_label_button_press), 
		   desktop_plugin);
  GtkWidget* backward_icon = gtk_image_new_from_file(BACKWARD_ICON);
  GtkWidget* forward_icon = gtk_image_new_from_file(FORWARD_ICON);
  desktop_plugin->priv->label_background = gtk_image_new_from_file(LABEL_BACKGROUND_ICON);
  g_signal_connect(desktop_plugin->priv->label_background, "expose-event", G_CALLBACK(_on_label_expose), 
		   desktop_plugin);
  desktop_plugin->priv->background_pixbuf = gdk_pixbuf_new_from_file(LABEL_BACKGROUND_ICON, NULL);
  desktop_plugin->priv->play_pause_icon = gtk_image_new_from_file(PLAY_ICON);
  desktop_plugin->priv->album_pixbuf = gdk_pixbuf_new_from_file_at_size(DEFAULT_ALBUM_ICON,
									64, 64, NULL);
  desktop_plugin->priv->album_icon = gtk_image_new_from_pixbuf(desktop_plugin->priv->album_pixbuf);

  gtk_container_add(GTK_CONTAINER(backward), backward_icon);
  gtk_container_add(GTK_CONTAINER(forward), forward_icon);
  gtk_container_add(GTK_CONTAINER(play_pause), desktop_plugin->priv->play_pause_icon);
  gtk_container_add(GTK_CONTAINER(album), desktop_plugin->priv->album_icon);
  gtk_container_add(GTK_CONTAINER(label), desktop_plugin->priv->label_background);
  gtk_widget_set_size_request(label, 206, 64);
  GtkWidget* label_box = gtk_hbox_new(FALSE, 0);
  GtkWidget* button_box = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(button_box), backward, TRUE, TRUE, 0); 
  gtk_box_pack_start(GTK_BOX(button_box), play_pause, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(button_box), forward, TRUE, TRUE, 0); 
  gtk_box_pack_start(GTK_BOX(label_box), album, TRUE, TRUE, 0); 
  gtk_box_pack_start(GTK_BOX(label_box), label, TRUE, TRUE, 0); 
  GtkWidget* contents = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(contents), label_box, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(contents), button_box, TRUE, TRUE, 0);
  gtk_widget_show_all(contents);
  gtk_container_add(GTK_CONTAINER(desktop_plugin), contents);
}

static void
media_player_widget_home_plugin_init(MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  desktop_plugin->priv = MEDIA_PLAYER_WIDGET_HOME_PLUGIN_GET_PRIVATE(desktop_plugin);
  desktop_plugin->priv->mafw_registry = NULL;
  desktop_plugin->priv->mafw_renderer = NULL;
  desktop_plugin->priv->mafw_playlist = NULL;
  desktop_plugin->priv->mafw_playlist_manager = NULL;
  desktop_plugin->priv->album_pixbuf = NULL;
  desktop_plugin->priv->state = -1;
  desktop_plugin->priv->title = NULL;
  desktop_plugin->priv->artist = NULL;
  desktop_plugin->priv->album = NULL;
  desktop_plugin->priv->on_media_changed_handler_id = 0;
  desktop_plugin->priv->on_state_changed_handler_id = 0;
  desktop_plugin->priv->on_metadata_changed_handler_id = 0;
  desktop_plugin->priv->on_playlist_changed_handler_id = 0;
  desktop_plugin->priv->on_error_handler_id = 0;
  desktop_plugin->priv->on_renderer_added_id = 0;
  desktop_plugin->priv->on_renderer_removed_id = 0;
  _init_ui(desktop_plugin);
  //  hd_home_plugin_item_set_settings(HD_HOME_PLUGIN_ITEM (desktop_plugin), TRUE);
  //  g_signal_connect(desktop_plugin, "show-settings", G_CALLBACK(_show_settings_dialog), NULL);
  g_signal_connect(desktop_plugin, "notify::is-on-current-desktop", G_CALLBACK(_on_current_desktop), NULL);
}

static void
media_player_widget_finalize(GObject* object)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(object);
  if(desktop_plugin->priv->mafw_renderer)
  {
    _disconnect_renderer_signal_handler(desktop_plugin->priv->mafw_renderer, desktop_plugin);
    g_object_unref(desktop_plugin->priv->mafw_renderer);
    desktop_plugin->priv->mafw_renderer = NULL;
  }
  if(desktop_plugin->priv->mafw_registry)
  {
    _disconnect_registry_signal_handler(desktop_plugin->priv->mafw_registry, desktop_plugin);
  }

  if(desktop_plugin->priv->title)
  {
    g_free(desktop_plugin->priv->title);
    desktop_plugin->priv->title = NULL;
  }
  if(desktop_plugin->priv->album)
  {
    g_free(desktop_plugin->priv->album);
    desktop_plugin->priv->album = NULL;
  }
  if(desktop_plugin->priv->artist)
  {
    g_free(desktop_plugin->priv->artist);
    desktop_plugin->priv->artist = NULL;
  }
  if(desktop_plugin->priv->background_pixbuf)
  {
    g_object_unref(desktop_plugin->priv->background_pixbuf);
    desktop_plugin->priv->background_pixbuf = NULL;
  }
  if(desktop_plugin->priv->album_pixbuf)
  {
    g_object_unref(desktop_plugin->priv->album_pixbuf);
    desktop_plugin->priv->album_pixbuf = NULL;
  }
  G_OBJECT_CLASS(media_player_widget_home_plugin_parent_class)->finalize(object);
}

const gchar *
g_module_check_init (GModule *module)
{
     g_module_make_resident(module);
     return NULL;
}

static void
media_player_widget_home_plugin_class_init(MediaPlayerWidgetHomePluginClass* klass)
{
  GtkWidgetClass* widget_class = GTK_WIDGET_CLASS(klass);
  g_type_class_add_private(klass, sizeof(MediaPlayerWidgetHomePluginPrivate));
  widget_class->realize = media_player_widget_realize;
  G_OBJECT_CLASS(klass)->finalize = media_player_widget_finalize;
}

static void
media_player_widget_home_plugin_class_finalize(MediaPlayerWidgetHomePluginClass* klass)
{
}
