/*
 *  media player widget for the maemo desktop.
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <libintl.h>
#include <locale.h>
#include <gconf/gconf-client.h>	
#include <gdk/gdkx.h>
#include <libmafw/mafw.h>
#include <libmafw/mafw-log.h>
#include <libmafw/mafw-registry.h>
#include <libmafw-shared/mafw-shared.h>

#include "media-player-widget.h"

#define MEDIA_PLAYER_WIDGET_HOME_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MEDIA_PLAYER_WIDGET_TYPE_HOME_PLUGIN, MediaPlayerWidgetHomePluginPrivate))

HD_DEFINE_PLUGIN_MODULE(MediaPlayerWidgetHomePlugin, media_player_widget_home_plugin, HD_TYPE_HOME_PLUGIN_ITEM)
#define DEFAULT_WIDGET_WIDTH 256
#define DEFAULT_WIDGET_HEIGHT 64
#define MEDIAPLAYER_RENDERER "Mafw-Gst-Renderer"

struct _MediaPlayerWidgetHomePluginPrivate
{
  gchar* iD;
  MafwRenderer* mafw_renderer;
  MafwPlaylist* mafw_playlist;
  GtkWidget* play_pause_icon;
  GtkWidget* album_icon;
  GtkWidget* label_background;
  GdkPixbuf* background_pixbuf;
  gchar* title;
  gchar* artist;
  gchar* album;
  MafwPlayState state;
  gulong on_media_changed_handler_id;
  gulong on_state_changed_handler_id;
  gulong on_metadata_changed_handler_id;
  gulong on_playlist_changed_handler_id;
  gulong on_error_handler_id;
};


static gboolean
media_player_widget_expose(GtkWidget* widget, GdkEventExpose* event)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(widget);
  cairo_t *cr;
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_region(cr, event->region);
  cairo_clip(cr);
  cairo_destroy(cr);
  return GTK_WIDGET_CLASS(media_player_widget_home_plugin_parent_class)->expose_event(widget, event);
}

static void
_set_image_for_state(MafwPlayState state, MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  if(state == Playing)
  {
    gtk_image_set_from_file(GTK_IMAGE(desktop_plugin->priv->play_pause_icon),
			    "/usr/share/pixmaps/media-player-widget/pause.png");
  }
  else if(state == Stopped || state == Paused)
  {
    gtk_image_set_from_file(GTK_IMAGE(desktop_plugin->priv->play_pause_icon),
			    "/usr/share/pixmaps/media-player-widget/play.png");
  }
}

static void
_playlist_item_metadata_cb(MafwPlaylist *pls,
			   guint index,
			   const gchar *object_id,
			   GHashTable *metadata,
			   gpointer user_data)
{
  GError* error = NULL;
  GValue* value;
  const gchar* title;
  
  value = mafw_metadata_first(metadata, MAFW_METADATA_KEY_TITLE);
  title = g_value_get_string(value);
  g_warning("title %s \n", title);
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  if(title != NULL)
  {
    if(desktop_plugin->priv->title)
    {
      g_free(desktop_plugin->priv->title);
    }
    desktop_plugin->priv->title = g_strdup(title);
  }
  value = mafw_metadata_first(metadata, MAFW_METADATA_KEY_ARTIST);
  title = g_value_get_string(value);
  if(title != NULL)
  {
    if(desktop_plugin->priv->artist)
    {
      g_free(desktop_plugin->priv->artist);
    }
    desktop_plugin->priv->artist = g_strdup(title);
  }
  value = mafw_metadata_first(metadata, MAFW_METADATA_KEY_ARTIST);
  title = g_value_get_string(value);
  if(title != NULL)
  {
    if(desktop_plugin->priv->album)
    {
      g_free(desktop_plugin->priv->album);
    }
    desktop_plugin->priv->album = g_strdup(title);
  }
  gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
}

static void
_on_media_changed_cb(MafwRenderer* mafw_renderer, gint index, gchar* object_id, gpointer user_data)
{
  g_warning("media_changed %s \n", object_id);
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  if(desktop_plugin->priv->mafw_playlist)
  {
    const gchar *const *keys;
    keys = MAFW_SOURCE_LIST(MAFW_METADATA_KEY_TITLE,
			    MAFW_METADATA_KEY_ARTIST,
			    MAFW_METADATA_KEY_ALBUM);
  
    mafw_playlist_get_items_md(desktop_plugin->priv->mafw_playlist,
			       index, index,
			       keys,
			       _playlist_item_metadata_cb,
			       desktop_plugin, 
			       NULL);
  }  
  //  gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
}

static void
_on_metadata_changed_cb(MafwRenderer* mafw_renderer, gchar *name, GValueArray *value, gpointer user_data)
{
  g_warning("metadata_changed %s \n", name);
}

static void
_on_state_changed_cb(MafwRenderer* mafw_renderer, gint state, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  g_warning("state %d\n", state);
  desktop_plugin->priv->state = state;
  _set_image_for_state(state, MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data));
}

static void
_on_error_cb(MafwExtension* mafw_extension, guint domain, gint code,
	     gchar *message,
	     gpointer user_data)
{
  g_warning("error message %s\n", message);
}

static void
_on_renderer_get_status_cb(MafwRenderer* mafw_renderer,
			   MafwPlaylist* playlist,
			   guint index,
			   MafwPlayState state,
			   const gchar* object_id,
			   gpointer user_data,
			   const GError* error)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  desktop_plugin->priv->mafw_playlist = playlist;
  if(error)
  {
    g_warning("error on playbackstate cb %s\n", error->message);
  }
  const gchar *const *keys;
  keys = MAFW_SOURCE_LIST(MAFW_METADATA_KEY_TITLE,
			  MAFW_METADATA_KEY_ARTIST,
			  MAFW_METADATA_KEY_ALBUM);
  
  mafw_playlist_get_items_md(playlist,
			     index, index,
			     keys,
			     _playlist_item_metadata_cb,
			     desktop_plugin, 
			     NULL);
  desktop_plugin->priv->state = state;
  _set_image_for_state(state, MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data));
}

static void
_on_playlist_changed_cb(MafwRenderer* mafw_renderer, GObject* playlist, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  desktop_plugin->priv->mafw_playlist = MAFW_PLAYLIST(playlist);
  mafw_renderer_get_status(desktop_plugin->priv->mafw_renderer,
			   _on_renderer_get_status_cb, 
			   desktop_plugin);
}

static void
_connect_signal_handler(MafwRenderer* mafw_renderer, MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  desktop_plugin->priv->on_media_changed_handler_id = g_signal_connect(mafw_renderer,
								       "media-changed",
								       G_CALLBACK (_on_media_changed_cb),
								       desktop_plugin);
  desktop_plugin->priv->on_state_changed_handler_id = g_signal_connect(mafw_renderer,
								       "state-changed",
								       G_CALLBACK (_on_state_changed_cb),
								       desktop_plugin);
  desktop_plugin->priv->on_metadata_changed_handler_id = g_signal_connect(mafw_renderer,
									  "metadata-changed",
									  G_CALLBACK (_on_metadata_changed_cb),
									  desktop_plugin);
  desktop_plugin->priv->on_playlist_changed_handler_id = g_signal_connect(mafw_renderer,
									  "playlist-changed",
									  G_CALLBACK (_on_playlist_changed_cb),
									  desktop_plugin);
  desktop_plugin->priv->on_error_handler_id = g_signal_connect(mafw_renderer,
							       "error",
							       G_CALLBACK (_on_error_cb),
							       desktop_plugin);
}

static void
_disconnect_signal_handler(MafwRenderer* mafw_renderer, MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  if(desktop_plugin->priv->on_media_changed_handler_id)
  {
    g_signal_handler_disconnect(mafw_renderer,
				desktop_plugin->priv->on_media_changed_handler_id);
    desktop_plugin->priv->on_media_changed_handler_id = 0;
  }
  if(desktop_plugin->priv->on_state_changed_handler_id)
  {
    g_signal_handler_disconnect (mafw_renderer,
				 desktop_plugin->priv->on_state_changed_handler_id);
    desktop_plugin->priv->on_state_changed_handler_id = 0;
  }
  if(desktop_plugin->priv->on_metadata_changed_handler_id)
  {
    g_signal_handler_disconnect (mafw_renderer,
				 desktop_plugin->priv->on_metadata_changed_handler_id);
    desktop_plugin->priv->on_metadata_changed_handler_id = 0;
  }
  if(desktop_plugin->priv->on_playlist_changed_handler_id)
  {
    g_signal_handler_disconnect(mafw_renderer,
				desktop_plugin->priv->on_playlist_changed_handler_id);
    desktop_plugin->priv->on_playlist_changed_handler_id = 0;
  }
  if(desktop_plugin->priv->on_error_handler_id)
  {
    g_signal_handler_disconnect (mafw_renderer,
				 desktop_plugin->priv->on_error_handler_id);
    desktop_plugin->priv->on_error_handler_id = 0;
  }
}

static void
_on_renderer_added(MafwRegistry* mafw_registry,
		   GObject* renderer,
		   gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  if(g_strcmp0(mafw_extension_get_name(MAFW_EXTENSION(renderer)), MEDIAPLAYER_RENDERER) == 0)
  {
    MafwRenderer* mafw_renderer = MAFW_RENDERER(renderer);
    desktop_plugin->priv->mafw_renderer = mafw_renderer;
    _connect_signal_handler(mafw_renderer, desktop_plugin);
    mafw_renderer_get_status(mafw_renderer,
			     _on_renderer_get_status_cb, 
			     desktop_plugin);
  }
  g_warning("renderer added %s\n", mafw_extension_get_name(MAFW_EXTENSION(renderer)));
}

static void
_on_renderer_removed(MafwRegistry* mafw_registry,
		     GObject* renderer,
		     gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  _disconnect_signal_handler(MAFW_RENDERER(renderer), desktop_plugin);
  desktop_plugin->priv->mafw_renderer = NULL;
}

static void
_init_mafw(MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  MafwRegistry* mafw_registry = NULL;
  GError* error = NULL;
  mafw_log_init("mafw-example:ALL");
  mafw_registry = MAFW_REGISTRY(mafw_registry_get_instance());
  if(mafw_registry == NULL)
  {
    g_warning("can not init mafw registry\n");
  }
  
  mafw_shared_init(mafw_registry, &error);
  if(error)
  {
    g_warning("mafw out-of-process discorvery failed: %s\n", error->message);
    g_error_free(error);
    error = NULL;
  }
  GList* renderers = mafw_registry_get_renderers(mafw_registry);
  if(renderers)
  {
    GList* renderer = renderers;
    while(renderer)
    {
      MafwRenderer* mafw_renderer = MAFW_RENDERER(renderer->data);
      g_warning("renderer %s\n", mafw_extension_get_name(MAFW_EXTENSION(mafw_renderer)));
      if(g_strcmp0(mafw_extension_get_name(MAFW_EXTENSION(mafw_renderer)), MEDIAPLAYER_RENDERER) == 0)
      {
	desktop_plugin->priv->mafw_renderer = mafw_renderer;
	_connect_signal_handler(mafw_renderer, desktop_plugin);
	mafw_renderer_get_status(mafw_renderer,
				 _on_renderer_get_status_cb, 
				 desktop_plugin);
      }
      renderer = renderer->next;
    }
  }
  else
  {
    g_warning("no renderers\n");
  }
  g_signal_connect(mafw_registry, "renderer_added", G_CALLBACK(_on_renderer_added), desktop_plugin);
  g_signal_connect(mafw_registry, "renderer_removed", G_CALLBACK(_on_renderer_removed), desktop_plugin);
  //  g_signal_connect(mafw_registry, "source_added", G_CALLBACK(source_added_cb), self);
  //  g_signal_connect(mafw_registry, "source_removed", G_CALLBACK(source_removed_cb), NULL);
}

static void
media_player_widget_realize(GtkWidget* widget)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(widget);
  GdkScreen *screen = gtk_widget_get_screen(widget);
  gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
  gtk_widget_set_app_paintable(widget, TRUE);
  //  desktop_plugin->priv->iD = hd_home_plugin_item_get_applet_id(HD_HOME_PLUGIN_ITEM(widget));
  _init_mafw(desktop_plugin);
  GTK_WIDGET_CLASS(media_player_widget_home_plugin_parent_class)->realize(widget);
}

static void
media_player_widget_finalize(GObject* object)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(object);
  if(desktop_plugin->priv->mafw_renderer)
    _disconnect_signal_handler(desktop_plugin->priv->mafw_renderer, desktop_plugin);
  if(desktop_plugin->priv->title)
  {
    g_free(desktop_plugin->priv->title);
  }
  if(desktop_plugin->priv->album)
  {
    g_free(desktop_plugin->priv->album);
  }
  if(desktop_plugin->priv->artist)
  {
    g_free(desktop_plugin->priv->artist);
  }
  if(desktop_plugin->priv->background_pixbuf)
    g_object_unref(desktop_plugin->priv->background_pixbuf);
  //  g_free(desktop_plugin->priv->iD);
  G_OBJECT_CLASS(media_player_widget_home_plugin_parent_class)->finalize(object);
}

static void
_on_current_desktop(GtkWidget* widget,
		    gpointer data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(widget);
  gboolean on;
  g_object_get(widget, "is-on-current-desktop", &on, NULL);
  if(on)
  {
    gtk_widget_queue_draw(widget);
  }
}

static void
_on_playback_state_changed(MafwRenderer* renderer,
			   gpointer user_data,
			   const GError* error)
{
  if(error)
  {
    g_warning("error on playbackstate cb %s\n", error->message);
  }
  else
  {
    if(MAFW_IS_RENDERER(renderer))
    {
      /* MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data); */
      /* mafw_renderer_get_status(renderer, */
      /* 			       _on_renderer_get_status_cb, */
      /* 			       desktop_plugin); */
    }
  }
}

static void
_on_backward_button_press(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  if(desktop_plugin->priv->mafw_renderer)
  {
    mafw_renderer_next(desktop_plugin->priv->mafw_renderer,
		       _on_playback_state_changed, desktop_plugin);
  }
}

static void
_on_forward_button_press(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  mafw_renderer_previous(desktop_plugin->priv->mafw_renderer,
			 _on_playback_state_changed, desktop_plugin);
}

static void
_on_play_pause_button_press(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  if(desktop_plugin->priv->state == Playing)
  {
    mafw_renderer_pause(desktop_plugin->priv->mafw_renderer,
			_on_playback_state_changed, desktop_plugin);
  }
  else if(desktop_plugin->priv->state == Stopped)
  {
    mafw_renderer_play(desktop_plugin->priv->mafw_renderer,
		       _on_playback_state_changed, desktop_plugin);

  }
  else if(desktop_plugin->priv->state == Paused)
  {
    mafw_renderer_resume(desktop_plugin->priv->mafw_renderer,
			 _on_playback_state_changed, desktop_plugin);

  }
  else
  {
    g_warning("current state %d\n", desktop_plugin->priv->state);
  }
}

static void
_on_album_button_press(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);

}

static gboolean
_on_label_expose(GtkWidget* widget, GdkEventExpose* event, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);
  cairo_t *cr;
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_region(cr, event->region);
  cairo_translate(cr, 64,0);
  cairo_clip(cr);
  cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
  gdk_cairo_set_source_pixbuf(cr, desktop_plugin->priv->background_pixbuf, 0, 0);
  cairo_paint(cr);
  PangoLayout* layout;
  PangoFontDescription* desc;
  layout = pango_cairo_create_layout(cr);
  g_warning("xpose %s\n", desktop_plugin->priv->title);
  if(desktop_plugin->priv->title)
    pango_layout_set_text(layout, desktop_plugin->priv->title, -1);
  else
    pango_layout_set_text(layout, "", -1);
  desc = pango_font_description_from_string("Sans Bold 8");
  pango_layout_set_font_description(layout ,desc);
  pango_font_description_free(desc);
  cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0); 
  cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
  cairo_move_to(cr, 0, 32);
  pango_cairo_show_layout(cr, layout);
  if(desktop_plugin->priv->artist)
    pango_layout_set_text(layout, desktop_plugin->priv->artist, -1);
  else
    pango_layout_set_text(layout, "", -1);
  cairo_move_to(cr, 0, 8);
  pango_cairo_show_layout(cr, layout);
  if(desktop_plugin->priv->album)
    pango_layout_set_text(layout, desktop_plugin->priv->album, -1);
  else
    pango_layout_set_text(layout, "", -1);
  cairo_move_to(cr, 0, 8);
  pango_cairo_show_layout(cr, layout);
  g_object_unref(layout);
  cairo_destroy(cr);
  //  return GTK_WIDGET_CLASS(media_player_widget_home_plugin_parent_class)->expose_event(widget, event);
  return TRUE;
}

static void
_on_label_button_press(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
  MediaPlayerWidgetHomePlugin* desktop_plugin = MEDIA_PLAYER_WIDGET_HOME_PLUGIN(user_data);

}

static void
_init_ui(MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  GtkWidget* backward = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(backward), FALSE); 
  gtk_container_set_border_width(GTK_CONTAINER(backward), 0);
  g_signal_connect(GTK_CONTAINER(backward), "button-release-event", G_CALLBACK(_on_backward_button_press), 
		   desktop_plugin);
  GtkWidget* forward = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(forward), FALSE); 
  gtk_container_set_border_width(GTK_CONTAINER(forward), 0);
  g_signal_connect(GTK_CONTAINER(forward), "button-release-event", G_CALLBACK(_on_forward_button_press), 
		   desktop_plugin);
  GtkWidget* play_pause = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(play_pause), FALSE); 
  gtk_container_set_border_width(GTK_CONTAINER(play_pause), 0);
  g_signal_connect(GTK_CONTAINER(play_pause), "button-release-event", G_CALLBACK(_on_play_pause_button_press), 
		   desktop_plugin);
  GtkWidget* album = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(album), FALSE); 
  gtk_container_set_border_width(GTK_CONTAINER(album), 0);
  g_signal_connect(GTK_CONTAINER(album), "button-release-event", G_CALLBACK(_on_album_button_press), 
		   desktop_plugin);
  GtkWidget* label = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(label), FALSE); 
  gtk_container_set_border_width(GTK_CONTAINER(label), 0);
  g_signal_connect(GTK_CONTAINER(album), "button-release-event", G_CALLBACK(_on_label_button_press), 
		   desktop_plugin);
  GtkWidget* backward_icon = gtk_image_new_from_file("/usr/share/pixmaps/media-player-widget/backward.png");
  GtkWidget* forward_icon = gtk_image_new_from_file("/usr/share/pixmaps/media-player-widget/forward.png");
  desktop_plugin->priv->label_background = gtk_image_new_from_file("/usr/share/pixmaps/media-player-widget/label_background.png");
  g_signal_connect(desktop_plugin->priv->label_background, "expose-event", G_CALLBACK(_on_label_expose), 
		   desktop_plugin);
  desktop_plugin->priv->background_pixbuf = gdk_pixbuf_new_from_file("/usr/share/pixmaps/media-player-widget/label_background.png", NULL);
  desktop_plugin->priv->play_pause_icon = gtk_image_new_from_file("/usr/share/pixmaps/media-player-widget/pause.png");
  desktop_plugin->priv->album_icon = gtk_image_new_from_file("/usr/share/pixmaps/media-player-widget/player.png");
  g_warning("build container\n");
  gtk_container_add(GTK_CONTAINER(backward), backward_icon);
  gtk_container_add(GTK_CONTAINER(forward), forward_icon);
  gtk_container_add(GTK_CONTAINER(play_pause), desktop_plugin->priv->play_pause_icon);
  gtk_container_add(GTK_CONTAINER(album), desktop_plugin->priv->album_icon);
  gtk_container_add(GTK_CONTAINER(label), desktop_plugin->priv->label_background);
  gtk_widget_set_size_request(label,
			      206,
			      64);
  g_warning("build container done\n");
  GtkWidget* label_box = gtk_hbox_new(FALSE, 0);
  GtkWidget* button_box = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(button_box), backward, TRUE, TRUE, 0); 
  gtk_box_pack_start(GTK_BOX(button_box), play_pause, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(button_box), forward, TRUE, TRUE, 0); 
  gtk_box_pack_start(GTK_BOX(label_box), album, TRUE, TRUE, 0); 
  gtk_box_pack_start(GTK_BOX(label_box), label, TRUE, TRUE, 0); 
  GtkWidget* contents = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(contents), label_box, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(contents), button_box, TRUE, TRUE, 0);
  gtk_widget_show_all(contents);
  gtk_container_add(GTK_CONTAINER(desktop_plugin), contents);
}

static void
media_player_widget_home_plugin_init(MediaPlayerWidgetHomePlugin* desktop_plugin)
{
  desktop_plugin->priv = MEDIA_PLAYER_WIDGET_HOME_PLUGIN_GET_PRIVATE(desktop_plugin);
  desktop_plugin->priv->mafw_renderer = NULL;
  desktop_plugin->priv->mafw_playlist = NULL;
  desktop_plugin->priv->on_media_changed_handler_id = 0;
  desktop_plugin->priv->on_state_changed_handler_id = 0;
  desktop_plugin->priv->on_metadata_changed_handler_id = 0;
  desktop_plugin->priv->on_playlist_changed_handler_id = 0;
  desktop_plugin->priv->on_error_handler_id = 0;
  desktop_plugin->priv->title = NULL;
  desktop_plugin->priv->artist = NULL;
  desktop_plugin->priv->album = NULL;
  _init_ui(desktop_plugin);
  //  hd_home_plugin_item_set_settings(HD_HOME_PLUGIN_ITEM (desktop_plugin), TRUE);
  //  g_signal_connect(desktop_plugin, "show-settings", G_CALLBACK(_show_settings_dialog), NULL);
  g_signal_connect(desktop_plugin, "notify::is-on-current-desktop", G_CALLBACK(_on_current_desktop), NULL);
}

static void
media_player_widget_home_plugin_class_init(MediaPlayerWidgetHomePluginClass* klass)
{
  GtkWidgetClass* widget_class = GTK_WIDGET_CLASS(klass);
  g_type_class_add_private(klass, sizeof(MediaPlayerWidgetHomePluginPrivate));
  widget_class->realize = media_player_widget_realize;
  //  widget_class->expose_event = media_player_widget_expose;
  G_OBJECT_CLASS(klass)->finalize = media_player_widget_finalize;
}

static void
media_player_widget_home_plugin_class_finalize(MediaPlayerWidgetHomePluginClass* klass)
{
}
