#! /usr/bin/env python
# -*- coding: UTF-8 -*-

import os, sys
import gobject, dbus, dbus.glib, telepathy
import ConfigParser

APP_CONFIG_PATH = '/home/user/.media-im-status-updater.ini'
IFACE_PRESENCE = 'org.freedesktop.Telepathy.Connection.Interface.SimplePresence'

def debug(message):
  if sys.stdout.isatty():
    sys.stdout.write(message + "\n")

class DomainConfig:
  template    = None
  accounts    = None

  __filename  = None

  def __init__(self, filename):
    self.__filename = filename

    self.template   = 'Listening to "%t" by "%a" via Nokia N900'
    self.accounts   = list()

  def load(self):
    if os.path.isfile(self.__filename):
      parser = ConfigParser.RawConfigParser()
      parser.read(self.__filename)

      self.template = parser.get('Config', 'Template')
      self.accounts = parser.get('Config', 'Accounts').split(',')

  def save(self):
    parser = ConfigParser.RawConfigParser()

    parser.add_section('Config')
    parser.set('Config', 'Template', self.template)
    parser.set('Config', 'Accounts', str.join(',', self.accounts))

    f = open(self.__filename, 'wb')
    parser.write(f)
    f.close()
    
class MediaMonitor:
  _object_path  = None
  _session_bus  = None

  def __init__(self, bus, object_path = '/com/nokia/mafw/renderer/gstrenderer'):
    self._object_path = object_path
    self._session_bus = bus

    self._session_bus.add_signal_receiver(self._on_state_change,
                                          path = object_path,
                                          dbus_interface = 'com.nokia.mafw.renderer',
                                          signal_name = 'state_changed')

    self._session_bus.add_signal_receiver(self._on_metadata_change,
                                          path = object_path,
                                          dbus_interface = 'com.nokia.mafw.renderer',
                                          signal_name = 'metadata_changed')

  def _on_state_change(self, state):
    if state == 0:
      self.on_media_stopped()

    elif state == 1:
      self.on_media_started()

    elif state == 2:
      self.on_media_paused()

    elif state == 3:
      self.on_media_transition()

    else:
      debug('*** Unknown state: %d' % state)

  def _on_metadata_change(self, key, state, data):
    self.on_metadata(key, data)

  def on_metadata(self, key, data):
    pass

  def on_media_started(self):
    pass

  def on_media_stopped(self):
    pass

  def on_media_paused(self):
    pass

  def on_media_transition(self):
    pass

class MediaIMStatusUpdater(MediaMonitor):
  __keys      = ['title', 'artist']
  __bus       = None
  __config    = None
  __metadata  = None

  def __init__(self, bus, config):
    MediaMonitor.__init__(self, bus)

    self.__bus      = bus
    self.__config   = config
    self.__metadata = None

  def on_media_transition(self):
    self.__metadata = dict()

  def on_metadata(self, key, data):
    if self.__metadata != None and self.__keys.count(key) > 0:
      self.__metadata[key] = data

      # Check to see if we've got a complete set of data...
      for k in self.__keys:
        if not self.__metadata.has_key(k):
          return

      # If we've got all the data we requre...
      self.on_metadata_complete()

  def on_metadata_complete(self):
    self.__config.load()

    result = self.__config.template
    result = result.replace('%t', self.__metadata['title'])
    result = result.replace('%a', self.__metadata['artist'])

    debug(result)

    self.__update_im_status(result)

  def __update_im_status(self, status_mesg):
    am        = self.__bus.get_object(telepathy.ACCOUNT_MANAGER, '/org/freedesktop/Telepathy/AccountManager')
    am_iface  = dbus.Interface(am, dbus.PROPERTIES_IFACE)

    for account in am_iface.Get(telepathy.ACCOUNT_MANAGER, 'ValidAccounts'):
      account_name  = account.replace('/org/freedesktop/Telepathy/Account/', '')

      # If we're configured to update this account, then do it.
      if self.__config.accounts.count(account_name) > 0:
        ac        = self.__bus.get_object(telepathy.ACCOUNT_MANAGER, account)
        ac_iface  = dbus.Interface(ac, dbus.PROPERTIES_IFACE)

        debug("Updating presence information on account: %s" % ac_iface.Get(telepathy.ACCOUNT, 'NormalizedName'))

        presence, availability, message = ac_iface.Get(telepathy.ACCOUNT, 'RequestedPresence')
        ac_iface.Set(telepathy.ACCOUNT, 'RequestedPresence', dbus.Struct((presence, availability, status_mesg), signature = 'uss'))

if __name__ == '__main__':
  config  = DomainConfig(APP_CONFIG_PATH)
  config.load()

  bus     = dbus.SessionBus()
  monitor = MediaIMStatusUpdater(bus, config)

  main = gobject.MainLoop()
  main.run()

