#! /usr/bin/env python
# -*- coding: UTF-8 -*-

import os, sys
import ConfigParser

import dbus, dbus.glib
import telepathy

from PySide.QtCore import *
from PySide.QtGui  import *

IFACE_PRESENCE  = 'org.freedesktop.Telepathy.Connection.Interface.SimplePresence'
APP_CONFIG_PATH = '/home/user/.media-im-status-updater.ini'
PROFILES_PATH   = '/usr/share/osso-rtcom'
ACCOUNTS_CONFIG = '/home/user/.rtcom-accounts/accounts.cfg'
ICONS_PATH      = '/opt/usr/share/icons/hicolor/48x48/hildon'

class DomainConfig:
  template    = None
  accounts    = None

  __filename  = None

  def __init__(self, filename):
    self.__filename = filename

    self.template   = 'Listening to "%t" by "%a" via Nokia N900'
    self.accounts   = list()

  def load(self):
    if os.path.isfile(self.__filename):
      parser = ConfigParser.RawConfigParser()
      parser.read(self.__filename)

      self.template = parser.get('Config', 'Template')
      self.accounts = parser.get('Config', 'Accounts').split(',')

  def save(self):
    parser = ConfigParser.RawConfigParser()

    parser.add_section('Config')
    parser.set('Config', 'Template', self.template)
    parser.set('Config', 'Accounts', str.join(',', self.accounts))

    f = open(self.__filename, 'wb')
    parser.write(f)
    f.close()

class ConfigForm(QScrollArea):
  buttons = None

  def __init__(self, bus, config):
    QScrollArea.__init__(self)

    vbox    = QVBoxLayout()
    hbox    = QHBoxLayout()
    widget  = QWidget()

    self.setWidgetResizable(True)

    self.buttons = list()

    self.template_input = template_input = QLineEdit()
    template_input.setText(config.template)

    vbox.addWidget(template_input)

    accounts = ConfigParser.RawConfigParser()
    accounts.read(ACCOUNTS_CONFIG)

    am        = bus.get_object(telepathy.ACCOUNT_MANAGER, '/org/freedesktop/Telepathy/AccountManager')
    am_iface  = dbus.Interface(am, dbus.PROPERTIES_IFACE)

    for account in am_iface.Get(telepathy.ACCOUNT_MANAGER, 'ValidAccounts'):
      account_name = account.replace('/org/freedesktop/Telepathy/Account/', '')
      profile_name = accounts.get(account_name, 'Profile')

      if account_name == 'ring/tel/ring':
        continue

      profile = ConfigParser.RawConfigParser()
      profile.read(PROFILES_PATH + '/' + profile_name + '.profile')

      ac          = bus.get_object(telepathy.ACCOUNT_MANAGER, account)
      ac_iface    = dbus.Interface(ac, dbus.PROPERTIES_IFACE)

      button  = QCheckBox(ac_iface.Get(telepathy.ACCOUNT, 'NormalizedName'))
      button.setProperty('account-path', account_name)
      button.setCheckable(True)
      button.setChecked(config.accounts.count(account_name) > 0)
      button.setIcon(QIcon(ICONS_PATH + '/' + profile.get('Profile', 'IconName') + '.png'))

      vbox.addWidget(button)
      self.buttons.append(button)

    widget.setLayout(vbox)
    self.setWidget(widget)

  def get_template(self):
    return self.template_input.text()

  def get_accounts(self):
    results = list()

    for button in self.buttons:
      if button.isChecked():
        results.append(str(button.property('account-path').toString()))

    return results

class MainDialog(QDialog):
  __config  = None

  def __init__(self, session_bus, config):
    QDialog.__init__(self)

    self.setWindowTitle('Media IM Status Updater')

    self.__config = config

    hbox = QHBoxLayout()

    save_btn = QPushButton("Save")
    save_btn.setDefault(True)

    self.connect(save_btn, SIGNAL('clicked()'), self, SLOT('on_save()'))

    button_box  = QDialogButtonBox(Qt.Vertical)
    button_box.addButton(save_btn, QDialogButtonBox.ActionRole)

    self.form = config_form = ConfigForm(session_bus, config)

    hbox.addWidget(config_form)
    hbox.addWidget(button_box)
    self.setLayout(hbox)

  def on_save(self):
    self.__config.template  = self.form.get_template()
    self.__config.accounts  = self.form.get_accounts()
    self.__config.save()
    self.done(0)

if __name__ == '__main__':
  bus         = dbus.SessionBus()

  config      = DomainConfig(APP_CONFIG_PATH)
  config.load()

  application = QApplication(sys.argv)
  main_dialog = MainDialog(bus, config)

  main_dialog.show()
  sys.exit(main_dialog.exec_())

