/**
 * This software is distributed under the GNU Lesser General Public License,
 * which can be found at http://www.gnu.org/licenses/old-licenses/lgpl-2.1.txt.
 * 
 * Copyright (c) 2010 Eero af Heurlin
 */

//Using GLib;

public class mceparser : Object
{
    private string filename;
    private string[] stock_pattern_names;
    private List<string> file_lines;

    public mceparser()
    {
        /* Constructor */
        this.filename = "/etc/mce/mce.ini";
        this.stock_pattern_names = {
            "PatternError",
            "PatternDeviceOn",
            "PatternPowerOn",
            "PatternPowerOff",
            "PatternCommunicationCall",
            "PatternCommunicationIM",
            "PatternCommunicationSMS",
            "PatternCommunicationEmail",
            "PatternCommonNotification",
            "PatternWebcamActive",
            "PatternBatteryCharging",
            "PatternBatteryFull",
            "PatternDeviceSoftOff"
        };
    }

    /**
     * Read the file line-by-line to array we'll be analyzing later
     */
    public void read_file() throws Error
    {
        var fp = File.new_for_path(this.filename);
        var seekable = new DataInputStream (fp.read (null));
        string line;
        while ((line = seekable.read_line (null, null)) != null)
        {
            // We want to include the newline character as part of line string
            this.file_lines.append(line + "\n");
        }
    }
    
    public void write_file() throws Error
    {
        // FIXME: Figure out a way to create unique temp file name
        string new_name = this.filename + ".new";
        var fp = File.new_for_path(new_name);
        try
        {
            {
                /* Extra block to allow the file resource to be closed before rename, see http://live.gnome.org/Vala/GIOSamples */
                var seekable = new DataOutputStream(fp.create(FileCreateFlags.REPLACE_DESTINATION, null));
                foreach (var line in this.file_lines)
                {
                    // The lines SHOULD have the newline character in place already
                    seekable.put_string(line, null);
                }
            }
        }
        catch (Error e)
        {
            // Re-throw weird errors
            throw e;
        }
        var fp_orig = File.new_for_path(this.filename);
        try
        {
            // Docs say the move method returns boolen but compiler complains if we don't catch errors too
            if (!fp.move(fp_orig, FileCopyFlags.OVERWRITE, null, null))
            {
                throw new IOError.FAILED("Could not replace file " + this.filename);
            }
        }
        catch (Error e)
        {
            // Re-throw weird errors
            throw e;
        }
    }

    /**
     * Sanity check pattern format
     */
    public bool sanity_check(string pattern)
    {
        try
        {
            var sane_format = new Regex("^[0-9]{1,3};[0-5];[0-9]{1,2};[rgb][RGB]?;9d80[0-9a-f]{4,60};9d80[0-9a-f]{4,60}$");
            if (!sane_format.match(pattern))
            {
                // Basic format sanity check failed 
                return false;
            }
        }
        catch (RegexError e)
        {
            // Got weird error
            warning ("%s", e.message);
            return false;
        }

        //TODO: Other sanity checks
        return true;
    }

    public bool add_pattern(string pattern_name, string pattern)
    {
        if (!this.sanity_check(pattern))
        {
            stderr.printf("Pattern '%s' is not sane!\n", pattern);
            return false;
        }
        try
        {
            this.read_file();
        }
        catch (Error e)
        {
            warning ("%s", e.message);
            return false;
        }
        string? section_open = null;
        var i = 0;
        var pattern_found = false;
        var file_changed = false;
        foreach (var line in this.file_lines)
        {
            if (line == "[LED]\n")
            {
                section_open = "led";
            }
            else if (line == "[LEDPatternLystiRX51]\n")
            {
                section_open = "pattern";
            }
            else if (line.has_prefix("["))
            {
                section_open = null;
            }
            //stdout.printf("DEBUG: section=%s\n", section_open);

            // Add pattern to LEDPatterns if required
            if (   section_open == "led"
                && line.len() > 12
                && line[0:12] == "LEDPatterns="
                && !(pattern_name in line[12:-1]))
            {
                var new_line = "LEDPatterns=%s;%s\n".printf(line[12:-1], pattern_name);
                this.file_lines.nth(i).data = new_line;
                file_changed = true;
                //stdout.printf("DEBUG: Changed line %d to: %s", i, new_line);
            }

            // Previous instance found, comment out and add new one
            if (   section_open == "pattern"
                && line.has_prefix(pattern_name + "="))
            {
                pattern_found = true;
                var now = TimeVal();
                // This is a bit sneaky since we're actually inserting many lines...
                var new_line = "# Commented out on " + now.to_iso8601() + " by mceledpattern\n#" + line;
                new_line += "# Added on " + now.to_iso8601() + " by mceledpattern\n";
                new_line += pattern_name + "=" + pattern + "\n";
                this.file_lines.nth(i).data = new_line;
                file_changed = true;
                //stdout.printf("DEBUG: Changed line %d to: %s", i, new_line);
            }

            // Pattern not found and reaching end of section
            if (   section_open == "pattern"
                && this.file_lines.nth(i+1).data.has_prefix("[")
                && !pattern_found)
            {
                var now = TimeVal();
                var new_line = "# Added on " + now.to_iso8601() + " by mceledpattern\n";
                new_line += pattern_name + "=" + pattern + "\n";
                pattern_found = true;
                this.file_lines.insert(new_line, i);
                //stdout.printf("DEBUG: inserted to line %d: %s", i, new_line);
                i += 1;
                file_changed = true;
            }
            
            // End of loop
            i++;
        }
        if (file_changed)
        {
            try
            {
                this.write_file();
            }
            catch (Error e)
            {
                // Got weird error
                warning ("%s", e.message);
                return false;
            }
        }
        return true;
    }

    /**
     * There must be a saner way...
     */
    private bool is_stock_pattern(string pattern_name)
    {
        foreach(var stock_name in this.stock_pattern_names)
        {
            if (pattern_name == stock_name)
            {
                return true;
            }
        }
        return false;
    }

    public bool remove_pattern(string pattern_name)
    {
        try
        {
            this.read_file();
        }
        catch (Error e)
        {
            warning ("%s", e.message);
            return false;
        }

        string? section_open = null;
        var i = 0;
        var pattern_found = false;
        var file_changed = false;
        int? enabled_patterns_line = null;
        var active_pattern_lines = new List<int>();
        var commented_pattern_lines = new List<List<int>>();
        foreach (var line in this.file_lines)
        {
            if (line == "[LED]\n")
            {
                section_open = "led";
            }
            else if (line == "[LEDPatternLystiRX51]\n")
            {
                section_open = "pattern";
            }
            else if (line.has_prefix("["))
            {
                section_open = null;
            }
            //stdout.printf("DEBUG: section=%s\n", section_open);

            // Find the line for LEDPatterns
            if (   section_open == "led"
                && line.len() > 12
                && line[0:12] == "LEDPatterns=")
            {
                enabled_patterns_line = i;
                if (!(pattern_name in line[12:-1]))
                {
                    stderr.printf("Cannot remove pattern that is not enabled in LEDPatterns\n");
                    return false;
                }
            }

            // Existing (not commented out pattern)
            if (   section_open == "pattern"
                && line.has_prefix(pattern_name + "="))
            {
                pattern_found = true;
                var previous_line = this.file_lines.nth(i-1).data;
                if (   previous_line.has_prefix("# Added on")
                    && previous_line.has_suffix(" by mceledpattern\n"))
                {
                    active_pattern_lines.append(i-1);
                }
                active_pattern_lines.append(i);
            }

            // Existing (not commented out pattern)
            if (   section_open == "pattern"
                && line.has_prefix("#" + pattern_name + "="))
            {
                var commented_pattern_lines_inner = new List<int>();
                var previous_line = this.file_lines.nth(i-1).data;
                if (   previous_line.has_prefix("# Commented out on")
                    && previous_line.has_suffix(" by mceledpattern\n"))
                {
                    commented_pattern_lines_inner.append(i-1);
                }
                commented_pattern_lines_inner.append(i);
                commented_pattern_lines.append((owned)commented_pattern_lines_inner);
            }

            // Pattern not found and reaching end of section
            if (   section_open == "pattern"
                && this.file_lines.nth(i+1).data.has_prefix("[")
                && !pattern_found)
            {
                stderr.printf("Pattern was in LEDPatterns but not found in LEDPatternLystiRX51-section, something is messed up!\n");
                return false;
            }
            
            // End of loop
            i++;
        }

        // Do not allow removal of last instance of a stock pattern
        if (   commented_pattern_lines.length() == 0
            && this.is_stock_pattern(pattern_name))
        {
            stderr.printf("Refusing to remove last instance of a stock pattern\n");
            return false;
        }

        // Removing last instance of non-stock pattern, remove from LEDPatterns too
        if (   commented_pattern_lines.length() == 0
            && !this.is_stock_pattern(pattern_name)
            && pattern_name in this.file_lines.nth(enabled_patterns_line).data)
        {
            var new_line = this.file_lines.nth(enabled_patterns_line).data.replace(";" + pattern_name, "");
            this.file_lines.nth(enabled_patterns_line).data = new_line;
            file_changed = true;
        }
        
        foreach(var line_num in active_pattern_lines)
        {
            this.file_lines.nth(line_num).data = "";
        }

        if (commented_pattern_lines.length() > 0)
        {
            unowned List<int> commented_pattern_lines_inner = commented_pattern_lines.last().data;
            foreach(int line_num in commented_pattern_lines_inner)
            {
                if (this.file_lines.nth(line_num).data.has_prefix("#" + pattern_name + "="))
                {
                    // Commented out pattern line, re-enable
                    var new_line = this.file_lines.nth(line_num).data.offset(1);
                    this.file_lines.nth(line_num).data = new_line;
                    file_changed = true;
                }
                else
                {
                    // Plain comment line, remove
                    this.file_lines.nth(line_num).data = "";
                    file_changed = true;
                }
            }
        }

        if (file_changed)
        {
            try
            {
                this.write_file();
            }
            catch (Error e)
            {
                // Got weird error
                warning ("%s", e.message);
                return false;
            }
        }
        return true;
    }

    
}

void usage(string arg)
{
    var fp = File.new_for_path(arg);
    var name = fp.get_basename();
    stdout.printf(@"Usage:\n  $name add <pattern_name> <pattern>\n  $name remove <pattern_name>\n\n  To replace existing pattern use add, to restore replaced\n  pattern back to previous value use remove\n");
}

int main(string[] args)
{
    var parser = new mceparser();
    if (   args.length == 3
        && args[1] == "remove")
    {
        if (!parser.remove_pattern(args[2]))
        {
            return 1;
        }
    }
    else if (   args.length == 4
             && args[1] == "add")
    {
        if (!parser.add_pattern(args[2], args[3]))
        {
            return 1;
        }
    }
    else if (args.length == 1)
    {
        usage(args[0]);
    }
    else
    {
        usage(args[0]);
        return 1;
    }

    return 0;
}
