//Maemo Barcode Reader and Interpreter (mbarcode or maemo-barcode)
//Copyright (C) 2010 Simon Pickering
//Copyright (C) 2010 Svenn-Arne Dragly
//
//Some source code obtained from other individuals/companies (not affiliated with the project), such as
//Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
//
//Various parts of barcode recognition and GStreamer manipulation code written by:
//      Timothy Terriberry
//      Adam Harwell
//      Jonas Hurrelmann
//
//Original GStreamer code based on the maemo-examples package:
//Copyright (C) 2007-2008 Nokia Corporation. All rights reserved.
//Copyright (C) 2006 INdT.
//@author Talita Menezes <talita.menezes@indt.org.br>
//@author Cidorvan Leite <cidorvan.leite@indt.org.br>
//@author Jami Pekkanen <jami.pekkanen@nokia.com>
//
//This program is free software: you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation, either version 3 of the License, or
//(at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "videowidget.h"
// gstreamer includes
#include <gst/gst.h>
#include <gst/gstbin.h>
#include <gst/gstelement.h>
#include <gst/interfaces/xoverlay.h>
#include <gst/interfaces/photography.h>
// qt includes
#include <QPainter>
#include <QWidget>
#include <QApplication>
#include <QTime>
#include <QTimer>

#include <QFileDialog>
#include <QFileInfo>

// local includes
#include "common.h"
#include "barcodedetector.h"
#include "maemobarcodewindow.h"

VideoWidget *videoWidget_callback;
VideoWidget::VideoWidget(MaemoBarcodeWindow *parent) : QGLWidget(parent) {
    BarcodeDetector *newdecoder = new BarcodeDetector(); // create a new decoder object
    VideoWidget(parent, newdecoder);
}
VideoWidget::VideoWidget(MaemoBarcodeWindow *parent, BarcodeDetector *decoder) : QGLWidget(parent) {
    qDebug() << "VideoWidget::VideoWidget(): Called";
    pipeline = NULL;
    this->parent = parent;
    keepRunning = true;
    setAttribute(Qt::WA_PaintOnScreen);
    setAttribute(Qt::WA_NoSystemBackground);
    setAutoBufferSwap(false);
    frame_width = WIDTH;
    frame_height = HEIGHT; // 300x160 should be the correct aspect ratio
    decoder->setFrameSize(frame_width, frame_height);
    cold_start = true;
    appSinkScanning = false;
    QApplication::syncX();
    // Set up the size of the widget
    QSizePolicy sizing(QSizePolicy::Preferred, QSizePolicy::Preferred);
    sizing.setHeightForWidth(true);
    setSizePolicy(sizing);

    pipelineReady = false;
    focusing = false;
    videoWidget_callback = this; // set the global pointer to this object
    qDebug() << "VideoWidget::VideoWidget(): Getting device type";
    getDeviceType(); // work out which device we are so we can specify what type of pipeline to setup
    qDebug() << "VideoWidget::VideoWidget(): Initializing pipeline";
    initPipeline(); // create the GStreamer pipeline
    qDebug() << "VideoWidget::VideoWidget(): Pipeline initialized";
    this->decoder = decoder; // create a new decoder object

    continuousFocusInterval = 3000;
    refocusTimer = new QTimer(this);
    connect(refocusTimer, SIGNAL(timeout()), this, SLOT(refocus()));
    refocusTimer->setInterval(continuousFocusInterval);

    qDebug() << "VideoWidget::VideoWidget(): Setting up images";
    // set up GUI stuff
    imgExit = QImage(":/images/gui/cross.png");
    imgExit = imgExit.scaledToWidth(48, Qt::SmoothTransformation);
    imgSave = QImage(":/images/gui/save.png");
    imgSave = imgSave.scaledToWidth(48, Qt::SmoothTransformation);
    imgTaskManager = QImage(":/images/gui/task-manager.png");
    imgTaskManager = imgTaskManager.scaledToWidth(48, Qt::SmoothTransformation);
    imgRecent = QImage(":/images/gui/recent.png");
    imgRecent = imgRecent.scaledToWidth(48, Qt::SmoothTransformation);
    imgSettings = QImage(":/images/gui/settings.png");
    imgSettings = imgSettings.scaledToWidth(48, Qt::SmoothTransformation);

    qDebug() << "VideoWidget::VideoWidget(): Setting up frame counts";
    // timer, should be set last, just in case
    frames = 0;
    frametime.start();
    timerOpenGL = new QTimer(this);
    QObject::connect(timerOpenGL, SIGNAL(timeout()), this, SLOT(updateGL()));
    timerOpenGL->setInterval(PAINTGL_INTERVAL_HIGH); //update the window every x milliseconds
    timerOpenGL->start();
    qDebug() << "VideoWidget::VideoWidget(): Finished";
}

VideoWidget::~VideoWidget() {
    destroyPipeline();
}

void VideoWidget::resizeGL(int w, int h) {
    Q_UNUSED(h);
    // right hand menu
    if(parent->settings->value("use_full_screen", true).toBool()) {
        imgExitPos = QPoint(w - imgExit.width() - 15, 10);
        imgSavePos = QPoint(w - imgSave.width() - 15, imgExitPos.y() + imgExit.height() + 30);
    } else {
        imgExitPos = QPoint(0,0);
        imgSavePos = QPoint(w - imgSave.width() - 15, 10);
    }
    imgRecentPos = QPoint(w - imgRecent.width() - 15, imgSavePos.y() + imgSave.height() + 30);
    imgSettingsPos = QPoint(w - imgSettings.width() - 15, imgRecentPos.y() + imgRecent.height() + 30);
    // task manager button
    imgTaskManagerPos = QPoint(25, 10);
}

void VideoWidget::newBufferAdded (GstAppSink *_appsink, gpointer user_data)
{
    Q_UNUSED(user_data);
    qDebug() << "New buffer added!";
    if(cold_start)
    {
        //initialize appsink
        qDebug() << "Appsink initialized!";
        appsink=_appsink;
        gst_app_sink_set_drop(appsink,true);
        //turn off emit new_buffer_added
        gst_app_sink_set_emit_signals(appsink,false);
        gst_app_sink_set_max_buffers(appsink,1);
        cold_start=false;
        buffer_ready=true;
        appSinkScanning=true;
    }
}

void VideoWidget::refreshBuffer() {
    if(appsink!=NULL && isPipelineReady()){
        gstBuffer = gst_app_sink_pull_buffer(appsink);
        buffer_ready=true;
    }
}

void VideoWidget::refreshTexture()
{
    if(appSinkScanning && isPipelineReady()) {
        refreshBuffer(); //manually call appsink, not through the callback function

        if(gstBuffer!=NULL)
        {
            //                        QImage img(data, frame_width, frame_height, QImage::Format_Indexed8);
            //            m_uiTexture = bindTexture(img);
            unsigned char* data=(unsigned char *) GST_BUFFER_DATA (gstBuffer);
            glBindTexture( GL_TEXTURE_2D, m_uiTexture );
            glTexImage2D( GL_TEXTURE_2D, 0, GL_RGB, frame_width, frame_height, 0, GL_RGB, GL_UNSIGNED_BYTE, data );
            char* cdata=(char *) GST_BUFFER_DATA (gstBuffer);
            bufferProbeCallback(cdata);
        }
        gst_buffer_unref(gstBuffer);
    }
}

void VideoWidget::initializeGL() {
    program2 = new QGLShaderProgram(this);

    glClearColor(0.1f, 0.1f, 0.2f, 1.0f);
//    glEnable(GL_TEXTURE_2D);
    glGenTextures(1, &m_uiTexture);
    m_uiTexture = bindTexture(QImage()); // needed to init the texture drawing

    QGLShader *vshader1 = new QGLShader(QGLShader::Vertex, this);
    const char *vertexShader =
        "attribute highp vec4 vertex;\n"
                "attribute highp vec4 texCoord;\n"
                "attribute mediump vec3 normal;\n"
                "uniform mediump mat4 matrix;\n"
                "varying highp vec4 texc;\n"
                "varying mediump float angle;\n"
                "void main(void)\n"
                "{\n"
                "    vec3 toLight = normalize(vec3(0.0, 0.3, 1.0));\n"
                "    angle = max(dot(normal, toLight), 0.0);\n"
                "    gl_Position = matrix * vertex;\n"
                "    texc = texCoord;\n"
                "}\n";
    if(!vshader1->compileSourceCode(vertexShader)) {
        qCritical() << "VideoWidget::initializeGL(): Could not compile vertex shader!";
    }

    QGLShader *fshader1 = new QGLShader(QGLShader::Fragment, this);
    const char *fragmentShader =
        "varying highp vec4 texc;\n"
                "uniform sampler2D tex;\n"
                "varying mediump float angle;\n"
                "void main(void)\n"
                "{\n"
                "    highp vec3 color = texture2D(tex, texc.st).rgb;\n"
                "    color = color * 0.2 + color * 0.8 * angle;\n"
                "    gl_FragColor = vec4(clamp(color, 0.0, 1.0), 1.0);\n"
                "}\n";
    if(!fshader1->compileSourceCode(fragmentShader)) {
        qCritical() << "VideoWidget::initializeGL(): Could not compile fragment shader!";
    }

    if(!program2->addShader(vshader1)) {
        qCritical() << "VideoWidget::initializeGL(): Could not add vertex shader!";
    }
    if(!program2->addShader(fshader1)) {
        qCritical() << "VideoWidget::initializeGL(): Could not add fragment shader!";
    }
    if(!program2->link()) {
        qCritical() << "VideoWidget::initializeGL(): Could not link shaders!";
    }

    vertexAttr = program2->attributeLocation("vertex");
    normalAttr = program2->attributeLocation("normal");
    texCoordAttr = program2->attributeLocation("texCoord");
    matrixUniform = program2->uniformLocation("matrix");
    textureUniform = program2->uniformLocation("tex");
    if(vertexAttr == -1) {
        qCritical() << "VideoWidget::initializeGL(): Failed to find vertexAttr";
    }
    if(normalAttr == -1) {
        qCritical() << "VideoWidget::initializeGL(): Failed to find normalAttr";
    }
    if(texCoordAttr == -1) {
        qCritical() << "VideoWidget::initializeGL(): Failed to find texCoordAttr";
    }
    if(matrixUniform == -1) {
        qCritical() << "VideoWidget::initializeGL(): Failed to find matrixUniform";
    }
    if(textureUniform == -1) {
        qCritical() << "VideoWidget::initializeGL(): Failed to find textureUniform";
    }

    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
}

void VideoWidget::paintTexturedCube()
{
    refreshTexture();
    glBindTexture(GL_TEXTURE_2D, m_uiTexture);
    GLfloat afVertices[] = {
        -1.0,-1.0,-1.0, 1.0,-1.0,-1.0, -1.0,1.0,-1.0,
        1.0, 1.0,-1.0, -1.0,1.0,-1.0, 1.0,-1.0,-1.0,
    };
    program2->setAttributeArray(vertexAttr, afVertices, 3);
    GLfloat afTexCoord[] = {
        0.0f,1.0f, 1.0f,1.0f, 0.0f,0.0f,
        1.0f,0.0f, 0.0f,0.0f, 1.0f,1.0f,
    };
    program2->setAttributeArray(texCoordAttr, afTexCoord, 2);
    GLfloat afNormals[] = {
        0,0,1, 0,0,1, 0,0,1,
        0,0,1, 0,0,1, 0,0,1,
    };
    program2->setAttributeArray(normalAttr, afNormals, 3);

    program2->setUniformValue(textureUniform, 0);    // use texture unit 0

    program2->enableAttributeArray(vertexAttr);
    program2->enableAttributeArray(normalAttr);
    program2->enableAttributeArray(texCoordAttr);

    glDrawArrays(GL_TRIANGLES, 0, 6);

    program2->disableAttributeArray(vertexAttr);
    program2->disableAttributeArray(normalAttr);
    program2->disableAttributeArray(texCoordAttr);

//    glUniform1i(textureUniform, 0);    // use texture unit 0
}

void VideoWidget::paintGL() {

    QPainter painter;
    painter.begin(this);

    if(isPipelineReady()) { // if everything is ready, let's draw the video stream

        painter.beginNativePainting();

        glClearColor(0.0f, 0.0f, 0.0f, 1.0f);
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

        glFrontFace(GL_CW);
        glCullFace(GL_FRONT);
        glEnable(GL_CULL_FACE);
        glEnable(GL_DEPTH_TEST);

        QMatrix4x4 modelview2;

        program2->bind();
        program2->setUniformValue(matrixUniform, modelview2);
        paintTexturedCube();
        program2->release();
        glDisable(GL_DEPTH_TEST);
        glDisable(GL_CULL_FACE);
        painter.endNativePainting();

        // Draw a pretty line to pretend we are a barcode scanner
        QPen redPen;
        redPen.setColor(QColor(150,10,10,150));
        redPen.setWidth(3);
        painter.setPen(redPen);
        painter.drawLine(QPoint(0,height() / 2), QPoint(width(), height() / 2));
        // Draw some pretty rectangles to guide our user
        QBrush blackBrush;
        blackBrush.setColor(QColor(10, 10, 10, 80)); // half transparent rectangles
        blackBrush.setStyle(Qt::SolidPattern);
        painter.setPen(Qt::NoPen);
        painter.setBrush(blackBrush);
        // What's going on here? Well, we draw four rectangles to create an opening in the middle.
        // Each rectangle is drawn on a portion of the screen, dividing the screen into a 8x8 grid.
        painter.drawRect(0,0,width() / 8, height());
        painter.drawRect(width() / 8 * 7,0,width() / 8, height());
        painter.drawRect(width() / 8,0, width() / 8 * 6, height() / 8);
        painter.drawRect(width() / 8,height() / 8 * 7, width() / 8 * 6, height() / 8);

        // Give the user some info about what's going on, and place it in the top 8th part
        painter.setPen(Qt::white);
        painter.drawText(QRect(0,0,width(),height() / 8), Qt::AlignCenter, tr("Place a barcode in the square below to scan"));


    } else { // if the pipeline is not ready, let us write a message to our user

        painter.setPen(Qt::white);
        if(parent->isLensCoverOpen()) {
            painter.drawText(QRect(0,0,width(),height()), Qt::AlignCenter, tr("Scanning is paused..."));
        } else {
            painter.drawText(QRect(0,0,width(),height()), Qt::AlignCenter, tr("Lens cover is closed.\nOpen cover to start scanning."));
        }

    }
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_CULL_FACE);

    QString framesPerSecond;
    framesPerSecond.setNum(frames /(time.elapsed() / 1000.0), 'f', 2);

    // draw GUI

    if(parent->settings->value("use_full_screen",true).toBool()) {
        painter.drawImage(imgExitPos, imgExit);
        painter.drawImage(imgTaskManagerPos, imgTaskManager);
    }
    painter.drawImage(imgSavePos, imgSave);
    painter.drawImage(imgRecentPos, imgRecent);
    painter.drawImage(imgSettingsPos, imgSettings);

    // draw info
    QPen halfTransparentPen;
    halfTransparentPen.setColor(QColor(255,255,255,150));
    painter.setPen(halfTransparentPen);
    painter.drawText(20, height() - 20, framesPerSecond + " fps");

    painter.end();

    swapBuffers();

    if (!(frames % 100)) {
        time.start();
        frames = 0;
    }
    frames ++;
}
// These callback functions are static wrappers making it possible to callback member functions of this class
// Remember to set the videoWidget_callback pointer correctly before calling these functions!
void VideoWidget::new_buffer_added_callback (GstAppSink *_appsink, gpointer user_data) {
    return videoWidget_callback->newBufferAdded(_appsink, user_data);
}

//gboolean VideoWidget::bufferProbeCallback_callback (GstElement *image_sink, GstBuffer *buffer, GstPad *pad) {
//    return videoWidget_callback->bufferProbeCallback(image_sink, buffer, pad);
//}
gboolean VideoWidget::busCallback_callback (GstBus * bus, GstMessage * message) {
    return videoWidget_callback->busCallback(bus,message);
}
// End listing callback functions
void VideoWidget::start(bool evenIfNotYetStarted) {
    qDebug() << "VideoWidget::start(): Called with startEvenIfNotYetStarted =" << evenIfNotYetStarted << "and lensCoverOpen =" << parent->isLensCoverOpen();
    if((evenIfNotYetStarted || keepRunning) && parent->isLensCoverOpen()) {
        qDebug() << "VideoWidget::start(): Setting pipeline state";
        //    gst_x_overlay_set_xwindow_id(GST_X_OVERLAY(data), this->winId());
        //    g_object_set(G_OBJECT(input_selector), "active-pad", gst_element_get_pad(video_crop, "src"), NULL);
        gst_element_set_state(pipeline, GST_STATE_PLAYING);
        gst_photography_set_scene_mode (GST_PHOTOGRAPHY (camera_src), GST_PHOTOGRAPHY_SCENE_MODE_CLOSEUP); // uncommented to fix focus although we are using crop
        pipelineReady = true;
        //    refocus();
        if(parent->settings->value("use_autofocus", true).toBool()) {
            refocusTimer->start();
        }
        keepRunning = true;
        timerOpenGL->setInterval(PAINTGL_INTERVAL_HIGH);
    }
}
void VideoWidget::stop(bool stopCompletely) {
    qDebug() << "VideoWidget::stop(): Called with stopCompletely = " << stopCompletely;
    if(pipeline != NULL) {
        qDebug() << "VideoWidget::stop(): Setting pipeline to paused";
        gst_element_set_state(pipeline, GST_STATE_PAUSED);
    }
    pipelineReady = false;
    refocusTimer->stop();
    focusing = false;
    if(stopCompletely) {
        keepRunning = false;
    }
    timerOpenGL->setInterval(PAINTGL_INTERVAL_LOW);

}
bool VideoWidget::isPipelineReady() {
    return pipelineReady;
}

bool VideoWidget::initPipeline() {

    qDebug() << "VideoWidget::initPipeline() called";

    GstElement *screen_sink/*, *image_sink*/;
    GstElement *screen_queue, *image_queue;
    GstElement *csp_filter, /**tee,*/ *image_filter;
    GstCaps *caps;
    GstBus *bus;

    qDebug() << "VideoWidget::initPipeline(): initializing gstreamer";
    /* Initialize Gstreamer */
    gst_init (0, 0);

    /* Create pipeline and attach a callback to it's
     * message bus */
    qDebug() << "VideoWidget::initPipeline(): creating new pipeline";
    pipeline = gst_pipeline_new ("test-camera");
    if(!pipeline)
        qCritical("VideoWidget::initPipeline(): pipeline failed to init");

    qDebug() << "VideoWidget::initPipeline(): getting bus";
    bus = gst_pipeline_get_bus (GST_PIPELINE (pipeline));
    // for the xvimagesink stuff, from here: http://gstreamer.freedesktop.org/data/doc/gstreamer/head/gst-plugins-base-libs/html/gst-plugins-base-libs-gstxoverlay.html#gst-x-overlay-set-xwindow-id
    qDebug() << "VideoWidget::initPipeline(): adding watch";
    gst_bus_add_watch (bus, (GstBusFunc) VideoWidget::busCallback_callback, NULL);
    qDebug() << "VideoWidget::initPipeline(): unrefing bus";
    gst_object_unref (GST_OBJECT (bus));

    /* Camera video stream comes from a Video4Linux driver */
    qDebug() << "VideoWidget::initPipeline(): making factory";
    if (deviceType==DEVICE_N900){
        camera_src = gst_element_factory_make("v4l2camsrc", "camera_src"); // gst_photography_set_autofocus(GST_PHOTOGRAPHY(cameraSrc), TRUE)
        qDebug() << "VideoWidget::initPipeline(): setting object";
        g_object_set(G_OBJECT(camera_src), "driver-name", "omap3cam", NULL); // thanks BBNS_ @ maemo irc aka Yun-Ta Tsai
    } else if (deviceType==DEVICE_N810) {
        camera_src = gst_element_factory_make("gconfv4l2src", "camera_src");
    } else {
        camera_src = gst_element_factory_make("v4l2src", "camera_src");
    }

    if(!camera_src)
        qCritical("VideoWidget::initPipeline(): camera_src failed to init");
    else
        qDebug() << "VideoWidget::initPipeline(): camera_src is now loaded";

    qDebug() << "VideoWidget::initPipeline(): Setting file source";
    file_src = gst_element_factory_make("filesrc", "file_src");
    //useFilesrc("/usr/share/icons/hicolor/14x14/hildon/general_alarm_on.png"); // dummy file to do a quickfix
    qDebug() << "VideoWidget::initPipeline(): Setting input selector";
    input_selector = gst_element_factory_make("input-selector", "input_selector");
    //    g_object_set(G_OBJECT(input_selector), "n-pads", 2, NULL); // set 2 input pads

    /* Colorspace filter is needed to make sure that sinks understands
     * the stream coming from the camera */
    qDebug() << "VideoWidget::initPipeline(): Creating csp_filter";
    csp_filter = gst_element_factory_make ("ffmpegcolorspace", "csp_filter");
    if(!csp_filter)
        qCritical("VideoWidget::initPipeline(): csp_filter failed to init");
    else
        qDebug() << "VideoWidget::initPipeline(): csp_filter is now loaded";
    /* Filter to convert stream to use format that the gdkpixbuf library can use */
    image_filter = gst_element_factory_make("ffmpegcolorspace", "image_filter");

    video_crop = gst_element_factory_make ("videocrop", "videocrop");
    g_object_set (G_OBJECT (video_crop), "bottom", frame_height / 2, NULL);
    g_object_set (G_OBJECT (video_crop), "left", frame_width / 2, NULL);
    g_object_set (G_OBJECT (video_crop), "right", frame_width / 2, NULL);
    g_object_set (G_OBJECT (video_crop), "top", frame_height / 2, NULL);

    /* Queue creates new thread for the stream */
    screen_queue = gst_element_factory_make ("queue", "screen_queue");
    if(!screen_queue)
        qCritical("VideoWidget::initPipeline(): screen_queue failed to init");

    /* Sink that shows the image on screen. Xephyr doesn't support XVideo
     * extension, so it needs to use ximagesink, but the device uses
     * xvimagesink */
    screen_sink = gst_element_factory_make ("appsink", "screen_sink");
    if(!screen_sink)
        qCritical("VideoWidget::initPipeline(): screen_sink failed to init");
    g_object_set (G_OBJECT (screen_sink), "sync", FALSE, NULL);

    /* Creates separate thread for the stream from which the image
     * is captured */
    image_queue = gst_element_factory_make ("queue", "image_queue");
    if(!pipeline)
        qCritical("VideoWidget::initPipeline(): pipeline failed to init");

    /* Check that elements are correctly initialized */
    if (!(pipeline && camera_src && screen_sink && csp_filter && screen_queue && image_queue /*&& image_sink*/ && video_crop && image_filter/*&& file_src && input_selector*/)) {
        qCritical ("VideoWidget::initPipeline(): Couldn't create pipeline elements");
        return FALSE;
    }

    /* Set image sink to emit handoff-signal before throwing away
     * it's buffer */
    //    g_object_set (G_OBJECT (image_sink), "signal-handoffs", TRUE, "sync", FALSE, NULL);

    /* Add elements to the pipeline. This has to be done prior to
     * linking them */
    gst_bin_add_many (GST_BIN (pipeline), camera_src, /*file_src,*/ image_filter, csp_filter, video_crop, /*input_selector,*/ /*tee,*/ screen_queue, screen_sink, image_queue, NULL);

    /* Specify what kind of video is wanted from the camera */
    if (deviceType==DEVICE_N900){
        qDebug() << "VideoWidget::initPipeline(): Using N900 caps";
        caps = gst_caps_new_simple("video/x-raw-yuv",
                        "format",  GST_TYPE_FOURCC, GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y'),
                        "width", G_TYPE_INT, frame_width * 2,
                        "height", G_TYPE_INT, frame_height * 2,
                        "bpp", G_TYPE_INT, 24,
                        "framerate", GST_TYPE_FRACTION, 25, 1,
                        NULL);

        qDebug() << "VideoWidget::initPipeline(): Used N900 caps";

    }else{
        qDebug("Using N8x0 caps");
        caps = gst_caps_new_simple ("video/x-raw-yuv",
                                    "format", GST_TYPE_FOURCC, GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y'),
                                    "width", G_TYPE_INT, 320, "height", G_TYPE_INT, 240, "framerate", GST_TYPE_FRACTION, 8, 1,
                                    NULL); // change to 2 fps
    }
    /* Link the camera source and colorspace filter using capabilities
     * specified */
    qDebug() << "VideoWidget::initPipeline(): Linking filtered";
    if (!gst_element_link_filtered (camera_src, csp_filter, caps)) {
        qCritical() << "VideoWidget::initPipeline(): Failed to link caps!";
        return FALSE;
    }

    qDebug() << "VideoWidget::initPipeline(): Unref caps";
    gst_caps_unref (caps);

    qDebug() << "VideoWidget::initPipeline(): Using rgb caps";
    caps = gst_caps_new_simple("video/x-raw-rgb",
                    "width", G_TYPE_INT, frame_width,
                    "height", G_TYPE_INT, frame_height,
                    "bpp", G_TYPE_INT, 24,
                    NULL);

    qDebug() << "VideoWidget::initPipeline(): Linking again";
    if(!gst_element_link_many(csp_filter, video_crop, screen_queue, image_filter, NULL)) {
        qCritical() << "VideoWidget::initPipeline(): Failed to link many 1";
        return FALSE;
    }
    qDebug() << "VideoWidget::initPipeline(): Linking another filtered";
    if(!gst_element_link_filtered(image_filter, screen_sink, caps)) {
        qCritical() << "VideoWidget::initPipeline(): Failed to link filtered";
        return FALSE;
    }
    qDebug() << "VideoWidget::initPipeline(): Unrefing caps";
    gst_caps_unref(caps);

    qDebug() << "VideoWidget::initPipeline(): Ask for emitted signals";
    g_object_set (G_OBJECT (screen_sink), "emit-signals", TRUE, NULL);
    qDebug() << "VideoWidget::initPipeline(): Creating screen sink";
    g_signal_connect (screen_sink, "new-buffer", G_CALLBACK (new_buffer_added_callback), NULL);

    qDebug() << "Pipeline initialized with success!";
    return true;
}

void VideoWidget::destroyPipeline() {
    /* Free the pipeline. This automatically also unrefs all elements
     * added to the pipeline */
    gst_element_set_state (pipeline, GST_STATE_NULL);
    gst_object_unref (GST_OBJECT (pipeline));
}

// This is a callback function which retrieves any message from the GstBus
gboolean VideoWidget::busCallback (GstBus * bus, GstMessage * message) {
    Q_UNUSED(bus)
    gchar *message_str;
    //const gchar *message_name;
    GError *error;
    /* Report errors to the console */
    if (GST_MESSAGE_TYPE (message) == GST_MESSAGE_ERROR) {
        gst_message_parse_error (message, &error, &message_str);
        qDebug("GST error: %s", message_str);
        g_error ("GST error: %s", message_str);
        g_free (error);
        g_free (message_str);
    }

    /* Report warnings to the console */
    if (GST_MESSAGE_TYPE (message) == GST_MESSAGE_WARNING) {
        gst_message_parse_warning (message, &error, &message_str);
        qDebug("GST warning: %s", message_str);
        g_warning ("GST warning: %s", message_str);
        g_free (error);
        g_free (message_str);
    }
    if(GST_MESSAGE_TYPE (message) == GST_MESSAGE_ELEMENT) { // look for GST_MESSAGE_ELEMENT
        const GstStructure *structure = gst_message_get_structure (message);
        const gchar *message_name = gst_structure_get_name (structure);
        gint status = GST_PHOTOGRAPHY_FOCUS_STATUS_NONE;

        if (g_strcmp0 (message_name, GST_PHOTOGRAPHY_AUTOFOCUS_DONE) == 0) { // autofocus messages
            focusing = false;
            emit focusingValueChanged(false);
            gst_structure_get_int (structure, "status", &status);
            switch (status) {
            case GST_PHOTOGRAPHY_FOCUS_STATUS_FAIL:
                qDebug("VideoWidget::busCallback(): Autofocus failed message received.");
                break;
            case GST_PHOTOGRAPHY_FOCUS_STATUS_SUCCESS:
                qDebug("VideoWidget::busCallback(): Autofocus success message received.");
                break;
            case GST_PHOTOGRAPHY_FOCUS_STATUS_NONE:
                qDebug("VideoWidget::busCallback(): Autofocus none message received.");
                break;
            case GST_PHOTOGRAPHY_FOCUS_STATUS_RUNNING:
                qDebug("VideoWidget::busCallback(): Autofocus running message received.");
                break;
            default:
                qDebug("VideoWidget::busCallback(): Autofocus unknown message recieved.");
                break;
            }
        }
    }

    /* See if the message type is GST_MESSAGE_APPLICATION which means
     * thet the message is sent by the client code (this program) and
     * not by gstreamer. */

    if (GST_MESSAGE_TYPE (message) == GST_MESSAGE_APPLICATION) {
        //char EAN[14], ISBN[11], title[255], author[255];
        //float price;
        //char place[255];
        //float number;
        //char temp[50];
        //unsigned int datetime;
        //        int message_type;
        //        char *message_output;
        //        BarcodeData *bcdata = new BarcodeData;

        qDebug("Got a message from the GStreamer thread");
        //        scanning_status=0;


        /* Get name of the message's structure */
        //message_name = gst_structure_get_name (gst_message_get_structure (message));
        //        bcdata->message_type = g_value_get_int(gst_structure_get_value(gst_message_get_structure (message), "type"));
        //        bcdata->message_output = g_strdup(g_value_get_string(gst_structure_get_value (gst_message_get_structure (message), "output")));

        //gst_structure_get_int(gst_message_get_structure(message), "type", &message_type);
        //message_output = gst_structure_get_string(gst_message_get_structure(message), "output");

        //qDebug("busCallback(): message_name=%s", message_name);


        //        gtk_button_set_label(GTK_BUTTON(scan_button), "Scan!");

        //handle_barcode_data(message_output, message_type);
        //        g_idle_add(handle_barcode_data, bcdata);

        qDebug("Just finished processing barcode");

        // unref/free this message
        // gst_message_unref(message); // perhaps not needed....?

        //        processing=0;
        //#ifndef USE_ZBAR
        //        QR1D_timer=0;
        //#endif
        //        dmtx_timer=0;

        // do we need to free message_output now?
        //g_free (message_name);
        //g_free (message_output);


    }
    return TRUE;
}
/* This callback will be registered to the image sink
 * after user requests a photo */
gboolean VideoWidget::bufferProbeCallback (char *data) {
    //    GstMessage *message;
    //gchar *message_name;
    //    char *data = (char *) GST_BUFFER_DATA (buffer);
    int type = 0;
    //unsigned char output[14];
    char fourcc[5] = "RGB3";
    int bpp = 24;

    //    qDebug("In bufferProbeCallback");
    //
    //    if (processing==1) // make sure we don't get called again while we're already scanning
    //        return TRUE;
    //
    //    // check the frame is large enough here
    //
    //    processing=1; // set the var, to avoid being called again while we're still running the last one

    // write over ourselves
//    src = data;
//    dst = data; //image_buffer

//    do {
//        src++;
//        *(dst++) = (*(src++));
//    } while (--z);

    type = decoder->analyseImage(data, fourcc, bpp); // type is ignored anyway

    /* Returning TRUE means that the buffer can is OK to be
     * sent forward. When using fakesink this doesn't really
     * matter because the data is discarded anyway */
    return TRUE;
}

void VideoWidget::getDeviceType() {
    char d[255];
    FILE *f;
    f = fopen("/proc/component_version", "r");
    fgets(d,255,f);
    fclose(f);

    if (strstr(d, "SU-18")) {
        qDebug("We're a 770");
        deviceType = DEVICE_770;
        frame_height = HEIGHT;
        frame_width = WIDTH;
    } else if (strstr(d, "RX-34")) {
        qDebug( "We're an N800");
        deviceType = DEVICE_N800;
        frame_height = HEIGHT;
        frame_width = WIDTH;
    } else if (strstr(d, "RX-44")) {
        qDebug( "We're an N810");
        deviceType = DEVICE_N810;
        frame_height = HEIGHT;
        frame_width = WIDTH;
    } else if (strstr(d, "RX-48")) {
        qDebug( "We're an N810 WiMax");
        deviceType = DEVICE_N810; // wimax one, thanks Qwerty :)
        frame_height = HEIGHT;
        frame_width = WIDTH;
    } else if (strstr(d, "RX-51")) {
        qDebug( "We're an N900");
        deviceType = DEVICE_N900;
        frame_height = HEIGHT; //N900_HEIGHT;
        frame_width = WIDTH; //N900_WIDTH;
    } else
        qDebug( "Oh no... We're something else.");
}
void VideoWidget::refocus() {
    if(!focusing && pipelineReady) { // we are running and are not focusing.
        emit focusingValueChanged(true);
        gst_element_set_state (pipeline, GST_STATE_PAUSED);

        qDebug("VideoWidget::refocus(): Calling refocus");
        qDebug() << "VideoWidget::refocus(): Interval is" << refocusTimer->interval();
        focusing = true;
        gst_photography_set_autofocus(GST_PHOTOGRAPHY(camera_src), TRUE);
        qDebug("VideoWidget::refocus(): Refocus called");
        //        gst_element_set_state (pipeline, GST_STATE_READY); // causes some problems with flickering, should probably be uncommented
        gst_element_set_state (pipeline, GST_STATE_PLAYING);
        //        refocusTimer->start(); // try to refocus using the pre-set interval
    } else {
        qDebug("VideoWidget::refocus(): Already attempting autofocus or pipeline not ready... Please wait :)");
    }
}
void VideoWidget::setFocusing(bool enabled) {
    focusing = enabled;
    if(!enabled)
        refocusTimer->stop(); // stop the timer to save cycles
}
bool VideoWidget::isFocusing() {
    return focusing;
}

void VideoWidget::useVideosrc() {
    //    g_object_set(G_OBJECT(input_selector), "active-pad", gst_element_get_pad(video_crop, "src"), NULL); // make video_crop the source
}

void VideoWidget::useFilesrc(QString filename) {
    Q_UNUSED(filename)
    //    g_object_set(G_OBJECT(input_selector), "active-pad", gst_element_get_pad(file_src, "src"), NULL); // make filesrc the source
    //    g_object_set(G_OBJECT(file_src), "location", filename.toAscii(), NULL); // set the location
    }

void VideoWidget::setContinuousFocusTime(int milliseconds)
{
    if (refocusTimer->isActive()){
        refocusTimer->stop();
        refocusTimer->setInterval(milliseconds);
        if(parent->settings->value("use_autofocus", true).toBool()) {
            refocusTimer->start(); // try to refocus every one-and-a-half seconds
        }
    }else
        refocusTimer->setInterval(milliseconds);

}

void VideoWidget::mousePressEvent(QMouseEvent *event){
    Q_UNUSED(event)
    //    emit doSaveBuffer();
    qDebug() << "VideoWidget::mousePressEvent(): At " << event->pos();
    QRect imgRect;
    imgRect = QRect(imgExitPos, imgExit.size()); // get the rectangle surrounding the image
    imgRect.adjust(-10,-10,10,10); // make the hit rectangle a bit larger
    if(imgRect.contains(event->pos())) { // see whether the click hit this rectangle
        stop(true);
        parentWidget()->close(); // exit the application
    }
    imgRect = QRect(imgSavePos, imgSave.size()); // get the rectangle surrounding the image
    imgRect.adjust(-10,-10,10,10); // make the hit rectangle a bit larger
    if(imgRect.contains(event->pos())) { // see whether the click hit this rectangle
        emit doSaveBuffer();
    }
    imgRect = QRect(imgTaskManagerPos, imgTaskManager.size()); // get the rectangle surrounding the image
    imgRect.adjust(-10,-10,10,10); // make the hit rectangle a bit larger
    if(imgRect.contains(event->pos())) {
        QDBusConnection connection = QDBusConnection::sessionBus();
        QDBusMessage message = QDBusMessage::createSignal("/","com.nokia.hildon_desktop","exit_app_view");
        connection.send(message);
    }
    imgRect = QRect(imgRecentPos, imgRecent.size()); // get the rectangle surrounding the image
    imgRect.adjust(-10,-10,10,10); // make the hit rectangle a bit larger
    if(imgRect.contains(event->pos())) { // see whether the click hit this rectangle
        parent->showRecentDialog();
    }
    imgRect = QRect(imgSettingsPos, imgSettings.size()); // get the rectangle surrounding the image
    imgRect.adjust(-10,-10,10,10); // make the hit rectangle a bit larger
    if(imgRect.contains(event->pos())) { // see whether the click hit this rectangle
        parent->showSettingsWindow();
    }
}

